/***************************************************************************
 *   Copyright (C) 2006-2009 by Robert Keevil                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; under version 2 of the License.         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include "libscrobble.h"

#ifdef _MSC_VER
#include "xgetopt.h"
#else
#include <getopt.h>
#endif

Scrobble scrob;

void print_entry(scrob_entry tmp)
{
    std::cout<< "[" << tmp.artist << "] ";
    std::cout<< "[" << tmp.album << "] ";
    std::cout<< "[" << tmp.title << "] ";
    std::cout<< "[" << tmp.tracknum << "] ";
    std::cout<< "[" << tmp.length << "] ";
    std::cout<< "[" << tmp.played << "] ";
    std::cout<< "[" << tmp.when << "]" << std::endl;
}

void printusage(void)
{
    string dst;
    if (scrob.get_dst())
        dst = "Yes";
    else
        dst = "No";

    std::cout << "scrobbler-cli (v" << CLIENT_VERSION << "): a Last.fm uploader" << std::endl;
    std::cout << std::endl;
    std::cout << "Mandatory arguments to long options are mandatory for short options too." << std::endl;
    std::cout << std::endl;
    std::cout << "Required Options: (-u and -p OR -u and -5)" << std::endl;
    std::cout << "  -u, --username=USER      Last.fm username" << std::endl;
    std::cout << "  -p, --password=PASS      Last.fm password" << std::endl;
    std::cout << "  -5, --md5=HASH           Last.fm password MD5 hash" << std::endl;
    std::cout << "  -l, --location=PATH      Path to logfile or database" << std::endl;
    std::cout << "                           Required only for ipod database or log file" << std::endl;
    std::cout << std::endl;
    std::cout << "Proxy Options:" << std::endl;
    std::cout << "  -w, --proxy_host=HOST    Proxy hostname or IP" << std::endl;
    std::cout << "  -x, --proxy_port=PORT    Proxy port" << std::endl;
    std::cout << "  -y, --proxy_user=USER    Proxy username (optional)" << std::endl;
    std::cout << "  -z, --proxy_pass=PASS    Proxy password (optional)" << std::endl;
#ifdef WIN32
    std::cout << "  -s, --proxy_winauth      Use domain login for proxy auth (NTLM)" << std::endl;
#endif
    std::cout << std::endl;
    std::cout << "Extras:" << std::endl;
    std::cout << "  -t, --timezone=OFFSET    Timezone offset from UTC" << std::endl;
    std::cout << "  -d, --database           Submit ipod database instead of log file" << std::endl;
    std::cout << "  -c, --playcounts         Remove the ipod database Play Counts after submission" << std::endl;
#ifdef HAVE_MTP
    std::cout << "  -m, --mtp                Submit MTP device" << std::endl;
#endif
    std::cout << "  -n, --now                Re-calculate the play time to now" << std::endl;
    std::cout << "  -r, --recalc=TIME        Re-calculate the time when tracks were played" << std::endl;
    std::cout << "  -v, --verbose=LEVEL      Verbosity level" << std::endl;
    std::cout << std::endl;
    std::cout << "Auto-detected timezone info:" << std::endl;
    std::cout << "Timezone: " << scrob.get_zonename() << std::endl;
    std::cout << "Offset: " << scrob.offset_str() << std::endl;
    std::cout << "Daylight saving: " << dst << std::endl;
    std::cout << std::endl;
}

int main(int argc, char **argv)
{
    // parse command line
    int c;

    std::string user, pass, path, md5;
    std::string proxy_host, proxy_user, proxy_pass;
    int proxy_port = 0;
    bool do_time = false;
    bool do_tzoverride = false;
    int do_db = 0;
#ifdef HAVE_MTP
    int do_mtp = 0;
#endif
    int do_now = 0;
    int do_help = 0;
    int proxy_winauth = 0;
    int del_playcounts = 0;
    int new_time = -1;
    double tz = 0;
    int verbose = 0;
    int timeout = 20;
    int connect_timeout = 10;
    proxy_host = proxy_user = proxy_pass = "";
    user = pass = md5 = path = "";

    while (1)
    {
        static struct option long_options[] =
        {
            {"help",            no_argument,       &do_help, 1},
            {"database",        no_argument,       &do_db,   1},
#ifdef HAVE_MTP
            {"mtp",             no_argument,       &do_mtp,  1},
#endif
            {"now",             no_argument,       &do_now,  1},
            {"proxy_winauth",   no_argument,       &proxy_winauth, 1},
            {"playcounts",      no_argument,       &del_playcounts, 1},
            {"username",        required_argument, 0,        'u'},
            {"password",        required_argument, 0,        'p'},
            {"md5",             required_argument, 0,        '5'},
            {"timezone",        required_argument, 0,        't'},
            {"location",        required_argument, 0,        'l'},
            {"verbose",         required_argument, 0,        'v'},
            {"recalc",          required_argument, 0,        'r'},
            {"proxy_host",      required_argument, 0,        'w'},
            {"proxy_port",      required_argument, 0,        'x'},
            {"proxy_user",      required_argument, 0,        'y'},
            {"proxy_pass",      required_argument, 0,        'z'},
            {"timeout",         required_argument, 0,        'a'},
            {"connect_timeout", required_argument, 0,        'b'},
            {0, 0, 0, 0}
        };

        int option_index = 0;

        c = getopt_long(argc, argv, "5:a:b:dnml:p:r:t:u:v:w:x:y:z:",
                        long_options, &option_index);

        if (c == -1)
            break;

        switch (c)
        {
            case 0: //long option
                break;
            case '5':
                md5 = optarg;
                break;
            case 'a':
                timeout = strtol(optarg, NULL, 10);
                break;
            case 'b':
                connect_timeout = strtol(optarg, NULL, 10);
                break;
            case 'c':
                del_playcounts = 1;
                break;
            case 'd':
                do_db = 1;
                break;
            case 'h':
                do_help = true;
                break;
            case 'l':
                path = optarg;
                break;
#ifdef HAVE_MTP
            case 'm':
                do_mtp = 1;
                break;
#endif
            case 'n':
                do_now = 1;
                break;
            case 'p':
                pass = optarg;
                break;
            case 'r':
                do_time = true;
                new_time = strtol(optarg, NULL, 10);
                break;
            case 's':
                proxy_winauth = 1;
                break;
            case 't':
                do_tzoverride = true;
                tz = atof(optarg);
                break;
            case 'u':
                user = optarg;
                break;
            case 'v':
                verbose = strtol(optarg, NULL, 10);
                break;
            case 'w':
                proxy_host = optarg;
                break;
            case 'x':
                proxy_port = strtol(optarg, NULL, 10);
                break;
            case 'y':
                proxy_user = optarg;
                break;
            case 'z':
                proxy_pass = optarg;
                break;
            case '?':
                break;
            default:
                abort ();
        }
    }

    if (optind < argc)
    {
        std::cout << "Unknown option argument:";
        while (optind < argc)
            std::cout << " " << argv[optind++];
        std::cout << std::endl;
        exit(1);
    }

    if (do_time && do_now)
    {
        std::cout << "You can only use one of -n and -r" << std::endl;
        exit(1);
    }

    if (del_playcounts && !do_db)
    {
        std::cout << "You can only use remove the PlayCounts file if using the database" << std::endl;
        exit(1);
    }

    if (do_help)
    {
        printusage();
        exit(0);
    }

    // check host value exists
    if (user == "" || ((pass == "") && (md5 == "")) ||  (
#ifdef HAVE_MTP
do_mtp != 1 &&
#endif
        path == "" ))
    {
        std::cout << "Error - missing required arguments\n" << std::endl;
        printusage();
        exit(1);
    }

    scrob.set_log_level((LOG_LEVEL)verbose);

    if (!scrob.init(timeout, connect_timeout))
    {
        std::cout << "Init failed: " << scrob.get_error_str() << std::endl;
        return 1;
    }

    if (do_db)
    {
        if (!scrob.parse_db(path, (do_tzoverride)?tz*3600:scrob.get_offset()))
        {
            std::cout << "DB parsing failed: " << scrob.get_error_str() << std::endl;
            return 1;
        }
    }
#ifdef HAVE_MTP
    else if (do_mtp)
    {
        if (!scrob.mtp_connect())
        {
            std::cout << "MTP parsing failed: " << scrob.get_error_str() << std::endl;
            return 1;
        }
        scrob.mtp_get_tracks();
    }
#endif
    else 
    {
        if (!scrob.parse_file(path, (do_tzoverride)?tz*3600:scrob.get_offset()))
        {
            std::cout << "File parsing failed: " << scrob.get_error_str() << std::endl;
            return 1;
        }
    }

    if (do_time)
        scrob.recalc_dt(new_time);

    if (do_now)
        scrob.recalc_now();

    // Remove tracks that are too short, or haven't been listened too enough
    scrob.cleanup_tracks();

    if (verbose)
        std::cout << "Submitting " << scrob.get_num_tracks() << " tracks" << std::endl;

    if (verbose > 1)
    {
        for (size_t i = 0; i < scrob.get_num_tracks(); i++)
            print_entry(scrob.get_track(i));
    }

    if (proxy_host.length())
        scrob.set_proxy(proxy_host, proxy_port, proxy_user, proxy_pass, proxy_winauth);

    if (!scrob.submit(user, (pass=="")?md5:MD5Digest(pass.c_str())))
    {
        std::cout << "Submission failed: " << scrob.get_error_str() << std::endl;
        return 1;
    } else {
        std::cout << "Submission complete" << std::endl;
        switch (scrob.get_last_method())
        {
            case SCROBBLE_LOG:
                remove(path.c_str());
                break;
            case SCROBBLE_IPOD:
                if (del_playcounts)
                {
                    path = path + "/iPod_Control/iTunes/Play Counts";
                    remove(path.c_str());
                }
                break;
#ifdef HAVE_MTP
            case SCROBBLE_MTP:
                scrob.mtp_clear_tracks();
                scrob.mtp_disconnect();
                break;
#endif
            default:
                break;
        }
    }

    return 0;
}
