/**********************************************************************************************
    Copyright (C) 2007 Oliver Eichler oliver.eichler@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

  Garmin and MapSource are registered trademarks or trademarks of Garmin Ltd.
  or one of its subsidiaries.

**********************************************************************************************/

#ifndef CGARMINTRACK_H
#define CGARMINTRACK_H
#include <QObject>
#include <QVector>
#include <QPolygon>
#include <QColor>
#include <QDateTime>
#include <IDevice.h>

class CGarminTrack : public QObject, public Garmin::Track_t
{
    Q_OBJECT
        public:
        CGarminTrack(QObject * parent);
        CGarminTrack(QObject * parent, Garmin::Track_t trk);
        virtual ~CGarminTrack();

        static const QColor colors[];

        enum TrkPtFlag_e
        {
            eSelected  = 1       ///< selected by track info view
            ,eCursor    = 2      ///< selected by cursor
            ,eDeleted   = 4      ///< mark point as deleted
            ,eFocus     = 8      ///< mark current point of user focus

        };

        struct TrkPt_t : Garmin::TrkPt_t
        {
            TrkPt_t();
            TrkPt_t(const Garmin::TrkPt_t& pt, quint32 idx);

            ///index position of track point
            quint32 idx;
            /// secondary data: the speed between this and the previous point
            double speed;
            /// secondary data: the distance between this and the previous point
            double delta;
            /// secondary data: the azimuth to the next point
            double azimuth;
            /// secondary data: the total distance of all visible points up to this point
            double distance;
            /// secondary data: the total ascend of all visible points up to this point
            double ascend;
            /// secondary data: the total descend of all visible points up to this point
            double descend;
            /// display flags
            quint32 flags;
            /// the point on the canvas. Used for fast cursor interaction
            QPoint point;
        };

        /// update all secondary track data from primary track data
        /**
            This will emit sigChanged(), too, and should be called each time
            any data has been changed. If you removed track points via erase()
            you should call this with reindex set to true to restore continous
            item indexing.

            @param reindex set true to re-index all track points
        */
        void update(bool reindex = false);

        /// set identification string
        void setName(const QString& name){ident = name;}
        /// get identification string
        const QString& getName(){return ident;}
        /// create a sibling name
        const QString& getSiblingName();

        /// set Qt color as well as Garmin color
        void setColor(const unsigned i);
        /// get track's Qt color
        const QColor& getColor(){return color;}

        /// get r/w access to canvas polyline
        QPolygon& polyline(){return pline;}

        /// set the highlight flag
        void setHighlight(bool yes){highlight = yes;}
        /// get the value of the highlight flag
        bool isHighlighted(){return highlight;}

        /// get iterator access to track point list
        QVector<TrkPt_t>::iterator begin(){return track.begin();}
        /// get iterator access to track point list
        QVector<TrkPt_t>::iterator end(){return track.end();}
        /// remove a tarck point (don't forget to call update(true))
        QVector<TrkPt_t>::iterator erase(QVector<TrkPt_t>::iterator i){return track.erase(i);}
        /// add track points to track
        void push_back(TrkPt_t& pt);
        /// access track points by index
        TrkPt_t& operator[](int i){return track[i];}

        /// set the point of user focus
        void  setPointOfFocus(qint32 idx);
        /// get the current point of focus
        QVector<TrkPt_t>::iterator getPointOfFocus();

        /// combine two tracks
        CGarminTrack& operator+=(CGarminTrack& t);

        /// return time between first and last visible trackpoint
        quint32 getTotalTime(){return totalTime;}
        ///
        double getTotalDistance(){return totalDistance;}
        ///
        double getAvgSpeed(){return totalTime ? totalDistance / totalTime * 3.6: -1;}
        /// add a track point to the internal trackpoint list
        /**
            This will take care of the point's index.
        */
        void addTrkPt(TrkPt_t& pt);

        QDateTime getStartTimestamp();
        QDateTime getEndTimestamp();

        double getTotalAscend(){return totalAscend;}
        double getTotalDescend(){return totalDescend;}

        signals:
        void sigChanged();

    protected:
        /// set true to draw track highlighted
        bool highlight;
        /// copy of Garmin::track with additional secondary information
        QVector<TrkPt_t> track;
        /// Qt copy of identification string
        QString ident;
        /// all track points as polyline (pixel). Used for fast cursor settings.
        QPolygon pline;
        /// the track color as Qt color
        QColor color;
        /// sibling counter
        quint32 cntSibling;
        /// last sibling name
        QString nameSibling;

        quint32 totalTime;
        double  totalDistance;
        double  totalAscend;
        double  totalDescend;
};
#endif                           //CGARMINTRACK_H
