/***************************************************************************
  qbrewview.cpp
  -------------------
  Client view class for QBrew
  -------------------
  begin         September 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qtabwidget.h>

#include "resource.h"
#include "qbrewview.h"
#include "qbrewdoc.h"
#include "qbrewcalc.h"

#include "fixedspinbox.h"
#include "grainpage.h"
#include "hopspage.h"
#include "miscpage.h"
#include "style.h"

using AppResource::CHAR_NBSP;
using AppResource::CHAR_LATIN_DEGREE;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// QBrewView()
// -----------
// Constructor

QBrewView::QBrewView(QWidget *parent, const char* name, QBrewDoc* doc, QBrewCalc* calc)
    : QWidget(parent, name), document_(doc), calc_(calc)
{
    initWidgets();
    initLayouts();

    mainlayout_->activate();
    // the following allows enough room to display all widgets
    ingredienttab_->setMinimumHeight(148);

    // connect doc with the view
    connect(document_, SIGNAL(documentChanged()), this, SLOT(slotDocumentChanged()));
    connect(document_, SIGNAL(documentChanged()), grainpage_, SLOT(slotDocumentChanged()));
    connect(document_, SIGNAL(documentChanged()), hoppage_, SLOT(slotDocumentChanged()));
    connect(document_, SIGNAL(documentChanged()), miscpage_, SLOT(slotDocumentChanged()));
    connect(document_, SIGNAL(styleChanged()), this, SLOT(slotStyleChanged()));
    // set widgets to a new doc
    slotDocumentChanged();
}

QBrewView::~QBrewView() { ; } // child widgets automatically freed

//////////////////////////////////////////////////////////////////////////////
// initWidget()
// ------------
// Initialize the widgets

void QBrewView::initWidgets()
{
    // title
    titlelabel_ = new QLabel(this, "titlelabel_");
    titlelabel_->setText(tr("Title"));
    titleedit_ = new QLineEdit(this, "titleedit_");
    connect(titleedit_, SIGNAL(textChanged(const QString &)), document_, SLOT(setTitle(const QString &)));
    // style
    stylelabel_ = new QLabel(this, "stylelabel_");
    stylelabel_->setText(tr("Style"));
    stylecombo_ = new QComboBox(false, this, "stylecombo_");
    stylecombo_->insertStringList(calc_->stylesList());
    connect(stylecombo_, SIGNAL(activated(const QString &)), document_, SLOT(setStyle(const QString &)));
    // brewer
    brewerlabel_ = new QLabel(this, "brewerlabel_");
    brewerlabel_->setText(tr("Brewer"));
    breweredit_ = new QLineEdit(this, "breweredit_");
    connect(breweredit_, SIGNAL(textChanged(const QString &)), document_, SLOT(setBrewer(const QString &)));
    // size
    sizelabel_ = new QLabel(this, "sizelabel_");
    sizelabel_->setText(tr("Size"));
    sizespin_ = new FixedSpinBox(0.00, 100.00, 0.25, 2, this, "sizespin_");
    sizespin_->setWrapping(true);
    sizespin_->setSuffix(tr(" Gallons"));
    connect(sizespin_, SIGNAL(valueChanged(double)), document_, SLOT(setSize(double)));
     // mash
    mashcheck_ = new QCheckBox(tr("Mash"), this, "mashcheck_");
    connect(mashcheck_, SIGNAL(toggled(bool)), document_, SLOT(setMash(bool)));

    // style layout widgets
    stylebox_ = new QGroupBox(this, "stylebox_");
    stylebox_->setFrameStyle(QFrame::Box | QFrame::Sunken);
    ogminimum_ = new QLabel(stylebox_, "ogminimum_");
    ogmaximum_ = new QLabel(stylebox_, "ogmaximum_");
    ogrecipe_ = new QLabel(stylebox_, "ogrecipe_");
    ibuminimum_ = new QLabel(stylebox_, "ibuminimum_");
    ibumaximum_ = new QLabel(stylebox_, "ibumaximum_");
    iburecipe_ = new QLabel(stylebox_, "iburecipe_");
    srmminimum_ = new QLabel(stylebox_, "srmminimum_");
    srmmaximum_ = new QLabel(stylebox_, "srmmaximum_");
    srmrecipe_ = new QLabel(stylebox_, "srmrecipe_");
    vertseparator_ = new QFrame(stylebox_, "vertseparator_");
    vertseparator_->setFrameStyle(QFrame::VLine | QFrame::Sunken);
    finalgravity_ = new QLabel(stylebox_, "finalgravity_");
    alcoholbyvol_ =  new QLabel(stylebox_, "alcoholbyvol_");
    alcoholbywgt_ =  new QLabel(stylebox_, "alcoholbywgt_");

    // ingredient widgets
    ingredienttab_ = new QTabWidget(this, "ingredienttab_");
    grainpage_ = new GrainPage(this, "grainpage_", document_, calc_);
    hoppage_ = new HopsPage(this, "hoppage_", document_, calc_);
    miscpage_ = new MiscPage(this, "miscpage_", document_, calc_);
    ingredienttab_->addTab(grainpage_, "&Grains");
    ingredienttab_->addTab(hoppage_, "Ho&ps");
    ingredienttab_->addTab(miscpage_, "&Miscellaneous");
}

//////////////////////////////////////////////////////////////////////////////
// initLayouts()
// -------------
// Initialize the layouts

void QBrewView::initLayouts()
{
    mainlayout_ = new QVBoxLayout(this, 6, 6, "mainlayout_");
    recipelayout_ = new QGridLayout(2, 10, 3, "recipelayout_");
    stylelayout_ = new QGridLayout(stylebox_, 4, 9, 9, 3, "stylelayout_");

    // mainlayout_
    mainlayout_->addLayout(recipelayout_, 0);
    mainlayout_->addWidget(stylebox_, 0);
    mainlayout_->addWidget(ingredienttab_, 1);

    // recipelayout_
    recipelayout_->addColSpacing(1, 3);
    recipelayout_->addColSpacing(3, 6);
    recipelayout_->addColSpacing(5, 3);
    recipelayout_->addColSpacing(7, 3);
    recipelayout_->setColStretch(2, 3);
    recipelayout_->setColStretch(6, 1);
    recipelayout_->setColStretch(8, 1);
    recipelayout_->addWidget(titlelabel_, 0, 0, Qt::AlignRight);
    recipelayout_->addWidget(titleedit_, 0, 2);
    recipelayout_->addWidget(stylelabel_, 0, 4, Qt::AlignRight);
    recipelayout_->addMultiCellWidget(stylecombo_, 0, 0, 6, 8); //span 4 cols
    recipelayout_->addWidget(brewerlabel_, 1, 0, Qt::AlignRight);
    recipelayout_->addWidget(breweredit_, 1, 2);
    recipelayout_->addWidget(sizelabel_, 1, 4, Qt::AlignRight);
    recipelayout_->addWidget(sizespin_, 1, 6);
    recipelayout_->addWidget(mashcheck_, 1, 8, Qt::AlignRight);

    // stylelayout_
    stylelayout_->addRowSpacing(0, 9);
    stylelayout_->addColSpacing(1, 3);
    stylelayout_->addColSpacing(3, 3);
    stylelayout_->addColSpacing(5, 3);
    stylelayout_->addColSpacing(7, 3);
    stylelayout_->setColStretch(0, 1);
    stylelayout_->setColStretch(2, 1);
    stylelayout_->setColStretch(4, 1);
    stylelayout_->setColStretch(8, 1);
    stylelayout_->addWidget(ogminimum_, 1, 0);
    stylelayout_->addWidget(ibuminimum_, 1, 2);
    stylelayout_->addWidget(srmminimum_, 1, 4);
    stylelayout_->addWidget(ogmaximum_, 2, 0);
    stylelayout_->addWidget(ibumaximum_, 2, 2);
    stylelayout_->addWidget(srmmaximum_, 2, 4);
    stylelayout_->addWidget(ogrecipe_, 3, 0);
    stylelayout_->addWidget(iburecipe_, 3, 2);
    stylelayout_->addWidget(srmrecipe_, 3, 4);
    stylelayout_->addMultiCellWidget(vertseparator_, 1, 3, 6, 6);
    stylelayout_->addWidget(finalgravity_, 1, 8);
    stylelayout_->addWidget(alcoholbyvol_, 2, 8);
    stylelayout_->addWidget(alcoholbywgt_, 3, 8);
}

//////////////////////////////////////////////////////////////////////////////
// Slot Implementations                                                     //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotDocumentChanged()
// ---------------------
// Received when the document is modified

void QBrewView::slotDocumentChanged()
{
    // TODO: need some way to shortcircuit this, so not every widget is updated
    // update recipe widgets
    titleedit_->setText(document_->title());
    breweredit_->setText(document_->brewer());
    stylecombo_->clear();
    stylecombo_->insertStringList(calc_->stylesList());
    setComboItem(stylecombo_, document_->style());
    sizespin_->setValue(document_->size());
    mashcheck_->setChecked(document_->mash());
    // update style widgets
    slotStyleChanged();
}

//////////////////////////////////////////////////////////////////////////////
// slotStyleChanged()
// ------------------
// Received when the style or recipe changes

void QBrewView::slotStyleChanged()
{
    Style* style = calc_->style(document_->style());

    // beer style has changed, so update labels in stylelayout_
    stylebox_->setTitle(stylecombo_->currentText() + tr(" Characteristics"));
    ogminimum_->setText(tr("Minimum Gravity: ") + QString::number(style->OGLow(), 'f', 3));
    ogmaximum_->setText(tr("Maximum Gravity: ") + QString::number(style->OGHi(), 'f', 3));
    ogrecipe_->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Gravity:") + CHAR_NBSP + QString::number(calc_->OG(), 'f', 3) + "</b>");
    ibuminimum_->setText(tr("Minimum Bitterness: ") + QString::number(style->IBULow()));
    ibumaximum_->setText(tr("Maximum Bitterness: ") + QString::number(style->IBUHi()));
    iburecipe_->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Bitterness:") + CHAR_NBSP + QString::number(calc_->IBU(), 'f', 0) + "</b>");
    srmminimum_->setText(tr("Minimum Color: ") + QString::number(style->SRMLow()) + CHAR_LATIN_DEGREE);
    srmmaximum_->setText(tr("Maximum Color: ") + QString::number(style->SRMHi()) + CHAR_LATIN_DEGREE);
    srmrecipe_->setText("<b>" + tr("Recipe") + CHAR_NBSP + tr("Color:") + CHAR_NBSP + QString::number(calc_->SRM(), 'f', 0) + CHAR_LATIN_DEGREE + "</b>");
    finalgravity_->setText(tr("Estimated FG: ") + QString::number(calc_->FGEstimate(), 'f', 3));
    alcoholbyvol_->setText("<b>" + tr("Alcohol") + CHAR_NBSP + tr("by") + CHAR_NBSP + tr("Volume:") + CHAR_NBSP + QString::number(calc_->ABV() * 100.0, 'f', 1) + "%</b>");
    alcoholbywgt_->setText("<b>" + tr("Alcohol") + CHAR_NBSP + tr("by") + CHAR_NBSP + tr("Weight:") + CHAR_NBSP + QString::number(calc_->ABW() * 100.0, 'f', 1) + "%</b>");
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                             //
 ///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// setComboItem()
// --------------
// Set a given combo box to the given entry.

void QBrewView::setComboItem(QComboBox *combo, const QString &txt)
{
    // search combo box for style string
    for (int item = 0; item < combo->count(); ++item) {
        if (combo->text(item) == txt) {
            combo->setCurrentItem(item);
            return;
        }
    }
    // if item not found, will set it to the first one
    combo->setCurrentItem(0);
}
