/***************************************************************************
  qbrewdoc.cpp
  -------------------
  Recipe document class for QBrew
  -------------------
  begin         September 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qdom.h>
#include <qtextstream.h>

#include "qbrew.h"
#include "qbrewdoc.h"
#include "calcresource.h"
#include "fileresource.h"
#include "oldstore.h"

using namespace CalcResource;
using namespace FileResource;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// QBrewDoc()
// ----------
// Constructor

QBrewDoc::QBrewDoc(QWidget *parent)
    : groupmap_(), fieldmap_(), parent_(parent), modified_(false), new_(true),
    title_(""), brewer_(""), style_(""), size_(0), mash_(false),
    grainlist_(), hoplist_(), misclist_()

{
    // set up group map
    groupmap_[groupRecipe] = gidRecipe;
    groupmap_[groupDocGrains] = gidDocGrains;
    groupmap_[groupDocHops] = gidDocHops;
    groupmap_[groupDocMisc] = gidDocMisc;

    // set up recipe attributes
    fieldmap_[fieldTitle] = fidTitle;
    fieldmap_[fieldBrewer] = fidBrewer;
    fieldmap_[fieldStyle] = fidStyle;
    fieldmap_[fieldSize] = fidSize;
    fieldmap_[fieldMash] = fidMash;

    if (parent) size_ = ((QBrew *)parent_)->getBatchSetting();
}

QBrewDoc::~QBrewDoc() {}

//////////////////////////////////////////////////////////////////////////////
// Recipe Access                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setTitle()
// ----------
// Set the recipe title

void QBrewDoc::setTitle(const QString &title)
{
    title_ = title;
    emit documentModified();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// setBrewer()
// -----------
// Set the recipe's brewer

void QBrewDoc::setBrewer(const QString &brewer)
{
    brewer_ = brewer;
    emit documentModified();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// setStyle()
// ----------
// Set the recipe's style

void QBrewDoc::setStyle(const QString &style)
{
    style_ = style;
    emit documentModified();
    emit styleChanged();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// setSize()
// ---------
// Set the recipe's size

void QBrewDoc::setSize(double size)
{
    size_ = size;
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// setMash()
// ---------
// Set whether the recipe is mashed

void QBrewDoc::setMash(bool mash)
{
    mash_ = mash;
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// AddGrain()
// ----------
// Add a grain ingredient to the recipe

GrainIterator QBrewDoc::addGrain(const QString &name, const double &quantity,
    const double &extract, const double &color, const QString &use)
{
    GrainList::Iterator it;
    Grain newgrain = Grain(name, quantity, extract, color, Grain::useStringToEnum(use));
    it = grainlist_.append(newgrain);
    emit documentModified();
    emit recalc();
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// ApplyGrain()
// ------------
// Apply (change) a grain ingredient to the recipe

void QBrewDoc::applyGrain(GrainIterator it, const QString &name, const double &quantity,
    const double &extract, const double &color, const QString &use)
{
    Grain newgrain(name, quantity, extract, color, Grain::useStringToEnum(use));
    (*it) = newgrain;
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// RemoveGrain()
// -------------
// Remove a grain ingredient from the recipe

void QBrewDoc::removeGrain(GrainIterator it)
{
    grainlist_.remove(it);
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// addHop()
// --------
// Add a hop ingredient to the recipe

HopIterator QBrewDoc::addHop(const QString &name, const double &quantity, const QString &form,
    const double &alpha, const unsigned &time)
{
    HopsList::Iterator it;
    Hops newhop = Hops(name, quantity, form, alpha, time);
    it = hoplist_.append(newhop);
    emit documentModified();
    emit recalc();
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// applyHop()
// ----------
// Apply (change) a hop ingredient to the recipe

void QBrewDoc::applyHop(HopIterator it, const QString &name, const double &quantity,
    const QString &form, const double &alpha, const unsigned &time)
{
    Hops newhop = Hops(name, quantity, form, alpha, time);
    (*it) = newhop;
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// removeHop()
// ----------
// Remove a hop ingredient from the recipe

void QBrewDoc::removeHop(HopIterator it)
{
    hoplist_.remove(it);
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// addMiscIngredient()
// -------------------
// Add a misc ingredient to the recipe

MiscIngredientIterator QBrewDoc::addMiscIngredient(const QString &name, const double &quantity,
    const QString &notes)
{
    MiscIngredientList::Iterator it;
    MiscIngredient newmisc = MiscIngredient(name, quantity, notes);
    it = misclist_.append(newmisc);
    emit documentModified();
    emit recalc();
    modified_ = true;
    return it;
}

//////////////////////////////////////////////////////////////////////////////
// applyMiscIngredient()
// -------------------
// Apply (change) a misc ingredient to the recipe

void QBrewDoc::applyMiscIngredient(MiscIngredientIterator it, const QString &name,
    const double &quantity, const QString &notes)
{
    MiscIngredient newmisc = MiscIngredient(name, quantity, notes);
    (*it) = newmisc;
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// removeMiscIngredient()
// -------------------
// Remove a misc ingredient from the recipe

void QBrewDoc::removeMiscIngredient(MiscIngredientIterator it)
{
    misclist_.remove(it);
    emit documentModified();
    emit recalc();
    modified_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// File Oriented Methods                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// newDoc()
// --------
// Creates a new recipe (clears current recipe)

void QBrewDoc::newDoc(const QString &style)
{
    // reset appropriate elements
    title_ = "";
    brewer_ = "";
    style_ = style;
    size_ = ((QBrew *)parent_)->getBatchSetting();
    if (!grainlist_.isEmpty()) grainlist_.clear();
    if (!hoplist_.isEmpty()) hoplist_.clear();
    if (!misclist_.isEmpty()) misclist_.clear();

    emit documentChanged();
    modified_ = false;
    new_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// load()
// ------
// load the recipe from file

bool QBrewDoc::load(const QString &filename)
{
    // TODO: need more error checking on tags and elements

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        qWarning(tr("Error: Cannot open ") + filename);
        datafile->close();
        delete (datafile);
        return false;
    }

    // open dom document
    QDomDocument doc;
    doc.setContent(datafile);
    datafile->close();
    delete (datafile);

    // check the doc type and stuff
    if (doc.doctype().name() != tagRecipe) {
        // wrong file type
        qWarning(tr("Error: Wrong file type ") + filename);
        return false;
    }
    QDomElement root = doc.documentElement();

    // check generator
    if (root.attribute(attrGenerator) != PACKAGE) {
        // right file type, wrong generator
        qWarning("Not a recipe file for " + QString(PACKAGE));
        return false;
    }

    // check file version
    if (root.attribute(attrVersion) < QBREW_PREVIOUS) {
        // too old of a version
        qWarning(tr("Error: Unsupported version ") + filename);
        return false;
    }

    // clear lists
    grainlist_.clear();
    hoplist_.clear();
    misclist_.clear();

    // Note: for some of these tags, only one in a document makes sense. But if there is
    // more than one, process them in order, with later ones overwriting the earlier

    // get title
    QDomNodeList nodes = root.elementsByTagName(tagTitle);
    QDomElement element;
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            title_ = element.text();
        }
    }
    // get brewer
    nodes = root.elementsByTagName(tagBrewer);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            brewer_ = element.text();
        }
    }
    // get style
    nodes = root.elementsByTagName(tagStyle);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            style_ = element.text();
        }
    }
    // get batch settings
    nodes = root.elementsByTagName(tagBatch);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            size_ = element.attribute(attrSize).toDouble();
            mash_ = (element.attribute(attrSize).lower() == "true") ? true : false;
        }
    }

    // get all grains tags
    nodes = root.elementsByTagName(tagGrains);
    QDomNodeList subnodes;
    QDomElement subelement;
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all grain tags
            subnodes = element.elementsByTagName(tagGrain);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    grainlist_.append(Grain(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrExtract).toDouble(),
                        subelement.attribute(attrColor).toDouble(),
                        subelement.attribute(attrUse)));
                }
            }
        }
    }

    // get all hops tags
    nodes = root.elementsByTagName(tagHops);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all hop tags
            subnodes = element.elementsByTagName(tagHop);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    hoplist_.append(Hops(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrForm),
                        subelement.attribute(attrAlpha).toDouble(),
                        subelement.attribute(attrTime).toUInt()));
                }
            }
        }
    }

    // get all miscingredients tags
    nodes = root.elementsByTagName(tagMiscIngredients);
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            element = nodes.item(n).toElement();
            // get all miscingredient tags
            subnodes = element.elementsByTagName(tagMiscIngredient);
            for (unsigned m=0; m<subnodes.count(); m++) {
                if (subnodes.item(m).isElement()) {
                    subelement = subnodes.item(m).toElement();
                    misclist_.append(MiscIngredient(subelement.text(),
                        subelement.attribute(attrQuantity).toDouble(),
                        subelement.attribute(attrNotes)));
                }
            }
        }
    }
    // we have now opened a document
    emit documentChanged();
    emit recalc();
    modified_ = false;
    new_ = false;
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// save()
// ------
// Save the recipe out to file

bool QBrewDoc::save(const QString &filename)
{
    QDomDocument doc(tagRecipe);

    // create the root element
    QDomElement root = doc.createElement(doc.doctype().name());
    root.setAttribute(attrGenerator, PACKAGE);
    root.setAttribute(attrVersion, VERSION);
    doc.appendChild(root);

    // title
    QDomElement element = doc.createElement(tagTitle);
    element.appendChild(doc.createTextNode(title_));
    root.appendChild(element);
    // brewer
    element = doc.createElement(tagBrewer);
    element.appendChild(doc.createTextNode(brewer_));
    root.appendChild(element);
    // style
    element = doc.createElement(tagStyle);
    element.appendChild(doc.createTextNode(style_));
    root.appendChild(element);
    // batch settings
    element = doc.createElement(tagBatch);
    element.setAttribute(attrSize, size_);
    element.setAttribute(attrMash, size_ ? "true" : "false");
    root.appendChild(element);

    // grains elements
    element = doc.createElement(tagGrains);
    GrainIterator git;
    QDomElement subelement;
    // iterate through _grains list
    for (git=grainlist_.begin(); git!=grainlist_.end(); ++git) {
        subelement = doc.createElement(tagGrain);
        subelement.appendChild(doc.createTextNode((*git).name()));
        subelement.setAttribute(attrQuantity, (*git).quantity());
        subelement.setAttribute(attrExtract, (*git).extract());
        subelement.setAttribute(attrColor, (*git).color());
        subelement.setAttribute(attrUse, (*git).useString());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // hops elements
    element = doc.createElement(tagHops);
    HopIterator hit;
    // iterate through _hops list
    for (hit=hoplist_.begin(); hit!=hoplist_.end(); ++hit) {
        subelement = doc.createElement(tagHop);
        subelement.appendChild(doc.createTextNode((*hit).name()));
        subelement.setAttribute(attrQuantity, (*hit).quantity());
        subelement.setAttribute(attrForm, (*hit).form());
        subelement.setAttribute(attrAlpha, (*hit).alpha());
        subelement.setAttribute(attrTime, (*hit).time());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // miscingredients elements
    element = doc.createElement(tagMiscIngredients);
    MiscIngredientIterator mit;
    // iterate through _misc list
    for (mit=misclist_.begin(); mit!=misclist_.end(); ++mit) {
        subelement = doc.createElement(tagMiscIngredient);
        subelement.appendChild(doc.createTextNode((*mit).name()));
        subelement.setAttribute(attrQuantity, (*mit).quantity());
        subelement.setAttribute(attrNotes, (*mit).notes());
        element.appendChild(subelement);
    }
    root.appendChild(element);

    // open file
    QFile* datafile = new QFile(filename);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning(tr("Error: Cannot open file ") + filename);
        datafile->close();
        return false;
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 0);
    datafile->close();
    delete (datafile);
    // document is saved, so set flags accordingly
    modified_ = false;
    new_ = false;
    return true;
}


//////////////////////////////////////////////////////////////////////////////
// convert()
// ---------
// Convert and load from old format

bool QBrewDoc::convert(const QString &filename)
{
    // open file for reading
    OldStore* _store = new OldStore(IO_ReadOnly, filename, QBREW_OLDFORMAT, QBREW_OLDVERSION);
    if (!_store->good()) {
        qWarning("Error: Cannot open " + filename);
        delete _store;
        return false;
    } else {
        // file opened without error
        if (_store->getVersion() < QBREW_OLDPREVIOUS) {
            qWarning("Error: Unsupported version " + _store->getVersion());
            delete _store;
            return false;
        }
        // clear lists
        grainlist_.clear();
        hoplist_.clear();
        misclist_.clear();
        // read in file line by line
        int ID;
        Grain newgrain;
        Hops newhop;
        MiscIngredient newmisc;
        while (_store->getLine()) {
            ID = groupmap_[_store->getGroup()];
            if (_store->getGroup() != _store->getName()) {
                switch (ID) {
                    case gidRecipe:
                        ID = fieldmap_[_store->getName()];
                        switch (ID) {
                            case fidTitle:
                                title_ = _store->getValue().copy();
                                break;
                            case fidBrewer:
                                brewer_ = _store->getValue().copy();
                                break;
                            case fidStyle:
                                style_ = _store->getValue().copy();
                                break;
                            case fidSize:
                                size_ = _store->getValue().toDouble() / 100.0;
                                break;
                            case fidMash:
                                mash_ = (_store->getValue().lower() == "true");
                                break;
                            default:
                                qWarning("Warning: " + _store->getName() + " is not a valid field");
                                break;
                        }
                        break;
                    case gidDocGrains:
                        newgrain.obsoleteSerializeIn(_store->getName(), _store->getValue());
                        grainlist_.append(newgrain);
                        break;
                    case gidDocHops:
                        newhop.obsoleteSerializeIn(_store->getName(), _store->getValue());
                        hoplist_.append(newhop);
                        break;
                    case gidDocMisc:
                        newmisc.obsoleteSerializeIn(_store->getName(), _store->getValue());
                        misclist_.append(newmisc);
                        break;
                    default:
                        qWarning("Warning: " + _store->getGroup() + " is not a valid group");
                        break;
                }
            }
        }
    }
    delete _store;

    // we have now opened a document
    emit documentChanged();
    emit recalc();
    return save(filename);
}
