/***************************************************************************
  qbrewcalc.h
  -------------------
  Handles calculations for QBrew
  -------------------
  begin         September 26th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef QBREWCALC_H
#define QBREWCALC_H

#include <qstringlist.h>

#include "grain.h"
#include "hops.h"
#include "miscingredient.h"
#include "style.h"

class QBrewDoc;
class Preferences;

// utilization table entry structure
struct UEntry {
    unsigned time;
    unsigned utilization;
};

class QBrewCalc : public QObject {
    Q_OBJECT
public:
    // constructor
    QBrewCalc(Preferences *preferences=0, QBrewDoc *doc=0);
    // destructor
    ~QBrewCalc();
    
    // return a list of styles
    QStringList stylesList();
    // return given style from name
    Style* style(const QString &name);
    // return a list of grains
    QStringList grainsList();
    // return given grain from name
    Grain* grain(const QString &name);
    // return a list of hops
    QStringList hopsList();
    // return a list of hop styles (this should be in Hops class)
    QStringList formsList();
    // return given hop from name
    Hops* hop(const QString &name);
    // return a list of misc ingredients
    QStringList miscList();
    // return given misc ingredient from name
    MiscIngredient* misc(const QString &name);
    
    // return original gravity
    double OG();
    // return estimated final gravity
    double FGEstimate();
    // return estimated final gravity
    double ABV();
    // return alcohol %v
    double ABW();
    // return bitterness
    double IBU();
    // return alcohol %w
    double SRM();

    // convert filename to the new database format
    bool convert(const QString& filename);

public slots:
    // received if stuff needs recalculating
    void slotRecalc();
    
signals:
    // send when recalculation done
    void calcDone();
    
private:    
    // load calc data
    bool loadData(const QString &filename, bool quiet=false);
    // save the information to the calc data file
    void saveData(const QString &filename);

    // calculate original gravity
    double calcOG();
    // calculate color
    double calcSRM();
    // calculate bitterness
    double calcIBU();
    // calculate bitterness from the rager method
    double calcRagerIBU();
    // calculate bitterness from the tinseth method
    double calcTinsethIBU();
    // look up hop utilization for time
    double utilization(const unsigned &time);
    
private:
    Preferences* preferences_;
    QBrewDoc* document_;
    
    double og_;
    double ibu_;
    double srm_;
    
    QMap<QString, int> groupmap_;
    QValueList<UEntry> utable_;
    QMap<QString, Style> styles_;
    GrainList grains_;
    HopsList hops_;
    MiscIngredientList misc_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline double QBrewCalc::OG() { return og_; }

inline double QBrewCalc::IBU() { return ibu_; }

inline double QBrewCalc::SRM() { return srm_; }

#endif // QBREWCALC_H
