/***************************************************************************
  miscpage.cpp
  -------------------
  A dialog page for misc ingredients
  -------------------
  begin         October 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "miscpage.h"

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include "fixedspinbox.h"

#include "qbrewview.h"
#include "qbrewcalc.h"
#include "qbrewdoc.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// MiscPage()
// ----------
// Constructor

MiscPage::MiscPage(QWidget *parent, const char *name, QObject *doc, QObject *calc)
    : IngredientPage(parent, name, doc, calc), noteslabel_(0), notesedit_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn(tr("Notes"));
    itemslistview_->setColumnAlignment(2, AlignLeft);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);

    // setup widgets
    quantityspin_->setSuffix(tr(" units"));
    noteslabel_ = new QLabel(tr("Notes"), this, "noteslabel_");
    notesedit_ = new QLineEdit(this, "notesedit_");

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(noteslabel_, 0);
    rowtwolayout_->addWidget(notesedit_, 1);

    delete spacer_; // don't need it
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, notesedit_);

    // setup connections
    connect(itemcombo_,SIGNAL(activated(const QString &)), this, SLOT(slotUpdateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(slotAddMiscIngredient()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(slotApplyMiscIngredient()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(slotRemoveMiscIngredient()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)), SLOT(slotItemSelected(QListViewItem *)));
    // initialize widgets with values
    slotDocumentChanged();
}

MiscPage::~MiscPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotUpdateEntryItems()
// ----------------------
// Received to update the misc ingredient entry widgets

void MiscPage::slotUpdateEntryItems(const QString &name)
{
    // highlight the appropriate entry in the listview, if available
    QListViewItem *item = itemslistview_->firstChild();
    while (item != 0) {
        if (item->text(0) == name) {
            itemslistview_->setSelected(item, true);
            // this sends the signal selectionChanged()
            return;
        }
        item = item->nextSibling();
    }
    // wasn't in the listview, update from database
    MiscIngredient *thisMiscIngredient = ((QBrewCalc *)calc_)->misc(name);
    quantityspin_->setValue(thisMiscIngredient->quantity());
    notesedit_->setText(thisMiscIngredient->notes());
}

//////////////////////////////////////////////////////////////////////////////
// slotDocumentChanged()
// ---------------------
// Received when document has changed

void MiscPage::slotDocumentChanged()
{
    itemcombo_->clear();
    itemcombo_->insertStringList(((QBrewCalc *)calc_)->miscList());
    slotUpdateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// slotAddMiscIngredient()
// -----------------------
// Received when user adds a misc ingredient to recipe

void MiscPage::slotAddMiscIngredient()
{
    MiscIngredientIterator it;
    MiscIngredientListItem *item;
    it = ((QBrewDoc *)document_)->addMiscIngredient(itemcombo_->currentText(), quantityspin_->value(),
        notesedit_->text());
    item = new MiscIngredientListItem( itemslistview_, (*it).name(),
        (*it).quantityString(),
        (*it).notes() );
    item->setMiscIngredientIterator(it);
}

//////////////////////////////////////////////////////////////////////////////
// slotApplyMiscIngredient()
// -------------------------
// Received when user applies a misc ingredient to recipe

void MiscPage::slotApplyMiscIngredient()
{
     if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->applyMiscIngredient(currentitem_->miscIngredientIterator(),
                itemcombo_->currentText(), quantityspin_->value(), notesedit_->text());
        currentitem_->setToIterator();
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotRemoveMiscIngredient()
// --------------------------
// Received when user removes a misc ingredient from recipe

void MiscPage::slotRemoveMiscIngredient()
{
    if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->removeMiscIngredient(currentitem_->miscIngredientIterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotItemSelected()
// ------------------
// Received when user selects an item in the list view

void MiscPage::slotItemSelected(QListViewItem *item)
{
    currentitem_ = (MiscIngredientListItem *)item;
    if (item != 0) {
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1).toDouble());
        notesedit_->setText(item->text(2));
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the listview

void MiscPage::updateListView()
{
    MiscIngredientList *list = ((QBrewDoc *)document_)->miscIngredientList();
    MiscIngredientIterator it;
    MiscIngredientListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new MiscIngredientListItem( itemslistview_, (*it).name(),
            (*it).quantityString(),
            (*it).notes() );
        item->setMiscIngredientIterator(it);
    }
}
