/***************************************************************************
  miscingredient.h
  -------------------
  A miscellaneous ingredient class
  -------------------
  begin         October 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef MISCINGREDIENT_H
#define MISCINGREDIENT_H

#include <qmap.h>
#include <qlistview.h>
#include <qstring.h>

class MiscIngredient {
public:
    // full constructor
    MiscIngredient(const QString &name="misc", const double &quantity=1.0, const QString &notes="");
    // copy constructor
    MiscIngredient(const MiscIngredient &m);
    // assignment operator
    MiscIngredient operator=(const MiscIngredient &m);
    // comparison operator
    bool operator==(const MiscIngredient &m);
    // destructor
    ~MiscIngredient();

    // return name
    QString name();
    // return quantity (in ounces)
    double quantity();
    // return MiscIngredient notes
    QString notes();
        
    // return quantity as string
    QString quantityString();
    
    // return object as string, for serialization (deprecated)
    QString obsoleteSerializeOut();
    // set object to string, for serialization (deprecated)
    void obsoleteSerializeIn(const QString &name, const QString &value);

private:
    QString name_;
    double quantity_; // measured in pounds
    QString notes_;
};

typedef QValueList<MiscIngredient> MiscIngredientList;
typedef QValueList<MiscIngredient>::Iterator MiscIngredientIterator;

class MiscIngredientListItem : public QListViewItem {
public:
    // constructor
    MiscIngredientListItem(QListView * parent, QString, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null );
    // destructor
    ~MiscIngredientListItem();
    // get iterator to misc item
    MiscIngredientIterator miscIngredientIterator();
    // set iterator to misc item
    void setMiscIngredientIterator(MiscIngredientIterator &it);
    // set misc to match iterator
    void setToIterator();
private:
    MiscIngredientIterator mit_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline QString MiscIngredient::name() { return name_; }

inline double MiscIngredient::quantity() { return quantity_; }

inline QString MiscIngredient::notes() { return notes_; }

inline QString MiscIngredient::quantityString() { return QString::number(quantity_, 'f', 2); }

inline MiscIngredientIterator MiscIngredientListItem::miscIngredientIterator() { return mit_; }

inline void MiscIngredientListItem::setMiscIngredientIterator(MiscIngredientIterator &it) { mit_ = it; }

#endif // MISCINGREDIENT_H
