/***************************************************************************
  main.cpp
  -------------------
  A brewing recipe calculator for Unix/X11
  -------------------
  begin         September 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#include <cstdlib>
#include <iostream>

#include <qapplication.h>
#include <qdir.h>
#include <qstring.h>

#include "resource.h"
#include "calcresource.h"
#include "qbrewcalc.h"
#include "qbrewdoc.h"
#include "preferences.h"
#include "oldsettings.h"
#include "qbrew.h"

using namespace std;
using namespace AppResource;
using namespace CalcResource;

//////////////////////////////////////////////////////////////////////////////
// doversion()
// -----------
// Print out copyright and version info to stdout

void doversion()
{
    cout << AppResource::ID_TITLE << " " << VERSION << "\n";
    cout << QObject::tr("A Brewing Recipe Calculator\n");
    cout << QObject::tr("Copyright 1999-2001 by David Johnson (david@usermode.org)\n");
}

//////////////////////////////////////////////////////////////////////////////
// dohelp()
// --------
// Print out help info to stdout

void dohelp()
{
    // print out version of program while we're at it
    cout << AppResource::ID_TITLE << " " << VERSION << '\n';

    // print out list of command line options
    cout << QObject::tr("\nGeneral Options\n\n");
    cout << QObject::tr("file\n");
    cout << QObject::tr("\tOpen the specified file in QBrew\n");
    cout << QObject::tr("-help\n");
    cout << QObject::tr("\tPrint the command line options.\n");
    cout << QObject::tr("-version\n");
    cout << QObject::tr("\tPrint the version number of QBrew.\n");
    cout << QObject::tr("\nConversion Options\n\n");
    cout << QObject::tr("-convert file\n");
    cout << QObject::tr("\tConverts the file from an older qbrew format to the current format.\n");
    cout << QObject::tr("\tThis will convert recipe files, the .qbrewrc file, or the calcdb file\n");
    cout << QObject::tr("\tto the new qbrewdata file.\n");

    cout << QObject::tr("\nMiscellaneous Options\n\n");
    cout << QObject::tr("\tThese are options for the Qt library and the X Windows System.\n");
    cout << QObject::tr("-style=style\n");
    cout << QObject::tr("\tSets the application GUI style. Possible values are cde, motif,\n");
    cout << QObject::tr("\tmotifplus, platinum, sgi and windows. Note that this option has no\n");
    cout << QObject::tr("\teffect if a style other than 'default' was selected in the options menu\n");
    cout << QObject::tr("-display display\n");
    cout << QObject::tr("\tSets the X display (default is $DISPLAY).\n");
    cout << QObject::tr("-geometry geometry\n");
    cout << QObject::tr("\tSets the client geometry of the main widget.\n");
    cout << QObject::tr("-fn or -font font\n");
    cout << QObject::tr("\tDefines the application font.\n");
    cout << QObject::tr("-bg or -background color\n");
    cout << QObject::tr("\tSets the default background color and an application palette (light and\n");
    cout << QObject::tr("\tdark shades are calculated).\n");
    cout << QObject::tr("-fg or -foreground color\n");
    cout << QObject::tr("\tSets the default foreground color.\n");
    cout << QObject::tr("-btn or -button color\n");
    cout << QObject::tr("\tSets the default button color.\n");
    cout << QObject::tr("-visual TrueColor\n");
    cout << QObject::tr("\tForces the application to use a TrueColor visual on an 8-bit display.\n");
    cout << QObject::tr("-cmap\n");
    cout << QObject::tr("\tCauses the application to install a private color map on an 8-bit\n");
    cout << QObject::tr("\tdisplay.\n");
    cout << QObject::tr("-name name\n");
    cout << QObject::tr("\tSets the application name.\n");
    cout << QObject::tr("-title title\n");
    cout << QObject::tr("\tSets the application title (caption).\n");
}

void doconvert(const QString &filename)
{
    Preferences* pref = new Preferences(QDir::homeDirPath() + "/" + ID_PREFERENCES_FILE, PACKAGE, VERSION);
    if (filename.right(8) == ID_PREFERENCES_FILE) {
        // check if conversion is even needed
        if (pref->formatState()) {
            cout << QObject::tr("Information: No need to convert ");
            cout << ID_PREFERENCES_FILE << endl;
        } else {
            if (pref->fileState()) {
                OldSettings* settings = new OldSettings();
                settings->convert(pref);
                pref->flush();
                delete settings;
                cout << QObject::tr("Information: Conversion complete") << endl;
            }
        }
    } else if (filename.right(6) == ID_OLD_CALC_FILE) {
        QBrewCalc* calc = new QBrewCalc(pref);
        if (calc->convert(filename))
            cout << QObject::tr("Information: Conversion complete") << endl;
        else
            cout << QObject::tr("Information: Could not convert") << endl;
        delete calc;
    } else if (filename.right(6) == ".qbrew") {
        QBrewDoc* recipe = new QBrewDoc();
        if (recipe->convert(filename))
            cout << QObject::tr("Information: Conversion complete") << endl;
        else
            cout << QObject::tr("Information: Could not convert") << endl;
    } else {
        cout << QObject::tr("Warning: Unknown file") << endl;
    }
    delete pref;
}

//////////////////////////////////////////////////////////////////////////////
// doargs()
// --------
// Process command line arguments that QApplication doesn't take care of

QString doargs(int argc, char **argv)
{
    for (int n=1; n<argc; n++) {
        QString arg = argv[n];
        // detect all common forms of help requests
        if ((arg.lower() == "-h") || (arg == "-help") || (arg == "--help") || (arg == "-?")) {
            dohelp();
            exit(0);    // TODO: do this another way...
        } else if ((arg.lower() == "-v") || (arg == "-version")) {
            doversion();
            exit(0);
        } else if (arg.lower() == "-convert") {
            doconvert(QString(argv[n+1]));
            exit(0);
        } else {
            // it's either a filename or nothing...
            return arg; // which means no args after filename...
        }
    }
    return (AppResource::ID_DEFAULT_FILE);
}

//////////////////////////////////////////////////////////////////////////////
// main()
// ------
// Entry point of application

int main(int argc, char ** argv) {
    QApplication app(argc, argv);

    // check for additional command line options before we go further
    QString filename = doargs(app.argc(), app.argv());

    QBrew* qbrew = new QBrew(filename);
    app.setMainWidget(qbrew);
    qbrew->setCaption(AppResource::ID_TITLE + " " +  VERSION);

    qbrew->show();
    return app.exec();
}
