/***************************************************************************
  grain.h
  -------------------
  A grain class
  -------------------
  begin         October 9th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef GRAIN_H
#define GRAIN_H

#include <qlistview.h>
#include <qmap.h>
#include <qstring.h>

enum { GRAIN_OTHER, GRAIN_MASHED, GRAIN_STEEPED, GRAIN_EXTRACT };

class Grain {
public:
    // TODO: get rid of the first (enumerated use) version
    // constructor
    Grain(const QString &name="grain", const double &quantity=1.0, const double &extract=1.025,
        const double &color=10.0, const int &use=GRAIN_OTHER);
    // constructor
    Grain(const QString &name, const double &quantity, const double &extract,
        const double &color, const QString &use);
    // copy constructor
    Grain(const Grain &g);
    // assignment operator
    Grain operator=(const Grain &g);
    // comparison operator
    bool operator==(const Grain &g);
    // destructor
    ~Grain();

    // return name
    QString name();
    // return quantity (in pounds)
    double quantity();
    // return grain extract
    double extract();
    // return grain color
    double color();
    // return how grain is used (mash, steep, extract)
    int use();
    // return the yield (quantity times extract)
    double yield();
    // return the HCU (quantity times color)
    double HCU();
        
    // return quantity as string
    QString quantityString();
    // return extract as string
    QString extractString();
    // return color as string
    QString colorString();
    // return use as string
    QString useString();

    // return a list of use strings
    static QStringList useStringList();
    // convert use string to enum
    static int useStringToEnum(const QString &name);
    
    // return object as string, for serialization (deprecated)
    QString obsoleteSerializeOut();
    // set object to string, for serialization (deprecated)
    void obsoleteSerializeIn(const QString &name, const QString &value);
    
private:
    QString name_;
    double quantity_;
    double extract_;
    double color_;
    int use_;
    double yield_;
    double hcu_;    // color yield for grain
};

typedef QValueList<Grain> GrainList;
typedef QValueList<Grain>::Iterator GrainIterator;

class GrainListItem : public QListViewItem {
public:
    // constructor
    GrainListItem(QListView * parent, QString, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null, QString = QString::null,
            QString = QString::null, QString = QString::null );
    // destructor
    ~GrainListItem();
    // get iterator to Grain item
    GrainIterator grainIterator();
    // set iterator to Grain item
    void setGrainIterator(GrainIterator &it);
    // set grain to match iterator
    void setToIterator();
private:
    GrainIterator git_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline QString Grain::name() { return name_; }

inline double Grain::quantity() { return quantity_; }

inline double Grain::extract() { return extract_; }

inline double Grain::color() { return color_; }

inline int Grain::use() { return use_; }

inline QString Grain::quantityString() { return QString::number(quantity_, 'f', 2); }

inline QString Grain::extractString() { return QString::number(extract_, 'f', 3); }

inline double Grain::yield() { return yield_; }

inline double Grain::HCU() { return hcu_; }

inline GrainIterator GrainListItem::grainIterator() { return git_; }

inline void GrainListItem::setGrainIterator(GrainIterator &it) { git_ = it; }

#endif // GRAIN_H
