# arch-tag: f3985cae-28c6-4edd-87e2-c8611a109d5d
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Limited.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for LogMessage.
"""

import os

import pybaz as arch

import framework
from framework import TestCase
import fixtures


class LogMessage(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        self.params.create_working_tree(self.params.version)
        nonarch = self.params.version.nonarch
        archname = self.params.version.archive.name
        self.file_name = '++log.%s--%s' % (nonarch, archname)

    tests = []

    def create(self):
        """LogMessage creation."""
        wt = self.params.working_tree
        # message is None if not created
        m = wt.log_message(create=False)
        self.failUnlessEqual(m, None)
        # message is created by default
        m = wt.log_message()
        self.failIfEqual(m, None)
    tests.append('create')

    def name(self):
        """LogMessage.name attribute is read-only."""
        m = self.params.working_tree.log_message()
        self.assertEqual(m.name, self.params.working_tree.abspath()/m.name)
        def setLogName(): m.name = 'error'
        self.failUnlessRaises(AttributeError, setLogName)
    tests.append('name')

    def template(self):
        """LogMessage template has expected value and can be parsed."""
        m = self.params.working_tree.log_message()
        template = '\n'.join(('Summary: ', 'Keywords: ', '', ''))
        self.failUnlessEqual(file(m.name).read(), template)
        self.failUnlessEqual(m['Summary'], '')
        self.failUnlessEqual(m['Keywords'], '')
        self.failUnlessEqual(m.description, '')
    tests.append('template')

    def setup_message(self):
        self.description = '\n'.join(("Hello,  ", "  World!"))
        # leading space in summary dropped, trailing preserved
        self.summary = "Short summary  "
        self.message = '\n'.join(('Summary: ' + self.summary,
                                  'Keywords: ', '', self.description))
        m = self.message
        m = self.params.working_tree.log_message()
        m.description = self.description
        m['Summary'] = self.summary
        return m

    def check_message_file(self, m):
        self.failUnlessEqual(file(m.name).read(), self.message)

    def check_message_object(self, m):
        self.failUnlessEqual(m['Summary'], self.summary)
        self.failUnlessEqual(m['Keywords'], '')
        self.failUnlessEqual(m.description, self.description)

    def save_load(self):
        """LogMessage save() and load() behave consistently."""
        m = self.setup_message()
        self.check_message_object(m)
        m.save()
        self.check_message_file(m)
        m.load()
        self.check_message_object(m)
    tests.append('save_load')

    def recreate(self):
        """Recreating LogMessage does not overwrite existing message."""
        m = self.setup_message()
        m.save()
        del m
        m = self.params.working_tree.log_message()
        self.check_message_file(m)
        self.check_message_object(m)
    tests.append('recreate')

    def reload_(self):
        """Reloading LogMessage discards unsaved changes."""
        m = self.setup_message()
        m.save()
        m['Summary'] = 'foo'
        m.load()
        self.check_message_file(m)
        self.check_message_object(m)
    tests.append('reload_')

    def clear(self):
        """LogMessage.clear() method."""
        m = self.setup_message()
        m.clear()
        self.failUnlessEqual(m['Summary'], None)
        self.failUnlessEqual(m.description, None)
        m.save()
        self.failUnlessEqual(file(m.name).read(), '\n')
    tests.append('clear')


class LogMessageWhitespace(TestCase):

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()

    tests = []

    def log_message_whitespace(self):
        """LogMessage files created in a path containing a space."""
        working_dir_newline = self.params.working_dir_newline
        os.mkdir(working_dir_newline)
        wt = arch.init_tree(working_dir_newline, self.params.version)
        m = wt.log_message()
        template = '\n'.join(('Summary: ', 'Keywords: ', '', ''))
        self.failUnlessEqual(file(m.name).read(), template)
    tests.append('log_message_whitespace')


framework.register(__name__)
