//
// This is part of dvd+rw-tools by Andy Polyakov <appro@fy.chalmers.se>
//
// Use-it-on-your-own-risk, GPL bless...
//
// For further details see http://fy.chalmers.se/~appro/linux/DVD+RW/
//

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <poll.h>
#include <sys/time.h>

inline long getmsecs()
{ struct timeval tv;
    gettimeofday (&tv,NULL);
  return tv.tv_sec*1000+tv.tv_usec/1000;
}

#include <errno.h>

#ifndef EMEDIUMTYPE
#define EMEDIUMTYPE	EINVAL
#endif
#ifndef	ENOMEDIUM
#define	ENOMEDIUM	ENODEV
#endif



#define CREAM_ON_ERRNO_NAKED(s)				\
    switch ((s)[12])					\
    {	case 0x04:	errno=EAGAIN;	break;		\
	case 0x20:	errno=ENODEV;	break;		\
	case 0x21:	if ((s)[13]==0)	errno=ENOSPC;	\
			else		errno=EINVAL;	\
			break;				\
	case 0x30:	errno=EMEDIUMTYPE;	break;	\
	case 0x3A:	errno=ENOMEDIUM;	break;	\
    }
#define CREAM_ON_ERRNO(s)	do { CREAM_ON_ERRNO_NAKED(s) } while(0)

#define	FATAL_START(er)	(0x80|(er))
#define ERRCODE(s)	((((s)[2]&0x0F)<<16)|((s)[12]<<8)|((s)[13]))
#define	SK(errcode)	(((errcode)>>16)&0xF)
#define	ASC(errcode)	(((errcode)>>8)&0xFF)
#define ASCQ(errcode)	((errcode)&0xFF)

static void sperror (const char *cmd,int err)
{ int saved_errno=errno;

    if (err==-1)
	fprintf (stderr,":-( unable to %s: ",cmd);
    else
	fprintf (stderr,":-[ %s failed with SK=%Xh/ASC=%02Xh/ACQ=%02Xh]: ",
			cmd,SK(err),ASC(err),ASCQ(err));
    errno=saved_errno, perror (NULL);
}

class autofree {
    private:
	unsigned char *ptr;
    public:
	autofree()			{ ptr=NULL; }
	~autofree()			{ if (ptr) free(ptr); }
	unsigned char *operator=(unsigned char *str)
					{ return ptr=str; }
	operator unsigned char *()	{ return ptr; }
};


#include <sys/ioctl.h>
#include <linux/cdrom.h>
#include <mntent.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <scsi/sg.h>
#if !defined(SG_FLAG_LUN_INHIBIT)
# if defined(SG_FLAG_UNUSED_LUN_INHIBIT)
#  define SG_FLAG_LUN_INHIBIT SG_FLAG_UNUSED_LUN_INHIBIT
# else
#  define SG_FLAG_LUN_INHIBIT 0
# endif
#endif
#ifndef CHECK_CONDITION
#define CHECK_CONDITION 0x01
#endif

typedef enum {	NONE=CGC_DATA_NONE,	// 3
		READ=CGC_DATA_READ,	// 2
		WRITE=CGC_DATA_WRITE	// 1
	     } Direction;
#ifdef SG_IO
static const int Dir_xlate [4] = {	// should have been defined
					// private in USE_SG_IO scope,
					// but it appears to be too
		0,			// implementation-dependent...
		SG_DXFER_TO_DEV,	// 1,CGC_DATA_WRITE
		SG_DXFER_FROM_DEV,	// 2,CGC_DATA_READ
		SG_DXFER_NONE	};	// 3,CGC_DATA_NONE
static const class USE_SG_IO {
private:
    int	yes_or_no;
public:
    USE_SG_IO()	{ struct utsname buf;
		    uname (&buf);
		    // was CDROM_SEND_PACKET declared dead in 2.5?
		    yes_or_no=(strcmp(buf.release,"2.5.43")>=0);
		}
    ~USE_SG_IO(){}
    operator int()			const	{ return yes_or_no; }
    int operator[] (Direction dir)	const	{ return Dir_xlate[dir]; }
} use_sg_io;
#endif

class Scsi_Command {
private:
    int fd,autoclose;
    char *filename;
    struct cdrom_generic_command cgc;
    union {
	struct request_sense	s;
	unsigned char		u[18];
    } _sense;
#ifdef SG_IO
    struct sg_io_hdr		sg_io;
#else
    struct { int cmd_len,timeout; }	sg_io;
#endif
public:
    Scsi_Command()	{ fd=-1, autoclose=1; filename=NULL; }
    Scsi_Command(int f)	{ fd=f,  autoclose=0; filename=NULL; }
    Scsi_Command(void*f){ fd=(long)f, autoclose=0; filename=NULL; }
    ~Scsi_Command()	{ if (fd>=0 && autoclose) close(fd),fd=-1;
			  if (filename) free(filename),filename=NULL;
			}
    int associate (const char *file,const struct stat *ref=NULL)
    { struct stat sb;

	/*
	 * O_RDWR is expected to provide for none set-root-uid
	 * execution under Linux kernel 2.6[.8]. Under 2.4 it
	 * falls down to O_RDONLY...
	 */
	if ((fd=open (file,O_RDWR|O_NONBLOCK)) < 0 &&
	    (fd=open (file,O_RDONLY|O_NONBLOCK)) < 0)	return 0;
	if (fstat(fd,&sb) < 0)				return 0;
	if (!S_ISBLK(sb.st_mode))	{ errno=ENOTBLK;return 0; }

	if (ref && (!S_ISBLK(ref->st_mode) || ref->st_rdev!=sb.st_rdev))
	{   errno=ENXIO; return 0;   }

	filename=strdup(file);

	return 1;
    }
    unsigned char &operator[] (size_t i)
    {	if (i==0)
	{   memset(&cgc,0,sizeof(cgc)), memset(&_sense,0,sizeof(_sense));
	    cgc.quiet = 1;
	    cgc.sense = &_sense.s;
#ifdef SG_IO
	    if (use_sg_io)
	    {	memset(&sg_io,0,sizeof(sg_io));
		sg_io.interface_id= 'S';
		sg_io.mx_sb_len	= sizeof(_sense);
		sg_io.cmdp	= cgc.cmd;
		sg_io.sbp	= _sense.u;
		sg_io.flags	= SG_FLAG_LUN_INHIBIT|SG_FLAG_DIRECT_IO;
	    }
#endif
	}
	sg_io.cmd_len = i+1;
	return cgc.cmd[i];
    }
    unsigned char &operator()(size_t i)	{ return _sense.u[i]; }
    unsigned char *sense()		{ return _sense.u;    }
    void timeout(int i)			{ cgc.timeout=sg_io.timeout=i*1000; }
#ifdef SG_IO
    size_t residue()			{ return use_sg_io?sg_io.resid:0; }
#else
    size_t residue()			{ return 0; }
#endif
    int transport(Direction dir=NONE,void *buf=NULL,size_t sz=0)
    { int ret = 0;

#ifdef SG_IO
#define KERNEL_BROKEN 0
	if (use_sg_io)
	{   sg_io.dxferp		= buf;
	    sg_io.dxfer_len		= sz;
	    sg_io.dxfer_direction	= use_sg_io[dir];
	    if (ioctl (fd,SG_IO,&sg_io)) return -1;

#if !KERNEL_BROKEN
	    if ((sg_io.info&SG_INFO_OK_MASK) != SG_INFO_OK)
#else
	    if (sg_io.status)
#endif
	    {	errno=EIO; ret=-1;
#if !KERNEL_BROKEN
		if (sg_io.masked_status&CHECK_CONDITION)
#endif
		{   ret = ERRCODE(sg_io.sbp);
		    if (ret==0) ret=-1;
		    else	CREAM_ON_ERRNO(sg_io.sbp);
		}
	    }
	    return ret;
	}
	else
#undef KERNEL_BROKEN
#endif
	{   cgc.buffer		= (unsigned char *)buf;
	    cgc.buflen		= sz;
	    cgc.data_direction	= dir;
	    if (ioctl (fd,CDROM_SEND_PACKET,&cgc))
	    {	ret = ERRCODE(_sense.u);
		if (ret==0) ret=-1;
	    }
	}
	return ret;
    }
    int umount(int f=-1)
    { struct stat    fsb,msb;
      struct mntent *mb;
      FILE          *fp;
      pid_t          pid,rpid;
      int            ret=0,rval;

	if (f==-1) f=fd;
	if (fstat (f,&fsb) < 0)				return -1;
	if ((fp=setmntent ("/proc/mounts","r"))==NULL)	return -1;

	while ((mb=getmntent (fp))!=NULL)
	{   if (stat (mb->mnt_fsname,&msb) < 0) continue; // corrupted line?
	    if (msb.st_rdev == fsb.st_rdev)
	    {	ret = -1;
		if ((pid = fork()) == (pid_t)-1)	break;
		if (pid == 0) execl ("/bin/umount","umount",mb->mnt_dir,NULL);
		while (1)
		{   rpid = waitpid (pid,&rval,0);
		    if (rpid == (pid_t)-1)
		    {	if (errno==EINTR)	continue;
			else			break;
		    }
		    else if (rpid != pid)
		    {	errno = ECHILD;
			break;
		    }
		    if (WIFEXITED(rval))
		    {	if (WEXITSTATUS(rval) == 0) ret=0;
			else			    errno=EBUSY; // most likely
			break;
		    }
		    else
		    {	errno = ENOLINK;	// some phony errno
			break;
		    }
		}
		break;
	    }
	}
	endmntent (fp);

	return ret;
    }
    int is_reload_needed ()
    {	return ioctl (fd,CDROM_MEDIA_CHANGED,CDSL_CURRENT) == 0;   }
};

#define DUMP_EVENTS 0
static int handle_events (Scsi_Command &cmd)
{ unsigned char  event[8];
  unsigned short profile=0,started=0;
  int err,ret=0;
  unsigned int descr;
  static unsigned char events=0xFF;	// "All events"

    while (events)
    {	cmd[0] = 0x4A;		// GET EVENT
	cmd[1] = 1;		// "Polled"
	cmd[4] = events;
	cmd[8] = sizeof(event);
	cmd[9] = 0;
	if ((err=cmd.transport(READ,event,sizeof(event))))
	{   events=0;
	    sperror ("GET EVENT",err);
	    return ret;
	}

	events = event[3];

	if ((event[2]&7) == 0			||
	    (event[0]<<8|event[1]) == 2	|| 
	    (event[4]&0xF) == 0			)	// No Changes
	    return ret;

	descr  = event[4]<<24|event[5]<<16|event[6]<<8|event[7];
#if DUMP_EVENTS
	fprintf(stderr,"< %d[%08x],%x >\n",event[2],descr,events);
#endif

	switch(event[2]&7)
	{   case 0: return ret;			// No [supported] events
	    case 1: ret |= 1<<1;		// Operational Change
		if ((descr&0xFFFF) < 3)
		    goto read_profile;
	    start_unit:
	    	if (!started)
		{   cmd[0]=0x1B;	// START STOP UNIT
		    cmd[4]=1;		// "Start"
		    cmd[5]=0;
		    if ((err=cmd.transport()) && err!=0x62800)
			sperror ("START UNIT",err);
		    started=1, profile=0;
		}
	    read_profile:
		if (!profile)
		{   cmd[0] = 0x46;	// GET CONFIGURATION
		    cmd[8] = sizeof(event);
		    cmd[9] = 0;
		    if (!cmd.transport(READ,event,sizeof(event)))
			profile=event[6]<<8|event[7];
		}
		break;
	    case 2: ret |= 1<<2;		// Power Management
		if (event[5]>1)		// State is other than Active
		    goto start_unit;
		break;
	    case 3: ret |= 1<<3; break;		// External Request
	    case 4: ret |= 1<<4;		// Media
		if (event[5]&2)		// Media in
		    goto  start_unit;
		break;
	    case 5: ret |= 1<<5; break;		// Multiple Initiators
	    case 6:				// Device Busy
		if ((event[4]&0xF)==1)	// Timeout occured
		{   poll(NULL,0,(descr&0xFFFF)*100+100);
		    cmd[0] = 0;		// TEST UNIT READY
		    cmd[5] = 0;
		    if ((err=cmd.transport()))
			sperror("TEST UNIT READY",err);
		    ret |= 1<<6;
		}
		break;
	    case 7: ret |= 1<<7; break;		// Reserved
	}
    }

  return ret;
}
#undef DUMP_EVENTS

static int wait_for_unit (Scsi_Command &cmd,volatile int *progress=NULL)
{ unsigned char *sense=cmd.sense(),sensebuf[18];
  int  err;
  long msecs=1000;

    while (1)
    {	if (msecs > 0) poll(NULL,0,msecs);
	msecs = getmsecs();
	cmd[0] = 0;	// TEST UNIT READY
	cmd[5] = 0;
	if (!(err=cmd.transport ())) break;
	// I wish I could test just for sense.valid, but (at least)
	// hp dvd100i returns 0 in valid bit at this point:-( So I
	// check for all bits...
	if ((sense[0]&0x70)==0)
	{   perror (":-( unable to TEST UNIT READY");
	    return -1;
	}
	else if (sense[12]==0x3A) // doesn't get any further than "no media"
	    return err;

	while (progress)
	{   if (sense[15]&0x80)
	    {	*progress = sense[16]<<8|sense[17];
		break;
	    }
	    // MMC-3 (draft) specification says that the unit should
	    // return progress indicator in key specific bytes even
	    // in reply to TEST UNIT READY. I.e. as above! But (at
	    // least) hp dvd100i doesn't do that and I have to fetch
	    // it separately:-(
	    cmd[0] = 0x03;	// REQUEST SENSE
	    cmd[4] = sizeof(sensebuf);
	    cmd[5] = 0;
	    if ((err=cmd.transport (READ,sensebuf,sizeof(sensebuf))))
	    {   sperror ("REQUEST SENSE",err);
		return err;
	    }
	    if (sensebuf[15]&0x80)
		*progress = sensebuf[16]<<8|sensebuf[17];
	    break;
	}
	msecs = 1000 - (getmsecs() - msecs);
    }

  return 0;
}

#define FEATURE21_BROKEN 1
static void page05_setup (Scsi_Command &cmd, unsigned short profile=0,
	unsigned char p32=0xC0)
	// 5 least significant bits of p32 go to p[2], Test Write&Write Type
	// 2 most significant bits go to p[3], Multi-session field
	// 0xC0 means "Multi-session, no Test Write, Incremental"
{ unsigned int   len,bdlen;
  unsigned char  header[12],track[32],*p;
#if !FEATURE21_BROKEN
  unsigned char  feature21[24];
#endif
  int            err;
  class autofree page05;

    if (profile==0)
    { unsigned char prof[8];

	cmd[0] = 0x46;	// GET CONFIGURATION
	cmd[8] = sizeof(prof);
	cmd[9] = 0;
	if ((err=cmd.transport(READ,prof,sizeof(prof))))
	    sperror ("GET CONFIGURATION",err), exit(FATAL_START(errno));

	profile = prof[6]<<8|prof[7];
    }

#if !FEATURE21_BROKEN
    if (profile==0x11 || profile==0x14)
    {	cmd[0] = 0x46;	// GET CONFIGURATION
	cmd[1] = 2;	// ask for the only feature...
	cmd[3] = 0x21;	// the "Incremental Streaming Writable" one
	cmd[8] = 8;	// read the header only to start with
	cmd[9] = 0;
	if ((err=cmd.transport(READ,feature21,8)))
	    sperror ("GET CONFIGURATION",err), exit(FATAL_START(errno));

	len = feature21[0]<<24|feature21[1]<<16|feature21[2]<<8|feature21[3];
	len += 4;
	if (len>sizeof(feature21))
	    len = sizeof(feature21);
	else if (len<(8+8))
	    fprintf (stderr,":-( READ FEATURE DESCRIPTOR 0021h: insane length\n"),
	    exit(FATAL_START(EINVAL));

	cmd[0] = 0x46;	// GET CONFIGURATION
	cmd[1] = 2;	// ask for the only feature...
	cmd[3] = 0x21;	// the "Incremental Streaming Writable" one
	cmd[8] = len;	// this time with real length
	cmd[9] = 0;
	if ((err=cmd.transport(READ,feature21,len)))
	    sperror ("READ FEATURE DESCRIPTOR 0021h",err),
	    exit(FATAL_START(errno));

	if ((feature21[8+2]&1)==0)
	    fprintf (stderr,":-( FEATURE 0021h is not in effect\n"),
	    exit(FATAL_START(EMEDIUMTYPE));
    }
    else
	feature21[8+2]=0;
#endif

    cmd[0] = 0x52;	// READ TRACK INFORMATION
    cmd[1] = 1;		// TRACK INFORMATION
    cmd[5] = 1;		// track#1, in DVD context it's safe to assume
    			//          that all tracks are in the same mode
    cmd[8] = sizeof(track);
    cmd[9] = 0;
    if ((err=cmd.transport(READ,track,sizeof(track))))
	sperror ("READ TRACK INFORMATION",err), exit(FATAL_START(errno));

    // WRITE PAGE SETUP //
    cmd[0] = 0x5A;		// MODE SENSE
    cmd[1] = 0x08;		// "Disable Block Descriptors"
    cmd[2] = 0x05;		// "Write Page"
    cmd[8] = sizeof(header);	// header only to start with
    cmd[9] = 0;
    if ((err=cmd.transport(READ,header,sizeof(header))))
	sperror ("MODE SENSE",err), exit(FATAL_START(errno));

    len   = (header[0]<<8|header[1])+2;
    bdlen = header[6]<<8|header[7];

    if (bdlen)	// should never happen as we set "DBD" above
    {	if (len <= (8+bdlen+14))
	    fprintf (stderr,":-( LUN is impossible to bear with...\n"),
	    exit(FATAL_START(EINVAL));
    }
    else if (len < (8+2+(unsigned int)header[9]))// SANYO does this.
	len = 8+2+header[9];

    page05 = (unsigned char *)malloc(len);
    if (page05 == NULL)
	fprintf (stderr,":-( memory exhausted\n"),
	exit(FATAL_START(ENOMEM));

    cmd[0] = 0x5A;		// MODE SENSE
    cmd[1] = 0x08;		// "Disable Block Descriptors"
    cmd[2] = 0x05;		// "Write Page"
    cmd[7] = len>>8;
    cmd[8] = len;		// real length this time
    cmd[9] = 0;
    if ((err=cmd.transport(READ,page05,len)))
	sperror("MODE SENSE",err), exit(FATAL_START(errno));

    len -= 2;
    if (len < ((unsigned int)page05[0]<<8|page05[1]))	// paranoia:-)
	page05[0] = len>>8, page05[1] = len;

    len   = (page05[0]<<8|page05[1])+2;
    bdlen = page05[6]<<8|page05[7];
    len  -= bdlen;
    if (len < (8+14))
	fprintf (stderr,":-( LUN is impossible to bear with...\n"),
	exit(FATAL_START(EINVAL));

    p = page05 + 8 + bdlen;

    memset (p-8,0,8);
    p[0] &= 0x7F;

    // copy "Test Write" and "Write Type" from p32
    p[2] &= ~0x1F, p[2] |= p32&0x1F;	
    p[2] |= 0x40;	// insist on BUFE on

    // setup Preferred Link Size
#if !FEATURE21_BROKEN
    if (feature21[8+2]&1)
    {	if (feature21[8+7])	p[2] |= 0x20,  p[5] = feature21[8+8];
	else			p[2] &= ~0x20, p[5] = 0;
    }
#else	// At least Pioneer DVR-104 returns some bogus data in
	// Preferred Link Size...
    if (profile==0x11 || profile==0x14)	// Sequential recordings...
	p[2] |= 0x20, p[5] = 0x10;
#endif
    else
	p[2] &= ~0x20, p[5] = 0;

    // copy Track Mode from TRACK INFORMATION
    // [some DVD-R units (most notably Panasonic LF-D310), insist
    // on Track Mode 5, even though it's effectively ignored]
    p[3] &= ~0x0F, p[3] |= profile==0x11?5:(track[5]&0x0F);

    // copy "Multi-session" bits from p32
    p[3] &= ~0xC0, p[3] |= p32&0xC0;
    if (profile == 0x13)	// DVD-RW Restricted Overwrite
    	p[3] &= 0x3F;		// always Single-session?

    // setup Data Block Type
    if ((track[6]&0x0F)==1)	p[4] = 8;
    else	fprintf (stderr,":-( none Mode 1 track\n"),
		exit(FATAL_START(EMEDIUMTYPE));

    // setup Packet Size
    // [some DVD-R units (most notably Panasonic LF-D310), insist
    // on fixed Packet Size of 16 blocks, even though it's effectively
    // ignored]
    p[3] |= 0x20, memset (p+10,0,4), p[13] = 0x10;
    if (track[6]&0x10)
	memcpy (p+10,track+20,4);	// Fixed
    else if (profile != 0x11)
	p[3] &= ~0x20, p[13] = 0;	// Variable

    switch (profile)
    {	case 0x13:	// DVD-RW Restricted Overwrite
	    if (!(track[6]&0x10))
		fprintf (stderr,":-( track is not formatted for fixed packet size\n"),
		exit(FATAL_START(EMEDIUMTYPE));
	    break;
	case 0x14:	// DVD-RW Sequential Recording
	case 0x11:	// DVD-R  Sequential Recording
	    if (track[6]&0x10)
		fprintf (stderr,":-( track is formatted for fixed packet size\n"),
		exit(FATAL_START(EMEDIUMTYPE));
	    break;
	default:
#if 0
	    fprintf (stderr,":-( invalid profile %04xh\n",profile);
	    exit(FATAL_START(EMEDIUMTYPE));
#endif
	    break;
    }

    p[8] = 0;		// "Session Format" should be ignored, but
			// I reset it just in case...

    cmd[0] = 0x55;	// MODE SELECT
    cmd[1] = 0x10;	// conformant
    cmd[7] = len>>8;
    cmd[8] = len;
    cmd[9] = 0;
    if ((err=cmd.transport(WRITE,p-8,len)))
	sperror ("MODE SELECT",err), exit(FATAL_START(errno));
    // END OF WRITE PAGE SETUP //
}
#undef FEATURE21_BROKEN

#undef ERRCODE
#undef CREAM_ON_ERRNO
#undef CREAM_ON_ERRNO_NAKED

int print_sense (int err) {
	char str[128];
	strcpy(str,"[unknown error]");
	switch (SK(err)) {
	case 0x1:
		switch (ASC(err)) {
		case 0x0B:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"WARNING"); break;
        		case 0x01: strcpy(str,"WARNING - SPECIFIED TEMPERATURE EXCEEDED"); break;
        		case 0x02: strcpy(str,"WARNING - ENCLOSURE DEGRADED"); break;

			default:  sprintf(str,"WARNING, ASCQ=%02X",ASCQ(err)); break;
			}
			break;
		case 0x17:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"RECOVERED DATA WITH NO ERROR CORRECTION APPLIED"); break;
			case 0x01: strcpy(str,"RECOVERED DATA WITH RETRIES"); break;
			case 0x02: strcpy(str,"RECOVERED DATA WITH POSITIVE HEAD OFFSET"); break;
			case 0x03: strcpy(str,"RECOVERED DATA WITH NEGATIVE HEAD OFFSET"); break;
			case 0x04: strcpy(str,"RECOVERED DATA WITH RETRIES AND/OR CIRC APPLIED"); break;
			case 0x05: strcpy(str,"RECOVERED DATA USING PREVIOUS SECTOR ID"); break;
			case 0x07: strcpy(str,"RECOVERED DATA WITHOUT ECC - RECOMMEND REASSIGNMENT"); break;
			case 0x08: strcpy(str,"RECOVERED DATA WITHOUT ECC - RECOMMEND REWRITE"); break;
			case 0x09: strcpy(str,"RECOVERED DATA WITHOUT ECC - DATA REWRITTEN"); break;

			default:   strcpy(str,"RECOVERED DATA WITH NO ERROR CORRECTION APPLIED"); break;
			}
			break;
		case 0x18:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"RECOVERED DATA WITH ERROR CORRECTION APPLIED"); break;
			case 0x01: strcpy(str,"RECOVERED DATA WITH ERROR CORR. & RETRIES APPLIED"); break;
			case 0x02: strcpy(str,"RECOVERED DATA - DATA AUTO-REALLOCATED"); break;
			case 0x03: strcpy(str,"RECOVERED DATA WITH CIRC"); break;
			case 0x04: strcpy(str,"RECOVERED DATA WITH L-EC"); break;
			case 0x05: strcpy(str,"RECOVERED DATA - RECOMMEND REASSIGNMENT"); break;
			case 0x06: strcpy(str,"RECOVERED DATA - RECOMMEND REWRITE"); break;
			case 0x08: strcpy(str,"RECOVERED DATA WITH LINKING"); break;

			default:   strcpy(str,"RECOVERED DATA WITH ERROR CORRECTION APPLIED"); break;
			}
			break;
		case 0x5D:
			switch (ASCQ(err)) {
			case 0x01: strcpy(str,"FAILURE PREDICTION THRESHOLD EXCEEDED - Predicted Media failure"); break;
			case 0x02: strcpy(str,"LOGICAL UNIT FAILURE PREDICTION THRESHOLD EXCEEDED"); break;
			case 0x03: strcpy(str,"FAILURE PREDICTION THRESHOLD EXCEEDED - Predicted Spare Area Exhaustion"); break;
			case 0xFF: strcpy(str,"FAILURE PREDICTION THRESHOLD EXCEEDED (FALSE)"); break;

			default:   strcpy(str,"LOGICAL UNIT FAILURE PREDICTION THRESHOLD EXCEEDED"); break;
			}
			break;
		case 0x73:
			switch (ASCQ(err)) {
			case 0x01: strcpy(str,"POWER CALIBRATION AREA ALMOST FULL"); break;
			case 0x06: strcpy(str,"RMA/PMA IS ALMOST FULL"); break;
			}
			break;
		}
	case 0x2:
		switch (ASC(err)) {
		case 0x04:
			switch (ASCQ(err)) {
        		case 0x00: strcpy(str,"LOGICAL UNIT NOT READY, CAUSE NOT REPORTABLE"); break;
        		case 0x01: strcpy(str,"LOGICAL UNIT IS IN PROCESS OF BECOMING READY"); break;
        		case 0x02: strcpy(str,"LOGICAL UNIT NOT READY, INITIALIZING CMD. REQUIRED"); break;
        		case 0x03: strcpy(str,"LOGICAL UNIT NOT READY, MANUAL INTERVENTION REQUIRED"); break;
        		case 0x04: strcpy(str,"LOGICAL UNIT NOT READY, FORMAT IN PROGRESS"); break;
        		case 0x07: strcpy(str,"LOGICAL UNIT NOT READY, OPERATION IN PROGRESS"); break;
        		case 0x08: strcpy(str,"LOGICAL UNIT NOT READY, LONG WRITE IN PROGRESS"); break;

        		default:   strcpy(str,"LOGICAL UNIT NOT READY, CAUSE NOT REPORTABLE"); break;
			}
			break;
		case 0x30:
			switch (ASCQ(err)) {
       			case 0x00: strcpy(str,"INCOMPATIBLE MEDIUM INSTALLED"); break;
       			case 0x01: strcpy(str,"CANNOT READ MEDIUM - UNKNOWN FORMAT"); break;
       			case 0x02: strcpy(str,"CANNOT READ MEDIUM - INCOMPATIBLE FORMAT"); break;
       			case 0x03: strcpy(str,"CLEANING CARTRIDGE INSTALLED"); break;
       			case 0x04: strcpy(str,"CANNOT WRITE MEDIUM - UNKNOWN FORMAT"); break;
       			case 0x05: strcpy(str,"CANNOT WRITE MEDIUM - INCOMPATIBLE FORMAT"); break;
       			case 0x06: strcpy(str,"CANNOT FORMAT MEDIUM - INCOMPATIBLE MEDIUM"); break;
       			case 0x07: strcpy(str,"CLEANING FAILURE"); break;
       			case 0x11: strcpy(str,"CANNOT WRITE MEDIUM - UNSUPPORTED MEDIUM VERSION"); break;

       			default:   strcpy(str,"INCOMPATIBLE MEDIUM INSTALLED"); break;
			}
			break;
		case 0x3A:
			switch (ASCQ(err)) {
       			case 0x00: strcpy(str,"MEDIUM NOT PRESENT"); break;
       			case 0x01: strcpy(str,"MEDIUM NOT PRESENT - TRAY CLOSED"); break;
       			case 0x02: strcpy(str,"MEDIUM NOT PRESENT - TRAY OPEN"); break;

       			default:   strcpy(str,"MEDIUM NOT PRESENT"); break;
			}
			break;
		case 0x3E: strcpy(str,"LOGICAL UNIT HAS NOT SELF-CONFIGURED YET"); break; /* ASCQ=00: */
		}
		break;
	case 0x3:
		switch (ASC(err)) {
		case 0x02: strcpy(str,"NO SEEK COMPLETE"); break; /* ASCQ = 0x00 */
		case 0x06: strcpy(str,"NO REFERENCE POSITION FOUND"); break;
		case 0x0C:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"WRITE ERROR"); break;
			case 0x07: strcpy(str,"WRITE ERROR - RECOVERY NEEDED"); break;
			case 0x08: strcpy(str,"WRITE ERROR - RECOVERY FAILED"); break;
			case 0x09: strcpy(str,"WRITE ERROR - LOSS OF STREAMING"); break;
			case 0x0A: strcpy(str,"WRITE ERROR - PADDING BLOCKS ADDED"); break;

			default:   strcpy(str,"WRITE ERROR"); break;
			}
			break;
		case 0x11:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"UNRECOVERED READ ERROR"); break;
			case 0x01: strcpy(str,"READ RETRIES EXHAUSTED"); break;
			case 0x02: strcpy(str,"ERROR TOO LONG TO CORRECT"); break;
			case 0x05: strcpy(str,"L-EC UNCORRECTABLE ERROR"); break;
			case 0x06: strcpy(str,"CIRC UNRECOVERED ERROR"); break;
			case 0x0F: strcpy(str,"ERROR READING UPC/EAN NUMBER"); break;
			case 0x10: strcpy(str,"ERROR READING ISRC NUMBER"); break;

			default:   strcpy(str,"UNRECOVERED READ ERROR"); break;
			}
			break;
		case 0x15:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"RANDOM POSITIONING ERROR"); break;
			case 0x01: strcpy(str,"MECHANICAL POSITIONING ERROR"); break;
			case 0x02: strcpy(str,"POSITIONING ERROR DETECTED BY READ OF MEDIUM"); break;

			default: strcpy(str,"RANDOM POSITIONING ERROR"); break;
			}
			break;
		case 0x31:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"MEDIUM FORMAT CORRUPTED"); break;
			case 0x01: strcpy(str,"FORMAT COMMAND FAILED"); break;
			case 0x02: strcpy(str,"ZONED FORMATTING FAILED DUE TO SPARE LINKING"); break;

			default:   strcpy(str,"MEDIUM FORMAT CORRUPTED"); break;
			}
			break;
		case 0x51:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"ERASE FAILURE"); break;
			case 0x01: strcpy(str,"ERASE FAILURE - INCOMPLETE ERASE OPERATION DETECTED"); break;

			default:   strcpy(str,"ERASE FAILURE"); break;
			}
			break;
		case 0x57: strcpy(str,"UNABLE TO RECOVER TABLE-OF-CONTENTS"); break; /* ASCQ = 00 */
		case 0x72:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"SESSION FIXATION ERROR"); break;
			case 0x01: strcpy(str,"SESSION FIXATION ERROR WRITING LEAD-IN"); break;
			case 0x02: strcpy(str,"SESSION FIXATION ERROR WRITING LEAD-OUT"); break;

			default:   strcpy(str,"SESSION FIXATION ERROR"); break;
			}
			break;
		case 0x73:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"CD CONTROL ERROR"); break;
			case 0x02: strcpy(str,"POWER CALIBRATION AREA IS FULL"); break;
			case 0x03: strcpy(str,"POWER CALIBRATION AREA ERROR"); break;
			case 0x04: strcpy(str,"PROGRAM MEMORY AREA UPDATE FAILURE"); break;
			case 0x05: strcpy(str,"PROGRAM MEMORY AREA IS FULL"); break;

			default:   strcpy(str,"CD CONTROL ERROR"); break;
			}
			break;
		}
		break;
	case 0x4:
		switch (ASC(err)) {
		case 0x00: strcpy(str,"CLEANING REQUESTED"); break;  /* ASCQ = 0x17 */
		case 0x05: strcpy(str,"LOGICAL UNIT DOES NOT RESPOND TO SELECTION"); break; /* ASCQ = 0x00 */
		case 0x08:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"LOGICAL UNIT COMMUNICATION FAILURE"); break;
			case 0x01: strcpy(str,"LOGICAL UNIT COMMUNICATION TIMEOUT"); break;
			case 0x02: strcpy(str,"LOGICAL UNIT COMMUNICATION PARITY ERROR"); break;
			case 0x03: strcpy(str,"LOGICAL UNIT COMMUNICATION CRC ERROR (ULTRA-DMA/32)"); break;
			}
			break;
		case 0x09:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"TRACK FOLLOWING ERROR"); break;
			case 0x01: strcpy(str,"TRACKING SERVO FAILURE"); break;
			case 0x02: strcpy(str,"FOCUS SERVO FAILURE"); break;
			case 0x03: strcpy(str,"SPINDLE SERVO FAILURE"); break;
			case 0x04: strcpy(str,"HEAD SELECT FAULT"); break;

			default:   strcpy(str,"TRACKING ERROR"); break;
			}
			break;
		case 0x15:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"RANDOM POSITIONING ERROR"); break;
			case 0x01: strcpy(str,"MECHANICAL POSITIONING ERROR"); break;

			default:   strcpy(str,"RANDOM POSITIONING ERROR"); break;
			}
			break;
		case 0x1B: strcpy(str,"SYNCHRONOUS DATA TRANSFER ERROR"); break; /* ASCQ = 0x00 */
		case 0x3B: strcpy(str,"MECHANICAL POSITIONING OR CHANGER ERROR"); break; /* ASCQ = 0x16 */
		case 0x3E:
			switch (ASCQ(err)) {
			case 0x01: strcpy(str,"LOGICAL UNIT FAILURE"); break;
			case 0x02: strcpy(str,"TIMEOUT ON LOGICAL UNIT"); break;

			default:   strcpy(str,"LOGICAL UNIT FAILURE"); break;
			}
			break;
		case 0x40: strcpy(str,"DIAGNOSTIC FAILURE ON COMPONENT NN (80H-FFH)"); break;
		case 0x44: strcpy(str,"INTERNAL TARGET FAILURE"); break;
		case 0x46: strcpy(str,"UNSUCCESSFUL SOFT RESET"); break;
		case 0x47: strcpy(str,"SCSI PARITY ERROR"); break;
		case 0x4A: strcpy(str,"COMMAND PHASE ERROR"); break;
		case 0x4B: strcpy(str,"DATA PHASE ERROR"); break;
		case 0x4C: strcpy(str,"LOGICAL UNIT FAILED SELF-CONFIGURATION"); break;
		case 0x53: strcpy(str,"MEDIA LOAD OR EJECT FAILED"); break;
		case 0x65: strcpy(str,"VOLTAGE FAULT"); break;
		}
		break;
	case 0x5:
		switch (ASC(err)) {
		case 0x07: strcpy(str,"MULTIPLE PERIPHERAL DEVICES SELECTED"); break; /* ASCQ = 0x00 */
		case 0x1A: strcpy(str,"PARAMETER LIST LENGTH ERROR"); break; /* ASCQ = 0x00 */
		case 0x20: strcpy(str,"INVALID COMMAND OPERATION CODE"); break; /* ASCQ = 0x00 */
		case 0x21:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"LOGICAL BLOCK ADDRESS OUT OF RANGE"); break;
			case 0x01: strcpy(str,"INVALID ELEMENT ADDRESS"); break;
			case 0x02: strcpy(str,"INVALID ADDRESS FOR WRITE"); break;

			default:   strcpy(str,"LOGICAL BLOCK ADDRESS OUT OF RANGE"); break;
			}
			break;
		case 0x24: strcpy(str,"INVALID FIELD IN CDB"); break;
		case 0x25: strcpy(str,"LOGICAL UNIT NOT SUPPORTED"); break;
		case 0x26:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"INVALID FIELD IN PARAMETER LIST"); break;
			case 0x01: strcpy(str,"PARAMETER NOT SUPPORTED"); break;
			case 0x02: strcpy(str,"PARAMETER VALUE INVALID"); break;
			case 0x03: strcpy(str,"THRESHOLD PARAMETERS NOT SUPPORTED"); break;
			}
			break;
		case 0x2B: strcpy(str,"COPY CANNOT EXECUTE SINCE INITIATOR CANNOT DISCONNECT"); break; /* ASCQ = 0x00 */
		case 0x2C:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"COMMAND SEQUENCE ERROR"); break;
			case 0x03: strcpy(str,"CURRENT PROGRAM AREA IS NOT EMPTY"); break;
			case 0x04: strcpy(str,"CURRENT PROGRAM AREA IS EMPTY"); break;
			}
			break;
		case 0x30:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"INCOMPATIBLE MEDIUM INSTALLED"); break;
			case 0x01: strcpy(str,"CANNOT READ MEDIUM - UNKNOWN FORMAT"); break;
			case 0x02: strcpy(str,"CANNOT READ MEDIUM - INCOMPATIBLE FORMAT"); break;
			case 0x03: strcpy(str,"CLEANING CARTRIDGE INSTALLED"); break;
			case 0x04: strcpy(str,"CANNOT WRITE MEDIUM - UNKNOWN FORMAT"); break;
			case 0x05: strcpy(str,"CANNOT WRITE MEDIUM - INCOMPATIBLE FORMAT"); break;
			case 0x06: strcpy(str,"CANNOT FORMAT MEDIUM - INCOMPATIBLE MEDIUM"); break;
			case 0x07: strcpy(str,"CLEANING FAILURE"); break;
			case 0x08: strcpy(str,"CANNOT WRITE - APPLICATION CODE MISMATCH"); break;
			case 0x09: strcpy(str,"CURRENT SESSION NOT FIXATED FOR APPEND"); break;
			case 0x10: strcpy(str,"MEDIUM NOT FORMATTED"); break;
			}
			break;
		case 0x39: strcpy(str,"SAVING PARAMETERS NOT SUPPORTED"); break;  /* ASCQ = 0x00 */
		case 0x3D: strcpy(str,"INVALID BITS IN IDENTIFY MESSAGE"); break; /* ASCQ = 0x00 */
		case 0x43: strcpy(str,"MESSAGE ERROR"); break; /* ASCQ = 0x00 */
		case 0x53: strcpy(str,"MEDIUM REMOVAL PREVENTED"); break; /* ASCQ = 0x02 */
		case 0x64:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"ILLEGAL MODE FOR THIS TRACK"); break;
			case 0x01: strcpy(str,"INVALID PACKET SIZE"); break;
			}
			break;
		case 0x6F:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"COPY PROTECTION KEY EXCHANGE FAILURE - AUTHENTICATION FAILURE"); break;
			case 0x01: strcpy(str,"COPY PROTECTION KEY EXCHANGE FAILURE - KEY NOT PRESENT"); break;
			case 0x02: strcpy(str,"COPY PROTECTION KEY EXCHANGE FAILURE - KEY NOT ESTABLISHED"); break;
			case 0x03: strcpy(str,"READ OF SCRAMBLED SECTOR WITHOUT AUTHENTICATION"); break;
			case 0x04: strcpy(str,"MEDIA REGION CODE IS MISMATCHED TO LOGICAL UNIT REGION"); break;
			case 0x05: strcpy(str,"LOGICAL UNIT REGION MUST BE PERMANENT/REGION RESET COUNT ERROR"); break;
			}
			break;
		case 0x72:
			switch (ASCQ(err)) {
			case 0x03: strcpy(str,"SESSION FIXATION ERROR . INCOMPLETE TRACK IN SESSION"); break;
			case 0x04: strcpy(str,"EMPTY OR PARTIALLY WRITTEN RESERVED TRACK"); break;
			case 0x05: strcpy(str,"NO MORE TRACK RESERVATIONS ALLOWED"); break;
			}
			break;
		}
		break;
	case 0x6:
		switch (ASC(err)) {
		case 0x28:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"NOT READY TO READY CHANGE, MEDIUM MAY HAVE CHANGED"); break;
			case 0x01: strcpy(str,"IMPORT OR EXPORT ELEMENT ACCESSED"); break;
			}
			break;
		case 0x29:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"POWER ON, RESET, OR BUS DEVICE RESET OCCURRED"); break;
			case 0x01: strcpy(str,"POWER ON OCCURRED"); break;
			case 0x02: strcpy(str,"BUS RESET OCCURRED"); break;
			case 0x03: strcpy(str,"BUS DEVICE RESET FUNCTION OCCURRED"); break;
			case 0x04: strcpy(str,"DEVICE INTERNAL RESET"); break;
			}
			break;
		case 0x2A:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"PARAMETERS CHANGED"); break;
			case 0x01: strcpy(str,"MODE PARAMETERS CHANGED"); break;
			case 0x02: strcpy(str,"LOG PARAMETERS CHANGED"); break;
			case 0x03: strcpy(str,"RESERVATIONS PREEMPTED"); break;
			}
			break;
		case 0x2E: strcpy(str,"INSUFFICIENT TIME FOR OPERATION"); break;
		case 0x2F: strcpy(str,"COMMANDS CLEARED BY ANOTHER INITIATOR"); break;
		case 0x3B:
			switch (ASCQ(err)) {
			case 0x0D: strcpy(str,"MEDIUM DESTINATION ELEMENT FULL"); break;
			case 0x0E: strcpy(str,"MEDIUM SOURCE ELEMENT EMPTY"); break;
			case 0x0F: strcpy(str,"END OF MEDIUM REACHED"); break;
			case 0x11: strcpy(str,"MEDIUM MAGAZINE NOT ACCESSIBLE"); break;
			case 0x12: strcpy(str,"MEDIUM MAGAZINE REMOVED"); break;
			case 0x13: strcpy(str,"MEDIUM MAGAZINE INSERTED"); break;
			case 0x14: strcpy(str,"MEDIUM MAGAZINE LOCKED"); break;
			case 0x15: strcpy(str,"MEDIUM MAGAZINE UNLOCKED"); break;
			}
			break;
		case 0x3F:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"TARGET OPERATING CONDITIONS HAVE CHANGED"); break;
			case 0x01: strcpy(str,"MICROCODE HAS BEEN CHANGED"); break;
			case 0x02: strcpy(str,"CHANGED OPERATING DEFINITION"); break;
			case 0x03: strcpy(str,"INQUIRY DATA HAS CHANGED"); break;
			}
			break;
		case 0x5A:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"OPERATOR REQUEST OR STATE CHANGE INPUT"); break;
			case 0x01: strcpy(str,"OPERATOR MEDIUM REMOVAL REQUEST"); break;
			case 0x02: strcpy(str,"OPERATOR SELECTED WRITE PROTECT"); break;
			case 0x03: strcpy(str,"OPERATOR SELECTED WRITE PERMIT"); break;
			}
			break;
		case 0x5B:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"LOG EXCEPTION"); break;
			case 0x01: strcpy(str,"THRESHOLD CONDITION MET"); break;
			case 0x02: strcpy(str,"LOG COUNTER AT MAXIMUM"); break;
			case 0x03: strcpy(str,"LOG LIST CODES EXHAUSTED"); break;
			}
			break;
		case 0x5E:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"LOW POWER CONDITION ON"); break;
			case 0x01: strcpy(str,"IDLE CONDITION ACTIVATED BY TIMER"); break;
			case 0x02: strcpy(str,"STANDBY CONDITION ACTIVATED BY TIMER"); break;
			case 0x03: strcpy(str,"IDLE CONDITION ACTIVATED BY COMMAND"); break;
			case 0x04: strcpy(str,"STANDBY CONDITION ACTIVATED BY COMMAND"); break;
			}
			break;
		}
		break;
	case 0x7:
		switch (ASC(err)) {
		case 0x27:
			switch (ASCQ(err)) {
			case 0x00: strcpy(str,"WRITE PROTECTED"); break;
			case 0x01: strcpy(str,"HARDWARE WRITE PROTECTED"); break;
			case 0x02: strcpy(str,"LOGICAL UNIT SOFTWARE WRITE PROTECTED"); break;
			case 0x03: strcpy(str,"ASSOCIATED WRITE PROTECT"); break;
			case 0x04: strcpy(str,"PERSISTENT WRITE PROTECT"); break;
			case 0x05: strcpy(str,"PERMANENT WRITE PROTECT"); break;
			case 0x06: strcpy(str,"CONDITIONAL WRITE PROTECT"); break;

			default:   strcpy(str,"WRITE PROTECTED"); break;
			}
			break;
		}
		break;
	case 0x8: strcpy(str,"BLANK CHECK"); break;
	case 0xB:
		switch (ASC(err)) {
		case 0x00: strcpy(str,"I/O PROCESS TERMINATED"); break; /* ASCQ = 06 */
        	case 0x11: strcpy(str,"READ ERROR - LOSS OF STREAMING"); break; /* ASCQ = 11 */
        	case 0x45: strcpy(str,"SELECT OR RESELECT FAILURE"); break; /* ASCQ = 00 */
        	case 0x48: strcpy(str,"INITIATOR DETECTED ERROR MESSAGE RECEIVED"); break; /* ASCQ = 00 */
        	case 0x49: strcpy(str,"INVALID MESSAGE ERROR"); break; /* ASCQ = 00 */
        	case 0x4D: strcpy(str,"TAGGED OVERLAPPED COMMANDS (NN = QUEUE TAG)"); break; /* ASCQ = xx */
		}
		break;
	}
	printf("[%05X]  %s", err, str);
	return 0;
}

int print_opcode (unsigned char opcode) {
	char str[128];
	switch (opcode) {

//	case 0x00: strcpy(str,"      "); break;
		case 0x00: strcpy(str,"MMC_TEST_UNIT_READY               "); break;
		case 0x03: strcpy(str,"MMC_REQUEST_SENSE                 "); break;
		case 0x04: strcpy(str,"MMC_FORMAT_UNIT                   "); break;
		case 0x12: strcpy(str,"MMC_INQUIRY                       "); break;
		case 0x15: strcpy(str,"MMC_MODE_SELECT6                  "); break;
		case 0x1A: strcpy(str,"MMC_MODE_SENSE6                   "); break;
		case 0x18: strcpy(str,"MMC_START_STOP_UNIT               "); break;
		case 0x1E: strcpy(str,"MMC_PREVENT_ALLOW_MEDIUM_REMIVAL  "); break;
		case 0x23: strcpy(str,"MMC_READ_FORMAT_CAPACITIES        "); break;
		case 0x25: strcpy(str,"MMC_READ_RECORDED_CAPACITY        "); break;
		case 0x28: strcpy(str,"MMC_READ                          "); break;
		case 0x2A: strcpy(str,"MMC_WRITE                         "); break;
		case 0x2B: strcpy(str,"MMC_SEEK                          "); break;
		case 0x2E: strcpy(str,"MMC_WRITE_AND_VERIFY              "); break;
		case 0x2F: strcpy(str,"MMC_VERIFY                        "); break;
		case 0x35: strcpy(str,"MMC_SYNC_CACHE                    "); break;
		case 0x42: strcpy(str,"MMC_READ_SUB_CHANNEL              "); break;
		case 0x43: strcpy(str,"MMC_READ_TOC_PMA_ATIP             "); break;
		case 0x44: strcpy(str,"MMC_READ_HEADER                   "); break;
		case 0x45: strcpy(str,"MMC_PLAY_AUDIO                    "); break;
		case 0x46: strcpy(str,"MMC_GET_CONFIGURATION             "); break;
		case 0x47: strcpy(str,"MMC_PLAY_AUDIO_MSF                "); break;
		case 0x4A: strcpy(str,"MMC_GET_EVENT_STATUS_NOTIFICATION "); break;
		case 0x4B: strcpy(str,"MMC_PAUSE_RESUME                  "); break;
		case 0x4E: strcpy(str,"MMC_STOP_PLAY_SCAN                "); break;
		case 0x51: strcpy(str,"MMC_READ_DISC_INFORMATION         "); break;
		case 0x52: strcpy(str,"MMC_READ_TRACK_INFORMATION        "); break;
		case 0x53: strcpy(str,"MMC_RESERVE_TRACK                 "); break;
		case 0x54: strcpy(str,"MMC_SEND_OPC_INFORMATION          "); break;
		case 0x55: strcpy(str,"MMC_MODE_SELECT10                 "); break;
		case 0x58: strcpy(str,"MMC_REPAIR_TRACK                  "); break;
		case 0x59: strcpy(str,"MMC_READ_MASTER_CUE               "); break;
		case 0x5A: strcpy(str,"MMC_MODE_SENSE10                  "); break;
		case 0x5B: strcpy(str,"MMC_CLOSE_TRACK_SESSION           "); break;
		case 0x5C: strcpy(str,"MMC_READ_BUFFER_CAPACITY          "); break;
		case 0x5D: strcpy(str,"MMC_SEND_CUE_SHEET                "); break;
		case 0xA1: strcpy(str,"MMC_BLANK                         "); break;
		case 0xA2: strcpy(str,"MMC_SEND_EVENT                    "); break;
		case 0xA3: strcpy(str,"MMC_SEND_KEY                      "); break;
		case 0xA4: strcpy(str,"MMC_REPORT_KEY                    "); break;
		case 0xA5: strcpy(str,"MMC_PLAY_AUDIO_12                 "); break;
		case 0xA6: strcpy(str,"MMC_LOAD_UNLOAD                   "); break;
		case 0xA8: strcpy(str,"MMC_READ_DVD                      "); break;
		case 0xAC: strcpy(str,"MMC_GET_PERFORMANCE               "); break;
		case 0xAD: strcpy(str,"MMC_READ_DVD_STRUCTURE            "); break;
		case 0xB6: strcpy(str,"MMC_SET_STREAMING                 "); break;
		case 0xB9: strcpy(str,"MMC_READ_CD_MSF                   "); break;
		case 0xBA: strcpy(str,"MMC_SCAN                          "); break;
		case 0xBB: strcpy(str,"MMC_SET_SPEED                     "); break;
		case 0xBC: strcpy(str,"MMC_PLAY_CD                       "); break;
		case 0xBD: strcpy(str,"MMC_MECHANISM_STATUS              "); break;
		case 0xBE: strcpy(str,"MMC_READ_CD                       "); break;

		case 0xD4: strcpy(str,"CMD_PLEX_GET_AUTH                 "); break;
		case 0xD5: strcpy(str,"CMD_PLEX_SEND_AUTH                "); break;

//const char  = 0xD8;

		case 0xE3: strcpy(str,"CMD_PLEX_ERASER                   "); break;
		case 0xE4: strcpy(str,"CMD_PLEX_AS_RD                    "); break;
		case 0xE5: strcpy(str,"CMD_PLEX_AS_WR                    "); break;

//const char FLUSH_CACHE = 0xE7;

		case 0xE9: strcpy(str,"CMD_PLEX_MODE                     "); break;
		case 0xEA: strcpy(str,"CMP_PLEX_QCHECK                   "); break;
		case 0xEB: strcpy(str,"CMD_PLEX_PREC_SPD                 "); break;
		case 0xED: strcpy(str,"CMD_PLEX_MODE2                    "); break;

		case 0xEE: strcpy(str,"CMD_PLEX     REBOOT ???           "); break;
		case 0xEF: strcpy(str,"CMD_PLEX     REBOOT ???           "); break;

		case 0xF1: strcpy(str,"CMD_PLEX_EEPROM_READ              "); break;

		case 0xF3: strcpy(str,"CMD_PLEX_SCAN_TA_FETE             "); break;
		case 0xF5: strcpy(str,"CMD_PLEX_FETE_READOUT             "); break;
		default:   strcpy(str,"*unknown*                         ");
	}
	printf("[%02X]  %s", opcode, str);
	return 0;
}
