/* 
   drvDXF.cpp : This file is part of pstoedit 

   Copyright (C) 1993 - 2001 Wolfgang Glunz, wglunz@pstoedit.net

	DXF Backend Version 0.9 ( LINEs only, no Text, no color, no linewidth )
	(see if polyaslines )
	Carsten Hammer	chammer@post.uni-bielefeld.de
   	CATS Gesellschaft fuer technische DV-Anwendungen mbH
	Carl-Zeiss-Strasse 65
	33334 Guetersloh

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "drvdxf.h"
#include I_stdio
#include I_string_h
#include I_iostream


// not used - the acad header also works with icad. later we may introduce an option
// which header to use #include "dxficad.h"
#include "dxfacad.h"

// try to use shorter headers
// but this doesn't work so far
static const char dxf14shortheader[] = 
			"  0\n"
			"SECTION\n"
			"2\nHEADER\n"
			"  9\n"
			"$ACADVER\n"
			"1\n"
			"AC1014\n"
			"  9\n"
			"$EXTMIN\n"
			"10\n"
			"0.0\n"
			"20\n"
			"0.0\n"
			"30\n"
			"0.0\n"
			"  9\n"
			"$EXTMAX\n"
			"10\n"
			"1000.0\n"
			"20\n"
			"1000.0\n"
			"30\n"
			"0.0\n"
			"  9\n"
			"$FILLMODE\n"
			"70\n"
			"0\n"
			"  9\n"
			"$SPLFRAME\n"
			"70\n"
			"1\n"
			"  0\n"
			"ENDSEC\n"
			"  0\n"
			"SECTION\n"
			"2\n"
			"TABLES\n"
// Layers
			"  0\n"
			"TABLE\n"
			"2\n"
			"LAYER\n"
			"70\n"
			"255\n"
			"  0\n"
			"LAYER\n"
			"2\n"
			"0\n"
			"70\n"
			"0\n"
			"62\n"
			"7\n"
			"6\n"
			"CONTINUOUS\n"
			"  0\n"
			"ENDTAB\n"
			"  0\n"
			"ENDSEC\n"
			"  0\n"
			"SECTION\n"
			"2\n"
			"ENTITIES\n";



static const char dxf14shorttrailer[] =
	"  0\n"
	"ENDSEC\n"
	"  0\n"
	"EOF\n";

// voloview doesn't like this header    outf << "999\nDXF generated by pstoedit\n";
static const char dxf9shortheader[] =
			"  0\n"
			"SECTION\n"
			"  2\n"
			"HEADER\n"
			"  9\n"
			"$ACADVER\n"
			"  1\n"
			"AC1009\n"
			"  9\n"
			"$EXTMIN\n"
			" 10\n"
			"0.0\n"
			" 20\n"
			"0.0\n"
			" 30\n"
			"0.0\n"
			"  9\n"
			"$EXTMAX\n"
			" 10\n"
			"1000.0\n"
			" 20\n"
			"1000.0\n"
			" 30\n"
			"0.0\n"
			"  9\n"
			"$FILLMODE\n"
			" 70\n"
			" 0\n"
			"  9\n"
			"$SPLFRAME\n"
			" 70\n"
			" 1\n"
			"  0\n"
			"ENDSEC\n"
			"  0\n"
			"SECTION\n"
			"  2\n"
			"TABLES\n"
// Layers
			"  0\n"
			"TABLE\n"
			"  2\n"
			"LAYER\n"
			" 70\n"
			"   255\n"
			"  0\n"
			"LAYER\n"
			"  2\n"
			"0\n"
			" 70\n"
			"     0\n"
			" 62\n"
			"     7\n"
			"  6\n"
			"CONTINUOUS\n"
			"  0\n"
			"ENDTAB\n"
			"  0\n"
			"ENDSEC\n"
			"  0\n"
			"SECTION\n"
			"  2\n"
			"ENTITIES\n";


static const char dxf9shorttrailer[] =
	"  0\n"
	"ENDSEC\n"
	"  0\n"
	"EOF\n";



class DXFColor {
public:
// The SparcCompiler wants to have this public in order to initialize DXFColors
	struct rgbcolor {
		unsigned short r;
		unsigned short g;
		unsigned short b;
	};

private:
	static const rgbcolor DXFColors[];
	static const unsigned short numberOfColors;
public:
	static unsigned int getDXFColor(float r, float g, float b);
};


unsigned int DXFColor::getDXFColor(float r, float g, float b)
{
	float mindist = 1.0f;
	unsigned int best = 0;
	for (unsigned int i = 0; i < numberOfColors; i++) {
		const float dxfr = DXFColors[i].r / 255.0f;
		const float dxfg = DXFColors[i].g / 255.0f;
		const float dxfb = DXFColors[i].b / 255.0f;
		const float dr = dxfr - r;
		const float dg = dxfg - g;
		const float db = dxfb - b;
		const float dist = dr * dr + dg * dg + db * db;
		if (dist == 0.0) {
//          errf << "Found exact match for " << r << " "
//              << g << " " << b << " " << i  << " " << numberOfColors << endl;
			return i;
		} else if (dist < mindist) {
			best = i;
			mindist = dist;
		}
	}
//  errf << "Found approximation for " << r << " "
//      << g << " " << b << " " << best  << " " << mindist  << " " << numberOfColors << endl;
	return best;
}

const DXFColor::rgbcolor DXFColor::DXFColors[] = {
	{0x00, 0x00, 0x00},			//0
	{0xFF, 0x00, 0x00},
	{0xFF, 0xFF, 0x00},
	{0x00, 0xFF, 0x00},
	{0x00, 0xFF, 0xFF},
	{0x00, 0x00, 0xFF},			//5
	{0xFF, 0x00, 0xFF},
	{0xFF, 0xFF, 0xFF},
	{0x41, 0x41, 0x41},
	{0x80, 0x80, 0x80},
	{0xFF, 0x00, 0x00},			//10
	{0xFF, 0xAA, 0xAA},
	{0xBD, 0x00, 0x00},
	{0xBD, 0x7E, 0x7E},
	{0x81, 0x00, 0x00},
	{0x81, 0x56, 0x56},			//15
	{0x68, 0x00, 0x00},
	{0x68, 0x45, 0x45},
	{0x4F, 0x00, 0x00},
	{0x4F, 0x35, 0x35},
	{0xFF, 0x3F, 0x00},			//20
	{0xFF, 0xBF, 0xAA},
	{0xBD, 0x2E, 0x00},
	{0xBD, 0x8D, 0x7E},
	{0x81, 0x1F, 0x00},
	{0x81, 0x60, 0x56},			//25
	{0x68, 0x19, 0x00},
	{0x68, 0x4E, 0x45},
	{0x4F, 0x13, 0x00},
	{0x4F, 0x3B, 0x35},
	{0xFF, 0x7F, 0x00},			//30
	{0xFF, 0xD4, 0xAA},
	{0xBD, 0x5E, 0x00},
	{0xBD, 0x9D, 0x7E},
	{0x81, 0x40, 0x00},
	{0x81, 0x6B, 0x56},			//35
	{0x68, 0x34, 0x00},
	{0x68, 0x56, 0x45},
	{0x4F, 0x27, 0x00},
	{0x4F, 0x42, 0x35},
	{0xFF, 0xBF, 0x00},			//40
	{0xFF, 0xEA, 0xAA},
	{0xBD, 0x8D, 0x00},
	{0xBD, 0xAD, 0x7E},
	{0x81, 0x60, 0x00},
	{0x81, 0x76, 0x56},			//45
	{0x68, 0x4E, 0x00},
	{0x68, 0x5F, 0x45},
	{0x4F, 0x3B, 0x00},
	{0x4F, 0x49, 0x35},
	{0xFF, 0xFF, 0x00},			//50
	{0xFF, 0xFF, 0xAA},
	{0xBD, 0xBD, 0x00},
	{0xBD, 0xBD, 0x7E},
	{0x81, 0x81, 0x00},
	{0x81, 0x81, 0x56},			//55
	{0x68, 0x68, 0x00},
	{0x68, 0x68, 0x45},
	{0x4F, 0x4F, 0x00},
	{0x4F, 0x4F, 0x35},
	{0xBF, 0xFF, 0x00},			//60
	{0xEA, 0xFF, 0xAA},
	{0x8D, 0xBD, 0x00},
	{0xAD, 0xBD, 0x7E},
	{0x60, 0x81, 0x00},
	{0x76, 0x81, 0x56},			//65
	{0x4E, 0x68, 0x00},
	{0x5F, 0x68, 0x45},
	{0x3B, 0x4F, 0x00},
	{0x49, 0x4F, 0x35},
	{0x7F, 0xFF, 0x00},			//70
	{0xD4, 0xFF, 0xAA},
	{0x5E, 0xBD, 0x00},
	{0x9D, 0xBD, 0x7E},
	{0x40, 0x81, 0x00},
	{0x6B, 0x81, 0x56},			//75
	{0x34, 0x68, 0x00},
	{0x56, 0x68, 0x45},
	{0x27, 0x4F, 0x00},
	{0x42, 0x4F, 0x35},
	{0x3F, 0xFF, 0x00},			//80
	{0xBF, 0xFF, 0xAA},
	{0x2E, 0xBD, 0x00},
	{0x8D, 0xBD, 0x7E},
	{0x1F, 0x81, 0x00},
	{0x60, 0x81, 0x56},			//85
	{0x19, 0x68, 0x00},
	{0x4E, 0x68, 0x45},
	{0x13, 0x4F, 0x00},
	{0x3B, 0x4F, 0x35},
	{0x00, 0xFF, 0x00},			//90
	{0xAA, 0xFF, 0xAA},
	{0x00, 0xBD, 0x00},
	{0x7E, 0xBD, 0x7E},
	{0x00, 0x81, 0x00},
	{0x56, 0x81, 0x56},			//95
	{0x00, 0x68, 0x00},
	{0x45, 0x68, 0x45},
	{0x00, 0x4F, 0x00},
	{0x35, 0x4F, 0x35},
	{0x00, 0xFF, 0x3F},			//100
	{0xAA, 0xFF, 0xBF},
	{0x00, 0xBD, 0x2E},
	{0x7E, 0xBD, 0x8D},
	{0x00, 0x81, 0x1F},
	{0x56, 0x81, 0x60},			//105
	{0x00, 0x68, 0x19},
	{0x45, 0x68, 0x4E},
	{0x00, 0x4F, 0x13},
	{0x35, 0x4F, 0x3B},
	{0x00, 0xFF, 0x7F},			//110
	{0xAA, 0xFF, 0xD4},
	{0x00, 0xBD, 0x5E},
	{0x7E, 0xBD, 0x9D},
	{0x00, 0x81, 0x40},
	{0x56, 0x81, 0x6B},			//115
	{0x00, 0x68, 0x34},
	{0x45, 0x68, 0x56},
	{0x00, 0x4F, 0x27},
	{0x35, 0x4F, 0x42},
	{0x00, 0xFF, 0xBF},			//120
	{0xAA, 0xFF, 0xEA},
	{0x00, 0xBD, 0x8D},
	{0x7E, 0xBD, 0xAD},
	{0x00, 0x81, 0x60},
	{0x56, 0x81, 0x76},			//125
	{0x00, 0x68, 0x4E},
	{0x45, 0x68, 0x5F},
	{0x00, 0x4F, 0x3B},
	{0x35, 0x4F, 0x49},
	{0x00, 0xFF, 0xFF},			//130
	{0xAA, 0xFF, 0xFF},
	{0x00, 0xBD, 0xBD},
	{0x7E, 0xBD, 0xBD},
	{0x00, 0x81, 0x81},
	{0x56, 0x81, 0x81},			//135
	{0x00, 0x68, 0x68},
	{0x45, 0x68, 0x68},
	{0x00, 0x4F, 0x4F},
	{0x35, 0x4F, 0x4F},
	{0x00, 0xBF, 0xFF},			//140
	{0xAA, 0xEA, 0xFF},
	{0x00, 0x8D, 0xBD},
	{0x7E, 0xAD, 0xBD},
	{0x00, 0x60, 0x81},
	{0x56, 0x76, 0x81},			//145
	{0x00, 0x4E, 0x68},
	{0x45, 0x5F, 0x68},
	{0x00, 0x3B, 0x4F},
	{0x35, 0x49, 0x4F},
	{0x00, 0x7F, 0xFF},			//150
	{0xAA, 0xD4, 0xFF},
	{0x00, 0x5E, 0xBD},
	{0x7E, 0x9D, 0xBD},
	{0x00, 0x40, 0x81},
	{0x56, 0x6B, 0x81},			//155
	{0x00, 0x34, 0x68},
	{0x45, 0x56, 0x68},
	{0x00, 0x27, 0x4F},
	{0x35, 0x42, 0x4F},
	{0x00, 0x3F, 0xFF},			//160
	{0xAA, 0xBF, 0xFF},
	{0x00, 0x2E, 0xBD},
	{0x7E, 0x8D, 0xBD},
	{0x00, 0x1F, 0x81},
	{0x56, 0x60, 0x81},			//165
	{0x00, 0x19, 0x68},
	{0x45, 0x4E, 0x68},
	{0x00, 0x13, 0x4F},
	{0x35, 0x3B, 0x4F},
	{0x00, 0x00, 0xFF},			//170
	{0xAA, 0xAA, 0xFF},
	{0x00, 0x00, 0xBD},
	{0x7E, 0x7E, 0xBD},
	{0x00, 0x00, 0x81},
	{0x56, 0x56, 0x81},			//175
	{0x00, 0x00, 0x68},
	{0x45, 0x45, 0x68},
	{0x00, 0x00, 0x4F},
	{0x35, 0x35, 0x4F},
	{0x3F, 0x00, 0xFF},			//180
	{0xBF, 0xAA, 0xFF},
	{0x2E, 0x00, 0xBD},
	{0x8D, 0x7E, 0xBD},
	{0x1F, 0x00, 0x81},
	{0x60, 0x56, 0x81},			//185
	{0x19, 0x00, 0x68},
	{0x4E, 0x45, 0x68},
	{0x13, 0x00, 0x4F},
	{0x3B, 0x35, 0x4F},
	{0x7F, 0x00, 0xFF},			//190
	{0xD4, 0xAA, 0xFF},
	{0x5E, 0x00, 0xBD},
	{0x9D, 0x7E, 0xBD},
	{0x40, 0x00, 0x81},
	{0x6B, 0x56, 0x81},			//195
	{0x34, 0x00, 0x68},
	{0x56, 0x45, 0x68},
	{0x27, 0x00, 0x4F},
	{0x42, 0x35, 0x4F},
	{0xBF, 0x00, 0xFF},			//200
	{0xEA, 0xAA, 0xFF},
	{0x8D, 0x00, 0xBD},
	{0xAD, 0x7E, 0xBD},
	{0x60, 0x00, 0x81},
	{0x76, 0x56, 0x81},			//205
	{0x4E, 0x00, 0x68},
	{0x5F, 0x45, 0x68},
	{0x3B, 0x00, 0x4F},
	{0x49, 0x35, 0x4F},
	{0xFF, 0x00, 0xFF},			//210
	{0xFF, 0xAA, 0xFF},
	{0xBD, 0x00, 0xBD},
	{0xBD, 0x7E, 0xBD},
	{0x81, 0x00, 0x81},
	{0x81, 0x56, 0x81},			//215
	{0x68, 0x00, 0x68},
	{0x68, 0x45, 0x68},
	{0x4F, 0x00, 0x4F},
	{0x4F, 0x35, 0x4F},
	{0xFF, 0x00, 0xBF},			//220
	{0xFF, 0xAA, 0xEA},
	{0xBD, 0x00, 0x8D},
	{0xBD, 0x7E, 0xAD},
	{0x81, 0x00, 0x60},
	{0x81, 0x56, 0x76},			//225
	{0x68, 0x00, 0x4E},
	{0x68, 0x45, 0x5F},
	{0x4F, 0x00, 0x3B},
	{0x4F, 0x35, 0x49},
	{0xFF, 0x00, 0x7F},			//230
	{0xFF, 0xAA, 0xD4},
	{0xBD, 0x00, 0x5E},
	{0xBD, 0x7E, 0x9D},
	{0x81, 0x00, 0x40},
	{0x81, 0x56, 0x6B},			//235
	{0x68, 0x00, 0x34},
	{0x68, 0x45, 0x56},
	{0x4F, 0x00, 0x27},
	{0x4F, 0x35, 0x42},
	{0xFF, 0x00, 0x3F},			//240
	{0xFF, 0xAA, 0xBF},
	{0xBD, 0x00, 0x2E},
	{0xBD, 0x7E, 0x8D},
	{0x81, 0x00, 0x1F},
	{0x81, 0x56, 0x60},			//245
	{0x68, 0x00, 0x19},
	{0x68, 0x45, 0x4E},
	{0x4F, 0x00, 0x13},
	{0x4F, 0x35, 0x3B},
	{0x33, 0x33, 0x33},			//250
	{0x50, 0x50, 0x50},
	{0x69, 0x69, 0x69},
	{0x82, 0x82, 0x82},
	{0xBE, 0xBE, 0xBE},
	{0xFF, 0xFF, 0xFF}			//255
};

const unsigned short DXFColor::numberOfColors =
	sizeof(DXFColor::DXFColors) / sizeof(DXFColor::rgbcolor);


static const OptionDescription driveroptions[] = {
	OptionDescription("-polyaslines","bool","use LINE instead of POLYLINE in DXF"),
	OptionDescription("-splineaspolyline","bool","approximate splines with PolyLines (only for -f dxf_s)"),
	OptionDescription("-splineasnurb","bool","experimental (only for -f dxf_s)"),
	OptionDescription("-splineasbspline","bool","experimental (only for -f dxf_s)"),
	OptionDescription("-splineassinglespline","bool","experimental (only for -f dxf_s)"),
	OptionDescription("-splineasmultispline","bool","experimental (only for -f dxf_s)"),
	OptionDescription("-splineasbezier","bool","use Bezier splines in DXF format (only for -f dxf_s)"),
	OptionDescription("-splineprecision","number",
	"number of samples to take from spline curve when doing approximation with -splineaspolyline or -splineasmultispline - should be >=2" ),
	endofoptions};

/* results: (25.12.2001) (v - VoloView ; i - IntelliCAD)
p2ed -f dxf_s:-splineasbezier 			curvetest.ps curve_bz.dxf v ok  i   ok
p2ed -f dxf_s:-splineaspolyline 		curvetest.ps curve_pl.dxf v ok  i   ok
#p2ed -f dxf_s:-splineasnurb 			curvetest.ps curve_nu.dxf v no  i   syntaktisch ja, aber die koordinaten werden falsch interpretiert (y als z)
#p2ed -f dxf_s:-splineasbspline 		curvetest.ps curve_bs.dxf v no  i   syntaktisch ja, aber die koordinaten werden falsch interpretiert (y als z)
#p2ed -f dxf_s:-splineassinglespline 	curvetest.ps curve_ss.dxf v no  i   syntaktisch ja, aber die koordinaten werden falsch interpretiert (y als z)
#p2ed -f dxf_s:-splineasmultispline 	curvetest.ps curve_ms.dxf v ok, but a bit strange  i same as acad
p2ed -f dxf 							curvetest.ps curve_np.dxf v ok  i   ok
p2ed -f dxf:-polyaslines 				curvetest.ps curve_nl.dxf v ok  i   ok
*/

drvDXF::derivedConstructor(drvDXF): 
	constructBase, 
	polyaslines(0), 
	splinemode(asbezier),
	formatis14(true),  // !!!! THESE TWO NEED TO BE CONSISTENT
	splineprecision(5),
#ifdef withshortheaders	
	header(dxf14shortheader),
	trailer(dxf14shorttrailer),
#else
	header(dxf14acadheader),
	trailer(dxf14acadtrailer),
#endif
	handleint(255)

{
	const bool withcurves = Pdriverdesc && Pdriverdesc->backendSupportsCurveto;
	if (!withcurves) { 
		formatis14 = false; // if no curves are generated by the frontent, no splines can occur and thus the old header is sufficient
		header  = dxf9shortheader;
		trailer = dxf9shorttrailer;
	}
	if (d_argc > 0) {
		if (verbose)
			errf << "% Driver options:" << endl;
		for (unsigned int i = 0; i < d_argc; i++) {
			assert(d_argv && d_argv[i]);
			if (verbose)
				errf << "% " << d_argv[i] << endl;
			if (strcmp(d_argv[i], "-polyaslines") == 0) {
				polyaslines = 1;
			} else if (strcmp(d_argv[i], "-splineasbezier") == 0) {
				splinemode = asbezier;
				formatis14 = true;
				if (!withcurves) errf << "Option -splineasbezier ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineaspolyline") == 0) {
				splinemode = aspolyline;	formatis14 = true;
				if (!withcurves) errf << "Option -splineaspolyline ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineasnurb") == 0) {
				splinemode = asnurb;		formatis14 = true;
				if (!withcurves) errf << "Option -splineasmultispline ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineasbspline") == 0) {
				splinemode = asbspline;		formatis14 = true;
				if (!withcurves) errf << "Option -splineasmultispline ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineasmultispline") == 0) {
				splinemode = asmultispline;	formatis14 = true;
				if (!withcurves) errf << "Option -splineasmultispline ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineassinglespline") == 0) {
				splinemode = assinglespline;formatis14 = true;
				if (!withcurves) errf << "Option -splineassinglespline ignored - only valid for dxf_c format" << endl;
			} else if (strcmp(d_argv[i], "-splineprecision") == 0) {
				i++;
				if (i >= d_argc) {
					errf << "-splineprecision requires a numeric argument >= 2 " << endl;
				} else {
					const int argi = atoi(d_argv[i]);
					if (argi < 2) {
						errf << "splineprecision argument should be >=2. Set to 2" << endl;
						splineprecision = 2;
					} else {
						splineprecision = (unsigned int) argi;
					}
				}
			} else {
				errf << "unknown option " << d_argv[i] << " ignored " << endl;
			}
		}
	}

	outf << header;

}

drvDXF::~drvDXF()
{
	outf << trailer;
	//lint -esym(1740,header) // no need to free header.
	//lint -esym(1740,trailer) // no need to free trailer.
	header = 0;
	trailer = 0;

}

void drvDXF::writehandle() {

	// VoloView needs the handled, intellicad doesn't need it but they don't harm
	outf << "  5\n" << hex << handleint << dec << endl;
	handleint++;
}


void drvDXF::close_page()
{
/*outf << "#Seite beendet.\n";*/
}

void drvDXF::open_page()
{
/*outf << "#Seite Nr. " << currentPageNumber << "\n";*/
}

void drvDXF::show_text(const TextInfo & textinfo)
{
	outf << "  0\n"
			"TEXT\n";

	if (formatis14) {
		writehandle();
		outf << "100\n"
				"AcDbEntity\n"
				"  8\n"
				"0\n"
				"100\n"
				"AcDbText\n";
	} else {
		outf << "  8\n"
				"0\n";
	}

	// color
	outf << " 62\n     " << DXFColor::getDXFColor(textinfo.currentR,
												  textinfo.currentG, 
												  textinfo.currentB)
		<< "\n";
	outf << " 10\n" << textinfo.x << "\n";
	outf << " 20\n" << textinfo.y << "\n";
	outf << " 30\n" << 0.0 << "\n";
	outf << " 40\n" << textinfo.currentFontSize << "\n";
	outf << "  1\n" << textinfo.thetext.value() << "\n";
	outf << " 50\n" << textinfo.currentFontAngle << "\n";
	if (formatis14) {
		outf << "100\n"
				"AcDbText\n";
	}
}

void drvDXF::printPoint(const Point & p, unsigned short add)
{
	outf << " " << add << "\n" << p.x_ << "\n";
	outf << " " << 10 + add << "\n" << p.y_ << "\n";
	outf << " " << 20 + add << "\n" << "0.0" << "\n";
}

void drvDXF::drawVertex(const Point & p, bool withlinewidth, int val70)
{

	outf << "  0\nVERTEX\n  8\n0\n";
	printPoint(p, 10);
	if (withlinewidth) {
		const float lineWidth = currentLineWidth();
		outf << " 40\n" << lineWidth << "\n 41\n" << lineWidth << "\n";
	}
	if (val70)					// used only for try with spline type polylines (which doesn't work anyway)
		outf << " 70\n    16\n";
}

void drvDXF::drawLine(const Point & start, const Point & end)
{

	outf << "  0\nLINE\n  8\n0\n";
	if (formatis14) {
		writehandle();
		//entity 
		outf <<	"100\n" "AcDbEntity\n";
		outf <<	"  8\n" "0\n" ;
		outf <<	"100\n" "AcDbLine" << endl;
	}
	// color
	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';
	printPoint(start, 10);
	printPoint(end, 11);
}

#if 0
0 LINE
	5
	64
	100
	AcDbEntity
	8 0 100 AcDbLine 10 - 13.272601 20 347.062801 30 0.0 11 89.636603 21 270.615964 31 0.0
#endif

#if 0
/* 
	Spline group codes Group codes Description 
100
 Subclass marker (AcDbSpline) 
 
210
 Normal vector (omitted if the spline is nonplanar)
DXF: X value; APP: 3D vector
 
220, 230
 DXF: Y and Z values of normal vector
 
70
 Spline flag (bit coded):
1 = Closed spline
2 = Periodic spline
4 = Rational spline
8 = Planar
16 = Linear (planar bit is also set) 
 
71
 Degree of the spline curve
 
72
 Number of knots
 
73
 Number of control points
 
74
 Number of fit points (if any)
 
42
 Knot tolerance (default = 0.0000001)
 
43
 Control-point tolerance (default = 0.0000001)
 
44
 Fit tolerance (default = 0.0000000001)
 
12
 Start tangent--may be omitted (in WCS).
DXF: X value; APP: 3D point.
 
22, 32
 DXF: Y and Z values of start tangent--may be omitted (in WCS).
 
13
 End tangent--may be omitted (in WCS).
DXF: X value; APP: 3D point.
 
23, 33
 DXF: Y and Z values of end tangent--may be omitted (in WCS)
 
40
 Knot value (one entry per knot)
 
41
 Weight (if not 1); with multiple group pairs, are present if all are not 1
 
10
 Control points (in WCS) one entry per control point.
DXF: X value; APP: 3D point
 
20, 30
 DXF: Y and Z values of control points (in WCS) (one entry per control point)
 
11
 Fit points (in WCS) one entry per fit point.
DXF: X value; APP: 3D point
 
21, 31
 DXF: Y and Z values of fit points (in WCS) (one entry per fit point)
 

*/

#endif


const unsigned short ClosedSpline	= 1 ; //= Closed spline
const unsigned short PeriodicSpline = 2 ; //= Periodic spline
const unsigned short RationalSpline = 4 ; //= Rational spline
const unsigned short PlanarSpline	= 8 ; //= Planar
const unsigned short LinearSpline	= 16 ; //= Linear (planar bit is also set) 


void drvDXF::writesplinetype(const unsigned short stype = 0) {
	outf << " 70\n     " << stype << "\n";	  /* spline type: */
}


void drvDXF::curvetoAsBezier(const basedrawingelement & elem, const Point & currentpoint)
{
	// single spline representation - 
	// each curveto is one SPLINE element
	// only the 4 points of the curveto are used
	//

	outf << "  0\nSPLINE\n";

	writehandle();
	outf <<"100\n" "AcDbEntity\n";

	outf << "  8\n0\n";			/* Layer */

	outf << "100\n" "AcDbSpline\n";

	outf << "210\n0.0\n220\n0.0\n230\n1.0\n";	/* Norm vector */

	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';

	writesplinetype( 0 );

	outf << " 71\n     3\n";            /* degree of curve */
	outf << " 72\n     8\n";            /* no of knots */
	outf << " 73\n" << 4 << "\n";	/* no of control points */

	
	// knot values 
	outf << " 40\n0.0\n"; 
	outf << " 40\n0.0\n";
	outf << " 40\n0.0\n";
	outf << " 40\n0.0\n";

	outf << " 40\n1.0\n"; 
	outf << " 40\n1.0\n";
	outf << " 40\n1.0\n";
	outf << " 40\n1.0\n";


	//  START_POINT(first).x, START_POINT(first).y);
	/* 10 control points
	   11 fit points
	   40 knot values
	 */
	const Point & cp1 = elem.getPoint(0);
	const Point & cp2 = elem.getPoint(1);
	const Point & ep  = elem.getPoint(2);

	// control points - same as the bezier points
	printPoint(currentpoint, 10);	
	printPoint(cp1, 10);		
	printPoint(cp2, 10);	
	printPoint(ep,  10);		

}

void drvDXF::curvetoAsNurb(const basedrawingelement & elem, const Point & currentpoint)
{
	// single spline representation - 
	// each curveto is one SPLINE element
	// only the 4 points of the curveto are used
	//
	// Here we use the mapping to Nurbs as described in Schneiders Nurbs tutorial
	// 
	// 
	outf << "  0\nSPLINE\n";
	writehandle();
	outf << "100\n" "AcDbEntity\n";

	outf << "  8\n0\n";			/* Layer */

	outf << "100\n" "AcDbSpline\n";

	outf << "210\n0.0\n220\n0.0\n230\n1.0\n";	/* Norm vector */

	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';

	writesplinetype( RationalSpline );

	outf << " 71\n     3\n";            /* degree of curve */
	outf << " 72\n     8\n";            /* no of knots */

	outf << " 73\n" << 4 << "\n";	/* no of control points */

	// knot values 
	outf << " 40\n0.0\n"; 
	outf << " 40\n0.0\n";
	outf << " 40\n0.0\n";
	outf << " 40\n0.0\n";

	outf << " 40\n1.0\n"; 
	outf << " 40\n1.0\n";
	outf << " 40\n1.0\n";
	outf << " 40\n1.0\n";


	//  START_POINT(first).x, START_POINT(first).y);
	/* 10 control points
	   11 fit points
	   40 knot values
	 */
	const Point & cp1 = elem.getPoint(0);
	const Point & cp2 = elem.getPoint(1);
	const Point & ep  = elem.getPoint(2);

	// control points - same as the bezier points
	printPoint(currentpoint, 10);	
	printPoint(cp1, 10);		
	printPoint(cp2, 10);	
	printPoint(ep,  10);		


}

void drvDXF::curvetoAsBSpline(const basedrawingelement & elem, const Point & currentpoint)
{
	// single spline representation - 
	// each curveto is one B-SPLINE element
	// only the 4 points of the curveto are used
	//
	// The Bezier points are transformed to the corresponding B-Spline control points.
	// 

	outf << "  0\nSPLINE\n";
	writehandle();
	outf << "100\n" "AcDbEntity\n";

	outf << "  8\n0\n";			/* Layer */

	outf << "100\n" "AcDbSpline\n";

	outf << "210\n0.0\n220\n0.0\n230\n1.0\n";	/* Norm vector */

	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';

	writesplinetype(RationalSpline);     
	outf << " 71\n     3\n";        /* degree of curve */
#define bsplinewithknots
#ifdef bsplinewithknots
	outf << " 72\n     8\n";	    /* no of knots */
#else
	outf << " 72\n     0\n";	    /* no of knots */
#endif
	outf << " 73\n" << 4 << "\n";	/*  no of control points */


#ifdef bsplinewithknots
		// knot values 
	outf << " 40\n0.0\n"; 
	outf << " 40\n1.0\n";
	outf << " 40\n2.0\n";
	outf << " 40\n3.0\n";

	outf << " 40\n4.0\n"; 
	outf << " 40\n5.0\n";
	outf << " 40\n6.0\n";
	outf << " 40\n7.0\n";

#endif



	/* 10 control points
	   11 fit points
	   40 knot values
	 */
	const Point & bp0 = currentpoint;
	const Point & bp1 = elem.getPoint(0);
	const Point & bp2 = elem.getPoint(1);
	const Point & bp3 = elem.getPoint(2);
/*
 bezier to bspline control point transformation matrix
 according to mail from Steve Noskow
6 -7 2 0
0 2 -1 0
0 -1 2 0
0 2 -7 6
*/
	//lint -save -e747
	//The conversions from int to float are OK here.
	const Point sp0 = (bp0* ( 6)) + (bp1* (-7)) + (bp2* ( 2)) ;
	const Point sp1 =               (bp1* ( 2)) + (bp2* (-1)) ;
	const Point sp2 =               (bp1* (-1)) + (bp2* ( 2)) ;
	const Point sp3 =               (bp1* ( 2)) + (bp2* (-7)) + (bp3* ( 6)) ;
	//lint -restore

		// b-spline control points 
	printPoint(sp0, 10);	
	printPoint(sp1, 10);		
	printPoint(sp2, 10);	
	printPoint(sp3, 10);	

}



void drvDXF::curvetoAsOneSpline(const basedrawingelement & elem, const Point & currentpoint)
{
	// single spline representation - 
	// each curveto is one SPLINE element
	// only the 4 points of the curveto are used
	//
	// this doesn's seem to work stable enough. It is unclear what are control points in DXF and
	// what are fit points. In all viewers I tried, the control points had no effect.
	// 


	// 
	outf << "  0\nSPLINE\n";
	writehandle();
	outf << "100\n" "AcDbEntity\n";

	outf << "  8\n0\n";			/* Layer */

	outf << "100\n" "AcDbSpline\n";

	outf << "210\n0.0\n220\n0.0\n230\n1.0\n";	/* Norm vector */


	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';

	writesplinetype(RationalSpline);
	/* 8: planar; 1 closed; 2 periodic; 16 linear */
	outf << " 71\n     3\n";             /* degree of curve */



#ifdef bsplinewithknots
	outf << " 72\n    10\n";	    /* no of knots */
			// knot values 
	outf << " 40\n0.0\n"; 
	outf << " 40\n1.0\n";
	outf << " 40\n2.0\n";
	outf << " 40\n3.0\n";

	outf << " 40\n4.0\n"; 
	outf << " 40\n5.0\n";
	outf << " 40\n6.0\n";
	outf << " 40\n7.0\n";
	outf << " 40\n8.0\n";
	outf << " 40\n9.0\n";

#else
	outf << " 72\n     0\n";	    /* no of knots */
#endif


// approach with the phantom nodes
		// (abcd)   ->   d1=(b-a);d2=(d-c) ; (a-d1),(a),(a+d1==b),(d-d2==c),(d),(d+d2)

	outf << " 73\n" << 6 << "\n";	  /* no of control points */


//	outf << " 74\n" << 2 << "\n";	  /* no of fit points */
//	outf << " 44\n0.0000000001\n";	//, 0.0000000001);           /* fit tolerance */


	/* 10 control points
	   11 fit points
	   40 knot values
	 */
	const Point & a = currentpoint;
	const Point & b = elem.getPoint(0);
	const Point & c = elem.getPoint(1);
	const Point & d = elem.getPoint(2);

	const Point delta1= b + (a * -1.0f);
	const Point delta2= d + (c * -1.0f);


	//  printPoint(cp1,12) ; // 12 22 32 // start tanget
	//  printPoint(cp2,13) ; // 13 23 33 // end tanget
	printPoint(a + (delta1 * -1.0f), 10);		// 12 22 32
	printPoint(a, 10);		// 13 23 33
	printPoint(b, 10);		// 13 23 33

	printPoint(c, 10);		// 12 22 32
	printPoint(d, 10);		// 13 23 33
	printPoint(d + delta2 , 10);		// 13 23 33

}

void drvDXF::curvetoAsMultiSpline(const basedrawingelement & elem, const Point & currentpoint)
{

	// multiple fit points on a single SPLINE
	const unsigned int fitpoints = splineprecision ; // 4;

	outf << "  0\nSPLINE\n";
	writehandle();
	outf <<"100\n" "AcDbEntity\n";
		

	outf << "  8\n0\n";			/* Layer */

	outf << "100\n" "AcDbSpline\n";

	outf << "210\n0.0\n220\n0.0\n230\n1.0\n";	/* Norm vector */


	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << '\n';

	writesplinetype( 0 );
	/* 8: planar; 1 closed; 2 periodic; 16 linear */
	outf << " 71\n     3\n";	// , 3);             /* degree of curve */
	outf << " 72\n     0\n";	//, 0);             /* no of knots */
	outf << " 73\n" << 0 << "\n";	// ????????? , SPLINE_LIST_LENGTH (list));  /* no of control points */
	outf << " 74\n" << fitpoints << "\n";	// , SPLINE_LIST_LENGTH (list));  /* no of fit points */


//wogl                      outf << " 42\n0.0000001\n"; // , 0.0000001);                /* knot tolerance */
//wogl                      outf << " 43\n0.0000001\n"; //, 0.0000001);             /* control-point tolerance */
	outf << " 44\n0.0000000001\n";	//, 0.0000000001);           /* fit tolerance */

	//  START_POINT(first).x, START_POINT(first).y);
	/* 10 control points
	   11 fit points
	   40 knot values
	 */
	const Point & cp1 = elem.getPoint(0);
	const Point & cp2 = elem.getPoint(1);
	const Point & ep = elem.getPoint(2);

	for (unsigned int s = 0; s < fitpoints; s++) {
		const float t = 1.0f * s / (fitpoints - 1);
		const Point pt = PointOnBezier(t, currentpoint, cp1, cp2, ep);
		printPoint(pt, 11);		// 12 22 32   fit points
	}


}

void drvDXF::curvetoAsPolyLine(const basedrawingelement & elem, const Point & currentpoint)
{
	// approximate spline with lines.
	const unsigned int sections = splineprecision ; // 20;

	outf << "  0\nLWPOLYLINE\n";
	writehandle();
	outf << "100\n"
			"AcDbEntity\n";
	outf << "  8\n0\n";
	outf << "100\n"
			"AcDbPolyline\n";
	outf << " 90\n"; // number of edges
	outf << (sections+1) << endl;
	outf << " 70\n"
			" 0\n";

	// color
	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << "\n";

	// the following lines marked with nolw are not needed with the lwpolyline object.
	// vertex flag (always 1)
//nolw	outf << " 66\n     1\n";
	// base point
	// outf << " 10\n0.0\n 20\n0.0\n 30\n0.0\n";
//nolw	printPoint(Point(0.0f,0.0f), 10);

	const Point & cp1 = elem.getPoint(0);
	const Point & cp2 = elem.getPoint(1);
	const Point & ep = elem.getPoint(2);

	for (unsigned int s = 0; s <= sections; s++) {
		const float t = 1.0f * s / sections;
		const Point pt = PointOnBezier(t, currentpoint, cp1, cp2, ep);
		//nolw drawVertex(pt, false);
		printPoint(pt,10);
	}
//nolw	outf << "  0\nSEQEND\n  8\n0\n";
}

#if 0
const polysplines = false;		// old try using polyline and splines - but doesn't seem to work
					// Polylines with type spline seem to have as well the approximating lines.

					// draw them just as polygon through the edge and control points (I know, this is not correct)

if (0) {
	outf << "  0\nPOLYLINE\n  8\n0\n";
	// color
	outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB()) << "\n";
	// vertex flag (always 1)
	outf << " 66\n     1\n";
	// base point
	printPoint(Point(0.0f,0.0f) 10));
	// outf << " 10\n0.0\n 20\n0.0\n 30\n0.0\n";
	if (polysplines) {
		// start and end line width
		outf << " 40\n" << lineWidth << "\n 41\n" << lineWidth << "\n";
		// polytype flag : always open curve
		outf << " 70\n     4\n";
		// degree 5 means spline of degree 3
		outf << " 75\n     5\n";
	}
	drawVertex(currentPoint, polysplines, polysplines ? 48 : 0);
	for (unsigned int cp = 0; cp < 3; cp++) {
		const Point & p = elem.getPoint(cp);
		drawVertex(p, polysplines, polysplines ? 16 : 0);
		currentPoint = p;
	}
	outf << "  0\nSEQEND\n  8\n0\n";

}
#endif

void drvDXF::show_path()
{
	if (Pdriverdesc && Pdriverdesc->backendSupportsCurveto) {
		Point currentPoint(0.0f, 0.0f);
		const Point firstPoint = pathElement(0).getPoint(0);
		for (unsigned int n = 0; n < numberOfElementsInPath(); n++) {
			const basedrawingelement & elem = pathElement(n);
			switch (elem.getType()) {
			case moveto:{
					const Point & p = elem.getPoint(0);
					currentPoint = p;
				}
				break;
			case lineto:{
					const Point & p = elem.getPoint(0);
					drawLine(currentPoint, p);
					currentPoint = p;
				}
				break;
			case closepath:
				drawLine(currentPoint, firstPoint);
				break;
			case curveto:{
					switch (splinemode) {
					case aspolyline:
						curvetoAsPolyLine(elem, currentPoint);
						break;
					case assinglespline:
						curvetoAsOneSpline(elem, currentPoint);
						break;
					case asmultispline:
						curvetoAsMultiSpline(elem, currentPoint);
						break;
					case asnurb:
						curvetoAsNurb(elem, currentPoint);
						break;
					case asbspline:
						curvetoAsBSpline(elem, currentPoint);
						break;
					case asbezier:
						curvetoAsBezier(elem, currentPoint);
						break;

					}
					currentPoint = elem.getPoint(2);
				}
				break;
			default:
				errf << "\t\tFatal: unexpected case in drvdxf " << endl;
				abort();
				break;
			}					// end switch
		}						// end for
	} else if (polyaslines) {
// Curveto not supported

// not used, since lines can't have a width
//             ||
		// it's just a simple line
//             ((numberOfElementsInPath() == 2)
//              && (pathElement(0).getType() == moveto)
//              && (pathElement(1).getType() == lineto))


		for (unsigned int t = 1; t < numberOfElementsInPath(); t++) {
			const Point & p = pathElement(t - 1).getPoint(0);
			const Point & q = pathElement(t).getPoint(0);
			drawLine(p, q);
		}
	} else {
		// no curveto and not forced to draw LINEs - use PolyLine/VERTEX then
		const float lineWidth = currentLineWidth();
		outf << "  0\nPOLYLINE\n";

		// layer
		outf << "8\n0\n";
		// color
		outf << " 62\n     " << DXFColor::getDXFColor(currentR(), currentG(), currentB())
			<< "\n";
		// vertex flag (always 1)
		outf << " 66\n     1\n";
		// base point
		printPoint(Point(0.0f,0.0f), 10);
		// outf << " 10\n0.0\n 20\n0.0\n 30\n0.0\n";
		if (isPolygon()) {
			outf << " 70\n     1\n";
		}
		// start and end line width
		outf << " 40\n" << lineWidth << "\n 41\n" << lineWidth << "\n";
		for (unsigned int t = 0; t < numberOfElementsInPath(); t++) {
			const Point & p = pathElement(t).getPoint(0);
			drawVertex(p, true, 0);
		}
		outf << "  0\nSEQEND\n  8\n0\n";
	}
}

void drvDXF::show_rectangle(const float llx, const float lly, const float urx, const float ury)
{
// just do show_polyline for a first guess
	unused(&llx);
	unused(&lly);
	unused(&urx);
	unused(&ury);
	show_path();
}

static DriverDescriptionT < drvDXF > D_dxf("dxf", "CAD exchange format", "dxf", false,	// if backend supports subpathes, else 0
										   // if subpathes are supported, the backend must deal with
										   // sequences of the following form
										   // moveto (start of subpath)
										   // lineto (a line segment)
										   // lineto 
										   // moveto (start of a new subpath)
										   // lineto (a line segment)
										   // lineto 
										   //
										   // If this argument is set to 0 each subpath is drawn 
										   // individually which might not necessarily represent
										   // the original drawing.
										   false,	// if backend supports curves, else 0
										   false,	// if backend supports elements with fill and edges
										   true,	// if backend supports text, else 0
										   false,	// if backend supports Images
										   false,	// no support for PNG file images
										   DriverDescription::normalopen, false,	// if format supports multiple pages in one file
										   false /*clipping */ ,
										   driveroptions);

static DriverDescriptionT < drvDXF > D_dxf_s("dxf_s", "CAD exchange format with splines", "dxf", false,	// if backend supports subpathes, else 0
											 // if subpathes are supported, the backend must deal with
											 // sequences of the following form
											 // moveto (start of subpath)
											 // lineto (a line segment)
											 // lineto 
											 // moveto (start of a new subpath)
											 // lineto (a line segment)
											 // lineto 
											 //
											 // If this argument is set to 0 each subpath is drawn 
											 // individually which might not necessarily represent
											 // the original drawing.
											 true,	// if backend supports curves, else 0
											 false,	// if backend supports elements with fill and edges
											 true,	// if backend supports text, else 0
											 false,	// if backend supports Images
											 false,	// no support for PNG file images
											 DriverDescription::normalopen, false,	// if format supports multiple pages in one file
											 false /*clipping */ ,
											 driveroptions);
 
 
 
