/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

#include <libmints/vector.h>
#include "P86_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

P86_CFunctional::P86_CFunctional()
{
    name_ = "P86_C";
    description_ = "    P86 Correlation (PZ81 LSDA + P86 GGA)\n";
    citation_ = "    J.P. Perdew, Phys. Rev. B., 33, 8822-8824, 1986\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = true;
    meta_ = false;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["two_13"] =   1.2599210498948732E+00;
    parameters_["EcPld_1"] =  -1.4230000000000001E-01;
    parameters_["EcPld_2"] =   1.0528999999999999E+00;
    parameters_["EcPld_3"] =   3.3339999999999997E-01;
    parameters_["EcFld_1"] =  -8.4300000000000000E-02;
    parameters_["EcFld_2"] =   1.3980999999999999E+00;
    parameters_["EcFld_3"] =   2.6110000000000000E-01;
    parameters_["EcPhd_1"] =   3.1099999999999999E-02;
    parameters_["EcPhd_2"] =  -4.8000000000000001E-02;
    parameters_["EcPhd_3"] =   2.0000000000000000E-03;
    parameters_["EcPhd_4"] =  -1.1599999999999999E-02;
    parameters_["EcFhd_1"] =   1.5550000000000000E-02;
    parameters_["EcFhd_2"] =  -2.6900000000000000E-02;
    parameters_["EcFhd_3"] =   6.9999999999999999E-04;
    parameters_["EcFhd_4"] =  -4.7999999999999996E-03;
    parameters_["Fg"] =   1.1000000000000000E-01;
    parameters_["Bg"] =   7.3889999999999999E-06;
    parameters_["Cx"] =   1.6670000000000001E-03;
    parameters_["Cinf"] =   4.2350000000000000E-03;
    parameters_["Cg_1"] =   2.5680000000000000E-03;
    parameters_["Cg_2"] =   2.3265999999999998E-02;
    parameters_["Cg_3"] =   8.7230000000000008E+00;
    parameters_["Cg_4"] =   4.7199999999999998E-01;
    parameters_["Pg_1"] =   1.7450000000000001E+00;
}
P86_CFunctional::~P86_CFunctional()
{
}
void P86_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c = parameters_["c"];
    double two_13 = parameters_["two_13"];
    double EcPld_1 = parameters_["EcPld_1"];
    double EcPld_2 = parameters_["EcPld_2"];
    double EcPld_3 = parameters_["EcPld_3"];
    double EcFld_1 = parameters_["EcFld_1"];
    double EcFld_2 = parameters_["EcFld_2"];
    double EcFld_3 = parameters_["EcFld_3"];
    double EcPhd_1 = parameters_["EcPhd_1"];
    double EcPhd_2 = parameters_["EcPhd_2"];
    double EcPhd_3 = parameters_["EcPhd_3"];
    double EcPhd_4 = parameters_["EcPhd_4"];
    double EcFhd_1 = parameters_["EcFhd_1"];
    double EcFhd_2 = parameters_["EcFhd_2"];
    double EcFhd_3 = parameters_["EcFhd_3"];
    double EcFhd_4 = parameters_["EcFhd_4"];
    double Fg = parameters_["Fg"];
    double Bg = parameters_["Bg"];
    double Cx = parameters_["Cx"];
    double Cinf = parameters_["Cinf"];
    double Cg_1 = parameters_["Cg_1"];
    double Cg_2 = parameters_["Cg_2"];
    double Cg_3 = parameters_["Cg_3"];
    double Cg_4 = parameters_["Cg_4"];
    double Pg_1 = parameters_["Pg_1"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {  
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    } 
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;
     
    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    } 
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {    
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables 
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }        
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }        
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }        

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t4882 = rho_a+rho_b;
                double t4883 = 1.0/pow(t4882,1.0/3.0);
                double t4884 = c*t4883;
                double t4885 = log(t4884);
                double t4886 = EcPhd_1*t4885;
                double t4887 = pow(2.0,1.0/3.0);
                double t4888 = t4887*2.0;
                double t4889 = t4888-2.0;
                double t4890 = two_13*2.0;
                double t4891 = t4890-2.0;
                double t4892 = 1.0/t4891;
                double t4893 = sqrt(t4884);
                double t4894 = EcPld_2*t4893;
                double t4895 = EcPld_3*c*t4883;
                double t4896 = t4894+t4895+1.0;
                double t4897 = 1.0/t4896;
                double t4898 = EcPld_1*t4897;
                double t4899 = c*c;
                double t4900 = 1.0/pow(t4882,2.0/3.0);
                double t4901 = Bg*t4899*t4900;
                double t4902 = Cg_2*c*t4883;
                double t4903 = Cg_1+t4901+t4902;
                double t4904 = 1.0/t4882;
                double t4905 = Bg*c*t4899*t4904*1.0E4;
                double t4906 = Cg_4*t4899*t4900;
                double t4907 = Cg_3*c*t4883;
                double t4908 = t4905+t4906+t4907+1.0;
                double t4909 = 1.0/t4908;
                double t4910 = t4903*t4909;
                double t4911 = Cx+t4910;
                double t4912 = gamma_ab*2.0;
                double t4913 = gamma_aa+gamma_bb+t4912;
                v[Q] += scale * (t4882*(heaviside(-c*t4883+1.0)*(EcPhd_2+t4886+t4892*t4889*(EcFhd_2-EcPhd_2-t4886+EcFhd_1*t4885+EcFhd_4*c*t4883-EcPhd_4*c*t4883+EcFhd_3*c*t4883*t4885-EcPhd_3*c*t4883*t4885)+EcPhd_4*c*t4883+EcPhd_3*c*t4883*t4885)+heaviside(t4884-1.0)*(t4898-t4892*t4889*(t4898-EcFld_1/(EcFld_2*t4893+EcFld_3*c*t4883+1.0))))+(1.0/pow(t4882,4.0/3.0)*t4911*t4913*exp(-(Cinf*Fg*Pg_1*1.0/pow(t4882,7.0/6.0)*sqrt(t4913))/t4911))/two_13);
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t4915 = rho_a+rho_b;
                double t4916 = 1.0/pow(t4915,4.0/3.0);
                double t4917 = 1.0/pow(t4915,1.0/3.0);
                double t4918 = c*t4917;
                double t4919 = 1.0/sqrt(t4918);
                double t4920 = sqrt(t4918);
                double t4921 = EcPld_3*c*t4916*(1.0/3.0);
                double t4922 = EcPld_2*c*t4916*t4919*(1.0/6.0);
                double t4923 = t4921+t4922;
                double t4924 = EcPld_2*t4920;
                double t4925 = EcPld_3*c*t4917;
                double t4926 = t4924+t4925+1.0;
                double t4927 = 1.0/(t4926*t4926);
                double t4928 = EcPld_1*t4923*t4927;
                double t4929 = pow(2.0,1.0/3.0);
                double t4930 = t4929*2.0;
                double t4931 = t4930-2.0;
                double t4932 = two_13*2.0;
                double t4933 = t4932-2.0;
                double t4934 = 1.0/t4933;
                double t4935 = 1.0/t4915;
                double t4936 = EcPhd_1*t4935*(1.0/3.0);
                double t4937 = log(t4918);
                double t4938 = EcPhd_3*c*t4916*(1.0/3.0);
                double t4939 = EcPhd_4*c*t4916*(1.0/3.0);
                double t4940 = EcPhd_3*c*t4916*t4937*(1.0/3.0);
                double t4941 = t4918-1.0;
                double t4942 = EcPhd_1*t4937;
                double t4943 = dirac(t4941);
                double t4944 = EcFld_2*t4920;
                double t4945 = EcFld_3*c*t4917;
                double t4946 = t4944+t4945+1.0;
                double t4947 = 1.0/t4926;
                double t4948 = EcPld_1*t4947;
                double t4949 = -t4918+1.0;
                double t4950 = heaviside(t4949);
                double t4951 = EcFhd_1*t4937;
                double t4952 = EcFhd_4*c*t4917;
                double t4953 = EcPhd_4*c*t4917;
                double t4954 = EcFhd_3*c*t4917*t4937;
                double t4955 = EcPhd_3*c*t4917*t4937;
                double t4956 = heaviside(t4941);
                double t4957 = 1.0/t4946;
                double t4958 = t4948-EcFld_1*t4957;
                double t4959 = t4948-t4931*t4934*t4958;
                double t4960 = c*c;
                double t4961 = 1.0/pow(t4915,2.0/3.0);
                double t4962 = Bg*c*t4960*t4935*1.0E4;
                double t4963 = Cg_4*t4960*t4961;
                double t4964 = Cg_3*c*t4917;
                double t4965 = t4962+t4963+t4964+1.0;
                double t4966 = 1.0/t4965;
                double t4967 = 1.0/pow(t4915,5.0/3.0);
                double t4968 = Bg*t4960*t4961;
                double t4969 = Cg_2*c*t4917;
                double t4970 = Cg_1+t4968+t4969;
                double t4971 = gamma_ab*2.0;
                double t4972 = gamma_aa+gamma_bb+t4971;
                double t4973 = 1.0/two_13;
                double t4974 = 1.0/pow(t4915,7.0/6.0);
                double t4975 = t4970*t4966;
                double t4976 = Cx+t4975;
                double t4977 = 1.0/t4976;
                double t4978 = sqrt(t4972);
                double t4980 = Cinf*Fg*Pg_1*t4974*t4977*t4978;
                double t4979 = exp(-t4980);
                double t4981 = Bg*t4960*t4967*(2.0/3.0);
                double t4982 = Cg_2*c*t4916*(1.0/3.0);
                double t4983 = t4981+t4982;
                double t4984 = t4983*t4966;
                double t4985 = 1.0/(t4915*t4915);
                double t4986 = Bg*c*t4960*t4985*1.0E4;
                double t4987 = Cg_4*t4960*t4967*(2.0/3.0);
                double t4988 = Cg_3*c*t4916*(1.0/3.0);
                double t4989 = t4986+t4987+t4988;
                double t4990 = 1.0/(t4965*t4965);
                double t4991 = t4984-t4970*t4990*t4989;
                v_rho_a[Q] += scale * (t4950*(EcPhd_2+t4942+t4953+t4955+t4931*t4934*(EcFhd_2-EcPhd_2-t4942+t4951+t4952-t4953+t4954-t4955))+t4956*t4959+t4915*(t4956*(t4928-t4931*t4934*(t4928-EcFld_1*1.0/(t4946*t4946)*(EcFld_3*c*t4916*(1.0/3.0)+EcFld_2*c*t4916*t4919*(1.0/6.0))))-t4950*(t4940+t4936+t4938+t4939-t4931*t4934*(t4940+t4936+t4938+t4939-EcFhd_1*t4935*(1.0/3.0)-EcFhd_3*c*t4916*(1.0/3.0)-EcFhd_4*c*t4916*(1.0/3.0)-EcFhd_3*c*t4916*t4937*(1.0/3.0)))-c*t4916*t4943*t4959*(1.0/3.0)+c*t4916*t4943*(EcPhd_2+t4942+t4953+t4955+t4931*t4934*(EcFhd_2-EcPhd_2-t4942+t4951+t4952+t4954-EcPhd_4*c*t4917-EcPhd_3*c*t4917*t4937))*(1.0/3.0))-1.0/pow(t4915,7.0/3.0)*t4972*t4973*t4976*t4979*(4.0/3.0)-t4916*t4972*t4973*t4991*t4979+t4916*t4972*t4973*t4976*t4979*(Cinf*Fg*Pg_1*1.0/pow(t4915,1.3E1/6.0)*t4977*t4978*(7.0/6.0)-Cinf*Fg*Pg_1*t4991*t4974*1.0/(t4976*t4976)*t4978));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t4993 = rho_a+rho_b;
                double t4994 = 1.0/pow(t4993,4.0/3.0);
                double t4995 = 1.0/pow(t4993,1.0/3.0);
                double t4996 = c*t4995;
                double t4997 = 1.0/sqrt(t4996);
                double t4998 = sqrt(t4996);
                double t4999 = EcPld_3*c*t4994*(1.0/3.0);
                double t5000 = EcPld_2*c*t4994*t4997*(1.0/6.0);
                double t5001 = t4999+t5000;
                double t5002 = EcPld_2*t4998;
                double t5003 = EcPld_3*c*t4995;
                double t5004 = t5002+t5003+1.0;
                double t5005 = 1.0/(t5004*t5004);
                double t5006 = EcPld_1*t5001*t5005;
                double t5007 = pow(2.0,1.0/3.0);
                double t5008 = t5007*2.0;
                double t5009 = t5008-2.0;
                double t5010 = two_13*2.0;
                double t5011 = t5010-2.0;
                double t5012 = 1.0/t5011;
                double t5013 = 1.0/t4993;
                double t5014 = EcPhd_1*t5013*(1.0/3.0);
                double t5015 = log(t4996);
                double t5016 = EcPhd_3*c*t4994*(1.0/3.0);
                double t5017 = EcPhd_4*c*t4994*(1.0/3.0);
                double t5018 = EcPhd_3*c*t4994*t5015*(1.0/3.0);
                double t5019 = t4996-1.0;
                double t5020 = EcPhd_1*t5015;
                double t5021 = dirac(t5019);
                double t5022 = EcFld_2*t4998;
                double t5023 = EcFld_3*c*t4995;
                double t5024 = t5022+t5023+1.0;
                double t5025 = 1.0/t5004;
                double t5026 = EcPld_1*t5025;
                double t5027 = -t4996+1.0;
                double t5028 = heaviside(t5027);
                double t5029 = EcFhd_1*t5015;
                double t5030 = EcFhd_4*c*t4995;
                double t5031 = EcPhd_4*c*t4995;
                double t5032 = EcFhd_3*c*t4995*t5015;
                double t5033 = EcPhd_3*c*t4995*t5015;
                double t5034 = heaviside(t5019);
                double t5035 = 1.0/t5024;
                double t5036 = t5026-EcFld_1*t5035;
                double t5037 = t5026-t5012*t5009*t5036;
                double t5038 = c*c;
                double t5039 = 1.0/pow(t4993,2.0/3.0);
                double t5040 = Bg*c*t5013*t5038*1.0E4;
                double t5041 = Cg_4*t5038*t5039;
                double t5042 = Cg_3*c*t4995;
                double t5043 = t5040+t5041+t5042+1.0;
                double t5044 = 1.0/t5043;
                double t5045 = 1.0/pow(t4993,5.0/3.0);
                double t5046 = Bg*t5038*t5039;
                double t5047 = Cg_2*c*t4995;
                double t5048 = Cg_1+t5046+t5047;
                double t5049 = gamma_ab*2.0;
                double t5050 = gamma_aa+gamma_bb+t5049;
                double t5051 = 1.0/two_13;
                double t5052 = 1.0/pow(t4993,7.0/6.0);
                double t5053 = t5044*t5048;
                double t5054 = Cx+t5053;
                double t5055 = 1.0/t5054;
                double t5056 = sqrt(t5050);
                double t5058 = Cinf*Fg*Pg_1*t5052*t5055*t5056;
                double t5057 = exp(-t5058);
                double t5059 = Bg*t5045*t5038*(2.0/3.0);
                double t5060 = Cg_2*c*t4994*(1.0/3.0);
                double t5061 = t5060+t5059;
                double t5062 = t5061*t5044;
                double t5063 = 1.0/(t4993*t4993);
                double t5064 = Bg*c*t5063*t5038*1.0E4;
                double t5065 = Cg_4*t5045*t5038*(2.0/3.0);
                double t5066 = Cg_3*c*t4994*(1.0/3.0);
                double t5067 = t5064+t5065+t5066;
                double t5068 = 1.0/(t5043*t5043);
                double t5069 = t5062-t5048*t5067*t5068;
                v_rho_b[Q] += scale * (t5028*(EcPhd_2+t5020+t5031+t5033+t5012*t5009*(EcFhd_2-EcPhd_2-t5020+t5030-t5031+t5032-t5033+t5029))+t5034*t5037+t4993*(t5034*(t5006-t5012*t5009*(t5006-EcFld_1*1.0/(t5024*t5024)*(EcFld_3*c*t4994*(1.0/3.0)+EcFld_2*c*t4994*t4997*(1.0/6.0))))-t5028*(t5014+t5016+t5017+t5018-t5012*t5009*(t5014+t5016+t5017+t5018-EcFhd_1*t5013*(1.0/3.0)-EcFhd_3*c*t4994*(1.0/3.0)-EcFhd_4*c*t4994*(1.0/3.0)-EcFhd_3*c*t4994*t5015*(1.0/3.0)))-c*t4994*t5021*t5037*(1.0/3.0)+c*t4994*t5021*(EcPhd_2+t5020+t5031+t5033+t5012*t5009*(EcFhd_2-EcPhd_2-t5020+t5030+t5032+t5029-EcPhd_4*c*t4995-EcPhd_3*c*t4995*t5015))*(1.0/3.0))-1.0/pow(t4993,7.0/3.0)*t5050*t5051*t5054*t5057*(4.0/3.0)-t4994*t5050*t5051*t5057*t5069+t4994*t5050*t5051*t5054*t5057*(Cinf*Fg*Pg_1*1.0/pow(t4993,1.3E1/6.0)*t5055*t5056*(7.0/6.0)-Cinf*Fg*Pg_1*t5052*1.0/(t5054*t5054)*t5056*t5069));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t5071 = rho_a+rho_b;
                double t5072 = c*c;
                double t5073 = 1.0/pow(t5071,2.0/3.0);
                double t5074 = 1.0/pow(t5071,1.0/3.0);
                double t5075 = Bg*t5072*t5073;
                double t5076 = Cg_2*c*t5074;
                double t5077 = Cg_1+t5075+t5076;
                double t5078 = 1.0/t5071;
                double t5079 = Bg*c*t5072*t5078*1.0E4;
                double t5080 = Cg_4*t5072*t5073;
                double t5081 = Cg_3*c*t5074;
                double t5082 = t5080+t5081+t5079+1.0;
                double t5083 = 1.0/t5082;
                double t5084 = t5083*t5077;
                double t5085 = Cx+t5084;
                double t5086 = 1.0/two_13;
                double t5087 = 1.0/pow(t5071,7.0/6.0);
                double t5088 = 1.0/t5085;
                double t5089 = gamma_ab*2.0;
                double t5090 = gamma_aa+gamma_bb+t5089;
                double t5091 = sqrt(t5090);
                double t5092 = exp(-Cinf*Fg*Pg_1*t5091*t5087*t5088);
                v_gamma_aa[Q] += scale * (1.0/pow(t5071,4.0/3.0)*t5092*t5085*t5086-Cinf*Fg*Pg_1*1.0/pow(t5071,5.0/2.0)*t5091*t5092*t5086*(1.0/2.0));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t5094 = rho_a+rho_b;
                double t5095 = c*c;
                double t5096 = 1.0/pow(t5094,2.0/3.0);
                double t5097 = 1.0/pow(t5094,1.0/3.0);
                double t5098 = Bg*t5095*t5096;
                double t5099 = Cg_2*c*t5097;
                double t5100 = Cg_1+t5098+t5099;
                double t5101 = 1.0/t5094;
                double t5102 = Bg*c*t5095*t5101*1.0E4;
                double t5103 = Cg_4*t5095*t5096;
                double t5104 = Cg_3*c*t5097;
                double t5105 = t5102+t5103+t5104+1.0;
                double t5106 = 1.0/t5105;
                double t5107 = t5100*t5106;
                double t5108 = Cx+t5107;
                double t5109 = 1.0/two_13;
                double t5110 = 1.0/pow(t5094,7.0/6.0);
                double t5111 = 1.0/t5108;
                double t5112 = gamma_ab*2.0;
                double t5113 = gamma_aa+gamma_bb+t5112;
                double t5114 = sqrt(t5113);
                double t5115 = exp(-Cinf*Fg*Pg_1*t5110*t5111*t5114);
                v_gamma_ab[Q] += scale * (1.0/pow(t5094,4.0/3.0)*t5115*t5108*t5109*2.0-Cinf*Fg*Pg_1*1.0/pow(t5094,5.0/2.0)*t5114*t5115*t5109);
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t5117 = rho_a+rho_b;
                double t5118 = c*c;
                double t5119 = 1.0/pow(t5117,2.0/3.0);
                double t5120 = 1.0/pow(t5117,1.0/3.0);
                double t5121 = Bg*t5118*t5119;
                double t5122 = Cg_2*c*t5120;
                double t5123 = Cg_1+t5121+t5122;
                double t5124 = 1.0/t5117;
                double t5125 = Bg*c*t5124*t5118*1.0E4;
                double t5126 = Cg_4*t5118*t5119;
                double t5127 = Cg_3*c*t5120;
                double t5128 = t5125+t5126+t5127+1.0;
                double t5129 = 1.0/t5128;
                double t5130 = t5123*t5129;
                double t5131 = Cx+t5130;
                double t5132 = 1.0/two_13;
                double t5133 = 1.0/pow(t5117,7.0/6.0);
                double t5134 = 1.0/t5131;
                double t5135 = gamma_ab*2.0;
                double t5136 = gamma_aa+gamma_bb+t5135;
                double t5137 = sqrt(t5136);
                double t5138 = exp(-Cinf*Fg*Pg_1*t5133*t5134*t5137);
                v_gamma_bb[Q] += scale * (t5131*t5132*1.0/pow(t5117,4.0/3.0)*t5138-Cinf*Fg*Pg_1*t5132*1.0/pow(t5117,5.0/2.0)*t5137*t5138*(1.0/2.0));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t5142 = rho_a+rho_b;
                double t5143 = 1.0/pow(t5142,4.0/3.0);
                double t5144 = 1.0/pow(t5142,1.0/3.0);
                double t5145 = c*t5144;
                double t5146 = 1.0/sqrt(t5145);
                double t5147 = sqrt(t5145);
                double t5148 = EcPld_3*c*t5143*(1.0/3.0);
                double t5149 = EcPld_2*c*t5143*t5146*(1.0/6.0);
                double t5150 = t5148+t5149;
                double t5151 = EcPld_2*t5147;
                double t5152 = EcPld_3*c*t5144;
                double t5153 = t5151+t5152+1.0;
                double t5154 = 1.0/(t5153*t5153);
                double t5155 = EcPld_1*t5150*t5154;
                double t5156 = t5145-1.0;
                double t5157 = heaviside(t5156);
                double t5158 = pow(2.0,1.0/3.0);
                double t5159 = t5158*2.0;
                double t5160 = t5159-2.0;
                double t5161 = two_13*2.0;
                double t5162 = t5161-2.0;
                double t5163 = 1.0/t5162;
                double t5164 = EcFld_3*c*t5143*(1.0/3.0);
                double t5165 = EcFld_2*c*t5143*t5146*(1.0/6.0);
                double t5166 = t5164+t5165;
                double t5167 = EcFld_2*t5147;
                double t5168 = EcFld_3*c*t5144;
                double t5169 = t5167+t5168+1.0;
                double t5170 = t5150*t5150;
                double t5171 = 1.0/(t5153*t5153*t5153);
                double t5172 = EcPld_1*t5170*t5171*2.0;
                double t5173 = 1.0/pow(t5142,7.0/3.0);
                double t5174 = 1.0/(t5169*t5169);
                double t5175 = c*c;
                double t5176 = 1.0/pow(t5142,8.0/3.0);
                double t5177 = 1.0/pow(t5145,3.0/2.0);
                double t5178 = EcPld_3*c*t5173*(4.0/9.0);
                double t5179 = EcPld_2*c*t5146*t5173*(2.0/9.0);
                double t5180 = t5178+t5179-EcPld_2*t5175*t5176*t5177*(1.0/3.6E1);
                double t5181 = EcPld_1*t5180*t5154;
                double t5182 = 1.0/(t5142*t5142);
                double t5183 = EcPhd_1*t5182*(1.0/3.0);
                double t5184 = log(t5145);
                double t5185 = EcPhd_3*c*t5173*(5.0/9.0);
                double t5186 = EcPhd_4*c*t5173*(4.0/9.0);
                double t5187 = EcPhd_3*c*t5173*t5184*(4.0/9.0);
                double t5188 = 1.0/t5153;
                double t5189 = EcPld_1*t5188;
                double t5190 = t5155-EcFld_1*t5174*t5166;
                double t5191 = t5155-t5160*t5163*t5190;
                double t5192 = dirac(t5156);
                double t5193 = EcPhd_1*t5184;
                double t5194 = 1.0/t5142;
                double t5195 = EcPhd_1*t5194*(1.0/3.0);
                double t5196 = EcPhd_3*c*t5143*(1.0/3.0);
                double t5197 = EcPhd_4*c*t5143*(1.0/3.0);
                double t5198 = EcPhd_3*c*t5143*t5184*(1.0/3.0);
                double t5199 = 1.0/t5169;
                double t5215 = EcFld_1*t5199;
                double t5200 = t5189-t5215;
                double t5216 = t5160*t5163*t5200;
                double t5201 = t5189-t5216;
                double t5202 = dirac(t5156,1.0);
                double t5203 = EcFhd_1*t5184;
                double t5204 = EcFhd_4*c*t5144;
                double t5205 = EcPhd_4*c*t5144;
                double t5206 = EcFhd_3*c*t5144*t5184;
                double t5207 = EcPhd_3*c*t5144*t5184;
                double t5208 = EcFhd_2-EcPhd_2-t5193+t5203+t5204-t5205+t5206-t5207;
                double t5209 = t5160*t5163*t5208;
                double t5210 = EcPhd_2+t5193+t5205+t5207+t5209;
                double t5211 = -t5145+1.0;
                double t5212 = heaviside(t5211);
                double t5213 = t5195+t5196+t5197+t5198-EcFhd_1*t5194*(1.0/3.0)-EcFhd_3*c*t5143*(1.0/3.0)-EcFhd_4*c*t5143*(1.0/3.0)-EcFhd_3*c*t5143*t5184*(1.0/3.0);
                double t5214 = t5195+t5196+t5197+t5198-t5160*t5163*t5213;
                double t5217 = 1.0/pow(t5142,2.0/3.0);
                double t5218 = Bg*c*t5175*t5194*1.0E4;
                double t5219 = Cg_4*t5175*t5217;
                double t5220 = Cg_3*c*t5144;
                double t5221 = t5220+t5218+t5219+1.0;
                double t5222 = 1.0/t5221;
                double t5223 = 1.0/pow(t5142,5.0/3.0);
                double t5224 = Bg*t5175*t5217;
                double t5225 = Cg_2*c*t5144;
                double t5226 = Cg_1+t5224+t5225;
                double t5227 = gamma_ab*2.0;
                double t5228 = gamma_aa+gamma_bb+t5227;
                double t5229 = 1.0/two_13;
                double t5230 = 1.0/pow(t5142,7.0/6.0);
                double t5231 = t5222*t5226;
                double t5232 = Cx+t5231;
                double t5233 = 1.0/t5232;
                double t5234 = sqrt(t5228);
                double t5236 = Cinf*Fg*Pg_1*t5230*t5233*t5234;
                double t5235 = exp(-t5236);
                double t5237 = Bg*t5175*t5223*(2.0/3.0);
                double t5238 = Cg_2*c*t5143*(1.0/3.0);
                double t5239 = t5237+t5238;
                double t5240 = Bg*c*t5182*t5175*1.0E4;
                double t5241 = Cg_4*t5175*t5223*(2.0/3.0);
                double t5242 = Cg_3*c*t5143*(1.0/3.0);
                double t5243 = t5240+t5241+t5242;
                double t5244 = 1.0/(t5221*t5221);
                double t5245 = t5222*t5239;
                double t5260 = t5243*t5226*t5244;
                double t5246 = -t5260+t5245;
                double t5247 = Bg*t5175*t5176*(1.0E1/9.0);
                double t5248 = Cg_2*c*t5173*(4.0/9.0);
                double t5249 = t5247+t5248;
                double t5250 = t5222*t5249;
                double t5251 = 1.0/(t5142*t5142*t5142);
                double t5252 = Bg*c*t5175*t5251*2.0E4;
                double t5253 = Cg_4*t5175*t5176*(1.0E1/9.0);
                double t5254 = Cg_3*c*t5173*(4.0/9.0);
                double t5255 = t5252+t5253+t5254;
                double t5256 = t5243*t5243;
                double t5257 = 1.0/(t5221*t5221*t5221);
                double t5258 = t5226*t5256*t5257*2.0;
                double t5259 = t5250+t5258-t5226*t5244*t5255-t5243*t5244*t5239*2.0;
                double t5261 = 1.0/(t5232*t5232);
                double t5262 = 1.0/pow(t5142,1.3E1/6.0);
                double t5263 = Cinf*Fg*Pg_1*t5233*t5234*t5262*(7.0/6.0);
                double t5265 = Cinf*Fg*Pg_1*t5230*t5234*t5261*t5246;
                double t5264 = t5263-t5265;
                v_rho_a_rho_a[Q] += scale * (-t5142*(-t5212*(t5183+t5185+t5186+t5187-t5160*t5163*(t5183+t5185+t5186+t5187-EcFhd_1*t5182*(1.0/3.0)-EcFhd_3*c*t5173*(5.0/9.0)-EcFhd_4*c*t5173*(4.0/9.0)-EcFhd_3*c*t5173*t5184*(4.0/9.0)))+t5157*(-t5172+t5181+t5160*t5163*(t5172-t5181-EcFld_1*(t5166*t5166)*1.0/(t5169*t5169*t5169)*2.0+EcFld_1*t5174*(EcFld_3*c*t5173*(4.0/9.0)-EcFld_2*t5175*t5176*t5177*(1.0/3.6E1)+EcFld_2*c*t5146*t5173*(2.0/9.0))))+c*t5143*t5191*t5192*(2.0/3.0)-c*t5173*t5192*t5201*(4.0/9.0)+c*t5173*t5192*t5210*(4.0/9.0)+c*t5143*t5192*t5214*(2.0/3.0)-t5175*t5176*t5201*t5202*(1.0/9.0)+t5175*t5176*t5210*t5202*(1.0/9.0))+t5191*t5157*2.0-t5212*t5214*2.0-c*t5143*t5192*t5201*(2.0/3.0)+c*t5143*t5192*t5210*(2.0/3.0)+1.0/pow(t5142,1.0E1/3.0)*t5232*t5235*t5228*t5229*(2.8E1/9.0)+t5173*t5235*t5228*t5246*t5229*(8.0/3.0)+t5143*t5235*t5228*t5229*t5259-t5173*t5232*t5235*t5228*t5264*t5229*(8.0/3.0)-t5143*t5235*t5228*t5246*t5264*t5229*2.0+t5143*t5232*t5235*t5228*(t5264*t5264)*t5229-t5143*t5232*t5235*t5228*t5229*(Cinf*Fg*Pg_1*1.0/pow(t5142,1.9E1/6.0)*t5233*t5234*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t5230*t5234*t5261*t5259-Cinf*Fg*Pg_1*t5234*t5261*t5262*t5246*(7.0/3.0)+Cinf*Fg*Pg_1*t5230*1.0/(t5232*t5232*t5232)*t5234*(t5246*t5246)*2.0));
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t5267 = rho_a+rho_b;
                double t5268 = 1.0/pow(t5267,4.0/3.0);
                double t5269 = 1.0/pow(t5267,1.0/3.0);
                double t5270 = c*t5269;
                double t5271 = 1.0/sqrt(t5270);
                double t5272 = sqrt(t5270);
                double t5273 = EcPld_3*c*t5268*(1.0/3.0);
                double t5274 = EcPld_2*c*t5271*t5268*(1.0/6.0);
                double t5275 = t5273+t5274;
                double t5276 = EcPld_2*t5272;
                double t5277 = EcPld_3*c*t5269;
                double t5278 = t5276+t5277+1.0;
                double t5279 = 1.0/(t5278*t5278);
                double t5280 = EcPld_1*t5275*t5279;
                double t5281 = t5270-1.0;
                double t5282 = heaviside(t5281);
                double t5283 = pow(2.0,1.0/3.0);
                double t5284 = t5283*2.0;
                double t5285 = t5284-2.0;
                double t5286 = two_13*2.0;
                double t5287 = t5286-2.0;
                double t5288 = 1.0/t5287;
                double t5289 = EcFld_3*c*t5268*(1.0/3.0);
                double t5290 = EcFld_2*c*t5271*t5268*(1.0/6.0);
                double t5291 = t5290+t5289;
                double t5292 = EcFld_2*t5272;
                double t5293 = EcFld_3*c*t5269;
                double t5294 = t5292+t5293+1.0;
                double t5295 = t5275*t5275;
                double t5296 = 1.0/(t5278*t5278*t5278);
                double t5297 = EcPld_1*t5295*t5296*2.0;
                double t5298 = 1.0/pow(t5267,7.0/3.0);
                double t5299 = 1.0/(t5294*t5294);
                double t5300 = c*c;
                double t5301 = 1.0/pow(t5267,8.0/3.0);
                double t5302 = 1.0/pow(t5270,3.0/2.0);
                double t5303 = EcPld_3*c*t5298*(4.0/9.0);
                double t5304 = EcPld_2*c*t5271*t5298*(2.0/9.0);
                double t5305 = t5303+t5304-EcPld_2*t5300*t5301*t5302*(1.0/3.6E1);
                double t5306 = EcPld_1*t5279*t5305;
                double t5307 = 1.0/(t5267*t5267);
                double t5308 = EcPhd_1*t5307*(1.0/3.0);
                double t5309 = log(t5270);
                double t5310 = EcPhd_3*c*t5298*(5.0/9.0);
                double t5311 = EcPhd_4*c*t5298*(4.0/9.0);
                double t5312 = EcPhd_3*c*t5298*t5309*(4.0/9.0);
                double t5313 = 1.0/t5278;
                double t5314 = EcPld_1*t5313;
                double t5315 = t5280-EcFld_1*t5291*t5299;
                double t5316 = t5280-t5285*t5288*t5315;
                double t5317 = dirac(t5281);
                double t5318 = EcPhd_1*t5309;
                double t5319 = 1.0/t5267;
                double t5320 = EcPhd_1*t5319*(1.0/3.0);
                double t5321 = EcPhd_3*c*t5268*(1.0/3.0);
                double t5322 = EcPhd_4*c*t5268*(1.0/3.0);
                double t5323 = EcPhd_3*c*t5268*t5309*(1.0/3.0);
                double t5324 = 1.0/t5294;
                double t5340 = EcFld_1*t5324;
                double t5325 = -t5340+t5314;
                double t5326 = t5314-t5285*t5288*t5325;
                double t5327 = dirac(t5281,1.0);
                double t5328 = EcFhd_1*t5309;
                double t5329 = EcFhd_4*c*t5269;
                double t5330 = EcPhd_4*c*t5269;
                double t5331 = EcFhd_3*c*t5269*t5309;
                double t5332 = EcPhd_3*c*t5269*t5309;
                double t5333 = EcFhd_2-EcPhd_2-t5330+t5331-t5332-t5318+t5328+t5329;
                double t5334 = t5285*t5288*t5333;
                double t5335 = EcPhd_2+t5330+t5332+t5334+t5318;
                double t5336 = -t5270+1.0;
                double t5337 = heaviside(t5336);
                double t5338 = t5320+t5321+t5322+t5323-EcFhd_1*t5319*(1.0/3.0)-EcFhd_3*c*t5268*(1.0/3.0)-EcFhd_4*c*t5268*(1.0/3.0)-EcFhd_3*c*t5268*t5309*(1.0/3.0);
                double t5339 = t5320+t5321+t5322+t5323-t5285*t5288*t5338;
                double t5341 = 1.0/pow(t5267,2.0/3.0);
                double t5342 = Bg*c*t5300*t5319*1.0E4;
                double t5343 = Cg_4*t5300*t5341;
                double t5344 = Cg_3*c*t5269;
                double t5345 = t5342+t5343+t5344+1.0;
                double t5346 = 1.0/t5345;
                double t5347 = 1.0/pow(t5267,5.0/3.0);
                double t5348 = Bg*t5300*t5341;
                double t5349 = Cg_2*c*t5269;
                double t5350 = Cg_1+t5348+t5349;
                double t5351 = gamma_ab*2.0;
                double t5352 = gamma_aa+gamma_bb+t5351;
                double t5353 = 1.0/two_13;
                double t5354 = 1.0/pow(t5267,7.0/6.0);
                double t5355 = t5350*t5346;
                double t5356 = Cx+t5355;
                double t5357 = 1.0/t5356;
                double t5358 = sqrt(t5352);
                double t5360 = Cinf*Fg*Pg_1*t5354*t5357*t5358;
                double t5359 = exp(-t5360);
                double t5361 = Bg*t5300*t5347*(2.0/3.0);
                double t5362 = Cg_2*c*t5268*(1.0/3.0);
                double t5363 = t5361+t5362;
                double t5364 = Bg*c*t5300*t5307*1.0E4;
                double t5365 = Cg_4*t5300*t5347*(2.0/3.0);
                double t5366 = Cg_3*c*t5268*(1.0/3.0);
                double t5367 = t5364+t5365+t5366;
                double t5368 = 1.0/(t5345*t5345);
                double t5369 = t5363*t5346;
                double t5384 = t5350*t5367*t5368;
                double t5370 = -t5384+t5369;
                double t5371 = Bg*t5300*t5301*(1.0E1/9.0);
                double t5372 = Cg_2*c*t5298*(4.0/9.0);
                double t5373 = t5371+t5372;
                double t5374 = t5346*t5373;
                double t5375 = 1.0/(t5267*t5267*t5267);
                double t5376 = Bg*c*t5300*t5375*2.0E4;
                double t5377 = Cg_4*t5300*t5301*(1.0E1/9.0);
                double t5378 = Cg_3*c*t5298*(4.0/9.0);
                double t5379 = t5376+t5377+t5378;
                double t5380 = t5367*t5367;
                double t5381 = 1.0/(t5345*t5345*t5345);
                double t5382 = t5350*t5380*t5381*2.0;
                double t5383 = t5382+t5374-t5350*t5368*t5379-t5363*t5367*t5368*2.0;
                double t5385 = 1.0/(t5356*t5356);
                double t5386 = 1.0/pow(t5267,1.3E1/6.0);
                double t5387 = Cinf*Fg*Pg_1*t5357*t5358*t5386*(7.0/6.0);
                double t5389 = Cinf*Fg*Pg_1*t5370*t5354*t5358*t5385;
                double t5388 = t5387-t5389;
                v_rho_a_rho_b[Q] += scale * (-t5267*(-t5337*(t5310+t5311+t5312+t5308-t5285*t5288*(t5310+t5311+t5312+t5308-EcFhd_1*t5307*(1.0/3.0)-EcFhd_3*c*t5298*(5.0/9.0)-EcFhd_4*c*t5298*(4.0/9.0)-EcFhd_3*c*t5298*t5309*(4.0/9.0)))+t5282*(-t5297+t5306+t5285*t5288*(t5297-t5306-EcFld_1*(t5291*t5291)*1.0/(t5294*t5294*t5294)*2.0+EcFld_1*t5299*(EcFld_3*c*t5298*(4.0/9.0)-EcFld_2*t5300*t5301*t5302*(1.0/3.6E1)+EcFld_2*c*t5271*t5298*(2.0/9.0))))+c*t5268*t5316*t5317*(2.0/3.0)-c*t5298*t5317*t5326*(4.0/9.0)+c*t5298*t5317*t5335*(4.0/9.0)+c*t5268*t5317*t5339*(2.0/3.0)-t5300*t5301*t5326*t5327*(1.0/9.0)+t5300*t5301*t5335*t5327*(1.0/9.0))+t5282*t5316*2.0-t5337*t5339*2.0+c*t5268*t5317*t5335*(2.0/3.0)-c*t5268*t5317*(t5314+t5285*t5288*(t5340-t5314))*(2.0/3.0)+1.0/pow(t5267,1.0E1/3.0)*t5352*t5353*t5356*t5359*(2.8E1/9.0)+t5298*t5352*t5370*t5353*t5359*(8.0/3.0)+t5268*t5352*t5353*t5383*t5359-t5268*t5352*t5370*t5353*t5359*t5388*2.0-t5298*t5352*t5353*t5356*t5359*t5388*(8.0/3.0)+t5268*t5352*t5353*t5356*t5359*(t5388*t5388)-t5268*t5352*t5353*t5356*t5359*(Cinf*Fg*Pg_1*1.0/pow(t5267,1.9E1/6.0)*t5357*t5358*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t5354*t5383*t5358*t5385-Cinf*Fg*Pg_1*t5370*t5358*t5385*t5386*(7.0/3.0)+Cinf*Fg*Pg_1*(t5370*t5370)*t5354*1.0/(t5356*t5356*t5356)*t5358*2.0));
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t5391 = rho_a+rho_b;
                double t5392 = 1.0/pow(t5391,4.0/3.0);
                double t5393 = 1.0/pow(t5391,1.0/3.0);
                double t5394 = c*t5393;
                double t5395 = 1.0/sqrt(t5394);
                double t5396 = sqrt(t5394);
                double t5397 = EcPld_3*c*t5392*(1.0/3.0);
                double t5398 = EcPld_2*c*t5392*t5395*(1.0/6.0);
                double t5399 = t5397+t5398;
                double t5400 = EcPld_2*t5396;
                double t5401 = EcPld_3*c*t5393;
                double t5402 = t5400+t5401+1.0;
                double t5403 = 1.0/(t5402*t5402);
                double t5404 = EcPld_1*t5399*t5403;
                double t5405 = t5394-1.0;
                double t5406 = heaviside(t5405);
                double t5407 = pow(2.0,1.0/3.0);
                double t5408 = t5407*2.0;
                double t5409 = t5408-2.0;
                double t5410 = two_13*2.0;
                double t5411 = t5410-2.0;
                double t5412 = 1.0/t5411;
                double t5413 = EcFld_3*c*t5392*(1.0/3.0);
                double t5414 = EcFld_2*c*t5392*t5395*(1.0/6.0);
                double t5415 = t5413+t5414;
                double t5416 = EcFld_2*t5396;
                double t5417 = EcFld_3*c*t5393;
                double t5418 = t5416+t5417+1.0;
                double t5419 = t5399*t5399;
                double t5420 = 1.0/(t5402*t5402*t5402);
                double t5421 = EcPld_1*t5420*t5419*2.0;
                double t5422 = 1.0/pow(t5391,7.0/3.0);
                double t5423 = 1.0/(t5418*t5418);
                double t5424 = c*c;
                double t5425 = 1.0/pow(t5391,8.0/3.0);
                double t5426 = 1.0/pow(t5394,3.0/2.0);
                double t5427 = EcPld_3*c*t5422*(4.0/9.0);
                double t5428 = EcPld_2*c*t5395*t5422*(2.0/9.0);
                double t5429 = t5427+t5428-EcPld_2*t5424*t5425*t5426*(1.0/3.6E1);
                double t5430 = EcPld_1*t5403*t5429;
                double t5431 = 1.0/(t5391*t5391);
                double t5432 = EcPhd_1*t5431*(1.0/3.0);
                double t5433 = log(t5394);
                double t5434 = EcPhd_3*c*t5422*(5.0/9.0);
                double t5435 = EcPhd_4*c*t5422*(4.0/9.0);
                double t5436 = EcPhd_3*c*t5422*t5433*(4.0/9.0);
                double t5437 = 1.0/t5402;
                double t5438 = EcPld_1*t5437;
                double t5439 = t5404-EcFld_1*t5423*t5415;
                double t5440 = t5404-t5412*t5409*t5439;
                double t5441 = dirac(t5405);
                double t5442 = EcPhd_1*t5433;
                double t5443 = 1.0/t5391;
                double t5444 = EcPhd_1*t5443*(1.0/3.0);
                double t5445 = EcPhd_3*c*t5392*(1.0/3.0);
                double t5446 = EcPhd_4*c*t5392*(1.0/3.0);
                double t5447 = EcPhd_3*c*t5392*t5433*(1.0/3.0);
                double t5448 = 1.0/t5418;
                double t5464 = EcFld_1*t5448;
                double t5449 = -t5464+t5438;
                double t5450 = t5438-t5412*t5409*t5449;
                double t5451 = dirac(t5405,1.0);
                double t5452 = EcFhd_1*t5433;
                double t5453 = EcFhd_4*c*t5393;
                double t5454 = EcPhd_4*c*t5393;
                double t5455 = EcFhd_3*c*t5393*t5433;
                double t5456 = EcPhd_3*c*t5393*t5433;
                double t5457 = EcFhd_2-EcPhd_2-t5442+t5452+t5453-t5454+t5455-t5456;
                double t5458 = t5412*t5409*t5457;
                double t5459 = EcPhd_2+t5442+t5454+t5456+t5458;
                double t5460 = -t5394+1.0;
                double t5461 = heaviside(t5460);
                double t5462 = t5444+t5445+t5446+t5447-EcFhd_1*t5443*(1.0/3.0)-EcFhd_3*c*t5392*(1.0/3.0)-EcFhd_4*c*t5392*(1.0/3.0)-EcFhd_3*c*t5392*t5433*(1.0/3.0);
                double t5463 = t5444+t5445+t5446+t5447-t5412*t5462*t5409;
                double t5465 = 1.0/pow(t5391,2.0/3.0);
                double t5466 = Bg*c*t5424*t5443*1.0E4;
                double t5467 = Cg_4*t5424*t5465;
                double t5468 = Cg_3*c*t5393;
                double t5469 = t5466+t5467+t5468+1.0;
                double t5470 = 1.0/t5469;
                double t5471 = 1.0/pow(t5391,5.0/3.0);
                double t5472 = Bg*t5424*t5465;
                double t5473 = Cg_2*c*t5393;
                double t5474 = Cg_1+t5472+t5473;
                double t5475 = gamma_ab*2.0;
                double t5476 = gamma_aa+gamma_bb+t5475;
                double t5477 = 1.0/two_13;
                double t5478 = 1.0/pow(t5391,7.0/6.0);
                double t5479 = t5470*t5474;
                double t5480 = Cx+t5479;
                double t5481 = 1.0/t5480;
                double t5482 = sqrt(t5476);
                double t5484 = Cinf*Fg*Pg_1*t5481*t5482*t5478;
                double t5483 = exp(-t5484);
                double t5485 = Bg*t5424*t5471*(2.0/3.0);
                double t5486 = Cg_2*c*t5392*(1.0/3.0);
                double t5487 = t5485+t5486;
                double t5488 = Bg*c*t5431*t5424*1.0E4;
                double t5489 = Cg_4*t5424*t5471*(2.0/3.0);
                double t5490 = Cg_3*c*t5392*(1.0/3.0);
                double t5491 = t5490+t5488+t5489;
                double t5492 = 1.0/(t5469*t5469);
                double t5493 = t5470*t5487;
                double t5508 = t5491*t5474*t5492;
                double t5494 = t5493-t5508;
                double t5495 = Bg*t5424*t5425*(1.0E1/9.0);
                double t5496 = Cg_2*c*t5422*(4.0/9.0);
                double t5497 = t5495+t5496;
                double t5498 = t5470*t5497;
                double t5499 = 1.0/(t5391*t5391*t5391);
                double t5500 = Bg*c*t5424*t5499*2.0E4;
                double t5501 = Cg_4*t5424*t5425*(1.0E1/9.0);
                double t5502 = Cg_3*c*t5422*(4.0/9.0);
                double t5503 = t5500+t5501+t5502;
                double t5504 = t5491*t5491;
                double t5505 = 1.0/(t5469*t5469*t5469);
                double t5506 = t5474*t5504*t5505*2.0;
                double t5507 = t5498+t5506-t5491*t5492*t5487*2.0-t5474*t5492*t5503;
                double t5509 = 1.0/(t5480*t5480);
                double t5510 = 1.0/pow(t5391,1.3E1/6.0);
                double t5511 = Cinf*Fg*Pg_1*t5481*t5482*t5510*(7.0/6.0);
                double t5513 = Cinf*Fg*Pg_1*t5482*t5494*t5478*t5509;
                double t5512 = t5511-t5513;
                v_rho_b_rho_b[Q] += scale * (-t5391*(-t5461*(t5432+t5434+t5435+t5436-t5412*t5409*(t5432+t5434+t5435+t5436-EcFhd_1*t5431*(1.0/3.0)-EcFhd_3*c*t5422*(5.0/9.0)-EcFhd_4*c*t5422*(4.0/9.0)-EcFhd_3*c*t5422*t5433*(4.0/9.0)))+t5406*(-t5421+t5430+t5412*t5409*(t5421-t5430-EcFld_1*(t5415*t5415)*1.0/(t5418*t5418*t5418)*2.0+EcFld_1*t5423*(EcFld_3*c*t5422*(4.0/9.0)-EcFld_2*t5424*t5425*t5426*(1.0/3.6E1)+EcFld_2*c*t5395*t5422*(2.0/9.0))))+c*t5392*t5440*t5441*(2.0/3.0)+c*t5392*t5441*t5463*(2.0/3.0)-c*t5422*t5441*t5450*(4.0/9.0)+c*t5422*t5441*t5459*(4.0/9.0)-t5450*t5424*t5451*t5425*(1.0/9.0)+t5424*t5451*t5425*t5459*(1.0/9.0))+t5440*t5406*2.0-t5461*t5463*2.0+c*t5392*t5441*t5459*(2.0/3.0)-c*t5392*t5441*(t5438+t5412*t5409*(t5464-t5438))*(2.0/3.0)+1.0/pow(t5391,1.0E1/3.0)*t5480*t5483*t5476*t5477*(2.8E1/9.0)+t5422*t5483*t5476*t5494*t5477*(8.0/3.0)+t5392*t5483*t5476*t5477*t5507-t5392*t5483*t5476*t5494*t5477*t5512*2.0-t5422*t5480*t5483*t5476*t5477*t5512*(8.0/3.0)+t5392*t5480*t5483*t5476*t5477*(t5512*t5512)-t5392*t5480*t5483*t5476*t5477*(Cinf*Fg*Pg_1*1.0/pow(t5391,1.9E1/6.0)*t5481*t5482*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t5482*t5494*t5510*t5509*(7.0/3.0)-Cinf*Fg*Pg_1*t5482*t5478*t5507*t5509+Cinf*Fg*Pg_1*1.0/(t5480*t5480*t5480)*t5482*(t5494*t5494)*t5478*2.0));
            }
            
            // v_gamma_aa_gamma_aa
            if (deriv >= 2) {
                double t5779 = rho_a+rho_b;
                double t5780 = c*c;
                double t5781 = 1.0/pow(t5779,2.0/3.0);
                double t5782 = 1.0/pow(t5779,1.0/3.0);
                double t5783 = Bg*t5780*t5781;
                double t5784 = Cg_2*c*t5782;
                double t5785 = Cg_1+t5783+t5784;
                double t5786 = 1.0/t5779;
                double t5787 = Bg*c*t5780*t5786*1.0E4;
                double t5788 = Cg_4*t5780*t5781;
                double t5789 = Cg_3*c*t5782;
                double t5790 = t5787+t5788+t5789+1.0;
                double t5791 = 1.0/t5790;
                double t5792 = t5791*t5785;
                double t5793 = Cx+t5792;
                double t5794 = 1.0/t5793;
                double t5795 = 1.0/two_13;
                double t5796 = 1.0/pow(t5779,7.0/6.0);
                double t5797 = gamma_ab*2.0;
                double t5798 = gamma_aa+gamma_bb+t5797;
                double t5799 = sqrt(t5798);
                double t5800 = exp(-Cinf*Fg*Pg_1*t5794*t5796*t5799);
                v_gamma_aa_gamma_aa[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t5794*t5795*1.0/pow(t5779,1.1E1/3.0)*t5800*(1.0/4.0)-Cinf*Fg*Pg_1*t5795*1.0/pow(t5779,5.0/2.0)*1.0/sqrt(t5798)*t5800*(3.0/4.0));
            }
            
            // v_gamma_aa_gamma_ab
            if (deriv >= 2) {
                double t5802 = rho_a+rho_b;
                double t5803 = c*c;
                double t5804 = 1.0/pow(t5802,2.0/3.0);
                double t5805 = 1.0/pow(t5802,1.0/3.0);
                double t5806 = Bg*t5803*t5804;
                double t5807 = Cg_2*c*t5805;
                double t5808 = Cg_1+t5806+t5807;
                double t5809 = 1.0/t5802;
                double t5810 = Bg*c*t5803*t5809*1.0E4;
                double t5811 = Cg_4*t5803*t5804;
                double t5812 = Cg_3*c*t5805;
                double t5813 = t5810+t5811+t5812+1.0;
                double t5814 = 1.0/t5813;
                double t5815 = t5814*t5808;
                double t5816 = Cx+t5815;
                double t5817 = 1.0/t5816;
                double t5818 = 1.0/two_13;
                double t5819 = 1.0/pow(t5802,7.0/6.0);
                double t5820 = gamma_ab*2.0;
                double t5821 = gamma_aa+gamma_bb+t5820;
                double t5822 = sqrt(t5821);
                double t5823 = exp(-Cinf*Fg*Pg_1*t5822*t5817*t5819);
                v_gamma_aa_gamma_ab[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t5802,1.1E1/3.0)*t5823*t5817*t5818*(1.0/2.0)-Cinf*Fg*Pg_1*1.0/pow(t5802,5.0/2.0)*1.0/sqrt(t5821)*t5823*t5818*(3.0/2.0));
            }
            
            // v_gamma_aa_gamma_bb
            if (deriv >= 2) {
                double t5825 = rho_a+rho_b;
                double t5826 = c*c;
                double t5827 = 1.0/pow(t5825,2.0/3.0);
                double t5828 = 1.0/pow(t5825,1.0/3.0);
                double t5829 = Bg*t5826*t5827;
                double t5830 = Cg_2*c*t5828;
                double t5831 = Cg_1+t5830+t5829;
                double t5832 = 1.0/t5825;
                double t5833 = Bg*c*t5832*t5826*1.0E4;
                double t5834 = Cg_4*t5826*t5827;
                double t5835 = Cg_3*c*t5828;
                double t5836 = t5833+t5834+t5835+1.0;
                double t5837 = 1.0/t5836;
                double t5838 = t5831*t5837;
                double t5839 = Cx+t5838;
                double t5840 = 1.0/t5839;
                double t5841 = 1.0/two_13;
                double t5842 = 1.0/pow(t5825,7.0/6.0);
                double t5843 = gamma_ab*2.0;
                double t5844 = gamma_aa+gamma_bb+t5843;
                double t5845 = sqrt(t5844);
                double t5846 = exp(-Cinf*Fg*Pg_1*t5840*t5842*t5845);
                v_gamma_aa_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t5840*t5841*1.0/pow(t5825,1.1E1/3.0)*t5846*(1.0/4.0)-Cinf*Fg*Pg_1*t5841*1.0/pow(t5825,5.0/2.0)*1.0/sqrt(t5844)*t5846*(3.0/4.0));
            }
            
            // v_gamma_ab_gamma_ab
            if (deriv >= 2) {
                double t5848 = rho_a+rho_b;
                double t5849 = c*c;
                double t5850 = 1.0/pow(t5848,2.0/3.0);
                double t5851 = 1.0/pow(t5848,1.0/3.0);
                double t5852 = Bg*t5850*t5849;
                double t5853 = Cg_2*c*t5851;
                double t5854 = Cg_1+t5852+t5853;
                double t5855 = 1.0/t5848;
                double t5856 = Bg*c*t5855*t5849*1.0E4;
                double t5857 = Cg_4*t5850*t5849;
                double t5858 = Cg_3*c*t5851;
                double t5859 = t5856+t5857+t5858+1.0;
                double t5860 = 1.0/t5859;
                double t5861 = t5860*t5854;
                double t5862 = Cx+t5861;
                double t5863 = 1.0/t5862;
                double t5864 = 1.0/two_13;
                double t5865 = 1.0/pow(t5848,7.0/6.0);
                double t5866 = gamma_ab*2.0;
                double t5867 = gamma_aa+gamma_bb+t5866;
                double t5868 = sqrt(t5867);
                double t5869 = exp(-Cinf*Fg*Pg_1*t5863*t5865*t5868);
                v_gamma_ab_gamma_ab[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t5863*t5864*1.0/pow(t5848,1.1E1/3.0)*t5869-Cinf*Fg*Pg_1*t5864*1.0/pow(t5848,5.0/2.0)*1.0/sqrt(t5867)*t5869*3.0);
            }
            
            // v_gamma_ab_gamma_bb
            if (deriv >= 2) {
                double t5871 = rho_a+rho_b;
                double t5872 = c*c;
                double t5873 = 1.0/pow(t5871,2.0/3.0);
                double t5874 = 1.0/pow(t5871,1.0/3.0);
                double t5875 = Bg*t5872*t5873;
                double t5876 = Cg_2*c*t5874;
                double t5877 = Cg_1+t5875+t5876;
                double t5878 = 1.0/t5871;
                double t5879 = Bg*c*t5872*t5878*1.0E4;
                double t5880 = Cg_4*t5872*t5873;
                double t5881 = Cg_3*c*t5874;
                double t5882 = t5880+t5881+t5879+1.0;
                double t5883 = 1.0/t5882;
                double t5884 = t5883*t5877;
                double t5885 = Cx+t5884;
                double t5886 = 1.0/t5885;
                double t5887 = 1.0/two_13;
                double t5888 = 1.0/pow(t5871,7.0/6.0);
                double t5889 = gamma_ab*2.0;
                double t5890 = gamma_aa+gamma_bb+t5889;
                double t5891 = sqrt(t5890);
                double t5892 = exp(-Cinf*Fg*Pg_1*t5891*t5886*t5888);
                v_gamma_ab_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t5871,1.1E1/3.0)*t5892*t5886*t5887*(1.0/2.0)-Cinf*Fg*Pg_1*1.0/pow(t5871,5.0/2.0)*1.0/sqrt(t5890)*t5892*t5887*(3.0/2.0));
            }
            
            // v_gamma_bb_gamma_bb
            if (deriv >= 2) {
                double t5894 = rho_a+rho_b;
                double t5895 = c*c;
                double t5896 = 1.0/pow(t5894,2.0/3.0);
                double t5897 = 1.0/pow(t5894,1.0/3.0);
                double t5898 = Bg*t5895*t5896;
                double t5899 = Cg_2*c*t5897;
                double t5900 = Cg_1+t5898+t5899;
                double t5901 = 1.0/t5894;
                double t5902 = Bg*c*t5895*t5901*1.0E4;
                double t5903 = Cg_4*t5895*t5896;
                double t5904 = Cg_3*c*t5897;
                double t5905 = t5902+t5903+t5904+1.0;
                double t5906 = 1.0/t5905;
                double t5907 = t5900*t5906;
                double t5908 = Cx+t5907;
                double t5909 = 1.0/t5908;
                double t5910 = 1.0/two_13;
                double t5911 = 1.0/pow(t5894,7.0/6.0);
                double t5912 = gamma_ab*2.0;
                double t5913 = gamma_aa+gamma_bb+t5912;
                double t5914 = sqrt(t5913);
                double t5915 = exp(-Cinf*Fg*Pg_1*t5911*t5914*t5909);
                v_gamma_bb_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t5894,1.1E1/3.0)*t5910*t5915*t5909*(1.0/4.0)-Cinf*Fg*Pg_1*1.0/pow(t5894,5.0/2.0)*t5910*1.0/sqrt(t5913)*t5915*(3.0/4.0));
            }
            
            // v_rho_a_gamma_aa
            if (deriv >= 2) {
                double t5515 = rho_a+rho_b;
                double t5516 = c*c;
                double t5517 = 1.0/pow(t5515,2.0/3.0);
                double t5518 = 1.0/pow(t5515,1.0/3.0);
                double t5519 = 1.0/pow(t5515,4.0/3.0);
                double t5520 = 1.0/t5515;
                double t5521 = Bg*c*t5520*t5516*1.0E4;
                double t5522 = Cg_4*t5516*t5517;
                double t5523 = Cg_3*c*t5518;
                double t5524 = t5521+t5522+t5523+1.0;
                double t5525 = 1.0/t5524;
                double t5526 = 1.0/pow(t5515,5.0/3.0);
                double t5527 = Bg*t5516*t5517;
                double t5528 = Cg_2*c*t5518;
                double t5529 = Cg_1+t5527+t5528;
                double t5530 = 1.0/two_13;
                double t5531 = 1.0/pow(t5515,7.0/6.0);
                double t5532 = t5525*t5529;
                double t5533 = Cx+t5532;
                double t5534 = 1.0/t5533;
                double t5535 = gamma_ab*2.0;
                double t5536 = gamma_aa+gamma_bb+t5535;
                double t5537 = sqrt(t5536);
                double t5539 = Cinf*Fg*Pg_1*t5531*t5534*t5537;
                double t5538 = exp(-t5539);
                double t5540 = Bg*t5516*t5526*(2.0/3.0);
                double t5541 = Cg_2*c*t5519*(1.0/3.0);
                double t5542 = t5540+t5541;
                double t5543 = t5542*t5525;
                double t5544 = 1.0/(t5515*t5515);
                double t5545 = Bg*c*t5516*t5544*1.0E4;
                double t5546 = Cg_4*t5516*t5526*(2.0/3.0);
                double t5547 = Cg_3*c*t5519*(1.0/3.0);
                double t5548 = t5545+t5546+t5547;
                double t5549 = 1.0/(t5524*t5524);
                double t5552 = t5529*t5548*t5549;
                double t5550 = t5543-t5552;
                double t5551 = 1.0/pow(t5515,1.3E1/6.0);
                double t5553 = 1.0/(t5533*t5533);
                double t5554 = 1.0/sqrt(t5536);
                double t5555 = Cinf*Fg*Pg_1*t5551*t5534*t5537*(7.0/6.0);
                double t5556 = t5555-Cinf*Fg*Pg_1*t5531*t5550*t5553*t5537;
                double t5557 = 1.0/pow(t5515,5.0/2.0);
                v_rho_a_gamma_aa[Q] += scale * (-t5530*t5550*t5519*t5538-t5530*1.0/pow(t5515,7.0/3.0)*t5533*t5538*(4.0/3.0)+t5530*t5533*t5519*t5538*t5556+t5530*t5533*t5536*t5519*t5538*(Cinf*Fg*Pg_1*t5551*t5534*t5554*(7.0/1.2E1)-Cinf*Fg*Pg_1*t5531*t5550*t5553*t5554*(1.0/2.0))+Cinf*Fg*Pg_1*t5530*1.0/pow(t5515,7.0/2.0)*t5537*t5538*(2.0/3.0)-Cinf*Fg*Pg_1*t5530*t5537*t5538*t5556*t5557*(1.0/2.0)+Cinf*Fg*Pg_1*t5530*t5550*t5534*t5537*t5538*t5557*(1.0/2.0));
            }
            
            // v_rho_a_gamma_ab
            if (deriv >= 2) {
                double t5559 = rho_a+rho_b;
                double t5560 = c*c;
                double t5561 = 1.0/pow(t5559,2.0/3.0);
                double t5562 = 1.0/pow(t5559,1.0/3.0);
                double t5563 = 1.0/pow(t5559,4.0/3.0);
                double t5564 = 1.0/t5559;
                double t5565 = Bg*c*t5560*t5564*1.0E4;
                double t5566 = Cg_4*t5560*t5561;
                double t5567 = Cg_3*c*t5562;
                double t5568 = t5565+t5566+t5567+1.0;
                double t5569 = 1.0/t5568;
                double t5570 = 1.0/pow(t5559,5.0/3.0);
                double t5571 = Bg*t5560*t5561;
                double t5572 = Cg_2*c*t5562;
                double t5573 = Cg_1+t5571+t5572;
                double t5574 = 1.0/two_13;
                double t5575 = 1.0/pow(t5559,7.0/6.0);
                double t5576 = t5573*t5569;
                double t5577 = Cx+t5576;
                double t5578 = 1.0/t5577;
                double t5579 = gamma_ab*2.0;
                double t5580 = gamma_aa+gamma_bb+t5579;
                double t5581 = sqrt(t5580);
                double t5583 = Cinf*Fg*Pg_1*t5581*t5575*t5578;
                double t5582 = exp(-t5583);
                double t5584 = Bg*t5560*t5570*(2.0/3.0);
                double t5585 = Cg_2*c*t5563*(1.0/3.0);
                double t5586 = t5584+t5585;
                double t5587 = t5586*t5569;
                double t5588 = 1.0/(t5559*t5559);
                double t5589 = Bg*c*t5560*t5588*1.0E4;
                double t5590 = Cg_4*t5560*t5570*(2.0/3.0);
                double t5591 = Cg_3*c*t5563*(1.0/3.0);
                double t5592 = t5590+t5591+t5589;
                double t5593 = 1.0/(t5568*t5568);
                double t5596 = t5573*t5592*t5593;
                double t5594 = t5587-t5596;
                double t5595 = 1.0/pow(t5559,1.3E1/6.0);
                double t5597 = 1.0/(t5577*t5577);
                double t5598 = 1.0/sqrt(t5580);
                double t5599 = Cinf*Fg*Pg_1*t5581*t5595*t5578*(7.0/6.0);
                double t5600 = t5599-Cinf*Fg*Pg_1*t5581*t5575*t5594*t5597;
                double t5601 = 1.0/pow(t5559,5.0/2.0);
                v_rho_a_gamma_ab[Q] += scale * (t5563*t5582*t5574*t5594*-2.0-t5582*t5574*1.0/pow(t5559,7.0/3.0)*t5577*(8.0/3.0)+t5563*t5582*t5574*t5577*t5600*2.0+t5580*t5563*t5582*t5574*t5577*(Cinf*Fg*Pg_1*t5595*t5578*t5598*(7.0/6.0)-Cinf*Fg*Pg_1*t5575*t5594*t5597*t5598)+Cinf*Fg*Pg_1*t5581*t5582*t5574*1.0/pow(t5559,7.0/2.0)*(4.0/3.0)-Cinf*Fg*Pg_1*t5581*t5582*t5574*t5600*t5601+Cinf*Fg*Pg_1*t5581*t5582*t5574*t5594*t5578*t5601);
            }
            
            // v_rho_a_gamma_bb
            if (deriv >= 2) {
                double t5603 = rho_a+rho_b;
                double t5604 = c*c;
                double t5605 = 1.0/pow(t5603,2.0/3.0);
                double t5606 = 1.0/pow(t5603,1.0/3.0);
                double t5607 = 1.0/pow(t5603,4.0/3.0);
                double t5608 = 1.0/t5603;
                double t5609 = Bg*c*t5604*t5608*1.0E4;
                double t5610 = Cg_4*t5604*t5605;
                double t5611 = Cg_3*c*t5606;
                double t5612 = t5610+t5611+t5609+1.0;
                double t5613 = 1.0/t5612;
                double t5614 = 1.0/pow(t5603,5.0/3.0);
                double t5615 = Bg*t5604*t5605;
                double t5616 = Cg_2*c*t5606;
                double t5617 = Cg_1+t5615+t5616;
                double t5618 = 1.0/two_13;
                double t5619 = 1.0/pow(t5603,7.0/6.0);
                double t5620 = t5613*t5617;
                double t5621 = Cx+t5620;
                double t5622 = 1.0/t5621;
                double t5623 = gamma_ab*2.0;
                double t5624 = gamma_aa+gamma_bb+t5623;
                double t5625 = sqrt(t5624);
                double t5627 = Cinf*Fg*Pg_1*t5622*t5625*t5619;
                double t5626 = exp(-t5627);
                double t5628 = Bg*t5604*t5614*(2.0/3.0);
                double t5629 = Cg_2*c*t5607*(1.0/3.0);
                double t5630 = t5628+t5629;
                double t5631 = t5630*t5613;
                double t5632 = 1.0/(t5603*t5603);
                double t5633 = Bg*c*t5604*t5632*1.0E4;
                double t5634 = Cg_4*t5604*t5614*(2.0/3.0);
                double t5635 = Cg_3*c*t5607*(1.0/3.0);
                double t5636 = t5633+t5634+t5635;
                double t5637 = 1.0/(t5612*t5612);
                double t5640 = t5617*t5636*t5637;
                double t5638 = t5631-t5640;
                double t5639 = 1.0/pow(t5603,1.3E1/6.0);
                double t5641 = 1.0/(t5621*t5621);
                double t5642 = 1.0/sqrt(t5624);
                double t5643 = Cinf*Fg*Pg_1*t5622*t5625*t5639*(7.0/6.0);
                double t5644 = t5643-Cinf*Fg*Pg_1*t5641*t5625*t5619*t5638;
                double t5645 = 1.0/pow(t5603,5.0/2.0);
                v_rho_a_gamma_bb[Q] += scale * (-t5607*t5626*t5618*t5638-1.0/pow(t5603,7.0/3.0)*t5621*t5626*t5618*(4.0/3.0)+t5621*t5607*t5626*t5644*t5618+t5621*t5624*t5607*t5626*t5618*(Cinf*Fg*Pg_1*t5622*t5642*t5639*(7.0/1.2E1)-Cinf*Fg*Pg_1*t5641*t5642*t5619*t5638*(1.0/2.0))+Cinf*Fg*Pg_1*1.0/pow(t5603,7.0/2.0)*t5625*t5626*t5618*(2.0/3.0)-Cinf*Fg*Pg_1*t5625*t5626*t5644*t5618*t5645*(1.0/2.0)+Cinf*Fg*Pg_1*t5622*t5625*t5626*t5618*t5645*t5638*(1.0/2.0));
            }
            
            // v_rho_b_gamma_aa
            if (deriv >= 2) {
                double t5647 = rho_a+rho_b;
                double t5648 = c*c;
                double t5649 = 1.0/pow(t5647,2.0/3.0);
                double t5650 = 1.0/pow(t5647,1.0/3.0);
                double t5651 = 1.0/pow(t5647,4.0/3.0);
                double t5652 = 1.0/t5647;
                double t5653 = Bg*c*t5652*t5648*1.0E4;
                double t5654 = Cg_4*t5648*t5649;
                double t5655 = Cg_3*c*t5650;
                double t5656 = t5653+t5654+t5655+1.0;
                double t5657 = 1.0/t5656;
                double t5658 = 1.0/pow(t5647,5.0/3.0);
                double t5659 = Bg*t5648*t5649;
                double t5660 = Cg_2*c*t5650;
                double t5661 = Cg_1+t5660+t5659;
                double t5662 = 1.0/two_13;
                double t5663 = 1.0/pow(t5647,7.0/6.0);
                double t5664 = t5661*t5657;
                double t5665 = Cx+t5664;
                double t5666 = 1.0/t5665;
                double t5667 = gamma_ab*2.0;
                double t5668 = gamma_aa+gamma_bb+t5667;
                double t5669 = sqrt(t5668);
                double t5671 = Cinf*Fg*Pg_1*t5663*t5666*t5669;
                double t5670 = exp(-t5671);
                double t5672 = Bg*t5648*t5658*(2.0/3.0);
                double t5673 = Cg_2*c*t5651*(1.0/3.0);
                double t5674 = t5672+t5673;
                double t5675 = t5674*t5657;
                double t5676 = 1.0/(t5647*t5647);
                double t5677 = Bg*c*t5648*t5676*1.0E4;
                double t5678 = Cg_4*t5648*t5658*(2.0/3.0);
                double t5679 = Cg_3*c*t5651*(1.0/3.0);
                double t5680 = t5677+t5678+t5679;
                double t5681 = 1.0/(t5656*t5656);
                double t5684 = t5661*t5680*t5681;
                double t5682 = t5675-t5684;
                double t5683 = 1.0/pow(t5647,1.3E1/6.0);
                double t5685 = 1.0/(t5665*t5665);
                double t5686 = 1.0/sqrt(t5668);
                double t5687 = Cinf*Fg*Pg_1*t5683*t5666*t5669*(7.0/6.0);
                double t5688 = t5687-Cinf*Fg*Pg_1*t5663*t5682*t5685*t5669;
                double t5689 = 1.0/pow(t5647,5.0/2.0);
                v_rho_b_gamma_aa[Q] += scale * (-t5651*t5670*t5662*t5682-t5670*t5662*1.0/pow(t5647,7.0/3.0)*t5665*(4.0/3.0)+t5651*t5670*t5662*t5665*t5688+t5651*t5670*t5662*t5665*t5668*(Cinf*Fg*Pg_1*t5683*t5666*t5686*(7.0/1.2E1)-Cinf*Fg*Pg_1*t5663*t5682*t5685*t5686*(1.0/2.0))+Cinf*Fg*Pg_1*t5670*t5662*1.0/pow(t5647,7.0/2.0)*t5669*(2.0/3.0)-Cinf*Fg*Pg_1*t5670*t5662*t5669*t5688*t5689*(1.0/2.0)+Cinf*Fg*Pg_1*t5670*t5662*t5682*t5666*t5669*t5689*(1.0/2.0));
            }
            
            // v_rho_b_gamma_ab
            if (deriv >= 2) {
                double t5691 = rho_a+rho_b;
                double t5692 = c*c;
                double t5693 = 1.0/pow(t5691,2.0/3.0);
                double t5694 = 1.0/pow(t5691,1.0/3.0);
                double t5695 = 1.0/pow(t5691,4.0/3.0);
                double t5696 = 1.0/t5691;
                double t5697 = Bg*c*t5692*t5696*1.0E4;
                double t5698 = Cg_4*t5692*t5693;
                double t5699 = Cg_3*c*t5694;
                double t5700 = t5697+t5698+t5699+1.0;
                double t5701 = 1.0/t5700;
                double t5702 = 1.0/pow(t5691,5.0/3.0);
                double t5703 = Bg*t5692*t5693;
                double t5704 = Cg_2*c*t5694;
                double t5705 = Cg_1+t5703+t5704;
                double t5706 = 1.0/two_13;
                double t5707 = 1.0/pow(t5691,7.0/6.0);
                double t5708 = t5701*t5705;
                double t5709 = Cx+t5708;
                double t5710 = 1.0/t5709;
                double t5711 = gamma_ab*2.0;
                double t5712 = gamma_aa+gamma_bb+t5711;
                double t5713 = sqrt(t5712);
                double t5715 = Cinf*Fg*Pg_1*t5710*t5713*t5707;
                double t5714 = exp(-t5715);
                double t5716 = Bg*t5692*t5702*(2.0/3.0);
                double t5717 = Cg_2*c*t5695*(1.0/3.0);
                double t5718 = t5716+t5717;
                double t5719 = t5701*t5718;
                double t5720 = 1.0/(t5691*t5691);
                double t5721 = Bg*c*t5692*t5720*1.0E4;
                double t5722 = Cg_4*t5692*t5702*(2.0/3.0);
                double t5723 = Cg_3*c*t5695*(1.0/3.0);
                double t5724 = t5721+t5722+t5723;
                double t5725 = 1.0/(t5700*t5700);
                double t5728 = t5705*t5724*t5725;
                double t5726 = t5719-t5728;
                double t5727 = 1.0/pow(t5691,1.3E1/6.0);
                double t5729 = 1.0/(t5709*t5709);
                double t5730 = 1.0/sqrt(t5712);
                double t5731 = Cinf*Fg*Pg_1*t5710*t5713*t5727*(7.0/6.0);
                double t5732 = t5731-Cinf*Fg*Pg_1*t5713*t5707*t5726*t5729;
                double t5733 = 1.0/pow(t5691,5.0/2.0);
                v_rho_b_gamma_ab[Q] += scale * (t5695*t5714*t5706*t5726*-2.0-1.0/pow(t5691,7.0/3.0)*t5714*t5706*t5709*(8.0/3.0)+t5695*t5714*t5732*t5706*t5709*2.0+t5695*t5712*t5714*t5706*t5709*(Cinf*Fg*Pg_1*t5710*t5730*t5727*(7.0/6.0)-Cinf*Fg*Pg_1*t5730*t5707*t5726*t5729)+Cinf*Fg*Pg_1*1.0/pow(t5691,7.0/2.0)*t5713*t5714*t5706*(4.0/3.0)-Cinf*Fg*Pg_1*t5713*t5714*t5732*t5706*t5733+Cinf*Fg*Pg_1*t5710*t5713*t5714*t5706*t5733*t5726);
            }
            
            // v_rho_b_gamma_bb
            if (deriv >= 2) {
                double t5735 = rho_a+rho_b;
                double t5736 = c*c;
                double t5737 = 1.0/pow(t5735,2.0/3.0);
                double t5738 = 1.0/pow(t5735,1.0/3.0);
                double t5739 = 1.0/pow(t5735,4.0/3.0);
                double t5740 = 1.0/t5735;
                double t5741 = Bg*c*t5740*t5736*1.0E4;
                double t5742 = Cg_4*t5736*t5737;
                double t5743 = Cg_3*c*t5738;
                double t5744 = t5741+t5742+t5743+1.0;
                double t5745 = 1.0/t5744;
                double t5746 = 1.0/pow(t5735,5.0/3.0);
                double t5747 = Bg*t5736*t5737;
                double t5748 = Cg_2*c*t5738;
                double t5749 = Cg_1+t5747+t5748;
                double t5750 = 1.0/two_13;
                double t5751 = 1.0/pow(t5735,7.0/6.0);
                double t5752 = t5745*t5749;
                double t5753 = Cx+t5752;
                double t5754 = 1.0/t5753;
                double t5755 = gamma_ab*2.0;
                double t5756 = gamma_aa+gamma_bb+t5755;
                double t5757 = sqrt(t5756);
                double t5759 = Cinf*Fg*Pg_1*t5751*t5754*t5757;
                double t5758 = exp(-t5759);
                double t5760 = Bg*t5736*t5746*(2.0/3.0);
                double t5761 = Cg_2*c*t5739*(1.0/3.0);
                double t5762 = t5760+t5761;
                double t5763 = t5762*t5745;
                double t5764 = 1.0/(t5735*t5735);
                double t5765 = Bg*c*t5736*t5764*1.0E4;
                double t5766 = Cg_4*t5736*t5746*(2.0/3.0);
                double t5767 = Cg_3*c*t5739*(1.0/3.0);
                double t5768 = t5765+t5766+t5767;
                double t5769 = 1.0/(t5744*t5744);
                double t5772 = t5749*t5768*t5769;
                double t5770 = t5763-t5772;
                double t5771 = 1.0/pow(t5735,1.3E1/6.0);
                double t5773 = 1.0/(t5753*t5753);
                double t5774 = 1.0/sqrt(t5756);
                double t5775 = Cinf*Fg*Pg_1*t5771*t5754*t5757*(7.0/6.0);
                double t5776 = t5775-Cinf*Fg*Pg_1*t5751*t5770*t5773*t5757;
                double t5777 = 1.0/pow(t5735,5.0/2.0);
                v_rho_b_gamma_bb[Q] += scale * (-t5750*t5770*t5739*t5758-t5750*1.0/pow(t5735,7.0/3.0)*t5753*t5758*(4.0/3.0)+t5750*t5753*t5739*t5758*t5776+t5750*t5753*t5756*t5739*t5758*(Cinf*Fg*Pg_1*t5771*t5754*t5774*(7.0/1.2E1)-Cinf*Fg*Pg_1*t5751*t5770*t5773*t5774*(1.0/2.0))+Cinf*Fg*Pg_1*t5750*1.0/pow(t5735,7.0/2.0)*t5757*t5758*(2.0/3.0)-Cinf*Fg*Pg_1*t5750*t5757*t5758*t5776*t5777*(1.0/2.0)+Cinf*Fg*Pg_1*t5750*t5770*t5754*t5757*t5758*t5777*(1.0/2.0));
            }
            
        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t5930 = rho_a+rho_b;
                double t5931 = 1.0/pow(t5930,1.0/3.0);
                double t5932 = c*t5931;
                double t5933 = log(t5932);
                double t5934 = EcPhd_1*t5933;
                double t5935 = pow(2.0,1.0/3.0);
                double t5936 = t5935*2.0;
                double t5937 = t5936-2.0;
                double t5938 = two_13*2.0;
                double t5939 = t5938-2.0;
                double t5940 = 1.0/t5939;
                double t5941 = sqrt(t5932);
                double t5942 = EcPld_2*t5941;
                double t5943 = EcPld_3*c*t5931;
                double t5944 = t5942+t5943+1.0;
                double t5945 = 1.0/t5944;
                double t5946 = EcPld_1*t5945;
                double t5947 = c*c;
                double t5948 = 1.0/pow(t5930,2.0/3.0);
                double t5949 = Bg*t5947*t5948;
                double t5950 = Cg_2*c*t5931;
                double t5951 = Cg_1+t5950+t5949;
                double t5952 = 1.0/t5930;
                double t5953 = Bg*c*t5952*t5947*1.0E4;
                double t5954 = Cg_4*t5947*t5948;
                double t5955 = Cg_3*c*t5931;
                double t5956 = t5953+t5954+t5955+1.0;
                double t5957 = 1.0/t5956;
                double t5958 = t5951*t5957;
                double t5959 = Cx+t5958;
                double t5960 = gamma_ab*2.0;
                double t5961 = gamma_aa+gamma_bb+t5960;
                v[Q] += scale * (t5930*(heaviside(-c*t5931+1.0)*(EcPhd_2+t5934+t5940*t5937*(EcFhd_2-EcPhd_2-t5934+EcFhd_1*t5933+EcFhd_4*c*t5931-EcPhd_4*c*t5931+EcFhd_3*c*t5931*t5933-EcPhd_3*c*t5931*t5933)+EcPhd_4*c*t5931+EcPhd_3*c*t5931*t5933)+heaviside(t5932-1.0)*(t5946-t5940*t5937*(t5946-EcFld_1/(EcFld_2*t5941+EcFld_3*c*t5931+1.0))))+(1.0/pow(t5930,4.0/3.0)*t5961*t5959*exp(-(Cinf*Fg*Pg_1*1.0/pow(t5930,7.0/6.0)*sqrt(t5961))/t5959))/two_13);
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t5963 = rho_a+rho_b;
                double t5964 = 1.0/pow(t5963,4.0/3.0);
                double t5965 = 1.0/pow(t5963,1.0/3.0);
                double t5966 = c*t5965;
                double t5967 = 1.0/sqrt(t5966);
                double t5968 = sqrt(t5966);
                double t5969 = EcPld_3*c*t5964*(1.0/3.0);
                double t5970 = EcPld_2*c*t5964*t5967*(1.0/6.0);
                double t5971 = t5970+t5969;
                double t5972 = EcPld_2*t5968;
                double t5973 = EcPld_3*c*t5965;
                double t5974 = t5972+t5973+1.0;
                double t5975 = 1.0/(t5974*t5974);
                double t5976 = EcPld_1*t5971*t5975;
                double t5977 = pow(2.0,1.0/3.0);
                double t5978 = t5977*2.0;
                double t5979 = t5978-2.0;
                double t5980 = two_13*2.0;
                double t5981 = t5980-2.0;
                double t5982 = 1.0/t5981;
                double t5983 = 1.0/t5963;
                double t5984 = EcPhd_1*t5983*(1.0/3.0);
                double t5985 = log(t5966);
                double t5986 = EcPhd_3*c*t5964*(1.0/3.0);
                double t5987 = EcPhd_4*c*t5964*(1.0/3.0);
                double t5988 = EcPhd_3*c*t5964*t5985*(1.0/3.0);
                double t5989 = t5966-1.0;
                double t5990 = EcPhd_1*t5985;
                double t5991 = dirac(t5989);
                double t5992 = EcFld_2*t5968;
                double t5993 = EcFld_3*c*t5965;
                double t5994 = t5992+t5993+1.0;
                double t5995 = 1.0/t5974;
                double t5996 = EcPld_1*t5995;
                double t5997 = -t5966+1.0;
                double t5998 = heaviside(t5997);
                double t5999 = EcFhd_1*t5985;
                double t6000 = EcFhd_4*c*t5965;
                double t6001 = EcPhd_4*c*t5965;
                double t6002 = EcFhd_3*c*t5965*t5985;
                double t6003 = EcPhd_3*c*t5965*t5985;
                double t6004 = heaviside(t5989);
                double t6005 = 1.0/t5994;
                double t6006 = t5996-EcFld_1*t6005;
                double t6007 = t5996-t5982*t5979*t6006;
                double t6008 = c*c;
                double t6009 = 1.0/pow(t5963,2.0/3.0);
                double t6010 = Bg*c*t5983*t6008*1.0E4;
                double t6011 = Cg_4*t6008*t6009;
                double t6012 = Cg_3*c*t5965;
                double t6013 = t6010+t6011+t6012+1.0;
                double t6014 = 1.0/t6013;
                double t6015 = 1.0/pow(t5963,5.0/3.0);
                double t6016 = Bg*t6008*t6009;
                double t6017 = Cg_2*c*t5965;
                double t6018 = Cg_1+t6016+t6017;
                double t6019 = gamma_ab*2.0;
                double t6020 = gamma_aa+gamma_bb+t6019;
                double t6021 = 1.0/two_13;
                double t6022 = 1.0/pow(t5963,7.0/6.0);
                double t6023 = t6014*t6018;
                double t6024 = Cx+t6023;
                double t6025 = 1.0/t6024;
                double t6026 = sqrt(t6020);
                double t6028 = Cinf*Fg*Pg_1*t6022*t6025*t6026;
                double t6027 = exp(-t6028);
                double t6029 = Bg*t6015*t6008*(2.0/3.0);
                double t6030 = Cg_2*c*t5964*(1.0/3.0);
                double t6031 = t6030+t6029;
                double t6032 = t6031*t6014;
                double t6033 = 1.0/(t5963*t5963);
                double t6034 = Bg*c*t6033*t6008*1.0E4;
                double t6035 = Cg_4*t6015*t6008*(2.0/3.0);
                double t6036 = Cg_3*c*t5964*(1.0/3.0);
                double t6037 = t6034+t6035+t6036;
                double t6038 = 1.0/(t6013*t6013);
                double t6039 = t6032-t6018*t6037*t6038;
                v_rho_a[Q] += scale * (t5998*(EcPhd_2+t5990+t6001+t6003+t5982*t5979*(EcFhd_2-EcPhd_2-t5990+t5999+t6000-t6001+t6002-t6003))+t6004*t6007+t5963*(t6004*(t5976-t5982*t5979*(t5976-EcFld_1*1.0/(t5994*t5994)*(EcFld_3*c*t5964*(1.0/3.0)+EcFld_2*c*t5964*t5967*(1.0/6.0))))-t5998*(t5984+t5986+t5987+t5988-t5982*t5979*(t5984+t5986+t5987+t5988-EcFhd_1*t5983*(1.0/3.0)-EcFhd_3*c*t5964*(1.0/3.0)-EcFhd_4*c*t5964*(1.0/3.0)-EcFhd_3*c*t5964*t5985*(1.0/3.0)))-c*t5964*t5991*t6007*(1.0/3.0)+c*t5964*t5991*(EcPhd_2+t5990+t6001+t6003+t5982*t5979*(EcFhd_2-EcPhd_2-t5990+t5999+t6000+t6002-EcPhd_4*c*t5965-EcPhd_3*c*t5965*t5985))*(1.0/3.0))-1.0/pow(t5963,7.0/3.0)*t6020*t6021*t6024*t6027*(4.0/3.0)-t5964*t6020*t6021*t6027*t6039+t5964*t6020*t6021*t6024*t6027*(Cinf*Fg*Pg_1*1.0/pow(t5963,1.3E1/6.0)*t6025*t6026*(7.0/6.0)-Cinf*Fg*Pg_1*t6022*1.0/(t6024*t6024)*t6026*t6039));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t6041 = rho_a+rho_b;
                double t6042 = 1.0/pow(t6041,4.0/3.0);
                double t6043 = 1.0/pow(t6041,1.0/3.0);
                double t6044 = c*t6043;
                double t6045 = 1.0/sqrt(t6044);
                double t6046 = sqrt(t6044);
                double t6047 = EcPld_3*c*t6042*(1.0/3.0);
                double t6048 = EcPld_2*c*t6042*t6045*(1.0/6.0);
                double t6049 = t6047+t6048;
                double t6050 = EcPld_2*t6046;
                double t6051 = EcPld_3*c*t6043;
                double t6052 = t6050+t6051+1.0;
                double t6053 = 1.0/(t6052*t6052);
                double t6054 = EcPld_1*t6053*t6049;
                double t6055 = pow(2.0,1.0/3.0);
                double t6056 = t6055*2.0;
                double t6057 = t6056-2.0;
                double t6058 = two_13*2.0;
                double t6059 = t6058-2.0;
                double t6060 = 1.0/t6059;
                double t6061 = 1.0/t6041;
                double t6062 = EcPhd_1*t6061*(1.0/3.0);
                double t6063 = log(t6044);
                double t6064 = EcPhd_3*c*t6042*(1.0/3.0);
                double t6065 = EcPhd_4*c*t6042*(1.0/3.0);
                double t6066 = EcPhd_3*c*t6042*t6063*(1.0/3.0);
                double t6067 = t6044-1.0;
                double t6068 = EcPhd_1*t6063;
                double t6069 = dirac(t6067);
                double t6070 = EcFld_2*t6046;
                double t6071 = EcFld_3*c*t6043;
                double t6072 = t6070+t6071+1.0;
                double t6073 = 1.0/t6052;
                double t6074 = EcPld_1*t6073;
                double t6075 = -t6044+1.0;
                double t6076 = heaviside(t6075);
                double t6077 = EcFhd_1*t6063;
                double t6078 = EcFhd_4*c*t6043;
                double t6079 = EcPhd_4*c*t6043;
                double t6080 = EcFhd_3*c*t6043*t6063;
                double t6081 = EcPhd_3*c*t6043*t6063;
                double t6082 = heaviside(t6067);
                double t6083 = 1.0/t6072;
                double t6084 = t6074-EcFld_1*t6083;
                double t6085 = t6074-t6060*t6057*t6084;
                double t6086 = c*c;
                double t6087 = 1.0/pow(t6041,2.0/3.0);
                double t6088 = Bg*c*t6061*t6086*1.0E4;
                double t6089 = Cg_4*t6086*t6087;
                double t6090 = Cg_3*c*t6043;
                double t6091 = t6090+t6088+t6089+1.0;
                double t6092 = 1.0/t6091;
                double t6093 = 1.0/pow(t6041,5.0/3.0);
                double t6094 = Bg*t6086*t6087;
                double t6095 = Cg_2*c*t6043;
                double t6096 = Cg_1+t6094+t6095;
                double t6097 = gamma_ab*2.0;
                double t6098 = gamma_aa+gamma_bb+t6097;
                double t6099 = 1.0/two_13;
                double t6100 = 1.0/pow(t6041,7.0/6.0);
                double t6101 = t6092*t6096;
                double t6102 = Cx+t6101;
                double t6103 = 1.0/t6102;
                double t6104 = sqrt(t6098);
                double t6106 = Cinf*Fg*Pg_1*t6100*t6103*t6104;
                double t6105 = exp(-t6106);
                double t6107 = Bg*t6093*t6086*(2.0/3.0);
                double t6108 = Cg_2*c*t6042*(1.0/3.0);
                double t6109 = t6107+t6108;
                double t6110 = t6092*t6109;
                double t6111 = 1.0/(t6041*t6041);
                double t6112 = Bg*c*t6086*t6111*1.0E4;
                double t6113 = Cg_4*t6093*t6086*(2.0/3.0);
                double t6114 = Cg_3*c*t6042*(1.0/3.0);
                double t6115 = t6112+t6113+t6114;
                double t6116 = 1.0/(t6091*t6091);
                double t6117 = t6110-t6096*t6115*t6116;
                v_rho_b[Q] += scale * (t6076*(EcPhd_2+t6081+t6068+t6079+t6060*t6057*(EcFhd_2-EcPhd_2+t6080-t6081-t6068+t6077+t6078-t6079))+t6082*t6085+t6041*(t6082*(t6054-t6060*t6057*(t6054-EcFld_1*1.0/(t6072*t6072)*(EcFld_3*c*t6042*(1.0/3.0)+EcFld_2*c*t6042*t6045*(1.0/6.0))))-t6076*(t6062+t6064+t6065+t6066-t6060*t6057*(t6062+t6064+t6065+t6066-EcFhd_1*t6061*(1.0/3.0)-EcFhd_3*c*t6042*(1.0/3.0)-EcFhd_4*c*t6042*(1.0/3.0)-EcFhd_3*c*t6042*t6063*(1.0/3.0)))-c*t6042*t6085*t6069*(1.0/3.0)+c*t6042*t6069*(EcPhd_2+t6081+t6068+t6079+t6060*t6057*(EcFhd_2-EcPhd_2+t6080-t6068+t6077+t6078-EcPhd_4*c*t6043-EcPhd_3*c*t6043*t6063))*(1.0/3.0))-1.0/pow(t6041,7.0/3.0)*t6098*t6099*t6102*t6105*(4.0/3.0)-t6042*t6098*t6099*t6105*t6117+t6042*t6098*t6099*t6102*t6105*(Cinf*Fg*Pg_1*1.0/pow(t6041,1.3E1/6.0)*t6103*t6104*(7.0/6.0)-Cinf*Fg*Pg_1*t6100*1.0/(t6102*t6102)*t6104*t6117));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t6119 = rho_a+rho_b;
                double t6120 = c*c;
                double t6121 = 1.0/pow(t6119,2.0/3.0);
                double t6122 = 1.0/pow(t6119,1.0/3.0);
                double t6123 = Bg*t6120*t6121;
                double t6124 = Cg_2*c*t6122;
                double t6125 = Cg_1+t6123+t6124;
                double t6126 = 1.0/t6119;
                double t6127 = Bg*c*t6120*t6126*1.0E4;
                double t6128 = Cg_4*t6120*t6121;
                double t6129 = Cg_3*c*t6122;
                double t6130 = t6127+t6128+t6129+1.0;
                double t6131 = 1.0/t6130;
                double t6132 = t6131*t6125;
                double t6133 = Cx+t6132;
                double t6134 = 1.0/two_13;
                double t6135 = 1.0/pow(t6119,7.0/6.0);
                double t6136 = 1.0/t6133;
                double t6137 = gamma_ab*2.0;
                double t6138 = gamma_aa+gamma_bb+t6137;
                double t6139 = sqrt(t6138);
                double t6140 = exp(-Cinf*Fg*Pg_1*t6135*t6136*t6139);
                v_gamma_aa[Q] += scale * (t6140*t6133*t6134*1.0/pow(t6119,4.0/3.0)-Cinf*Fg*Pg_1*t6140*t6134*1.0/pow(t6119,5.0/2.0)*t6139*(1.0/2.0));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t6142 = rho_a+rho_b;
                double t6143 = c*c;
                double t6144 = 1.0/pow(t6142,2.0/3.0);
                double t6145 = 1.0/pow(t6142,1.0/3.0);
                double t6146 = Bg*t6143*t6144;
                double t6147 = Cg_2*c*t6145;
                double t6148 = Cg_1+t6146+t6147;
                double t6149 = 1.0/t6142;
                double t6150 = Bg*c*t6143*t6149*1.0E4;
                double t6151 = Cg_4*t6143*t6144;
                double t6152 = Cg_3*c*t6145;
                double t6153 = t6150+t6151+t6152+1.0;
                double t6154 = 1.0/t6153;
                double t6155 = t6154*t6148;
                double t6156 = Cx+t6155;
                double t6157 = 1.0/two_13;
                double t6158 = 1.0/pow(t6142,7.0/6.0);
                double t6159 = 1.0/t6156;
                double t6160 = gamma_ab*2.0;
                double t6161 = gamma_aa+gamma_bb+t6160;
                double t6162 = sqrt(t6161);
                double t6163 = exp(-Cinf*Fg*Pg_1*t6162*t6158*t6159);
                v_gamma_ab[Q] += scale * (1.0/pow(t6142,4.0/3.0)*t6163*t6156*t6157*2.0-Cinf*Fg*Pg_1*1.0/pow(t6142,5.0/2.0)*t6162*t6163*t6157);
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t6165 = rho_a+rho_b;
                double t6166 = c*c;
                double t6167 = 1.0/pow(t6165,2.0/3.0);
                double t6168 = 1.0/pow(t6165,1.0/3.0);
                double t6169 = Bg*t6166*t6167;
                double t6170 = Cg_2*c*t6168;
                double t6171 = Cg_1+t6170+t6169;
                double t6172 = 1.0/t6165;
                double t6173 = Bg*c*t6172*t6166*1.0E4;
                double t6174 = Cg_4*t6166*t6167;
                double t6175 = Cg_3*c*t6168;
                double t6176 = t6173+t6174+t6175+1.0;
                double t6177 = 1.0/t6176;
                double t6178 = t6171*t6177;
                double t6179 = Cx+t6178;
                double t6180 = 1.0/two_13;
                double t6181 = 1.0/pow(t6165,7.0/6.0);
                double t6182 = 1.0/t6179;
                double t6183 = gamma_ab*2.0;
                double t6184 = gamma_aa+gamma_bb+t6183;
                double t6185 = sqrt(t6184);
                double t6186 = exp(-Cinf*Fg*Pg_1*t6181*t6182*t6185);
                v_gamma_bb[Q] += scale * (t6180*1.0/pow(t6165,4.0/3.0)*t6186*t6179-Cinf*Fg*Pg_1*t6180*1.0/pow(t6165,5.0/2.0)*t6185*t6186*(1.0/2.0));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t6190 = rho_a+rho_b;
                double t6191 = 1.0/pow(t6190,4.0/3.0);
                double t6192 = 1.0/pow(t6190,1.0/3.0);
                double t6193 = c*t6192;
                double t6194 = 1.0/sqrt(t6193);
                double t6195 = sqrt(t6193);
                double t6196 = EcPld_3*c*t6191*(1.0/3.0);
                double t6197 = EcPld_2*c*t6191*t6194*(1.0/6.0);
                double t6198 = t6196+t6197;
                double t6199 = EcPld_2*t6195;
                double t6200 = EcPld_3*c*t6192;
                double t6201 = t6199+t6200+1.0;
                double t6202 = 1.0/(t6201*t6201);
                double t6203 = EcPld_1*t6198*t6202;
                double t6204 = t6193-1.0;
                double t6205 = heaviside(t6204);
                double t6206 = pow(2.0,1.0/3.0);
                double t6207 = t6206*2.0;
                double t6208 = t6207-2.0;
                double t6209 = two_13*2.0;
                double t6210 = t6209-2.0;
                double t6211 = 1.0/t6210;
                double t6212 = EcFld_3*c*t6191*(1.0/3.0);
                double t6213 = EcFld_2*c*t6191*t6194*(1.0/6.0);
                double t6214 = t6212+t6213;
                double t6215 = EcFld_2*t6195;
                double t6216 = EcFld_3*c*t6192;
                double t6217 = t6215+t6216+1.0;
                double t6218 = t6198*t6198;
                double t6219 = 1.0/(t6201*t6201*t6201);
                double t6220 = EcPld_1*t6218*t6219*2.0;
                double t6221 = 1.0/pow(t6190,7.0/3.0);
                double t6222 = 1.0/(t6217*t6217);
                double t6223 = c*c;
                double t6224 = 1.0/pow(t6190,8.0/3.0);
                double t6225 = 1.0/pow(t6193,3.0/2.0);
                double t6226 = EcPld_3*c*t6221*(4.0/9.0);
                double t6227 = EcPld_2*c*t6194*t6221*(2.0/9.0);
                double t6228 = t6226+t6227-EcPld_2*t6223*t6224*t6225*(1.0/3.6E1);
                double t6229 = EcPld_1*t6202*t6228;
                double t6230 = 1.0/(t6190*t6190);
                double t6231 = EcPhd_1*t6230*(1.0/3.0);
                double t6232 = log(t6193);
                double t6233 = EcPhd_3*c*t6221*(5.0/9.0);
                double t6234 = EcPhd_4*c*t6221*(4.0/9.0);
                double t6235 = EcPhd_3*c*t6221*t6232*(4.0/9.0);
                double t6236 = 1.0/t6201;
                double t6237 = EcPld_1*t6236;
                double t6238 = t6203-EcFld_1*t6222*t6214;
                double t6239 = t6203-t6211*t6208*t6238;
                double t6240 = dirac(t6204);
                double t6241 = EcPhd_1*t6232;
                double t6242 = 1.0/t6190;
                double t6243 = EcPhd_1*t6242*(1.0/3.0);
                double t6244 = EcPhd_3*c*t6191*(1.0/3.0);
                double t6245 = EcPhd_4*c*t6191*(1.0/3.0);
                double t6246 = EcPhd_3*c*t6191*t6232*(1.0/3.0);
                double t6247 = 1.0/t6217;
                double t6263 = EcFld_1*t6247;
                double t6248 = -t6263+t6237;
                double t6249 = t6237-t6211*t6208*t6248;
                double t6250 = dirac(t6204,1.0);
                double t6251 = EcFhd_1*t6232;
                double t6252 = EcFhd_4*c*t6192;
                double t6253 = EcPhd_4*c*t6192;
                double t6254 = EcFhd_3*c*t6192*t6232;
                double t6255 = EcPhd_3*c*t6192*t6232;
                double t6256 = EcFhd_2-EcPhd_2-t6241+t6251+t6252-t6253+t6254-t6255;
                double t6257 = t6211*t6208*t6256;
                double t6258 = EcPhd_2+t6241+t6253+t6255+t6257;
                double t6259 = -t6193+1.0;
                double t6260 = heaviside(t6259);
                double t6261 = t6243+t6244+t6245+t6246-EcFhd_1*t6242*(1.0/3.0)-EcFhd_3*c*t6191*(1.0/3.0)-EcFhd_4*c*t6191*(1.0/3.0)-EcFhd_3*c*t6191*t6232*(1.0/3.0);
                double t6262 = t6243+t6244+t6245+t6246-t6211*t6261*t6208;
                double t6264 = 1.0/pow(t6190,2.0/3.0);
                double t6265 = Bg*c*t6223*t6242*1.0E4;
                double t6266 = Cg_4*t6223*t6264;
                double t6267 = Cg_3*c*t6192;
                double t6268 = t6265+t6266+t6267+1.0;
                double t6269 = 1.0/t6268;
                double t6270 = 1.0/pow(t6190,5.0/3.0);
                double t6271 = Bg*t6223*t6264;
                double t6272 = Cg_2*c*t6192;
                double t6273 = Cg_1+t6271+t6272;
                double t6274 = gamma_ab*2.0;
                double t6275 = gamma_aa+gamma_bb+t6274;
                double t6276 = 1.0/two_13;
                double t6277 = 1.0/pow(t6190,7.0/6.0);
                double t6278 = t6273*t6269;
                double t6279 = Cx+t6278;
                double t6280 = 1.0/t6279;
                double t6281 = sqrt(t6275);
                double t6283 = Cinf*Fg*Pg_1*t6280*t6281*t6277;
                double t6282 = exp(-t6283);
                double t6284 = Bg*t6223*t6270*(2.0/3.0);
                double t6285 = Cg_2*c*t6191*(1.0/3.0);
                double t6286 = t6284+t6285;
                double t6287 = Bg*c*t6230*t6223*1.0E4;
                double t6288 = Cg_4*t6223*t6270*(2.0/3.0);
                double t6289 = Cg_3*c*t6191*(1.0/3.0);
                double t6290 = t6287+t6288+t6289;
                double t6291 = 1.0/(t6268*t6268);
                double t6292 = t6286*t6269;
                double t6307 = t6290*t6273*t6291;
                double t6293 = t6292-t6307;
                double t6294 = Bg*t6223*t6224*(1.0E1/9.0);
                double t6295 = Cg_2*c*t6221*(4.0/9.0);
                double t6296 = t6294+t6295;
                double t6297 = t6269*t6296;
                double t6298 = 1.0/(t6190*t6190*t6190);
                double t6299 = Bg*c*t6223*t6298*2.0E4;
                double t6300 = Cg_4*t6223*t6224*(1.0E1/9.0);
                double t6301 = Cg_3*c*t6221*(4.0/9.0);
                double t6302 = t6299+t6300+t6301;
                double t6303 = t6290*t6290;
                double t6304 = 1.0/(t6268*t6268*t6268);
                double t6305 = t6273*t6303*t6304*2.0;
                double t6306 = t6297+t6305-t6290*t6291*t6286*2.0-t6273*t6291*t6302;
                double t6308 = 1.0/(t6279*t6279);
                double t6309 = 1.0/pow(t6190,1.3E1/6.0);
                double t6310 = Cinf*Fg*Pg_1*t6280*t6281*t6309*(7.0/6.0);
                double t6312 = Cinf*Fg*Pg_1*t6281*t6293*t6277*t6308;
                double t6311 = t6310-t6312;
                v_rho_a_rho_a[Q] += scale * (-t6190*(-t6260*(t6231+t6233+t6234+t6235-t6211*t6208*(t6231+t6233+t6234+t6235-EcFhd_1*t6230*(1.0/3.0)-EcFhd_3*c*t6221*(5.0/9.0)-EcFhd_4*c*t6221*(4.0/9.0)-EcFhd_3*c*t6221*t6232*(4.0/9.0)))+t6205*(-t6220+t6229+t6211*t6208*(t6220-t6229-EcFld_1*(t6214*t6214)*1.0/(t6217*t6217*t6217)*2.0+EcFld_1*t6222*(EcFld_3*c*t6221*(4.0/9.0)-EcFld_2*t6223*t6224*t6225*(1.0/3.6E1)+EcFld_2*c*t6194*t6221*(2.0/9.0))))+c*t6191*t6240*t6262*(2.0/3.0)+c*t6191*t6240*t6239*(2.0/3.0)-c*t6221*t6240*t6249*(4.0/9.0)+c*t6221*t6240*t6258*(4.0/9.0)-t6223*t6250*t6224*t6249*(1.0/9.0)+t6223*t6250*t6224*t6258*(1.0/9.0))-t6260*t6262*2.0+t6205*t6239*2.0+c*t6191*t6240*t6258*(2.0/3.0)-c*t6191*t6240*(t6237+t6211*t6208*(t6263-t6237))*(2.0/3.0)+1.0/pow(t6190,1.0E1/3.0)*t6282*t6275*t6276*t6279*(2.8E1/9.0)+t6221*t6282*t6275*t6293*t6276*(8.0/3.0)+t6191*t6282*t6275*t6276*t6306-t6191*t6282*t6275*t6293*t6276*t6311*2.0-t6221*t6282*t6275*t6276*t6279*t6311*(8.0/3.0)+t6191*t6282*t6275*t6276*t6279*(t6311*t6311)-t6191*t6282*t6275*t6276*t6279*(Cinf*Fg*Pg_1*1.0/pow(t6190,1.9E1/6.0)*t6280*t6281*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t6281*t6277*t6306*t6308-Cinf*Fg*Pg_1*t6281*t6293*t6308*t6309*(7.0/3.0)+Cinf*Fg*Pg_1*t6281*(t6293*t6293)*t6277*1.0/(t6279*t6279*t6279)*2.0));
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t6314 = rho_a+rho_b;
                double t6315 = 1.0/pow(t6314,4.0/3.0);
                double t6316 = 1.0/pow(t6314,1.0/3.0);
                double t6317 = c*t6316;
                double t6318 = 1.0/sqrt(t6317);
                double t6319 = sqrt(t6317);
                double t6320 = EcPld_3*c*t6315*(1.0/3.0);
                double t6321 = EcPld_2*c*t6315*t6318*(1.0/6.0);
                double t6322 = t6320+t6321;
                double t6323 = EcPld_2*t6319;
                double t6324 = EcPld_3*c*t6316;
                double t6325 = t6323+t6324+1.0;
                double t6326 = 1.0/(t6325*t6325);
                double t6327 = EcPld_1*t6322*t6326;
                double t6328 = t6317-1.0;
                double t6329 = heaviside(t6328);
                double t6330 = pow(2.0,1.0/3.0);
                double t6331 = t6330*2.0;
                double t6332 = t6331-2.0;
                double t6333 = two_13*2.0;
                double t6334 = t6333-2.0;
                double t6335 = 1.0/t6334;
                double t6336 = EcFld_3*c*t6315*(1.0/3.0);
                double t6337 = EcFld_2*c*t6315*t6318*(1.0/6.0);
                double t6338 = t6336+t6337;
                double t6339 = EcFld_2*t6319;
                double t6340 = EcFld_3*c*t6316;
                double t6341 = t6340+t6339+1.0;
                double t6342 = t6322*t6322;
                double t6343 = 1.0/(t6325*t6325*t6325);
                double t6344 = EcPld_1*t6342*t6343*2.0;
                double t6345 = 1.0/pow(t6314,7.0/3.0);
                double t6346 = 1.0/(t6341*t6341);
                double t6347 = c*c;
                double t6348 = 1.0/pow(t6314,8.0/3.0);
                double t6349 = 1.0/pow(t6317,3.0/2.0);
                double t6350 = EcPld_3*c*t6345*(4.0/9.0);
                double t6351 = EcPld_2*c*t6318*t6345*(2.0/9.0);
                double t6352 = t6350+t6351-EcPld_2*t6347*t6348*t6349*(1.0/3.6E1);
                double t6353 = EcPld_1*t6352*t6326;
                double t6354 = 1.0/(t6314*t6314);
                double t6355 = EcPhd_1*t6354*(1.0/3.0);
                double t6356 = log(t6317);
                double t6357 = EcPhd_3*c*t6345*(5.0/9.0);
                double t6358 = EcPhd_4*c*t6345*(4.0/9.0);
                double t6359 = EcPhd_3*c*t6345*t6356*(4.0/9.0);
                double t6360 = 1.0/t6325;
                double t6361 = EcPld_1*t6360;
                double t6362 = t6327-EcFld_1*t6346*t6338;
                double t6363 = t6327-t6332*t6335*t6362;
                double t6364 = dirac(t6328);
                double t6365 = EcPhd_1*t6356;
                double t6366 = 1.0/t6314;
                double t6367 = EcPhd_1*t6366*(1.0/3.0);
                double t6368 = EcPhd_3*c*t6315*(1.0/3.0);
                double t6369 = EcPhd_4*c*t6315*(1.0/3.0);
                double t6370 = EcPhd_3*c*t6315*t6356*(1.0/3.0);
                double t6371 = 1.0/t6341;
                double t6387 = EcFld_1*t6371;
                double t6372 = t6361-t6387;
                double t6388 = t6332*t6335*t6372;
                double t6373 = t6361-t6388;
                double t6374 = dirac(t6328,1.0);
                double t6375 = EcFhd_1*t6356;
                double t6376 = EcFhd_4*c*t6316;
                double t6377 = EcPhd_4*c*t6316;
                double t6378 = EcFhd_3*c*t6316*t6356;
                double t6379 = EcPhd_3*c*t6316*t6356;
                double t6380 = EcFhd_2-EcPhd_2-t6365+t6375+t6376-t6377+t6378-t6379;
                double t6381 = t6332*t6335*t6380;
                double t6382 = EcPhd_2+t6381+t6365+t6377+t6379;
                double t6383 = -t6317+1.0;
                double t6384 = heaviside(t6383);
                double t6385 = t6370+t6367+t6368+t6369-EcFhd_1*t6366*(1.0/3.0)-EcFhd_3*c*t6315*(1.0/3.0)-EcFhd_4*c*t6315*(1.0/3.0)-EcFhd_3*c*t6315*t6356*(1.0/3.0);
                double t6386 = t6370+t6367+t6368+t6369-t6332*t6335*t6385;
                double t6389 = 1.0/pow(t6314,2.0/3.0);
                double t6390 = Bg*c*t6347*t6366*1.0E4;
                double t6391 = Cg_4*t6347*t6389;
                double t6392 = Cg_3*c*t6316;
                double t6393 = t6390+t6391+t6392+1.0;
                double t6394 = 1.0/t6393;
                double t6395 = 1.0/pow(t6314,5.0/3.0);
                double t6396 = Bg*t6347*t6389;
                double t6397 = Cg_2*c*t6316;
                double t6398 = Cg_1+t6396+t6397;
                double t6399 = gamma_ab*2.0;
                double t6400 = gamma_aa+gamma_bb+t6399;
                double t6401 = 1.0/two_13;
                double t6402 = 1.0/pow(t6314,7.0/6.0);
                double t6403 = t6394*t6398;
                double t6404 = Cx+t6403;
                double t6405 = 1.0/t6404;
                double t6406 = sqrt(t6400);
                double t6408 = Cinf*Fg*Pg_1*t6402*t6405*t6406;
                double t6407 = exp(-t6408);
                double t6409 = Bg*t6347*t6395*(2.0/3.0);
                double t6410 = Cg_2*c*t6315*(1.0/3.0);
                double t6411 = t6410+t6409;
                double t6412 = Bg*c*t6354*t6347*1.0E4;
                double t6413 = Cg_4*t6347*t6395*(2.0/3.0);
                double t6414 = Cg_3*c*t6315*(1.0/3.0);
                double t6415 = t6412+t6413+t6414;
                double t6416 = 1.0/(t6393*t6393);
                double t6417 = t6394*t6411;
                double t6432 = t6398*t6415*t6416;
                double t6418 = -t6432+t6417;
                double t6419 = Bg*t6347*t6348*(1.0E1/9.0);
                double t6420 = Cg_2*c*t6345*(4.0/9.0);
                double t6421 = t6420+t6419;
                double t6422 = t6394*t6421;
                double t6423 = 1.0/(t6314*t6314*t6314);
                double t6424 = Bg*c*t6347*t6423*2.0E4;
                double t6425 = Cg_4*t6347*t6348*(1.0E1/9.0);
                double t6426 = Cg_3*c*t6345*(4.0/9.0);
                double t6427 = t6424+t6425+t6426;
                double t6428 = t6415*t6415;
                double t6429 = 1.0/(t6393*t6393*t6393);
                double t6430 = t6398*t6428*t6429*2.0;
                double t6431 = t6430+t6422-t6398*t6416*t6427-t6411*t6415*t6416*2.0;
                double t6433 = 1.0/(t6404*t6404);
                double t6434 = 1.0/pow(t6314,1.3E1/6.0);
                double t6435 = Cinf*Fg*Pg_1*t6405*t6406*t6434*(7.0/6.0);
                double t6437 = Cinf*Fg*Pg_1*t6402*t6406*t6433*t6418;
                double t6436 = t6435-t6437;
                v_rho_a_rho_b[Q] += scale * (-t6314*(-t6384*(t6355+t6357+t6358+t6359-t6332*t6335*(t6355+t6357+t6358+t6359-EcFhd_1*t6354*(1.0/3.0)-EcFhd_3*c*t6345*(5.0/9.0)-EcFhd_4*c*t6345*(4.0/9.0)-EcFhd_3*c*t6345*t6356*(4.0/9.0)))+t6329*(-t6344+t6353+t6332*t6335*(t6344-t6353-EcFld_1*1.0/(t6341*t6341*t6341)*(t6338*t6338)*2.0+EcFld_1*t6346*(EcFld_3*c*t6345*(4.0/9.0)-EcFld_2*t6347*t6348*t6349*(1.0/3.6E1)+EcFld_2*c*t6318*t6345*(2.0/9.0))))+c*t6315*t6363*t6364*(2.0/3.0)-c*t6345*t6364*t6373*(4.0/9.0)+c*t6345*t6364*t6382*(4.0/9.0)+c*t6315*t6364*t6386*(2.0/3.0)-t6373*t6347*t6374*t6348*(1.0/9.0)+t6382*t6347*t6374*t6348*(1.0/9.0))+t6363*t6329*2.0-t6384*t6386*2.0-c*t6315*t6364*t6373*(2.0/3.0)+c*t6315*t6364*t6382*(2.0/3.0)+1.0/pow(t6314,1.0E1/3.0)*t6400*t6401*t6404*t6407*(2.8E1/9.0)+t6315*t6400*t6401*t6431*t6407+t6345*t6400*t6401*t6407*t6418*(8.0/3.0)-t6345*t6400*t6401*t6404*t6407*t6436*(8.0/3.0)-t6315*t6400*t6401*t6407*t6418*t6436*2.0+t6315*t6400*t6401*t6404*t6407*(t6436*t6436)-t6315*t6400*t6401*t6404*t6407*(Cinf*Fg*Pg_1*1.0/pow(t6314,1.9E1/6.0)*t6405*t6406*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t6402*t6431*t6406*t6433-Cinf*Fg*Pg_1*t6406*t6433*t6434*t6418*(7.0/3.0)+Cinf*Fg*Pg_1*t6402*1.0/(t6404*t6404*t6404)*t6406*(t6418*t6418)*2.0));
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t6439 = rho_a+rho_b;
                double t6440 = 1.0/pow(t6439,4.0/3.0);
                double t6441 = 1.0/pow(t6439,1.0/3.0);
                double t6442 = c*t6441;
                double t6443 = 1.0/sqrt(t6442);
                double t6444 = sqrt(t6442);
                double t6445 = EcPld_3*c*t6440*(1.0/3.0);
                double t6446 = EcPld_2*c*t6440*t6443*(1.0/6.0);
                double t6447 = t6445+t6446;
                double t6448 = EcPld_2*t6444;
                double t6449 = EcPld_3*c*t6441;
                double t6450 = t6448+t6449+1.0;
                double t6451 = 1.0/(t6450*t6450);
                double t6452 = EcPld_1*t6451*t6447;
                double t6453 = t6442-1.0;
                double t6454 = heaviside(t6453);
                double t6455 = pow(2.0,1.0/3.0);
                double t6456 = t6455*2.0;
                double t6457 = t6456-2.0;
                double t6458 = two_13*2.0;
                double t6459 = t6458-2.0;
                double t6460 = 1.0/t6459;
                double t6461 = EcFld_3*c*t6440*(1.0/3.0);
                double t6462 = EcFld_2*c*t6440*t6443*(1.0/6.0);
                double t6463 = t6461+t6462;
                double t6464 = EcFld_2*t6444;
                double t6465 = EcFld_3*c*t6441;
                double t6466 = t6464+t6465+1.0;
                double t6467 = t6447*t6447;
                double t6468 = 1.0/(t6450*t6450*t6450);
                double t6469 = EcPld_1*t6467*t6468*2.0;
                double t6470 = 1.0/pow(t6439,7.0/3.0);
                double t6471 = 1.0/(t6466*t6466);
                double t6472 = c*c;
                double t6473 = 1.0/pow(t6439,8.0/3.0);
                double t6474 = 1.0/pow(t6442,3.0/2.0);
                double t6475 = EcPld_3*c*t6470*(4.0/9.0);
                double t6476 = EcPld_2*c*t6443*t6470*(2.0/9.0);
                double t6477 = t6475+t6476-EcPld_2*t6472*t6473*t6474*(1.0/3.6E1);
                double t6478 = EcPld_1*t6451*t6477;
                double t6479 = 1.0/(t6439*t6439);
                double t6480 = EcPhd_1*t6479*(1.0/3.0);
                double t6481 = log(t6442);
                double t6482 = EcPhd_3*c*t6470*(5.0/9.0);
                double t6483 = EcPhd_4*c*t6470*(4.0/9.0);
                double t6484 = EcPhd_3*c*t6470*t6481*(4.0/9.0);
                double t6485 = 1.0/t6450;
                double t6486 = EcPld_1*t6485;
                double t6487 = t6452-EcFld_1*t6471*t6463;
                double t6488 = t6452-t6460*t6457*t6487;
                double t6489 = dirac(t6453);
                double t6490 = EcPhd_1*t6481;
                double t6491 = 1.0/t6439;
                double t6492 = EcPhd_1*t6491*(1.0/3.0);
                double t6493 = EcPhd_3*c*t6440*(1.0/3.0);
                double t6494 = EcPhd_4*c*t6440*(1.0/3.0);
                double t6495 = EcPhd_3*c*t6440*t6481*(1.0/3.0);
                double t6496 = 1.0/t6466;
                double t6512 = EcFld_1*t6496;
                double t6497 = t6486-t6512;
                double t6513 = t6460*t6457*t6497;
                double t6498 = t6486-t6513;
                double t6499 = dirac(t6453,1.0);
                double t6500 = EcFhd_1*t6481;
                double t6501 = EcFhd_4*c*t6441;
                double t6502 = EcPhd_4*c*t6441;
                double t6503 = EcFhd_3*c*t6441*t6481;
                double t6504 = EcPhd_3*c*t6441*t6481;
                double t6505 = EcFhd_2-EcPhd_2-t6490+t6500+t6501-t6502+t6503-t6504;
                double t6506 = t6460*t6457*t6505;
                double t6507 = EcPhd_2+t6490+t6502+t6504+t6506;
                double t6508 = -t6442+1.0;
                double t6509 = heaviside(t6508);
                double t6510 = t6492+t6493+t6494+t6495-EcFhd_1*t6491*(1.0/3.0)-EcFhd_3*c*t6440*(1.0/3.0)-EcFhd_4*c*t6440*(1.0/3.0)-EcFhd_3*c*t6440*t6481*(1.0/3.0);
                double t6511 = t6492+t6493+t6494+t6495-t6460*t6457*t6510;
                double t6514 = 1.0/pow(t6439,2.0/3.0);
                double t6515 = Bg*c*t6472*t6491*1.0E4;
                double t6516 = Cg_4*t6472*t6514;
                double t6517 = Cg_3*c*t6441;
                double t6518 = t6515+t6516+t6517+1.0;
                double t6519 = 1.0/t6518;
                double t6520 = 1.0/pow(t6439,5.0/3.0);
                double t6521 = Bg*t6472*t6514;
                double t6522 = Cg_2*c*t6441;
                double t6523 = Cg_1+t6521+t6522;
                double t6524 = gamma_ab*2.0;
                double t6525 = gamma_aa+gamma_bb+t6524;
                double t6526 = 1.0/two_13;
                double t6527 = 1.0/pow(t6439,7.0/6.0);
                double t6528 = t6523*t6519;
                double t6529 = Cx+t6528;
                double t6530 = 1.0/t6529;
                double t6531 = sqrt(t6525);
                double t6533 = Cinf*Fg*Pg_1*t6530*t6531*t6527;
                double t6532 = exp(-t6533);
                double t6534 = Bg*t6472*t6520*(2.0/3.0);
                double t6535 = Cg_2*c*t6440*(1.0/3.0);
                double t6536 = t6534+t6535;
                double t6537 = Bg*c*t6472*t6479*1.0E4;
                double t6538 = Cg_4*t6472*t6520*(2.0/3.0);
                double t6539 = Cg_3*c*t6440*(1.0/3.0);
                double t6540 = t6537+t6538+t6539;
                double t6541 = 1.0/(t6518*t6518);
                double t6542 = t6536*t6519;
                double t6557 = t6540*t6523*t6541;
                double t6543 = t6542-t6557;
                double t6544 = Bg*t6472*t6473*(1.0E1/9.0);
                double t6545 = Cg_2*c*t6470*(4.0/9.0);
                double t6546 = t6544+t6545;
                double t6547 = t6519*t6546;
                double t6548 = 1.0/(t6439*t6439*t6439);
                double t6549 = Bg*c*t6472*t6548*2.0E4;
                double t6550 = Cg_4*t6472*t6473*(1.0E1/9.0);
                double t6551 = Cg_3*c*t6470*(4.0/9.0);
                double t6552 = t6550+t6551+t6549;
                double t6553 = t6540*t6540;
                double t6554 = 1.0/(t6518*t6518*t6518);
                double t6555 = t6523*t6553*t6554*2.0;
                double t6556 = t6555+t6547-t6523*t6541*t6552-t6540*t6541*t6536*2.0;
                double t6558 = 1.0/(t6529*t6529);
                double t6559 = 1.0/pow(t6439,1.3E1/6.0);
                double t6560 = Cinf*Fg*Pg_1*t6530*t6531*t6559*(7.0/6.0);
                double t6562 = Cinf*Fg*Pg_1*t6531*t6543*t6527*t6558;
                double t6561 = t6560-t6562;
                v_rho_b_rho_b[Q] += scale * (-t6439*(-t6509*(t6480+t6482+t6483+t6484-t6460*t6457*(t6480+t6482+t6483+t6484-EcFhd_1*t6479*(1.0/3.0)-EcFhd_3*c*t6470*(5.0/9.0)-EcFhd_4*c*t6470*(4.0/9.0)-EcFhd_3*c*t6470*t6481*(4.0/9.0)))+t6454*(-t6469+t6478+t6460*t6457*(t6469-t6478-EcFld_1*(t6463*t6463)*1.0/(t6466*t6466*t6466)*2.0+EcFld_1*t6471*(EcFld_3*c*t6470*(4.0/9.0)-EcFld_2*t6472*t6473*t6474*(1.0/3.6E1)+EcFld_2*c*t6443*t6470*(2.0/9.0))))+c*t6440*t6488*t6489*(2.0/3.0)-c*t6470*t6489*t6498*(4.0/9.0)+c*t6440*t6489*t6511*(2.0/3.0)+c*t6470*t6489*t6507*(4.0/9.0)-t6472*t6473*t6498*t6499*(1.0/9.0)+t6472*t6473*t6499*t6507*(1.0/9.0))+t6454*t6488*2.0-t6511*t6509*2.0-c*t6440*t6489*t6498*(2.0/3.0)+c*t6440*t6489*t6507*(2.0/3.0)+1.0/pow(t6439,1.0E1/3.0)*t6532*t6525*t6526*t6529*(2.8E1/9.0)+t6470*t6532*t6525*t6543*t6526*(8.0/3.0)+t6440*t6532*t6525*t6526*t6556-t6440*t6532*t6525*t6543*t6561*t6526*2.0-t6470*t6532*t6525*t6561*t6526*t6529*(8.0/3.0)+t6440*t6532*t6525*(t6561*t6561)*t6526*t6529-t6440*t6532*t6525*t6526*t6529*(Cinf*Fg*Pg_1*1.0/pow(t6439,1.9E1/6.0)*t6530*t6531*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t6531*t6527*t6556*t6558-Cinf*Fg*Pg_1*t6531*t6543*t6558*t6559*(7.0/3.0)+Cinf*Fg*Pg_1*t6531*(t6543*t6543)*t6527*1.0/(t6529*t6529*t6529)*2.0));
            }
            
            // v_gamma_aa_gamma_aa
            if (deriv >= 2) {
                double t6828 = rho_a+rho_b;
                double t6829 = c*c;
                double t6830 = 1.0/pow(t6828,2.0/3.0);
                double t6831 = 1.0/pow(t6828,1.0/3.0);
                double t6832 = Bg*t6830*t6829;
                double t6833 = Cg_2*c*t6831;
                double t6834 = Cg_1+t6832+t6833;
                double t6835 = 1.0/t6828;
                double t6836 = Bg*c*t6835*t6829*1.0E4;
                double t6837 = Cg_4*t6830*t6829;
                double t6838 = Cg_3*c*t6831;
                double t6839 = t6836+t6837+t6838+1.0;
                double t6840 = 1.0/t6839;
                double t6841 = t6840*t6834;
                double t6842 = Cx+t6841;
                double t6843 = 1.0/t6842;
                double t6844 = 1.0/two_13;
                double t6845 = 1.0/pow(t6828,7.0/6.0);
                double t6846 = gamma_ab*2.0;
                double t6847 = gamma_aa+gamma_bb+t6846;
                double t6848 = sqrt(t6847);
                double t6849 = exp(-Cinf*Fg*Pg_1*t6843*t6845*t6848);
                v_gamma_aa_gamma_aa[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t6843*t6844*1.0/pow(t6828,1.1E1/3.0)*t6849*(1.0/4.0)-Cinf*Fg*Pg_1*t6844*1.0/pow(t6828,5.0/2.0)*1.0/sqrt(t6847)*t6849*(3.0/4.0));
            }
            
            // v_gamma_aa_gamma_ab
            if (deriv >= 2) {
                double t6851 = rho_a+rho_b;
                double t6852 = c*c;
                double t6853 = 1.0/pow(t6851,2.0/3.0);
                double t6854 = 1.0/pow(t6851,1.0/3.0);
                double t6855 = Bg*t6852*t6853;
                double t6856 = Cg_2*c*t6854;
                double t6857 = Cg_1+t6855+t6856;
                double t6858 = 1.0/t6851;
                double t6859 = Bg*c*t6852*t6858*1.0E4;
                double t6860 = Cg_4*t6852*t6853;
                double t6861 = Cg_3*c*t6854;
                double t6862 = t6860+t6861+t6859+1.0;
                double t6863 = 1.0/t6862;
                double t6864 = t6863*t6857;
                double t6865 = Cx+t6864;
                double t6866 = 1.0/t6865;
                double t6867 = 1.0/two_13;
                double t6868 = 1.0/pow(t6851,7.0/6.0);
                double t6869 = gamma_ab*2.0;
                double t6870 = gamma_aa+gamma_bb+t6869;
                double t6871 = sqrt(t6870);
                double t6872 = exp(-Cinf*Fg*Pg_1*t6871*t6866*t6868);
                v_gamma_aa_gamma_ab[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t6851,1.1E1/3.0)*t6872*t6866*t6867*(1.0/2.0)-Cinf*Fg*Pg_1*1.0/pow(t6851,5.0/2.0)*1.0/sqrt(t6870)*t6872*t6867*(3.0/2.0));
            }
            
            // v_gamma_aa_gamma_bb
            if (deriv >= 2) {
                double t6874 = rho_a+rho_b;
                double t6875 = c*c;
                double t6876 = 1.0/pow(t6874,2.0/3.0);
                double t6877 = 1.0/pow(t6874,1.0/3.0);
                double t6878 = Bg*t6875*t6876;
                double t6879 = Cg_2*c*t6877;
                double t6880 = Cg_1+t6878+t6879;
                double t6881 = 1.0/t6874;
                double t6882 = Bg*c*t6881*t6875*1.0E4;
                double t6883 = Cg_4*t6875*t6876;
                double t6884 = Cg_3*c*t6877;
                double t6885 = t6882+t6883+t6884+1.0;
                double t6886 = 1.0/t6885;
                double t6887 = t6880*t6886;
                double t6888 = Cx+t6887;
                double t6889 = 1.0/t6888;
                double t6890 = 1.0/two_13;
                double t6891 = 1.0/pow(t6874,7.0/6.0);
                double t6892 = gamma_ab*2.0;
                double t6893 = gamma_aa+gamma_bb+t6892;
                double t6894 = sqrt(t6893);
                double t6895 = exp(-Cinf*Fg*Pg_1*t6891*t6894*t6889);
                v_gamma_aa_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t6890*1.0/pow(t6874,1.1E1/3.0)*t6895*t6889*(1.0/4.0)-Cinf*Fg*Pg_1*t6890*1.0/pow(t6874,5.0/2.0)*1.0/sqrt(t6893)*t6895*(3.0/4.0));
            }
            
            // v_gamma_ab_gamma_ab
            if (deriv >= 2) {
                double t6897 = rho_a+rho_b;
                double t6898 = c*c;
                double t6899 = 1.0/pow(t6897,2.0/3.0);
                double t6900 = 1.0/pow(t6897,1.0/3.0);
                double t6901 = Bg*t6898*t6899;
                double t6902 = Cg_2*c*t6900;
                double t6903 = Cg_1+t6901+t6902;
                double t6904 = 1.0/t6897;
                double t6905 = Bg*c*t6898*t6904*1.0E4;
                double t6906 = Cg_4*t6898*t6899;
                double t6907 = Cg_3*c*t6900;
                double t6908 = t6905+t6906+t6907+1.0;
                double t6909 = 1.0/t6908;
                double t6910 = t6903*t6909;
                double t6911 = Cx+t6910;
                double t6912 = 1.0/t6911;
                double t6913 = 1.0/two_13;
                double t6914 = 1.0/pow(t6897,7.0/6.0);
                double t6915 = gamma_ab*2.0;
                double t6916 = gamma_aa+gamma_bb+t6915;
                double t6917 = sqrt(t6916);
                double t6918 = exp(-Cinf*Fg*Pg_1*t6912*t6914*t6917);
                v_gamma_ab_gamma_ab[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t6897,1.1E1/3.0)*t6912*t6913*t6918-Cinf*Fg*Pg_1*1.0/pow(t6897,5.0/2.0)*t6913*1.0/sqrt(t6916)*t6918*3.0);
            }
            
            // v_gamma_ab_gamma_bb
            if (deriv >= 2) {
                double t6920 = rho_a+rho_b;
                double t6921 = c*c;
                double t6922 = 1.0/pow(t6920,2.0/3.0);
                double t6923 = 1.0/pow(t6920,1.0/3.0);
                double t6924 = Bg*t6921*t6922;
                double t6925 = Cg_2*c*t6923;
                double t6926 = Cg_1+t6924+t6925;
                double t6927 = 1.0/t6920;
                double t6928 = Bg*c*t6921*t6927*1.0E4;
                double t6929 = Cg_4*t6921*t6922;
                double t6930 = Cg_3*c*t6923;
                double t6931 = t6930+t6928+t6929+1.0;
                double t6932 = 1.0/t6931;
                double t6933 = t6932*t6926;
                double t6934 = Cx+t6933;
                double t6935 = 1.0/t6934;
                double t6936 = 1.0/two_13;
                double t6937 = 1.0/pow(t6920,7.0/6.0);
                double t6938 = gamma_ab*2.0;
                double t6939 = gamma_aa+gamma_bb+t6938;
                double t6940 = sqrt(t6939);
                double t6941 = exp(-Cinf*Fg*Pg_1*t6940*t6935*t6937);
                v_gamma_ab_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t6920,1.1E1/3.0)*t6941*t6935*t6936*(1.0/2.0)-Cinf*Fg*Pg_1*1.0/pow(t6920,5.0/2.0)*t6941*t6936*1.0/sqrt(t6939)*(3.0/2.0));
            }
            
            // v_gamma_bb_gamma_bb
            if (deriv >= 2) {
                double t6943 = rho_a+rho_b;
                double t6944 = c*c;
                double t6945 = 1.0/pow(t6943,2.0/3.0);
                double t6946 = 1.0/pow(t6943,1.0/3.0);
                double t6947 = Bg*t6944*t6945;
                double t6948 = Cg_2*c*t6946;
                double t6949 = Cg_1+t6947+t6948;
                double t6950 = 1.0/t6943;
                double t6951 = Bg*c*t6950*t6944*1.0E4;
                double t6952 = Cg_4*t6944*t6945;
                double t6953 = Cg_3*c*t6946;
                double t6954 = t6951+t6952+t6953+1.0;
                double t6955 = 1.0/t6954;
                double t6956 = t6955*t6949;
                double t6957 = Cx+t6956;
                double t6958 = 1.0/t6957;
                double t6959 = 1.0/two_13;
                double t6960 = 1.0/pow(t6943,7.0/6.0);
                double t6961 = gamma_ab*2.0;
                double t6962 = gamma_aa+gamma_bb+t6961;
                double t6963 = sqrt(t6962);
                double t6964 = exp(-Cinf*Fg*Pg_1*t6960*t6963*t6958);
                v_gamma_bb_gamma_bb[Q] += scale * ((Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t6943,1.1E1/3.0)*t6964*t6958*t6959*(1.0/4.0)-Cinf*Fg*Pg_1*1.0/pow(t6943,5.0/2.0)*1.0/sqrt(t6962)*t6964*t6959*(3.0/4.0));
            }
            
            // v_rho_a_gamma_aa
            if (deriv >= 2) {
                double t6564 = rho_a+rho_b;
                double t6565 = c*c;
                double t6566 = 1.0/pow(t6564,2.0/3.0);
                double t6567 = 1.0/pow(t6564,1.0/3.0);
                double t6568 = 1.0/pow(t6564,4.0/3.0);
                double t6569 = 1.0/t6564;
                double t6570 = Bg*c*t6565*t6569*1.0E4;
                double t6571 = Cg_4*t6565*t6566;
                double t6572 = Cg_3*c*t6567;
                double t6573 = t6570+t6571+t6572+1.0;
                double t6574 = 1.0/t6573;
                double t6575 = 1.0/pow(t6564,5.0/3.0);
                double t6576 = Bg*t6565*t6566;
                double t6577 = Cg_2*c*t6567;
                double t6578 = Cg_1+t6576+t6577;
                double t6579 = 1.0/two_13;
                double t6580 = 1.0/pow(t6564,7.0/6.0);
                double t6581 = t6574*t6578;
                double t6582 = Cx+t6581;
                double t6583 = 1.0/t6582;
                double t6584 = gamma_ab*2.0;
                double t6585 = gamma_aa+gamma_bb+t6584;
                double t6586 = sqrt(t6585);
                double t6588 = Cinf*Fg*Pg_1*t6580*t6583*t6586;
                double t6587 = exp(-t6588);
                double t6589 = Bg*t6565*t6575*(2.0/3.0);
                double t6590 = Cg_2*c*t6568*(1.0/3.0);
                double t6591 = t6590+t6589;
                double t6592 = t6591*t6574;
                double t6593 = 1.0/(t6564*t6564);
                double t6594 = Bg*c*t6565*t6593*1.0E4;
                double t6595 = Cg_4*t6565*t6575*(2.0/3.0);
                double t6596 = Cg_3*c*t6568*(1.0/3.0);
                double t6597 = t6594+t6595+t6596;
                double t6598 = 1.0/(t6573*t6573);
                double t6601 = t6578*t6597*t6598;
                double t6599 = t6592-t6601;
                double t6600 = 1.0/pow(t6564,1.3E1/6.0);
                double t6602 = 1.0/(t6582*t6582);
                double t6603 = 1.0/sqrt(t6585);
                double t6604 = Cinf*Fg*Pg_1*t6583*t6586*t6600*(7.0/6.0);
                double t6605 = t6604-Cinf*Fg*Pg_1*t6580*t6586*t6599*t6602;
                double t6606 = 1.0/pow(t6564,5.0/2.0);
                v_rho_a_gamma_aa[Q] += scale * (-t6568*t6587*t6579*t6599-1.0/pow(t6564,7.0/3.0)*t6582*t6587*t6579*(4.0/3.0)+t6582*t6568*t6587*t6579*t6605+t6582*t6585*t6568*t6587*t6579*(Cinf*Fg*Pg_1*t6583*t6600*t6603*(7.0/1.2E1)-Cinf*Fg*Pg_1*t6580*t6599*t6602*t6603*(1.0/2.0))+Cinf*Fg*Pg_1*1.0/pow(t6564,7.0/2.0)*t6586*t6587*t6579*(2.0/3.0)-Cinf*Fg*Pg_1*t6586*t6587*t6579*t6605*t6606*(1.0/2.0)+Cinf*Fg*Pg_1*t6583*t6586*t6587*t6579*t6599*t6606*(1.0/2.0));
            }
            
            // v_rho_a_gamma_ab
            if (deriv >= 2) {
                double t6608 = rho_a+rho_b;
                double t6609 = c*c;
                double t6610 = 1.0/pow(t6608,2.0/3.0);
                double t6611 = 1.0/pow(t6608,1.0/3.0);
                double t6612 = 1.0/pow(t6608,4.0/3.0);
                double t6613 = 1.0/t6608;
                double t6614 = Bg*c*t6613*t6609*1.0E4;
                double t6615 = Cg_4*t6610*t6609;
                double t6616 = Cg_3*c*t6611;
                double t6617 = t6614+t6615+t6616+1.0;
                double t6618 = 1.0/t6617;
                double t6619 = 1.0/pow(t6608,5.0/3.0);
                double t6620 = Bg*t6610*t6609;
                double t6621 = Cg_2*c*t6611;
                double t6622 = Cg_1+t6620+t6621;
                double t6623 = 1.0/two_13;
                double t6624 = 1.0/pow(t6608,7.0/6.0);
                double t6625 = t6622*t6618;
                double t6626 = Cx+t6625;
                double t6627 = 1.0/t6626;
                double t6628 = gamma_ab*2.0;
                double t6629 = gamma_aa+gamma_bb+t6628;
                double t6630 = sqrt(t6629);
                double t6632 = Cinf*Fg*Pg_1*t6630*t6624*t6627;
                double t6631 = exp(-t6632);
                double t6633 = Bg*t6609*t6619*(2.0/3.0);
                double t6634 = Cg_2*c*t6612*(1.0/3.0);
                double t6635 = t6633+t6634;
                double t6636 = t6635*t6618;
                double t6637 = 1.0/(t6608*t6608);
                double t6638 = Bg*c*t6609*t6637*1.0E4;
                double t6639 = Cg_4*t6609*t6619*(2.0/3.0);
                double t6640 = Cg_3*c*t6612*(1.0/3.0);
                double t6641 = t6640+t6638+t6639;
                double t6642 = 1.0/(t6617*t6617);
                double t6645 = t6622*t6641*t6642;
                double t6643 = t6636-t6645;
                double t6644 = 1.0/pow(t6608,1.3E1/6.0);
                double t6646 = 1.0/(t6626*t6626);
                double t6647 = 1.0/sqrt(t6629);
                double t6648 = Cinf*Fg*Pg_1*t6630*t6644*t6627*(7.0/6.0);
                double t6649 = t6648-Cinf*Fg*Pg_1*t6630*t6624*t6643*t6646;
                double t6650 = 1.0/pow(t6608,5.0/2.0);
                v_rho_a_gamma_ab[Q] += scale * (t6612*t6631*t6623*t6643*-2.0-t6631*t6623*1.0/pow(t6608,7.0/3.0)*t6626*(8.0/3.0)+t6612*t6631*t6623*t6626*t6649*2.0+t6612*t6631*t6623*t6626*t6629*(Cinf*Fg*Pg_1*t6644*t6627*t6647*(7.0/6.0)-Cinf*Fg*Pg_1*t6624*t6643*t6646*t6647)+Cinf*Fg*Pg_1*t6630*t6631*t6623*1.0/pow(t6608,7.0/2.0)*(4.0/3.0)-Cinf*Fg*Pg_1*t6630*t6631*t6623*t6650*t6649+Cinf*Fg*Pg_1*t6630*t6631*t6623*t6650*t6643*t6627);
            }
            
            // v_rho_a_gamma_bb
            if (deriv >= 2) {
                double t6652 = rho_a+rho_b;
                double t6653 = c*c;
                double t6654 = 1.0/pow(t6652,2.0/3.0);
                double t6655 = 1.0/pow(t6652,1.0/3.0);
                double t6656 = 1.0/pow(t6652,4.0/3.0);
                double t6657 = 1.0/t6652;
                double t6658 = Bg*c*t6653*t6657*1.0E4;
                double t6659 = Cg_4*t6653*t6654;
                double t6660 = Cg_3*c*t6655;
                double t6661 = t6660+t6658+t6659+1.0;
                double t6662 = 1.0/t6661;
                double t6663 = 1.0/pow(t6652,5.0/3.0);
                double t6664 = Bg*t6653*t6654;
                double t6665 = Cg_2*c*t6655;
                double t6666 = Cg_1+t6664+t6665;
                double t6667 = 1.0/two_13;
                double t6668 = 1.0/pow(t6652,7.0/6.0);
                double t6669 = t6662*t6666;
                double t6670 = Cx+t6669;
                double t6671 = 1.0/t6670;
                double t6672 = gamma_ab*2.0;
                double t6673 = gamma_aa+gamma_bb+t6672;
                double t6674 = sqrt(t6673);
                double t6676 = Cinf*Fg*Pg_1*t6671*t6674*t6668;
                double t6675 = exp(-t6676);
                double t6677 = Bg*t6653*t6663*(2.0/3.0);
                double t6678 = Cg_2*c*t6656*(1.0/3.0);
                double t6679 = t6677+t6678;
                double t6680 = t6662*t6679;
                double t6681 = 1.0/(t6652*t6652);
                double t6682 = Bg*c*t6653*t6681*1.0E4;
                double t6683 = Cg_4*t6653*t6663*(2.0/3.0);
                double t6684 = Cg_3*c*t6656*(1.0/3.0);
                double t6685 = t6682+t6683+t6684;
                double t6686 = 1.0/(t6661*t6661);
                double t6689 = t6666*t6685*t6686;
                double t6687 = t6680-t6689;
                double t6688 = 1.0/pow(t6652,1.3E1/6.0);
                double t6690 = 1.0/(t6670*t6670);
                double t6691 = 1.0/sqrt(t6673);
                double t6692 = Cinf*Fg*Pg_1*t6671*t6674*t6688*(7.0/6.0);
                double t6693 = t6692-Cinf*Fg*Pg_1*t6690*t6674*t6668*t6687;
                double t6694 = 1.0/pow(t6652,5.0/2.0);
                v_rho_a_gamma_bb[Q] += scale * (-t6656*t6675*t6667*t6687-1.0/pow(t6652,7.0/3.0)*t6670*t6675*t6667*(4.0/3.0)+t6670*t6656*t6675*t6693*t6667+t6670*t6673*t6656*t6675*t6667*(Cinf*Fg*Pg_1*t6671*t6691*t6688*(7.0/1.2E1)-Cinf*Fg*Pg_1*t6690*t6691*t6668*t6687*(1.0/2.0))+Cinf*Fg*Pg_1*1.0/pow(t6652,7.0/2.0)*t6674*t6675*t6667*(2.0/3.0)-Cinf*Fg*Pg_1*t6674*t6675*t6693*t6667*t6694*(1.0/2.0)+Cinf*Fg*Pg_1*t6671*t6674*t6675*t6667*t6694*t6687*(1.0/2.0));
            }
            
            // v_rho_b_gamma_aa
            if (deriv >= 2) {
                double t6696 = rho_a+rho_b;
                double t6697 = c*c;
                double t6698 = 1.0/pow(t6696,2.0/3.0);
                double t6699 = 1.0/pow(t6696,1.0/3.0);
                double t6700 = 1.0/pow(t6696,4.0/3.0);
                double t6701 = 1.0/t6696;
                double t6702 = Bg*c*t6697*t6701*1.0E4;
                double t6703 = Cg_4*t6697*t6698;
                double t6704 = Cg_3*c*t6699;
                double t6705 = t6702+t6703+t6704+1.0;
                double t6706 = 1.0/t6705;
                double t6707 = 1.0/pow(t6696,5.0/3.0);
                double t6708 = Bg*t6697*t6698;
                double t6709 = Cg_2*c*t6699;
                double t6710 = Cg_1+t6708+t6709;
                double t6711 = 1.0/two_13;
                double t6712 = 1.0/pow(t6696,7.0/6.0);
                double t6713 = t6710*t6706;
                double t6714 = Cx+t6713;
                double t6715 = 1.0/t6714;
                double t6716 = gamma_ab*2.0;
                double t6717 = gamma_aa+gamma_bb+t6716;
                double t6718 = sqrt(t6717);
                double t6720 = Cinf*Fg*Pg_1*t6712*t6715*t6718;
                double t6719 = exp(-t6720);
                double t6721 = Bg*t6697*t6707*(2.0/3.0);
                double t6722 = Cg_2*c*t6700*(1.0/3.0);
                double t6723 = t6721+t6722;
                double t6724 = t6723*t6706;
                double t6725 = 1.0/(t6696*t6696);
                double t6726 = Bg*c*t6697*t6725*1.0E4;
                double t6727 = Cg_4*t6697*t6707*(2.0/3.0);
                double t6728 = Cg_3*c*t6700*(1.0/3.0);
                double t6729 = t6726+t6727+t6728;
                double t6730 = 1.0/(t6705*t6705);
                double t6733 = t6710*t6730*t6729;
                double t6731 = t6724-t6733;
                double t6732 = 1.0/pow(t6696,1.3E1/6.0);
                double t6734 = 1.0/(t6714*t6714);
                double t6735 = 1.0/sqrt(t6717);
                double t6736 = Cinf*Fg*Pg_1*t6732*t6715*t6718*(7.0/6.0);
                double t6737 = t6736-Cinf*Fg*Pg_1*t6712*t6731*t6734*t6718;
                double t6738 = 1.0/pow(t6696,5.0/2.0);
                v_rho_b_gamma_aa[Q] += scale * (-t6700*t6711*t6731*t6719-1.0/pow(t6696,7.0/3.0)*t6711*t6714*t6719*(4.0/3.0)+t6700*t6711*t6714*t6719*t6737+t6700*t6711*t6714*t6717*t6719*(Cinf*Fg*Pg_1*t6732*t6715*t6735*(7.0/1.2E1)-Cinf*Fg*Pg_1*t6712*t6731*t6734*t6735*(1.0/2.0))+Cinf*Fg*Pg_1*1.0/pow(t6696,7.0/2.0)*t6711*t6718*t6719*(2.0/3.0)-Cinf*Fg*Pg_1*t6711*t6718*t6719*t6737*t6738*(1.0/2.0)+Cinf*Fg*Pg_1*t6711*t6731*t6715*t6718*t6719*t6738*(1.0/2.0));
            }
            
            // v_rho_b_gamma_ab
            if (deriv >= 2) {
                double t6740 = rho_a+rho_b;
                double t6741 = c*c;
                double t6742 = 1.0/pow(t6740,2.0/3.0);
                double t6743 = 1.0/pow(t6740,1.0/3.0);
                double t6744 = 1.0/pow(t6740,4.0/3.0);
                double t6745 = 1.0/t6740;
                double t6746 = Bg*c*t6741*t6745*1.0E4;
                double t6747 = Cg_4*t6741*t6742;
                double t6748 = Cg_3*c*t6743;
                double t6749 = t6746+t6747+t6748+1.0;
                double t6750 = 1.0/t6749;
                double t6751 = 1.0/pow(t6740,5.0/3.0);
                double t6752 = Bg*t6741*t6742;
                double t6753 = Cg_2*c*t6743;
                double t6754 = Cg_1+t6752+t6753;
                double t6755 = 1.0/two_13;
                double t6756 = 1.0/pow(t6740,7.0/6.0);
                double t6757 = t6750*t6754;
                double t6758 = Cx+t6757;
                double t6759 = 1.0/t6758;
                double t6760 = gamma_ab*2.0;
                double t6761 = gamma_aa+gamma_bb+t6760;
                double t6762 = sqrt(t6761);
                double t6764 = Cinf*Fg*Pg_1*t6762*t6756*t6759;
                double t6763 = exp(-t6764);
                double t6765 = Bg*t6741*t6751*(2.0/3.0);
                double t6766 = Cg_2*c*t6744*(1.0/3.0);
                double t6767 = t6765+t6766;
                double t6768 = t6750*t6767;
                double t6769 = 1.0/(t6740*t6740);
                double t6770 = Bg*c*t6741*t6769*1.0E4;
                double t6771 = Cg_4*t6741*t6751*(2.0/3.0);
                double t6772 = Cg_3*c*t6744*(1.0/3.0);
                double t6773 = t6770+t6771+t6772;
                double t6774 = 1.0/(t6749*t6749);
                double t6777 = t6754*t6773*t6774;
                double t6775 = t6768-t6777;
                double t6776 = 1.0/pow(t6740,1.3E1/6.0);
                double t6778 = 1.0/(t6758*t6758);
                double t6779 = 1.0/sqrt(t6761);
                double t6780 = Cinf*Fg*Pg_1*t6762*t6776*t6759*(7.0/6.0);
                double t6781 = t6780-Cinf*Fg*Pg_1*t6762*t6756*t6775*t6778;
                double t6782 = 1.0/pow(t6740,5.0/2.0);
                v_rho_b_gamma_ab[Q] += scale * (t6744*t6763*t6755*t6775*-2.0-1.0/pow(t6740,7.0/3.0)*t6763*t6755*t6758*(8.0/3.0)+t6744*t6763*t6781*t6755*t6758*2.0+t6761*t6744*t6763*t6755*t6758*(Cinf*Fg*Pg_1*t6776*t6759*t6779*(7.0/6.0)-Cinf*Fg*Pg_1*t6756*t6775*t6778*t6779)+Cinf*Fg*Pg_1*1.0/pow(t6740,7.0/2.0)*t6762*t6763*t6755*(4.0/3.0)-Cinf*Fg*Pg_1*t6762*t6763*t6781*t6755*t6782+Cinf*Fg*Pg_1*t6762*t6763*t6755*t6782*t6775*t6759);
            }
            
            // v_rho_b_gamma_bb
            if (deriv >= 2) {
                double t6784 = rho_a+rho_b;
                double t6785 = c*c;
                double t6786 = 1.0/pow(t6784,2.0/3.0);
                double t6787 = 1.0/pow(t6784,1.0/3.0);
                double t6788 = 1.0/pow(t6784,4.0/3.0);
                double t6789 = 1.0/t6784;
                double t6790 = Bg*c*t6785*t6789*1.0E4;
                double t6791 = Cg_4*t6785*t6786;
                double t6792 = Cg_3*c*t6787;
                double t6793 = t6790+t6791+t6792+1.0;
                double t6794 = 1.0/t6793;
                double t6795 = 1.0/pow(t6784,5.0/3.0);
                double t6796 = Bg*t6785*t6786;
                double t6797 = Cg_2*c*t6787;
                double t6798 = Cg_1+t6796+t6797;
                double t6799 = 1.0/two_13;
                double t6800 = 1.0/pow(t6784,7.0/6.0);
                double t6801 = t6794*t6798;
                double t6802 = Cx+t6801;
                double t6803 = 1.0/t6802;
                double t6804 = gamma_ab*2.0;
                double t6805 = gamma_aa+gamma_bb+t6804;
                double t6806 = sqrt(t6805);
                double t6808 = Cinf*Fg*Pg_1*t6800*t6803*t6806;
                double t6807 = exp(-t6808);
                double t6809 = Bg*t6785*t6795*(2.0/3.0);
                double t6810 = Cg_2*c*t6788*(1.0/3.0);
                double t6811 = t6810+t6809;
                double t6812 = t6794*t6811;
                double t6813 = 1.0/(t6784*t6784);
                double t6814 = Bg*c*t6785*t6813*1.0E4;
                double t6815 = Cg_4*t6785*t6795*(2.0/3.0);
                double t6816 = Cg_3*c*t6788*(1.0/3.0);
                double t6817 = t6814+t6815+t6816;
                double t6818 = 1.0/(t6793*t6793);
                double t6821 = t6798*t6817*t6818;
                double t6819 = t6812-t6821;
                double t6820 = 1.0/pow(t6784,1.3E1/6.0);
                double t6822 = 1.0/(t6802*t6802);
                double t6823 = 1.0/sqrt(t6805);
                double t6824 = Cinf*Fg*Pg_1*t6820*t6803*t6806*(7.0/6.0);
                double t6825 = t6824-Cinf*Fg*Pg_1*t6800*t6822*t6806*t6819;
                double t6826 = 1.0/pow(t6784,5.0/2.0);
                v_rho_b_gamma_bb[Q] += scale * (-t6788*t6799*t6807*t6819-1.0/pow(t6784,7.0/3.0)*t6799*t6802*t6807*(4.0/3.0)+t6788*t6799*t6802*t6807*t6825+t6788*t6799*t6802*t6805*t6807*(Cinf*Fg*Pg_1*t6820*t6803*t6823*(7.0/1.2E1)-Cinf*Fg*Pg_1*t6800*t6822*t6823*t6819*(1.0/2.0))+Cinf*Fg*Pg_1*1.0/pow(t6784,7.0/2.0)*t6799*t6806*t6807*(2.0/3.0)-Cinf*Fg*Pg_1*t6799*t6806*t6807*t6825*t6826*(1.0/2.0)+Cinf*Fg*Pg_1*t6799*t6803*t6806*t6807*t6826*t6819*(1.0/2.0));
            }
            
        } else {
            // v
            if (deriv >= 0) {
                double t3518 = rho_a+rho_b;
                double t3519 = 1.0/pow(t3518,1.0/3.0);
                double t3520 = 1.0/t3518;
                double t3521 = rho_a-rho_b;
                double t3522 = t3520*t3521;
                double t3523 = c*t3519;
                double t3524 = log(t3523);
                double t3525 = EcPhd_1*t3524;
                double t3526 = two_13*2.0;
                double t3527 = t3526-2.0;
                double t3528 = 1.0/t3527;
                double t3529 = sqrt(t3523);
                double t3530 = EcPld_2*t3529;
                double t3531 = EcPld_3*c*t3519;
                double t3532 = t3530+t3531+1.0;
                double t3533 = 1.0/t3532;
                double t3534 = EcPld_1*t3533;
                double t3535 = t3522+1.0;
                double t3536 = pow(t3535,4.0/3.0);
                double t3537 = -t3522+1.0;
                double t3538 = pow(t3537,4.0/3.0);
                double t3539 = t3536+t3538-2.0;
                double t3540 = c*c;
                double t3541 = 1.0/pow(t3518,2.0/3.0);
                double t3542 = Bg*t3540*t3541;
                double t3543 = Cg_2*c*t3519;
                double t3544 = Cg_1+t3542+t3543;
                double t3545 = Bg*c*t3520*t3540*1.0E4;
                double t3546 = Cg_4*t3540*t3541;
                double t3547 = Cg_3*c*t3519;
                double t3548 = t3545+t3546+t3547+1.0;
                double t3549 = 1.0/t3548;
                double t3550 = t3544*t3549;
                double t3551 = Cx+t3550;
                double t3552 = gamma_ab*2.0;
                double t3553 = gamma_aa+gamma_bb+t3552;
                v[Q] += scale * (t3518*(heaviside(-c*t3519+1.0)*(EcPhd_2+t3525+t3528*t3539*(EcFhd_2-EcPhd_2-t3525+EcFhd_1*t3524+EcFhd_4*c*t3519-EcPhd_4*c*t3519+EcFhd_3*c*t3524*t3519-EcPhd_3*c*t3524*t3519)+EcPhd_4*c*t3519+EcPhd_3*c*t3524*t3519)+heaviside(t3523-1.0)*(t3534-t3528*t3539*(t3534-EcFld_1/(EcFld_2*t3529+EcFld_3*c*t3519+1.0))))+(t3551*t3553*1.0/pow(t3518,4.0/3.0)*exp(-(Cinf*Fg*Pg_1*sqrt(t3553)*1.0/pow(t3518,7.0/6.0))/t3551)*1.0/sqrt(pow(t3520*t3521*(-1.0/2.0)+1.0/2.0,5.0/3.0)+pow(t3520*t3521*(1.0/2.0)+1.0/2.0,5.0/3.0)))/two_13);
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t3555 = rho_a+rho_b;
                double t3556 = 1.0/pow(t3555,1.0/3.0);
                double t3557 = 1.0/t3555;
                double t3558 = rho_a-rho_b;
                double t3559 = t3557*t3558;
                double t3560 = c*t3556;
                double t3561 = log(t3560);
                double t3562 = EcPhd_1*t3561;
                double t3563 = 1.0/pow(t3555,4.0/3.0);
                double t3564 = two_13*2.0;
                double t3565 = t3564-2.0;
                double t3566 = 1.0/t3565;
                double t3567 = t3559+1.0;
                double t3568 = pow(t3567,4.0/3.0);
                double t3569 = -t3559+1.0;
                double t3570 = pow(t3569,4.0/3.0);
                double t3571 = t3570+t3568-2.0;
                double t3572 = EcPhd_1*t3557*(1.0/3.0);
                double t3573 = EcPhd_3*c*t3563*(1.0/3.0);
                double t3574 = EcPhd_4*c*t3563*(1.0/3.0);
                double t3575 = 1.0/(t3555*t3555);
                double t3586 = t3575*t3558;
                double t3576 = t3557-t3586;
                double t3577 = EcFhd_1*t3561;
                double t3578 = EcFhd_4*c*t3556;
                double t3579 = EcPhd_4*c*t3556;
                double t3580 = EcFhd_3*c*t3561*t3556;
                double t3581 = EcPhd_3*c*t3561*t3556;
                double t3582 = EcPhd_3*c*t3561*t3563*(1.0/3.0);
                double t3583 = 1.0/sqrt(t3560);
                double t3584 = sqrt(t3560);
                double t3585 = pow(t3567,1.0/3.0);
                double t3587 = t3576*t3585*(4.0/3.0);
                double t3588 = pow(t3569,1.0/3.0);
                double t3589 = t3587-t3576*t3588*(4.0/3.0);
                double t3590 = EcFld_2*t3584;
                double t3591 = EcFld_3*c*t3556;
                double t3592 = t3590+t3591+1.0;
                double t3593 = EcPld_2*t3584;
                double t3594 = EcPld_3*c*t3556;
                double t3595 = t3593+t3594+1.0;
                double t3596 = EcPld_3*c*t3563*(1.0/3.0);
                double t3597 = EcPld_2*c*t3563*t3583*(1.0/6.0);
                double t3598 = t3596+t3597;
                double t3599 = 1.0/(t3595*t3595);
                double t3600 = t3560-1.0;
                double t3601 = EcFhd_2-EcPhd_2-t3562+t3580-t3581+t3577+t3578-t3579;
                double t3602 = dirac(t3600);
                double t3603 = 1.0/t3595;
                double t3604 = EcPld_1*t3603;
                double t3605 = 1.0/t3592;
                double t3608 = EcFld_1*t3605;
                double t3606 = t3604-t3608;
                double t3607 = heaviside(t3600);
                double t3609 = t3604-t3571*t3566*t3606;
                double t3610 = c*c;
                double t3611 = 1.0/pow(t3555,2.0/3.0);
                double t3612 = Bg*c*t3557*t3610*1.0E4;
                double t3613 = Cg_4*t3610*t3611;
                double t3614 = Cg_3*c*t3556;
                double t3615 = t3612+t3613+t3614+1.0;
                double t3616 = 1.0/t3615;
                double t3617 = 1.0/pow(t3555,5.0/3.0);
                double t3618 = Bg*t3610*t3611;
                double t3619 = Cg_2*c*t3556;
                double t3620 = Cg_1+t3618+t3619;
                double t3621 = gamma_ab*2.0;
                double t3622 = gamma_aa+gamma_bb+t3621;
                double t3623 = 1.0/two_13;
                double t3624 = 1.0/pow(t3555,7.0/6.0);
                double t3625 = t3620*t3616;
                double t3626 = Cx+t3625;
                double t3627 = 1.0/t3626;
                double t3628 = sqrt(t3622);
                double t3633 = Cinf*Fg*Pg_1*t3624*t3627*t3628;
                double t3629 = exp(-t3633);
                double t3630 = t3557*t3558*(1.0/2.0);
                double t3631 = t3630+1.0/2.0;
                double t3632 = pow(t3631,5.0/3.0);
                double t3634 = -t3630+1.0/2.0;
                double t3635 = pow(t3634,5.0/3.0);
                double t3636 = t3632+t3635;
                double t3637 = 1.0/sqrt(t3636);
                double t3638 = Bg*t3610*t3617*(2.0/3.0);
                double t3639 = Cg_2*c*t3563*(1.0/3.0);
                double t3640 = t3638+t3639;
                double t3641 = t3640*t3616;
                double t3642 = Bg*c*t3575*t3610*1.0E4;
                double t3643 = Cg_4*t3610*t3617*(2.0/3.0);
                double t3644 = Cg_3*c*t3563*(1.0/3.0);
                double t3645 = t3642+t3643+t3644;
                double t3646 = 1.0/(t3615*t3615);
                double t3647 = t3641-t3620*t3645*t3646;
                double t3648 = t3557*(1.0/2.0);
                double t3649 = t3648-t3575*t3558*(1.0/2.0);
                v_rho_a[Q] += scale * (-t3555*(heaviside(-t3560+1.0)*(t3572+t3573+t3582+t3574-t3571*t3566*(t3572+t3573+t3582+t3574-EcFhd_1*t3557*(1.0/3.0)-EcFhd_3*c*t3563*(1.0/3.0)-EcFhd_4*c*t3563*(1.0/3.0)-EcFhd_3*c*t3561*t3563*(1.0/3.0))-t3566*t3589*t3601)-t3607*(t3571*t3566*(EcFld_1*1.0/(t3592*t3592)*(EcFld_3*c*t3563*(1.0/3.0)+EcFld_2*c*t3563*t3583*(1.0/6.0))-EcPld_1*t3598*t3599)+EcPld_1*t3598*t3599-t3566*t3589*t3606)+c*t3563*t3602*t3609*(1.0/3.0)-c*t3563*t3602*(EcPhd_2+t3562+t3581+t3579+t3571*t3566*t3601)*(1.0/3.0))+t3607*t3609+heaviside(-c*t3556+1.0)*(EcPhd_2+t3562+t3581+t3579+t3571*t3566*(EcFhd_2-EcPhd_2-t3562+t3580+t3577+t3578-EcPhd_4*c*t3556-EcPhd_3*c*t3561*t3556))-1.0/pow(t3555,7.0/3.0)*t3622*t3623*t3626*t3637*t3629*(4.0/3.0)-t3563*t3622*t3623*t3629*t3647*1.0/sqrt(t3632+pow(t3557*t3558*(-1.0/2.0)+1.0/2.0,5.0/3.0))-t3563*t3622*t3623*t3626*1.0/pow(t3636,3.0/2.0)*t3629*(pow(t3631,2.0/3.0)*t3649*(5.0/3.0)-pow(t3634,2.0/3.0)*t3649*(5.0/3.0))*(1.0/2.0)+t3563*t3622*t3623*t3626*t3637*t3629*(Cinf*Fg*Pg_1*1.0/pow(t3555,1.3E1/6.0)*t3627*t3628*(7.0/6.0)-Cinf*Fg*Pg_1*t3624*1.0/(t3626*t3626)*t3628*t3647));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t3651 = rho_a+rho_b;
                double t3652 = 1.0/pow(t3651,1.0/3.0);
                double t3653 = 1.0/t3651;
                double t3654 = rho_a-rho_b;
                double t3655 = t3653*t3654;
                double t3656 = c*t3652;
                double t3657 = log(t3656);
                double t3658 = EcPhd_1*t3657;
                double t3659 = 1.0/pow(t3651,4.0/3.0);
                double t3660 = two_13*2.0;
                double t3661 = t3660-2.0;
                double t3662 = 1.0/t3661;
                double t3663 = t3655+1.0;
                double t3664 = pow(t3663,4.0/3.0);
                double t3665 = -t3655+1.0;
                double t3666 = pow(t3665,4.0/3.0);
                double t3667 = t3664+t3666-2.0;
                double t3668 = EcPhd_1*t3653*(1.0/3.0);
                double t3669 = EcPhd_3*c*t3659*(1.0/3.0);
                double t3670 = EcPhd_4*c*t3659*(1.0/3.0);
                double t3671 = 1.0/(t3651*t3651);
                double t3672 = t3671*t3654;
                double t3673 = t3653+t3672;
                double t3674 = EcFhd_1*t3657;
                double t3675 = EcFhd_4*c*t3652;
                double t3676 = EcPhd_4*c*t3652;
                double t3677 = EcFhd_3*c*t3652*t3657;
                double t3678 = EcPhd_3*c*t3652*t3657;
                double t3679 = EcPhd_3*c*t3657*t3659*(1.0/3.0);
                double t3680 = 1.0/sqrt(t3656);
                double t3681 = sqrt(t3656);
                double t3682 = pow(t3663,1.0/3.0);
                double t3683 = t3673*t3682*(4.0/3.0);
                double t3684 = pow(t3665,1.0/3.0);
                double t3685 = t3683-t3673*t3684*(4.0/3.0);
                double t3686 = EcFld_2*t3681;
                double t3687 = EcFld_3*c*t3652;
                double t3688 = t3686+t3687+1.0;
                double t3689 = EcPld_2*t3681;
                double t3690 = EcPld_3*c*t3652;
                double t3691 = t3690+t3689+1.0;
                double t3692 = EcPld_3*c*t3659*(1.0/3.0);
                double t3693 = EcPld_2*c*t3680*t3659*(1.0/6.0);
                double t3694 = t3692+t3693;
                double t3695 = 1.0/(t3691*t3691);
                double t3696 = t3656-1.0;
                double t3697 = EcFhd_2-EcPhd_2+t3674+t3675-t3658-t3676+t3677-t3678;
                double t3698 = dirac(t3696);
                double t3699 = 1.0/t3691;
                double t3700 = EcPld_1*t3699;
                double t3701 = 1.0/t3688;
                double t3704 = EcFld_1*t3701;
                double t3702 = t3700-t3704;
                double t3703 = heaviside(t3696);
                double t3705 = t3700-t3662*t3667*t3702;
                double t3706 = c*c;
                double t3707 = 1.0/pow(t3651,2.0/3.0);
                double t3708 = Bg*c*t3653*t3706*1.0E4;
                double t3709 = Cg_4*t3706*t3707;
                double t3710 = Cg_3*c*t3652;
                double t3711 = t3710+t3708+t3709+1.0;
                double t3712 = 1.0/t3711;
                double t3713 = 1.0/pow(t3651,5.0/3.0);
                double t3714 = Bg*t3706*t3707;
                double t3715 = Cg_2*c*t3652;
                double t3716 = Cg_1+t3714+t3715;
                double t3717 = gamma_ab*2.0;
                double t3718 = gamma_aa+gamma_bb+t3717;
                double t3719 = 1.0/two_13;
                double t3720 = 1.0/pow(t3651,7.0/6.0);
                double t3721 = t3712*t3716;
                double t3722 = Cx+t3721;
                double t3723 = 1.0/t3722;
                double t3724 = sqrt(t3718);
                double t3729 = Cinf*Fg*Pg_1*t3720*t3723*t3724;
                double t3725 = exp(-t3729);
                double t3726 = t3653*t3654*(1.0/2.0);
                double t3727 = t3726+1.0/2.0;
                double t3728 = pow(t3727,5.0/3.0);
                double t3730 = -t3726+1.0/2.0;
                double t3731 = pow(t3730,5.0/3.0);
                double t3732 = t3731+t3728;
                double t3733 = 1.0/sqrt(t3732);
                double t3734 = Bg*t3713*t3706*(2.0/3.0);
                double t3735 = Cg_2*c*t3659*(1.0/3.0);
                double t3736 = t3734+t3735;
                double t3737 = t3712*t3736;
                double t3738 = Bg*c*t3671*t3706*1.0E4;
                double t3739 = Cg_4*t3713*t3706*(2.0/3.0);
                double t3740 = Cg_3*c*t3659*(1.0/3.0);
                double t3741 = t3740+t3738+t3739;
                double t3742 = 1.0/(t3711*t3711);
                double t3743 = t3737-t3741*t3742*t3716;
                double t3744 = t3653*(1.0/2.0);
                double t3745 = t3671*t3654*(1.0/2.0);
                double t3746 = t3744+t3745;
                v_rho_b[Q] += scale * (-t3651*(heaviside(-t3656+1.0)*(t3670+t3668+t3669+t3679-t3662*t3667*(t3670+t3668+t3669+t3679-EcFhd_1*t3653*(1.0/3.0)-EcFhd_3*c*t3659*(1.0/3.0)-EcFhd_4*c*t3659*(1.0/3.0)-EcFhd_3*c*t3657*t3659*(1.0/3.0))+t3662*t3685*t3697)-t3703*(t3662*t3667*(EcFld_1*1.0/(t3688*t3688)*(EcFld_3*c*t3659*(1.0/3.0)+EcFld_2*c*t3680*t3659*(1.0/6.0))-EcPld_1*t3694*t3695)+EcPld_1*t3694*t3695+t3662*t3685*t3702)+c*t3659*t3698*t3705*(1.0/3.0)-c*t3659*t3698*(EcPhd_2+t3658+t3676+t3678+t3662*t3667*t3697)*(1.0/3.0))+t3703*t3705+heaviside(-c*t3652+1.0)*(EcPhd_2+t3658+t3676+t3678+t3662*t3667*(EcFhd_2-EcPhd_2+t3674+t3675-t3658+t3677-EcPhd_4*c*t3652-EcPhd_3*c*t3652*t3657))-1.0/pow(t3651,7.0/3.0)*t3722*t3733*t3725*t3718*t3719*(4.0/3.0)-t3659*t3725*t3743*t3718*t3719*1.0/sqrt(t3728+pow(t3653*t3654*(-1.0/2.0)+1.0/2.0,5.0/3.0))-t3659*t3722*1.0/pow(t3732,3.0/2.0)*t3725*t3718*t3719*(pow(t3730,2.0/3.0)*t3746*(5.0/3.0)-pow(t3727,2.0/3.0)*t3746*(5.0/3.0))*(1.0/2.0)+t3659*t3722*t3733*t3725*t3718*t3719*(Cinf*Fg*Pg_1*1.0/pow(t3651,1.3E1/6.0)*t3723*t3724*(7.0/6.0)-Cinf*Fg*Pg_1*t3720*1.0/(t3722*t3722)*t3724*t3743));
            }
            
            // v_gamma_aa
            if (deriv >= 1) {
                double t3748 = rho_a+rho_b;
                double t3749 = c*c;
                double t3750 = 1.0/pow(t3748,2.0/3.0);
                double t3751 = 1.0/pow(t3748,1.0/3.0);
                double t3752 = 1.0/t3748;
                double t3753 = rho_a-rho_b;
                double t3754 = Bg*t3750*t3749;
                double t3755 = Cg_2*c*t3751;
                double t3756 = Cg_1+t3754+t3755;
                double t3757 = Bg*c*t3752*t3749*1.0E4;
                double t3758 = Cg_4*t3750*t3749;
                double t3759 = Cg_3*c*t3751;
                double t3760 = t3757+t3758+t3759+1.0;
                double t3761 = 1.0/t3760;
                double t3762 = t3761*t3756;
                double t3763 = Cx+t3762;
                double t3764 = 1.0/two_13;
                double t3765 = 1.0/pow(t3748,7.0/6.0);
                double t3766 = 1.0/t3763;
                double t3767 = gamma_ab*2.0;
                double t3768 = gamma_aa+gamma_bb+t3767;
                double t3769 = sqrt(t3768);
                double t3770 = exp(-Cinf*Fg*Pg_1*t3765*t3766*t3769);
                double t3771 = t3752*t3753*(1.0/2.0);
                double t3772 = t3771+1.0/2.0;
                double t3773 = pow(t3772,5.0/3.0);
                v_gamma_aa[Q] += scale * (t3770*t3763*t3764*1.0/pow(t3748,4.0/3.0)*1.0/sqrt(t3773+pow(t3752*t3753*(-1.0/2.0)+1.0/2.0,5.0/3.0))-Cinf*Fg*Pg_1*t3770*t3764*1.0/pow(t3748,5.0/2.0)*t3769*1.0/sqrt(t3773+pow(-t3771+1.0/2.0,5.0/3.0))*(1.0/2.0));
            }
            
            // v_gamma_ab
            if (deriv >= 1) {
                double t3775 = rho_a+rho_b;
                double t3776 = c*c;
                double t3777 = 1.0/pow(t3775,2.0/3.0);
                double t3778 = 1.0/pow(t3775,1.0/3.0);
                double t3779 = 1.0/t3775;
                double t3780 = rho_a-rho_b;
                double t3781 = Bg*t3776*t3777;
                double t3782 = Cg_2*c*t3778;
                double t3783 = Cg_1+t3781+t3782;
                double t3784 = Bg*c*t3776*t3779*1.0E4;
                double t3785 = Cg_4*t3776*t3777;
                double t3786 = Cg_3*c*t3778;
                double t3787 = t3784+t3785+t3786+1.0;
                double t3788 = 1.0/t3787;
                double t3789 = t3783*t3788;
                double t3790 = Cx+t3789;
                double t3791 = 1.0/two_13;
                double t3792 = 1.0/pow(t3775,7.0/6.0);
                double t3793 = 1.0/t3790;
                double t3794 = gamma_ab*2.0;
                double t3795 = gamma_aa+gamma_bb+t3794;
                double t3796 = sqrt(t3795);
                double t3797 = exp(-Cinf*Fg*Pg_1*t3792*t3793*t3796);
                double t3798 = t3780*t3779*(1.0/2.0);
                double t3799 = t3798+1.0/2.0;
                double t3800 = pow(t3799,5.0/3.0);
                v_gamma_ab[Q] += scale * (t3790*t3791*1.0/pow(t3775,4.0/3.0)*t3797*1.0/sqrt(t3800+pow(t3780*t3779*(-1.0/2.0)+1.0/2.0,5.0/3.0))*2.0-Cinf*Fg*Pg_1*t3791*1.0/pow(t3775,5.0/2.0)*t3796*t3797*1.0/sqrt(t3800+pow(-t3798+1.0/2.0,5.0/3.0)));
            }
            
            // v_gamma_bb
            if (deriv >= 1) {
                double t3802 = rho_a+rho_b;
                double t3803 = c*c;
                double t3804 = 1.0/pow(t3802,2.0/3.0);
                double t3805 = 1.0/pow(t3802,1.0/3.0);
                double t3806 = 1.0/t3802;
                double t3807 = rho_a-rho_b;
                double t3808 = Bg*t3803*t3804;
                double t3809 = Cg_2*c*t3805;
                double t3810 = Cg_1+t3808+t3809;
                double t3811 = Bg*c*t3803*t3806*1.0E4;
                double t3812 = Cg_4*t3803*t3804;
                double t3813 = Cg_3*c*t3805;
                double t3814 = t3811+t3812+t3813+1.0;
                double t3815 = 1.0/t3814;
                double t3816 = t3810*t3815;
                double t3817 = Cx+t3816;
                double t3818 = 1.0/two_13;
                double t3819 = 1.0/pow(t3802,7.0/6.0);
                double t3820 = 1.0/t3817;
                double t3821 = gamma_ab*2.0;
                double t3822 = gamma_aa+gamma_bb+t3821;
                double t3823 = sqrt(t3822);
                double t3824 = exp(-Cinf*Fg*Pg_1*t3820*t3823*t3819);
                double t3825 = t3806*t3807*(1.0/2.0);
                double t3826 = t3825+1.0/2.0;
                double t3827 = pow(t3826,5.0/3.0);
                v_gamma_bb[Q] += scale * (1.0/pow(t3802,4.0/3.0)*t3824*t3817*t3818*1.0/sqrt(t3827+pow(t3806*t3807*(-1.0/2.0)+1.0/2.0,5.0/3.0))-Cinf*Fg*Pg_1*1.0/pow(t3802,5.0/2.0)*t3823*t3824*t3818*1.0/sqrt(t3827+pow(-t3825+1.0/2.0,5.0/3.0))*(1.0/2.0));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t3831 = rho_a+rho_b;
                double t3832 = 1.0/pow(t3831,4.0/3.0);
                double t3833 = 1.0/pow(t3831,1.0/3.0);
                double t3834 = c*t3833;
                double t3841 = 1.0/sqrt(t3834);
                double t3843 = EcPld_3*c*t3832*(1.0/3.0);
                double t3844 = EcPld_2*c*t3832*t3841*(1.0/6.0);
                double t3835 = t3843+t3844;
                double t3836 = 1.0/t3831;
                double t3837 = rho_a-rho_b;
                double t3838 = t3836*t3837;
                double t3839 = 1.0/(t3831*t3831);
                double t3876 = t3837*t3839;
                double t3840 = t3836-t3876;
                double t3842 = sqrt(t3834);
                double t3845 = EcPld_2*t3842;
                double t3846 = EcPld_3*c*t3833;
                double t3847 = t3845+t3846+1.0;
                double t3848 = two_13*2.0;
                double t3849 = t3848-2.0;
                double t3850 = 1.0/t3849;
                double t3851 = t3838+1.0;
                double t3852 = -t3838+1.0;
                double t3853 = EcFld_3*c*t3832*(1.0/3.0);
                double t3854 = EcFld_2*c*t3832*t3841*(1.0/6.0);
                double t3855 = t3853+t3854;
                double t3856 = EcFld_2*t3842;
                double t3857 = EcFld_3*c*t3833;
                double t3858 = t3856+t3857+1.0;
                double t3859 = t3835*t3835;
                double t3860 = 1.0/(t3847*t3847*t3847);
                double t3861 = EcPld_1*t3860*t3859*2.0;
                double t3862 = 1.0/pow(t3831,7.0/3.0);
                double t3863 = 1.0/(t3858*t3858);
                double t3864 = c*c;
                double t3865 = 1.0/pow(t3831,8.0/3.0);
                double t3866 = 1.0/pow(t3834,3.0/2.0);
                double t3867 = 1.0/(t3847*t3847);
                double t3868 = EcPld_3*c*t3862*(4.0/9.0);
                double t3869 = EcPld_2*c*t3841*t3862*(2.0/9.0);
                double t3870 = t3868+t3869-EcPld_2*t3864*t3865*t3866*(1.0/3.6E1);
                double t3871 = pow(t3851,1.0/3.0);
                double t3872 = pow(t3852,1.0/3.0);
                double t3873 = t3839*2.0;
                double t3874 = 1.0/(t3831*t3831*t3831);
                double t3878 = t3837*t3874*2.0;
                double t3875 = t3873-t3878;
                double t3877 = t3840*t3840;
                double t3879 = t3872*t3875*(4.0/3.0);
                double t3880 = 1.0/pow(t3851,2.0/3.0);
                double t3881 = t3880*t3877*(4.0/9.0);
                double t3882 = 1.0/pow(t3852,2.0/3.0);
                double t3883 = t3882*t3877*(4.0/9.0);
                double t3884 = t3881+t3883+t3879-t3871*t3875*(4.0/3.0);
                double t3885 = log(t3834);
                double t3886 = pow(t3851,4.0/3.0);
                double t3887 = pow(t3852,4.0/3.0);
                double t3888 = t3886+t3887-2.0;
                double t3889 = EcPhd_1*t3839*(1.0/3.0);
                double t3890 = EcPhd_3*c*t3862*(5.0/9.0);
                double t3891 = EcPhd_4*c*t3862*(4.0/9.0);
                double t3892 = t3840*t3871*(4.0/3.0);
                double t3912 = t3840*t3872*(4.0/3.0);
                double t3893 = t3892-t3912;
                double t3894 = EcPhd_3*c*t3862*t3885*(4.0/9.0);
                double t3895 = t3834-1.0;
                double t3896 = 1.0/t3847;
                double t3897 = EcPld_1*t3896;
                double t3898 = 1.0/t3858;
                double t3918 = EcFld_1*t3898;
                double t3899 = t3897-t3918;
                double t3900 = EcFhd_1*t3885;
                double t3901 = EcPhd_1*t3885;
                double t3902 = EcFhd_4*c*t3833;
                double t3903 = EcFhd_3*c*t3833*t3885;
                double t3904 = dirac(t3895);
                double t3905 = EcFhd_1*t3836*(1.0/3.0);
                double t3906 = EcPhd_1*t3836*(1.0/3.0);
                double t3907 = EcFhd_3*c*t3832*(1.0/3.0);
                double t3908 = EcFhd_4*c*t3832*(1.0/3.0);
                double t3909 = EcPhd_3*c*t3832*(1.0/3.0);
                double t3910 = EcPhd_4*c*t3832*(1.0/3.0);
                double t3911 = EcFhd_3*c*t3832*t3885*(1.0/3.0);
                double t3913 = EcPhd_4*c*t3833;
                double t3914 = EcPhd_3*c*t3833*t3885;
                double t3915 = EcPhd_3*c*t3832*t3885*(1.0/3.0);
                double t3916 = EcFld_1*t3863*t3855;
                double t3919 = EcPld_1*t3835*t3867;
                double t3917 = t3916-t3919;
                double t3933 = t3850*t3888*t3899;
                double t3920 = t3897-t3933;
                double t3921 = dirac(t3895,1.0);
                double t3922 = EcFhd_2-EcPhd_2+t3900-t3901+t3902+t3903-t3913-t3914;
                double t3923 = EcPld_1*t3870*t3867;
                double t3924 = t3910-t3911-t3905+t3906+t3915-t3907-t3908+t3909;
                double t3925 = t3850*t3888*t3922;
                double t3926 = EcPhd_2+t3901+t3913+t3914+t3925;
                double t3927 = -t3834+1.0;
                double t3928 = heaviside(t3927);
                double t3929 = t3910+t3906+t3915+t3909-t3850*t3893*t3922-t3850*t3888*t3924;
                double t3930 = heaviside(t3895);
                double t3931 = t3850*t3888*t3917;
                double t3932 = t3931+t3919-t3850*t3893*t3899;
                double t3934 = 1.0/pow(t3831,2.0/3.0);
                double t3935 = gamma_ab*2.0;
                double t3936 = gamma_aa+gamma_bb+t3935;
                double t3937 = Bg*c*t3836*t3864*1.0E4;
                double t3938 = Cg_4*t3864*t3934;
                double t3939 = Cg_3*c*t3833;
                double t3940 = t3937+t3938+t3939+1.0;
                double t3941 = 1.0/t3940;
                double t3942 = 1.0/pow(t3831,5.0/3.0);
                double t3943 = Bg*t3864*t3934;
                double t3944 = Cg_2*c*t3833;
                double t3945 = Cg_1+t3943+t3944;
                double t3946 = 1.0/(t3940*t3940);
                double t3947 = Bg*c*t3864*t3839*1.0E4;
                double t3948 = Cg_4*t3864*t3942*(2.0/3.0);
                double t3949 = Cg_3*c*t3832*(1.0/3.0);
                double t3950 = t3947+t3948+t3949;
                double t3951 = 1.0/two_13;
                double t3952 = 1.0/pow(t3831,7.0/6.0);
                double t3953 = t3941*t3945;
                double t3954 = Cx+t3953;
                double t3955 = 1.0/t3954;
                double t3956 = sqrt(t3936);
                double t3964 = Cinf*Fg*Pg_1*t3952*t3955*t3956;
                double t3957 = exp(-t3964);
                double t3958 = t3836*t3837*(1.0/2.0);
                double t3959 = t3958+1.0/2.0;
                double t3960 = pow(t3959,5.0/3.0);
                double t3961 = Bg*t3864*t3942*(2.0/3.0);
                double t3962 = Cg_2*c*t3832*(1.0/3.0);
                double t3963 = t3961+t3962;
                double t3965 = -t3958+1.0/2.0;
                double t3966 = pow(t3965,5.0/3.0);
                double t3967 = t3960+t3966;
                double t3968 = 1.0/sqrt(t3967);
                double t3969 = t3836*(1.0/2.0);
                double t3975 = t3837*t3839*(1.0/2.0);
                double t3970 = -t3975+t3969;
                double t3977 = pow(t3965,2.0/3.0);
                double t3978 = pow(t3959,2.0/3.0);
                double t3997 = t3970*t3977*(5.0/3.0);
                double t3998 = t3970*t3978*(5.0/3.0);
                double t3971 = t3997-t3998;
                double t3972 = t3941*t3963;
                double t3974 = t3950*t3945*t3946;
                double t3973 = t3972-t3974;
                double t3976 = t3970*t3970;
                double t3979 = t3839-t3837*t3874;
                double t3980 = 1.0/(t3954*t3954);
                double t3981 = Bg*t3864*t3865*(1.0E1/9.0);
                double t3982 = Cg_2*c*t3862*(4.0/9.0);
                double t3983 = t3981+t3982;
                double t3984 = t3941*t3983;
                double t3985 = Bg*c*t3864*t3874*2.0E4;
                double t3986 = Cg_4*t3864*t3865*(1.0E1/9.0);
                double t3987 = Cg_3*c*t3862*(4.0/9.0);
                double t3988 = t3985+t3986+t3987;
                double t3989 = t3950*t3950;
                double t3990 = 1.0/(t3940*t3940*t3940);
                double t3991 = t3945*t3990*t3989*2.0;
                double t3992 = t3991+t3984-t3950*t3963*t3946*2.0-t3945*t3946*t3988;
                double t3993 = 1.0/pow(t3831,1.3E1/6.0);
                double t3994 = Cinf*Fg*Pg_1*t3955*t3956*t3993*(7.0/6.0);
                double t3999 = Cinf*Fg*Pg_1*t3952*t3980*t3973*t3956;
                double t3995 = t3994-t3999;
                double t3996 = 1.0/pow(t3967,3.0/2.0);
                v_rho_a_rho_a[Q] += scale * (-t3831*(t3930*(-t3861+t3923+t3850*t3888*(t3861-t3923-EcFld_1*(t3855*t3855)*1.0/(t3858*t3858*t3858)*2.0+EcFld_1*t3863*(EcFld_3*c*t3862*(4.0/9.0)-EcFld_2*t3864*t3865*t3866*(1.0/3.6E1)+EcFld_2*c*t3841*t3862*(2.0/9.0)))+t3850*t3884*t3899-t3850*t3893*t3917*2.0)-t3928*(t3890+t3891+t3894+t3889-t3850*t3888*(t3890+t3891+t3894+t3889-EcFhd_1*t3839*(1.0/3.0)-EcFhd_3*c*t3862*(5.0/9.0)-EcFhd_4*c*t3862*(4.0/9.0)-EcFhd_3*c*t3862*t3885*(4.0/9.0))+t3850*t3884*t3922+t3850*t3893*t3924*2.0)+c*t3832*t3904*t3932*(2.0/3.0)-c*t3862*t3920*t3904*(4.0/9.0)+c*t3832*t3904*t3929*(2.0/3.0)+c*t3862*t3904*t3926*(4.0/9.0)-t3864*t3865*t3920*t3921*(1.0/9.0)+t3864*t3865*t3921*t3926*(1.0/9.0))+t3930*t3932*2.0-t3928*t3929*2.0-c*t3832*t3920*t3904*(2.0/3.0)+c*t3832*t3904*t3926*(2.0/3.0)+t3862*t3951*t3936*t3973*t3957*t3968*(8.0/3.0)+1.0/pow(t3831,1.0E1/3.0)*t3951*t3936*t3954*t3957*t3968*(2.8E1/9.0)+t3832*t3951*t3936*t3992*t3957*1.0/sqrt(t3960+pow(t3836*t3837*(-1.0/2.0)+1.0/2.0,5.0/3.0))+t3832*t3951*(t3971*t3971)*t3936*t3954*t3957*1.0/pow(t3967,5.0/2.0)*(3.0/4.0)-t3832*t3951*t3936*t3954*t3957*t3968*(Cinf*Fg*Pg_1*1.0/pow(t3831,1.9E1/6.0)*t3955*t3956*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t3952*t3980*t3956*t3992-Cinf*Fg*Pg_1*t3980*t3973*t3956*t3993*(7.0/3.0)+Cinf*Fg*Pg_1*t3952*1.0/(t3954*t3954*t3954)*(t3973*t3973)*t3956*2.0)-t3832*t3951*t3936*t3954*t3957*t3996*(t3977*t3979*(5.0/3.0)-t3978*t3979*(5.0/3.0)+1.0/pow(t3965,1.0/3.0)*t3976*(1.0E1/9.0)+t3976*1.0/pow(t3959,1.0/3.0)*(1.0E1/9.0))*(1.0/2.0)-t3832*t3951*t3971*t3936*t3973*t3957*t3996-t3862*t3951*t3971*t3936*t3954*t3957*t3996*(4.0/3.0)-t3832*t3951*t3936*t3973*t3957*t3968*t3995*2.0-t3862*t3951*t3936*t3954*t3957*t3968*t3995*(8.0/3.0)+t3832*t3951*t3936*t3954*t3957*t3968*(t3995*t3995)+t3832*t3951*t3971*t3936*t3954*t3957*t3995*t3996);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t4001 = rho_a+rho_b;
                double t4002 = rho_a-rho_b;
                double t4003 = 1.0/t4001;
                double t4004 = t4002*t4003;
                double t4005 = 1.0/(t4001*t4001*t4001);
                double t4006 = t4004+1.0;
                double t4007 = 1.0/(t4001*t4001);
                double t4008 = t4002*t4007;
                double t4009 = -t4004+1.0;
                double t4010 = t4003+t4008;
                double t4011 = t4003-t4008;
                double t4012 = 1.0/pow(t4001,1.0/3.0);
                double t4013 = c*t4012;
                double t4014 = log(t4013);
                double t4015 = 1.0/pow(t4001,7.0/3.0);
                double t4016 = two_13*2.0;
                double t4017 = t4016-2.0;
                double t4018 = 1.0/t4017;
                double t4019 = EcPhd_1*t4007*(1.0/3.0);
                double t4020 = EcPhd_3*c*t4015*(5.0/9.0);
                double t4021 = EcPhd_4*c*t4015*(4.0/9.0);
                double t4022 = pow(t4006,1.0/3.0);
                double t4023 = pow(t4009,1.0/3.0);
                double t4024 = 1.0/pow(t4001,4.0/3.0);
                double t4025 = EcFhd_1*t4003*(1.0/3.0);
                double t4026 = EcFhd_3*c*t4024*(1.0/3.0);
                double t4027 = EcFhd_4*c*t4024*(1.0/3.0);
                double t4028 = EcFhd_3*c*t4014*t4024*(1.0/3.0);
                double t4081 = EcPhd_1*t4003*(1.0/3.0);
                double t4082 = EcPhd_3*c*t4024*(1.0/3.0);
                double t4083 = EcPhd_4*c*t4024*(1.0/3.0);
                double t4084 = EcPhd_3*c*t4014*t4024*(1.0/3.0);
                double t4029 = t4025+t4026+t4027-t4081+t4028-t4082-t4083-t4084;
                double t4030 = EcPhd_3*c*t4014*t4015*(4.0/9.0);
                double t4034 = 1.0/sqrt(t4013);
                double t4036 = EcPld_3*c*t4024*(1.0/3.0);
                double t4037 = EcPld_2*c*t4024*t4034*(1.0/6.0);
                double t4031 = t4036+t4037;
                double t4032 = t4010*t4022*(4.0/3.0);
                double t4085 = t4010*t4023*(4.0/3.0);
                double t4033 = t4032-t4085;
                double t4035 = sqrt(t4013);
                double t4038 = EcPld_2*t4035;
                double t4039 = EcPld_3*c*t4012;
                double t4040 = t4038+t4039+1.0;
                double t4041 = t4011*t4022*(4.0/3.0);
                double t4089 = t4011*t4023*(4.0/3.0);
                double t4042 = t4041-t4089;
                double t4043 = EcFld_3*c*t4024*(1.0/3.0);
                double t4044 = EcFld_2*c*t4024*t4034*(1.0/6.0);
                double t4045 = t4043+t4044;
                double t4046 = EcFld_2*t4035;
                double t4047 = EcFld_3*c*t4012;
                double t4048 = t4046+t4047+1.0;
                double t4049 = 1.0/(t4048*t4048);
                double t4050 = EcFld_1*t4045*t4049;
                double t4051 = 1.0/(t4040*t4040);
                double t4091 = EcPld_1*t4031*t4051;
                double t4052 = t4050-t4091;
                double t4053 = pow(t4006,4.0/3.0);
                double t4054 = pow(t4009,4.0/3.0);
                double t4055 = t4053+t4054-2.0;
                double t4056 = t4031*t4031;
                double t4057 = 1.0/(t4040*t4040*t4040);
                double t4058 = EcPld_1*t4056*t4057*2.0;
                double t4059 = c*c;
                double t4060 = 1.0/pow(t4001,8.0/3.0);
                double t4061 = 1.0/pow(t4013,3.0/2.0);
                double t4062 = EcPld_3*c*t4015*(4.0/9.0);
                double t4063 = EcPld_2*c*t4015*t4034*(2.0/9.0);
                double t4064 = t4062+t4063-EcPld_2*t4060*t4061*t4059*(1.0/3.6E1);
                double t4065 = t4002*t4005*t4023*(8.0/3.0);
                double t4066 = 1.0/pow(t4006,2.0/3.0);
                double t4067 = t4010*t4011*t4066*(4.0/9.0);
                double t4068 = 1.0/pow(t4009,2.0/3.0);
                double t4069 = t4010*t4011*t4068*(4.0/9.0);
                double t4070 = t4065+t4067+t4069-t4002*t4022*t4005*(8.0/3.0);
                double t4071 = t4013-1.0;
                double t4072 = 1.0/t4040;
                double t4073 = EcPld_1*t4072;
                double t4074 = 1.0/t4048;
                double t4092 = EcFld_1*t4074;
                double t4075 = t4073-t4092;
                double t4076 = EcFhd_1*t4014;
                double t4077 = EcPhd_1*t4014;
                double t4078 = EcFhd_4*c*t4012;
                double t4079 = EcFhd_3*c*t4012*t4014;
                double t4080 = dirac(t4071);
                double t4086 = EcPhd_4*c*t4012;
                double t4087 = EcPhd_3*c*t4012*t4014;
                double t4088 = t4018*t4055*t4029;
                double t4090 = EcFhd_2-EcPhd_2+t4076-t4077-t4086+t4078-t4087+t4079;
                double t4093 = t4052*t4018*t4055;
                double t4105 = t4018*t4055*t4075;
                double t4094 = t4073-t4105;
                double t4095 = dirac(t4071,1.0);
                double t4096 = t4018*t4090*t4055;
                double t4097 = EcPhd_2+t4077+t4086+t4087+t4096;
                double t4098 = -t4013+1.0;
                double t4099 = heaviside(t4098);
                double t4100 = t4033*t4018*t4090;
                double t4101 = t4081+t4082+t4083+t4084+t4088+t4100;
                double t4102 = t4081+t4082+t4083+t4084+t4088-t4042*t4018*t4090;
                double t4103 = heaviside(t4071);
                double t4104 = t4091+t4093-t4042*t4018*t4075;
                double t4106 = 1.0/pow(t4001,2.0/3.0);
                double t4107 = gamma_ab*2.0;
                double t4108 = gamma_aa+gamma_bb+t4107;
                double t4109 = Bg*c*t4003*t4059*1.0E4;
                double t4110 = Cg_4*t4059*t4106;
                double t4111 = Cg_3*c*t4012;
                double t4112 = t4110+t4111+t4109+1.0;
                double t4113 = 1.0/t4112;
                double t4114 = 1.0/pow(t4001,5.0/3.0);
                double t4115 = Bg*t4059*t4106;
                double t4116 = Cg_2*c*t4012;
                double t4117 = Cg_1+t4115+t4116;
                double t4118 = 1.0/(t4112*t4112);
                double t4119 = Bg*c*t4007*t4059*1.0E4;
                double t4120 = Cg_4*t4059*t4114*(2.0/3.0);
                double t4121 = Cg_3*c*t4024*(1.0/3.0);
                double t4122 = t4120+t4121+t4119;
                double t4123 = 1.0/two_13;
                double t4124 = 1.0/pow(t4001,7.0/6.0);
                double t4125 = t4113*t4117;
                double t4126 = Cx+t4125;
                double t4127 = 1.0/t4126;
                double t4128 = sqrt(t4108);
                double t4136 = Cinf*Fg*Pg_1*t4124*t4127*t4128;
                double t4129 = exp(-t4136);
                double t4130 = t4002*t4003*(1.0/2.0);
                double t4131 = t4130+1.0/2.0;
                double t4132 = pow(t4131,5.0/3.0);
                double t4133 = Bg*t4059*t4114*(2.0/3.0);
                double t4134 = Cg_2*c*t4024*(1.0/3.0);
                double t4135 = t4133+t4134;
                double t4137 = -t4130+1.0/2.0;
                double t4138 = pow(t4137,5.0/3.0);
                double t4139 = t4132+t4138;
                double t4140 = 1.0/sqrt(t4139);
                double t4141 = t4003*(1.0/2.0);
                double t4142 = t4002*t4007*(1.0/2.0);
                double t4143 = t4141+t4142;
                double t4144 = t4113*t4135;
                double t4146 = t4122*t4117*t4118;
                double t4145 = t4144-t4146;
                double t4147 = 1.0/(t4126*t4126);
                double t4148 = Bg*t4060*t4059*(1.0E1/9.0);
                double t4149 = Cg_2*c*t4015*(4.0/9.0);
                double t4150 = t4148+t4149;
                double t4151 = t4113*t4150;
                double t4152 = Bg*c*t4005*t4059*2.0E4;
                double t4153 = Cg_4*t4060*t4059*(1.0E1/9.0);
                double t4154 = Cg_3*c*t4015*(4.0/9.0);
                double t4155 = t4152+t4153+t4154;
                double t4156 = t4122*t4122;
                double t4157 = 1.0/(t4112*t4112*t4112);
                double t4158 = t4117*t4156*t4157*2.0;
                double t4159 = t4151+t4158-t4122*t4135*t4118*2.0-t4117*t4118*t4155;
                double t4160 = 1.0/pow(t4001,1.3E1/6.0);
                double t4161 = Cinf*Fg*Pg_1*t4160*t4127*t4128*(7.0/6.0);
                double t4171 = Cinf*Fg*Pg_1*t4124*t4145*t4128*t4147;
                double t4162 = t4161-t4171;
                double t4163 = 1.0/pow(t4139,3.0/2.0);
                double t4164 = pow(t4137,2.0/3.0);
                double t4165 = t4141-t4142;
                double t4166 = pow(t4131,2.0/3.0);
                double t4167 = t4164*t4165*(5.0/3.0);
                double t4172 = t4165*t4166*(5.0/3.0);
                double t4168 = -t4172+t4167;
                double t4169 = t4143*t4164*(5.0/3.0);
                double t4173 = t4143*t4166*(5.0/3.0);
                double t4170 = -t4173+t4169;
                v_rho_a_rho_b[Q] += scale * (t4103*(t4091+t4093+t4033*t4018*(t4073-t4092))-t4099*t4101-t4099*t4102+t4103*t4104-t4001*(-t4103*(t4058-EcPld_1*t4051*t4064-t4033*t4052*t4018+t4042*t4052*t4018+t4070*t4018*t4075-t4018*t4055*(t4058-EcFld_1*(t4045*t4045)*1.0/(t4048*t4048*t4048)*2.0-EcPld_1*t4051*t4064+EcFld_1*t4049*(EcFld_3*c*t4015*(4.0/9.0)-EcFld_2*t4060*t4061*t4059*(1.0/3.6E1)+EcFld_2*c*t4015*t4034*(2.0/9.0))))-t4099*(t4020+t4021+t4030+t4019-t4018*t4055*(t4020+t4021+t4030+t4019-EcFhd_1*t4007*(1.0/3.0)-EcFhd_3*c*t4015*(5.0/9.0)-EcFhd_4*c*t4015*(4.0/9.0)-EcFhd_3*c*t4014*t4015*(4.0/9.0))-t4070*t4018*t4090+t4033*t4018*t4029-t4042*t4018*t4029)-c*t4015*t4080*t4094*(4.0/9.0)+c*t4015*t4080*t4097*(4.0/9.0)+c*t4024*t4080*t4101*(1.0/3.0)+c*t4024*t4080*t4102*(1.0/3.0)+c*t4024*t4080*t4104*(1.0/3.0)-t4060*t4094*t4059*t4095*(1.0/9.0)+t4060*t4059*t4095*t4097*(1.0/9.0)+c*t4024*t4080*(t4091+t4093+t4033*t4018*t4075)*(1.0/3.0))-c*t4024*t4080*t4094*(2.0/3.0)+c*t4024*t4080*t4097*(2.0/3.0)+t4015*t4140*t4123*t4108*t4145*t4129*(8.0/3.0)+1.0/pow(t4001,1.0E1/3.0)*t4140*t4123*t4108*t4126*t4129*(2.8E1/9.0)+t4024*t4123*t4108*t4129*t4159*1.0/sqrt(t4132+pow(t4002*t4003*(-1.0/2.0)+1.0/2.0,5.0/3.0))-t4024*t4140*t4123*t4108*t4126*t4129*(Cinf*Fg*Pg_1*1.0/pow(t4001,1.9E1/6.0)*t4127*t4128*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t4160*t4145*t4128*t4147*(7.0/3.0)-Cinf*Fg*Pg_1*t4124*t4128*t4147*t4159+Cinf*Fg*Pg_1*t4124*1.0/(t4126*t4126*t4126)*(t4145*t4145)*t4128*2.0)-t4015*t4140*t4123*t4108*t4126*t4162*t4129*(8.0/3.0)-t4024*t4140*t4123*t4108*t4162*t4145*t4129*2.0+t4015*t4123*t4170*t4108*t4126*t4163*t4129*(2.0/3.0)+t4024*t4123*t4170*t4108*t4145*t4163*t4129*(1.0/2.0)-t4015*t4123*t4108*t4126*t4163*t4129*t4168*(2.0/3.0)-t4024*t4123*t4108*t4145*t4163*t4129*t4168*(1.0/2.0)+t4024*t4123*t4108*t4126*t4163*t4129*(1.0/pow(t4131,1.0/3.0)*t4143*t4165*(1.0E1/9.0)+t4002*t4005*t4164*(5.0/3.0)-t4002*t4005*t4166*(5.0/3.0)+t4143*1.0/pow(t4137,1.0/3.0)*(t4141-t4002*t4007*(1.0/2.0))*(1.0E1/9.0))*(1.0/2.0)+t4024*t4140*t4123*t4108*t4126*(t4162*t4162)*t4129-t4024*t4123*t4170*t4108*t4126*t4162*t4163*t4129*(1.0/2.0)+t4024*t4123*t4108*t4126*t4162*t4163*t4129*t4168*(1.0/2.0)-t4024*t4123*t4170*t4108*t4126*t4129*1.0/pow(t4139,5.0/2.0)*t4168*(3.0/4.0));
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t4175 = rho_a+rho_b;
                double t4176 = 1.0/pow(t4175,4.0/3.0);
                double t4177 = 1.0/pow(t4175,1.0/3.0);
                double t4178 = c*t4177;
                double t4186 = 1.0/sqrt(t4178);
                double t4188 = EcPld_3*c*t4176*(1.0/3.0);
                double t4189 = EcPld_2*c*t4176*t4186*(1.0/6.0);
                double t4179 = t4188+t4189;
                double t4180 = 1.0/t4175;
                double t4181 = rho_a-rho_b;
                double t4182 = t4180*t4181;
                double t4183 = 1.0/(t4175*t4175);
                double t4184 = t4181*t4183;
                double t4185 = t4180+t4184;
                double t4187 = sqrt(t4178);
                double t4190 = EcPld_2*t4187;
                double t4191 = EcPld_3*c*t4177;
                double t4192 = t4190+t4191+1.0;
                double t4193 = two_13*2.0;
                double t4194 = t4193-2.0;
                double t4195 = 1.0/t4194;
                double t4196 = t4182+1.0;
                double t4197 = -t4182+1.0;
                double t4198 = EcFld_3*c*t4176*(1.0/3.0);
                double t4199 = EcFld_2*c*t4176*t4186*(1.0/6.0);
                double t4200 = t4198+t4199;
                double t4201 = EcFld_2*t4187;
                double t4202 = EcFld_3*c*t4177;
                double t4203 = t4201+t4202+1.0;
                double t4204 = t4179*t4179;
                double t4205 = 1.0/(t4192*t4192*t4192);
                double t4206 = EcPld_1*t4204*t4205*2.0;
                double t4207 = 1.0/pow(t4175,7.0/3.0);
                double t4208 = 1.0/(t4203*t4203);
                double t4209 = c*c;
                double t4210 = 1.0/pow(t4175,8.0/3.0);
                double t4211 = 1.0/pow(t4178,3.0/2.0);
                double t4212 = 1.0/(t4192*t4192);
                double t4213 = EcPld_3*c*t4207*(4.0/9.0);
                double t4214 = EcPld_2*c*t4186*t4207*(2.0/9.0);
                double t4215 = t4213+t4214-EcPld_2*t4210*t4211*t4209*(1.0/3.6E1);
                double t4216 = pow(t4196,1.0/3.0);
                double t4217 = pow(t4197,1.0/3.0);
                double t4218 = t4183*2.0;
                double t4219 = 1.0/(t4175*t4175*t4175);
                double t4220 = t4181*t4219*2.0;
                double t4221 = t4220+t4218;
                double t4222 = t4185*t4185;
                double t4223 = t4221*t4216*(4.0/3.0);
                double t4224 = 1.0/pow(t4196,2.0/3.0);
                double t4225 = t4222*t4224*(4.0/9.0);
                double t4226 = 1.0/pow(t4197,2.0/3.0);
                double t4227 = t4222*t4226*(4.0/9.0);
                double t4228 = t4223+t4225+t4227-t4221*t4217*(4.0/3.0);
                double t4229 = log(t4178);
                double t4230 = pow(t4196,4.0/3.0);
                double t4231 = pow(t4197,4.0/3.0);
                double t4232 = t4230+t4231-2.0;
                double t4233 = EcPhd_1*t4183*(1.0/3.0);
                double t4234 = EcPhd_3*c*t4207*(5.0/9.0);
                double t4235 = EcPhd_4*c*t4207*(4.0/9.0);
                double t4236 = t4185*t4216*(4.0/3.0);
                double t4256 = t4185*t4217*(4.0/3.0);
                double t4237 = t4236-t4256;
                double t4238 = EcPhd_3*c*t4207*t4229*(4.0/9.0);
                double t4239 = t4178-1.0;
                double t4240 = 1.0/t4192;
                double t4241 = EcPld_1*t4240;
                double t4242 = 1.0/t4203;
                double t4262 = EcFld_1*t4242;
                double t4243 = t4241-t4262;
                double t4244 = EcFhd_1*t4229;
                double t4245 = EcPhd_1*t4229;
                double t4246 = EcFhd_4*c*t4177;
                double t4247 = EcFhd_3*c*t4177*t4229;
                double t4248 = dirac(t4239);
                double t4249 = EcFhd_1*t4180*(1.0/3.0);
                double t4250 = EcPhd_1*t4180*(1.0/3.0);
                double t4251 = EcFhd_3*c*t4176*(1.0/3.0);
                double t4252 = EcFhd_4*c*t4176*(1.0/3.0);
                double t4253 = EcPhd_3*c*t4176*(1.0/3.0);
                double t4254 = EcPhd_4*c*t4176*(1.0/3.0);
                double t4255 = EcFhd_3*c*t4176*t4229*(1.0/3.0);
                double t4257 = EcPhd_4*c*t4177;
                double t4258 = EcPhd_3*c*t4177*t4229;
                double t4259 = EcPhd_3*c*t4176*t4229*(1.0/3.0);
                double t4260 = EcFld_1*t4200*t4208;
                double t4263 = EcPld_1*t4179*t4212;
                double t4261 = t4260-t4263;
                double t4277 = t4195*t4232*t4243;
                double t4264 = t4241-t4277;
                double t4265 = dirac(t4239,1.0);
                double t4266 = EcFhd_2-EcPhd_2+t4244-t4245+t4246+t4247-t4257-t4258;
                double t4267 = EcPld_1*t4212*t4215;
                double t4268 = t4250-t4251-t4252+t4253+t4254-t4255-t4249+t4259;
                double t4269 = t4195*t4232*t4266;
                double t4270 = EcPhd_2+t4245+t4257+t4258+t4269;
                double t4271 = -t4178+1.0;
                double t4272 = heaviside(t4271);
                double t4273 = t4195*t4237*t4266;
                double t4274 = t4250+t4253+t4254+t4273+t4259-t4195*t4232*t4268;
                double t4275 = heaviside(t4239);
                double t4276 = t4195*t4232*t4261;
                double t4278 = 1.0/pow(t4175,2.0/3.0);
                double t4279 = gamma_ab*2.0;
                double t4280 = gamma_aa+gamma_bb+t4279;
                double t4281 = Bg*c*t4180*t4209*1.0E4;
                double t4282 = Cg_4*t4209*t4278;
                double t4283 = Cg_3*c*t4177;
                double t4284 = t4281+t4282+t4283+1.0;
                double t4285 = 1.0/t4284;
                double t4286 = 1.0/pow(t4175,5.0/3.0);
                double t4287 = Bg*t4209*t4278;
                double t4288 = Cg_2*c*t4177;
                double t4289 = Cg_1+t4287+t4288;
                double t4290 = 1.0/(t4284*t4284);
                double t4291 = Bg*c*t4183*t4209*1.0E4;
                double t4292 = Cg_4*t4209*t4286*(2.0/3.0);
                double t4293 = Cg_3*c*t4176*(1.0/3.0);
                double t4294 = t4291+t4292+t4293;
                double t4295 = 1.0/two_13;
                double t4296 = 1.0/pow(t4175,7.0/6.0);
                double t4297 = t4285*t4289;
                double t4298 = Cx+t4297;
                double t4299 = 1.0/t4298;
                double t4300 = sqrt(t4280);
                double t4308 = Cinf*Fg*Pg_1*t4296*t4299*t4300;
                double t4301 = exp(-t4308);
                double t4302 = t4180*t4181*(1.0/2.0);
                double t4303 = t4302+1.0/2.0;
                double t4304 = pow(t4303,5.0/3.0);
                double t4305 = Bg*t4209*t4286*(2.0/3.0);
                double t4306 = Cg_2*c*t4176*(1.0/3.0);
                double t4307 = t4305+t4306;
                double t4309 = -t4302+1.0/2.0;
                double t4310 = pow(t4309,5.0/3.0);
                double t4311 = t4310+t4304;
                double t4312 = 1.0/sqrt(t4311);
                double t4313 = t4180*(1.0/2.0);
                double t4314 = t4181*t4183*(1.0/2.0);
                double t4315 = t4313+t4314;
                double t4321 = pow(t4309,2.0/3.0);
                double t4322 = pow(t4303,2.0/3.0);
                double t4342 = t4321*t4315*(5.0/3.0);
                double t4343 = t4322*t4315*(5.0/3.0);
                double t4316 = -t4342+t4343;
                double t4317 = t4285*t4307;
                double t4319 = t4290*t4294*t4289;
                double t4318 = t4317-t4319;
                double t4320 = t4315*t4315;
                double t4323 = t4181*t4219;
                double t4324 = t4183+t4323;
                double t4325 = 1.0/(t4298*t4298);
                double t4326 = Bg*t4210*t4209*(1.0E1/9.0);
                double t4327 = Cg_2*c*t4207*(4.0/9.0);
                double t4328 = t4326+t4327;
                double t4329 = t4285*t4328;
                double t4330 = Bg*c*t4209*t4219*2.0E4;
                double t4331 = Cg_4*t4210*t4209*(1.0E1/9.0);
                double t4332 = Cg_3*c*t4207*(4.0/9.0);
                double t4333 = t4330+t4331+t4332;
                double t4334 = t4294*t4294;
                double t4335 = 1.0/(t4284*t4284*t4284);
                double t4336 = t4289*t4334*t4335*2.0;
                double t4337 = t4336+t4329-t4290*t4294*t4307*2.0-t4290*t4289*t4333;
                double t4338 = 1.0/pow(t4175,1.3E1/6.0);
                double t4339 = Cinf*Fg*Pg_1*t4299*t4300*t4338*(7.0/6.0);
                double t4345 = Cinf*Fg*Pg_1*t4296*t4300*t4325*t4318;
                double t4340 = -t4345+t4339;
                double t4341 = 1.0/pow(t4311,3.0/2.0);
                double t4344 = t4342-t4343;
                v_rho_b_rho_b[Q] += scale * (t4275*(t4263+t4276+t4195*t4237*(t4241-t4262))*2.0-t4272*t4274*2.0-t4175*(t4275*(-t4206+t4267+t4195*t4232*(t4206-t4267-EcFld_1*(t4200*t4200)*1.0/(t4203*t4203*t4203)*2.0+EcFld_1*t4208*(EcFld_3*c*t4207*(4.0/9.0)-EcFld_2*t4210*t4211*t4209*(1.0/3.6E1)+EcFld_2*c*t4186*t4207*(2.0/9.0)))+t4195*t4243*t4228+t4195*t4261*t4237*2.0)-t4272*(t4233+t4234+t4235+t4238-t4195*t4232*(t4233+t4234+t4235+t4238-EcFhd_1*t4183*(1.0/3.0)-EcFhd_3*c*t4207*(5.0/9.0)-EcFhd_4*c*t4207*(4.0/9.0)-EcFhd_3*c*t4207*t4229*(4.0/9.0))+t4195*t4228*t4266-t4195*t4237*t4268*2.0)+c*t4176*t4274*t4248*(2.0/3.0)+c*t4207*t4270*t4248*(4.0/9.0)-c*t4207*t4264*t4248*(4.0/9.0)+t4210*t4270*t4209*t4265*(1.0/9.0)-t4210*t4209*t4264*t4265*(1.0/9.0)+c*t4176*t4248*(t4263+t4276+t4195*t4243*t4237)*(2.0/3.0))+c*t4176*t4270*t4248*(2.0/3.0)-c*t4176*t4264*t4248*(2.0/3.0)+t4207*t4280*t4295*t4301*t4312*t4318*(8.0/3.0)+1.0/pow(t4175,1.0E1/3.0)*t4280*t4295*t4298*t4301*t4312*(2.8E1/9.0)+t4176*t4280*t4295*t4301*t4337*1.0/sqrt(t4304+pow(t4180*t4181*(-1.0/2.0)+1.0/2.0,5.0/3.0))+t4176*t4280*t4295*t4298*t4301*1.0/pow(t4311,5.0/2.0)*(t4316*t4316)*(3.0/4.0)-t4176*t4280*t4295*t4298*t4301*t4312*(Cinf*Fg*Pg_1*1.0/pow(t4175,1.9E1/6.0)*t4299*t4300*(9.1E1/3.6E1)-Cinf*Fg*Pg_1*t4296*t4300*t4325*t4337-Cinf*Fg*Pg_1*t4300*t4325*t4318*t4338*(7.0/3.0)+Cinf*Fg*Pg_1*t4296*1.0/(t4298*t4298*t4298)*t4300*(t4318*t4318)*2.0)-t4176*t4280*t4295*t4298*t4301*t4341*(t4321*t4324*(-5.0/3.0)+t4322*t4324*(5.0/3.0)+t4320*1.0/pow(t4303,1.0/3.0)*(1.0E1/9.0)+t4320*1.0/pow(t4309,1.0/3.0)*(1.0E1/9.0))*(1.0/2.0)-t4176*t4280*t4295*t4301*t4312*t4340*t4318*2.0-t4207*t4280*t4295*t4298*t4301*t4312*t4340*(8.0/3.0)+t4176*t4280*t4295*t4301*t4341*t4344*t4318+t4207*t4280*t4295*t4298*t4301*t4341*t4344*(4.0/3.0)+t4176*t4280*t4295*t4298*t4301*t4312*(t4340*t4340)-t4176*t4280*t4295*t4298*t4301*t4340*t4341*t4344);
            }
            
            // v_gamma_aa_gamma_aa
            if (deriv >= 2) {
                double t4707 = rho_a+rho_b;
                double t4708 = c*c;
                double t4709 = 1.0/pow(t4707,2.0/3.0);
                double t4710 = 1.0/pow(t4707,1.0/3.0);
                double t4711 = 1.0/t4707;
                double t4712 = rho_a-rho_b;
                double t4713 = Bg*t4708*t4709;
                double t4714 = Cg_2*c*t4710;
                double t4715 = Cg_1+t4713+t4714;
                double t4716 = Bg*c*t4711*t4708*1.0E4;
                double t4717 = Cg_4*t4708*t4709;
                double t4718 = Cg_3*c*t4710;
                double t4719 = t4716+t4717+t4718+1.0;
                double t4720 = 1.0/t4719;
                double t4721 = t4720*t4715;
                double t4722 = Cx+t4721;
                double t4723 = 1.0/t4722;
                double t4724 = 1.0/two_13;
                double t4725 = 1.0/pow(t4707,7.0/6.0);
                double t4726 = gamma_ab*2.0;
                double t4727 = gamma_aa+gamma_bb+t4726;
                double t4728 = sqrt(t4727);
                double t4729 = exp(-Cinf*Fg*Pg_1*t4723*t4725*t4728);
                double t4730 = t4711*t4712*(1.0/2.0);
                double t4731 = t4730+1.0/2.0;
                double t4732 = pow(t4731,5.0/3.0);
                v_gamma_aa_gamma_aa[Q] += scale * (Cinf*Fg*Pg_1*t4724*1.0/pow(t4707,5.0/2.0)*1.0/sqrt(t4727)*t4729*1.0/sqrt(t4732+pow(-t4730+1.0/2.0,5.0/3.0))*(-3.0/4.0)+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t4723*t4724*1.0/pow(t4707,1.1E1/3.0)*t4729*1.0/sqrt(t4732+pow(t4711*t4712*(-1.0/2.0)+1.0/2.0,5.0/3.0))*(1.0/4.0));
            }
            
            // v_gamma_aa_gamma_ab
            if (deriv >= 2) {
                double t4734 = rho_a+rho_b;
                double t4735 = c*c;
                double t4736 = 1.0/pow(t4734,2.0/3.0);
                double t4737 = 1.0/pow(t4734,1.0/3.0);
                double t4738 = 1.0/t4734;
                double t4739 = rho_a-rho_b;
                double t4740 = Bg*t4735*t4736;
                double t4741 = Cg_2*c*t4737;
                double t4742 = Cg_1+t4740+t4741;
                double t4743 = Bg*c*t4735*t4738*1.0E4;
                double t4744 = Cg_4*t4735*t4736;
                double t4745 = Cg_3*c*t4737;
                double t4746 = t4743+t4744+t4745+1.0;
                double t4747 = 1.0/t4746;
                double t4748 = t4742*t4747;
                double t4749 = Cx+t4748;
                double t4750 = 1.0/t4749;
                double t4751 = 1.0/two_13;
                double t4752 = 1.0/pow(t4734,7.0/6.0);
                double t4753 = gamma_ab*2.0;
                double t4754 = gamma_aa+gamma_bb+t4753;
                double t4755 = sqrt(t4754);
                double t4756 = exp(-Cinf*Fg*Pg_1*t4750*t4752*t4755);
                double t4757 = t4738*t4739*(1.0/2.0);
                double t4758 = t4757+1.0/2.0;
                double t4759 = pow(t4758,5.0/3.0);
                v_gamma_aa_gamma_ab[Q] += scale * (Cinf*Fg*Pg_1*t4751*1.0/pow(t4734,5.0/2.0)*1.0/sqrt(t4754)*t4756*1.0/sqrt(t4759+pow(-t4757+1.0/2.0,5.0/3.0))*(-3.0/2.0)+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t4750*t4751*1.0/pow(t4734,1.1E1/3.0)*t4756*1.0/sqrt(t4759+pow(t4738*t4739*(-1.0/2.0)+1.0/2.0,5.0/3.0))*(1.0/2.0));
            }
            
            // v_gamma_aa_gamma_bb
            if (deriv >= 2) {
                double t4761 = rho_a+rho_b;
                double t4762 = c*c;
                double t4763 = 1.0/pow(t4761,2.0/3.0);
                double t4764 = 1.0/pow(t4761,1.0/3.0);
                double t4765 = 1.0/t4761;
                double t4766 = rho_a-rho_b;
                double t4767 = Bg*t4762*t4763;
                double t4768 = Cg_2*c*t4764;
                double t4769 = Cg_1+t4767+t4768;
                double t4770 = Bg*c*t4762*t4765*1.0E4;
                double t4771 = Cg_4*t4762*t4763;
                double t4772 = Cg_3*c*t4764;
                double t4773 = t4770+t4771+t4772+1.0;
                double t4774 = 1.0/t4773;
                double t4775 = t4774*t4769;
                double t4776 = Cx+t4775;
                double t4777 = 1.0/t4776;
                double t4778 = 1.0/two_13;
                double t4779 = 1.0/pow(t4761,7.0/6.0);
                double t4780 = gamma_ab*2.0;
                double t4781 = gamma_aa+gamma_bb+t4780;
                double t4782 = sqrt(t4781);
                double t4783 = exp(-Cinf*Fg*Pg_1*t4782*t4777*t4779);
                double t4784 = t4765*t4766*(1.0/2.0);
                double t4785 = t4784+1.0/2.0;
                double t4786 = pow(t4785,5.0/3.0);
                v_gamma_aa_gamma_bb[Q] += scale * (Cinf*Fg*Pg_1*1.0/pow(t4761,5.0/2.0)*1.0/sqrt(t4781)*t4783*t4778*1.0/sqrt(t4786+pow(-t4784+1.0/2.0,5.0/3.0))*(-3.0/4.0)+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t4761,1.1E1/3.0)*t4783*t4777*t4778*1.0/sqrt(t4786+pow(t4765*t4766*(-1.0/2.0)+1.0/2.0,5.0/3.0))*(1.0/4.0));
            }
            
            // v_gamma_ab_gamma_ab
            if (deriv >= 2) {
                double t4788 = rho_a+rho_b;
                double t4789 = c*c;
                double t4790 = 1.0/pow(t4788,2.0/3.0);
                double t4791 = 1.0/pow(t4788,1.0/3.0);
                double t4792 = 1.0/t4788;
                double t4793 = rho_a-rho_b;
                double t4794 = Bg*t4790*t4789;
                double t4795 = Cg_2*c*t4791;
                double t4796 = Cg_1+t4794+t4795;
                double t4797 = Bg*c*t4792*t4789*1.0E4;
                double t4798 = Cg_4*t4790*t4789;
                double t4799 = Cg_3*c*t4791;
                double t4800 = t4797+t4798+t4799+1.0;
                double t4801 = 1.0/t4800;
                double t4802 = t4796*t4801;
                double t4803 = Cx+t4802;
                double t4804 = 1.0/t4803;
                double t4805 = 1.0/two_13;
                double t4806 = 1.0/pow(t4788,7.0/6.0);
                double t4807 = gamma_ab*2.0;
                double t4808 = gamma_aa+gamma_bb+t4807;
                double t4809 = sqrt(t4808);
                double t4810 = exp(-Cinf*Fg*Pg_1*t4804*t4806*t4809);
                double t4811 = t4792*t4793*(1.0/2.0);
                double t4812 = t4811+1.0/2.0;
                double t4813 = pow(t4812,5.0/3.0);
                v_gamma_ab_gamma_ab[Q] += scale * (Cinf*Fg*Pg_1*1.0/pow(t4788,5.0/2.0)*t4810*t4805*1.0/sqrt(t4808)*1.0/sqrt(t4813+pow(-t4811+1.0/2.0,5.0/3.0))*-3.0+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t4788,1.1E1/3.0)*t4810*t4804*t4805*1.0/sqrt(t4813+pow(t4792*t4793*(-1.0/2.0)+1.0/2.0,5.0/3.0)));
            }
            
            // v_gamma_ab_gamma_bb
            if (deriv >= 2) {
                double t4815 = rho_a+rho_b;
                double t4816 = c*c;
                double t4817 = 1.0/pow(t4815,2.0/3.0);
                double t4818 = 1.0/pow(t4815,1.0/3.0);
                double t4819 = 1.0/t4815;
                double t4820 = rho_a-rho_b;
                double t4821 = Bg*t4816*t4817;
                double t4822 = Cg_2*c*t4818;
                double t4823 = Cg_1+t4821+t4822;
                double t4824 = Bg*c*t4816*t4819*1.0E4;
                double t4825 = Cg_4*t4816*t4817;
                double t4826 = Cg_3*c*t4818;
                double t4827 = t4824+t4825+t4826+1.0;
                double t4828 = 1.0/t4827;
                double t4829 = t4823*t4828;
                double t4830 = Cx+t4829;
                double t4831 = 1.0/t4830;
                double t4832 = 1.0/two_13;
                double t4833 = 1.0/pow(t4815,7.0/6.0);
                double t4834 = gamma_ab*2.0;
                double t4835 = gamma_aa+gamma_bb+t4834;
                double t4836 = sqrt(t4835);
                double t4837 = exp(-Cinf*Fg*Pg_1*t4831*t4833*t4836);
                double t4838 = t4820*t4819*(1.0/2.0);
                double t4839 = t4838+1.0/2.0;
                double t4840 = pow(t4839,5.0/3.0);
                v_gamma_ab_gamma_bb[Q] += scale * (Cinf*Fg*Pg_1*t4832*1.0/pow(t4815,5.0/2.0)*1.0/sqrt(t4835)*t4837*1.0/sqrt(t4840+pow(-t4838+1.0/2.0,5.0/3.0))*(-3.0/2.0)+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*t4831*t4832*1.0/pow(t4815,1.1E1/3.0)*t4837*1.0/sqrt(t4840+pow(t4820*t4819*(-1.0/2.0)+1.0/2.0,5.0/3.0))*(1.0/2.0));
            }
            
            // v_gamma_bb_gamma_bb
            if (deriv >= 2) {
                double t4842 = rho_a+rho_b;
                double t4843 = c*c;
                double t4844 = 1.0/pow(t4842,2.0/3.0);
                double t4845 = 1.0/pow(t4842,1.0/3.0);
                double t4846 = 1.0/t4842;
                double t4847 = rho_a-rho_b;
                double t4848 = Bg*t4843*t4844;
                double t4849 = Cg_2*c*t4845;
                double t4850 = Cg_1+t4848+t4849;
                double t4851 = Bg*c*t4843*t4846*1.0E4;
                double t4852 = Cg_4*t4843*t4844;
                double t4853 = Cg_3*c*t4845;
                double t4854 = t4851+t4852+t4853+1.0;
                double t4855 = 1.0/t4854;
                double t4856 = t4850*t4855;
                double t4857 = Cx+t4856;
                double t4858 = 1.0/t4857;
                double t4859 = 1.0/two_13;
                double t4860 = 1.0/pow(t4842,7.0/6.0);
                double t4861 = gamma_ab*2.0;
                double t4862 = gamma_aa+gamma_bb+t4861;
                double t4863 = sqrt(t4862);
                double t4864 = exp(-Cinf*Fg*Pg_1*t4860*t4863*t4858);
                double t4865 = t4846*t4847*(1.0/2.0);
                double t4866 = t4865+1.0/2.0;
                double t4867 = pow(t4866,5.0/3.0);
                v_gamma_bb_gamma_bb[Q] += scale * (Cinf*Fg*Pg_1*1.0/pow(t4842,5.0/2.0)*1.0/sqrt(t4862)*t4864*t4859*1.0/sqrt(t4867+pow(-t4865+1.0/2.0,5.0/3.0))*(-3.0/4.0)+(Cinf*Cinf)*(Fg*Fg)*(Pg_1*Pg_1)*1.0/pow(t4842,1.1E1/3.0)*t4864*t4858*t4859*1.0/sqrt(t4867+pow(t4846*t4847*(-1.0/2.0)+1.0/2.0,5.0/3.0))*(1.0/4.0));
            }
            
            // v_rho_a_gamma_aa
            if (deriv >= 2) {
                double t4347 = rho_a+rho_b;
                double t4348 = c*c;
                double t4349 = 1.0/pow(t4347,2.0/3.0);
                double t4350 = 1.0/pow(t4347,1.0/3.0);
                double t4351 = 1.0/t4347;
                double t4352 = rho_a-rho_b;
                double t4353 = 1.0/pow(t4347,4.0/3.0);
                double t4354 = Bg*c*t4351*t4348*1.0E4;
                double t4355 = Cg_4*t4348*t4349;
                double t4356 = Cg_3*c*t4350;
                double t4357 = t4354+t4355+t4356+1.0;
                double t4358 = 1.0/t4357;
                double t4359 = 1.0/pow(t4347,5.0/3.0);
                double t4360 = Bg*t4348*t4349;
                double t4361 = Cg_2*c*t4350;
                double t4362 = Cg_1+t4360+t4361;
                double t4363 = 1.0/two_13;
                double t4364 = 1.0/pow(t4347,7.0/6.0);
                double t4365 = t4362*t4358;
                double t4366 = Cx+t4365;
                double t4367 = 1.0/t4366;
                double t4368 = gamma_ab*2.0;
                double t4369 = gamma_aa+gamma_bb+t4368;
                double t4370 = sqrt(t4369);
                double t4375 = Cinf*Fg*Pg_1*t4370*t4364*t4367;
                double t4371 = exp(-t4375);
                double t4372 = t4351*t4352*(1.0/2.0);
                double t4373 = t4372+1.0/2.0;
                double t4374 = pow(t4373,5.0/3.0);
                double t4376 = -t4372+1.0/2.0;
                double t4377 = pow(t4376,5.0/3.0);
                double t4378 = t4374+t4377;
                double t4379 = 1.0/sqrt(t4378);
                double t4380 = Bg*t4348*t4359*(2.0/3.0);
                double t4381 = Cg_2*c*t4353*(1.0/3.0);
                double t4382 = t4380+t4381;
                double t4383 = t4382*t4358;
                double t4384 = 1.0/(t4347*t4347);
                double t4385 = Bg*c*t4348*t4384*1.0E4;
                double t4386 = Cg_4*t4348*t4359*(2.0/3.0);
                double t4387 = Cg_3*c*t4353*(1.0/3.0);
                double t4388 = t4385+t4386+t4387;
                double t4389 = 1.0/(t4357*t4357);
                double t4394 = t4362*t4388*t4389;
                double t4390 = t4383-t4394;
                double t4391 = t4351*(1.0/2.0);
                double t4402 = t4352*t4384*(1.0/2.0);
                double t4392 = t4391-t4402;
                double t4393 = 1.0/pow(t4347,1.3E1/6.0);
                double t4395 = 1.0/(t4366*t4366);
                double t4396 = 1.0/sqrt(t4369);
                double t4397 = Cinf*Fg*Pg_1*t4370*t4393*t4367*(7.0/6.0);
                double t4398 = t4397-Cinf*Fg*Pg_1*t4370*t4390*t4364*t4395;
                double t4399 = 1.0/pow(t4378,3.0/2.0);
                double t4400 = 1.0/pow(t4347,5.0/2.0);
                double t4401 = pow(t4376,2.0/3.0);
                double t4403 = t4392*t4401*(5.0/3.0);
                double t4404 = pow(t4373,2.0/3.0);
                double t4405 = t4403-t4392*t4404*(5.0/3.0);
                v_rho_a_gamma_aa[Q] += scale * (t4371*t4363*1.0/pow(t4347,7.0/3.0)*t4366*t4379*(-4.0/3.0)-t4353*t4371*t4363*t4390*1.0/sqrt(t4374+pow(t4351*t4352*(-1.0/2.0)+1.0/2.0,5.0/3.0))+t4353*t4371*t4363*t4366*t4379*t4398+t4353*t4371*t4363*t4366*t4399*t4405*(1.0/2.0)+t4353*t4371*t4363*t4366*t4369*t4379*(Cinf*Fg*Pg_1*t4393*t4367*t4396*(7.0/1.2E1)-Cinf*Fg*Pg_1*t4390*t4364*t4395*t4396*(1.0/2.0))+Cinf*Fg*Pg_1*t4370*t4371*t4363*1.0/pow(t4347,7.0/2.0)*t4379*(2.0/3.0)-Cinf*Fg*Pg_1*t4370*t4371*t4363*t4379*t4398*t4400*(1.0/2.0)-Cinf*Fg*Pg_1*t4370*t4371*t4363*t4399*t4400*t4405*(1.0/4.0)+Cinf*Fg*Pg_1*t4370*t4371*t4363*t4390*t4367*t4379*t4400*(1.0/2.0));
            }
            
            // v_rho_a_gamma_ab
            if (deriv >= 2) {
                double t4407 = rho_a+rho_b;
                double t4408 = c*c;
                double t4409 = 1.0/pow(t4407,2.0/3.0);
                double t4410 = 1.0/pow(t4407,1.0/3.0);
                double t4411 = 1.0/t4407;
                double t4412 = rho_a-rho_b;
                double t4413 = 1.0/pow(t4407,4.0/3.0);
                double t4414 = Bg*c*t4411*t4408*1.0E4;
                double t4415 = Cg_4*t4408*t4409;
                double t4416 = Cg_3*c*t4410;
                double t4417 = t4414+t4415+t4416+1.0;
                double t4418 = 1.0/t4417;
                double t4419 = 1.0/pow(t4407,5.0/3.0);
                double t4420 = Bg*t4408*t4409;
                double t4421 = Cg_2*c*t4410;
                double t4422 = Cg_1+t4420+t4421;
                double t4423 = 1.0/two_13;
                double t4424 = 1.0/pow(t4407,7.0/6.0);
                double t4425 = t4422*t4418;
                double t4426 = Cx+t4425;
                double t4427 = 1.0/t4426;
                double t4428 = gamma_ab*2.0;
                double t4429 = gamma_aa+gamma_bb+t4428;
                double t4430 = sqrt(t4429);
                double t4435 = Cinf*Fg*Pg_1*t4430*t4424*t4427;
                double t4431 = exp(-t4435);
                double t4432 = t4411*t4412*(1.0/2.0);
                double t4433 = t4432+1.0/2.0;
                double t4434 = pow(t4433,5.0/3.0);
                double t4436 = -t4432+1.0/2.0;
                double t4437 = pow(t4436,5.0/3.0);
                double t4438 = t4434+t4437;
                double t4439 = 1.0/sqrt(t4438);
                double t4440 = Bg*t4408*t4419*(2.0/3.0);
                double t4441 = Cg_2*c*t4413*(1.0/3.0);
                double t4442 = t4440+t4441;
                double t4443 = t4442*t4418;
                double t4444 = 1.0/(t4407*t4407);
                double t4445 = Bg*c*t4408*t4444*1.0E4;
                double t4446 = Cg_4*t4408*t4419*(2.0/3.0);
                double t4447 = Cg_3*c*t4413*(1.0/3.0);
                double t4448 = t4445+t4446+t4447;
                double t4449 = 1.0/(t4417*t4417);
                double t4454 = t4422*t4448*t4449;
                double t4450 = t4443-t4454;
                double t4451 = t4411*(1.0/2.0);
                double t4462 = t4412*t4444*(1.0/2.0);
                double t4452 = t4451-t4462;
                double t4453 = 1.0/pow(t4407,1.3E1/6.0);
                double t4455 = 1.0/(t4426*t4426);
                double t4456 = 1.0/sqrt(t4429);
                double t4457 = Cinf*Fg*Pg_1*t4430*t4453*t4427*(7.0/6.0);
                double t4458 = t4457-Cinf*Fg*Pg_1*t4430*t4450*t4424*t4455;
                double t4459 = 1.0/pow(t4438,3.0/2.0);
                double t4460 = 1.0/pow(t4407,5.0/2.0);
                double t4461 = pow(t4436,2.0/3.0);
                double t4463 = t4452*t4461*(5.0/3.0);
                double t4464 = pow(t4433,2.0/3.0);
                double t4465 = t4463-t4452*t4464*(5.0/3.0);
                v_rho_a_gamma_ab[Q] += scale * (t4431*t4423*1.0/pow(t4407,7.0/3.0)*t4426*t4439*(-8.0/3.0)-t4413*t4431*t4423*t4450*1.0/sqrt(t4434+pow(t4411*t4412*(-1.0/2.0)+1.0/2.0,5.0/3.0))*2.0+t4413*t4431*t4423*t4426*t4465*t4459+t4413*t4431*t4423*t4426*t4439*t4458*2.0+t4413*t4431*t4423*t4426*t4429*t4439*(Cinf*Fg*Pg_1*t4453*t4427*t4456*(7.0/6.0)-Cinf*Fg*Pg_1*t4450*t4424*t4455*t4456)+Cinf*Fg*Pg_1*t4430*t4431*t4423*1.0/pow(t4407,7.0/2.0)*t4439*(4.0/3.0)-Cinf*Fg*Pg_1*t4430*t4431*t4423*t4460*t4465*t4459*(1.0/2.0)-Cinf*Fg*Pg_1*t4430*t4431*t4423*t4460*t4439*t4458+Cinf*Fg*Pg_1*t4430*t4431*t4423*t4450*t4460*t4427*t4439);
            }
            
            // v_rho_a_gamma_bb
            if (deriv >= 2) {
                double t4467 = rho_a+rho_b;
                double t4468 = c*c;
                double t4469 = 1.0/pow(t4467,2.0/3.0);
                double t4470 = 1.0/pow(t4467,1.0/3.0);
                double t4471 = 1.0/t4467;
                double t4472 = rho_a-rho_b;
                double t4473 = 1.0/pow(t4467,4.0/3.0);
                double t4474 = Bg*c*t4471*t4468*1.0E4;
                double t4475 = Cg_4*t4468*t4469;
                double t4476 = Cg_3*c*t4470;
                double t4477 = t4474+t4475+t4476+1.0;
                double t4478 = 1.0/t4477;
                double t4479 = 1.0/pow(t4467,5.0/3.0);
                double t4480 = Bg*t4468*t4469;
                double t4481 = Cg_2*c*t4470;
                double t4482 = Cg_1+t4480+t4481;
                double t4483 = 1.0/two_13;
                double t4484 = 1.0/pow(t4467,7.0/6.0);
                double t4485 = t4482*t4478;
                double t4486 = Cx+t4485;
                double t4487 = 1.0/t4486;
                double t4488 = gamma_ab*2.0;
                double t4489 = gamma_aa+gamma_bb+t4488;
                double t4490 = sqrt(t4489);
                double t4495 = Cinf*Fg*Pg_1*t4490*t4484*t4487;
                double t4491 = exp(-t4495);
                double t4492 = t4471*t4472*(1.0/2.0);
                double t4493 = t4492+1.0/2.0;
                double t4494 = pow(t4493,5.0/3.0);
                double t4496 = -t4492+1.0/2.0;
                double t4497 = pow(t4496,5.0/3.0);
                double t4498 = t4494+t4497;
                double t4499 = 1.0/sqrt(t4498);
                double t4500 = Bg*t4468*t4479*(2.0/3.0);
                double t4501 = Cg_2*c*t4473*(1.0/3.0);
                double t4502 = t4500+t4501;
                double t4503 = t4478*t4502;
                double t4504 = 1.0/(t4467*t4467);
                double t4505 = Bg*c*t4468*t4504*1.0E4;
                double t4506 = Cg_4*t4468*t4479*(2.0/3.0);
                double t4507 = Cg_3*c*t4473*(1.0/3.0);
                double t4508 = t4505+t4506+t4507;
                double t4509 = 1.0/(t4477*t4477);
                double t4514 = t4482*t4508*t4509;
                double t4510 = t4503-t4514;
                double t4511 = t4471*(1.0/2.0);
                double t4522 = t4472*t4504*(1.0/2.0);
                double t4512 = t4511-t4522;
                double t4513 = 1.0/pow(t4467,1.3E1/6.0);
                double t4515 = 1.0/(t4486*t4486);
                double t4516 = 1.0/sqrt(t4489);
                double t4517 = Cinf*Fg*Pg_1*t4490*t4487*t4513*(7.0/6.0);
                double t4518 = t4517-Cinf*Fg*Pg_1*t4490*t4484*t4510*t4515;
                double t4519 = 1.0/pow(t4498,3.0/2.0);
                double t4520 = 1.0/pow(t4467,5.0/2.0);
                double t4521 = pow(t4496,2.0/3.0);
                double t4523 = t4512*t4521*(5.0/3.0);
                double t4524 = pow(t4493,2.0/3.0);
                double t4525 = t4523-t4512*t4524*(5.0/3.0);
                v_rho_a_gamma_bb[Q] += scale * (t4491*t4483*1.0/pow(t4467,7.0/3.0)*t4486*t4499*(-4.0/3.0)-t4473*t4491*t4483*t4510*1.0/sqrt(t4494+pow(t4471*t4472*(-1.0/2.0)+1.0/2.0,5.0/3.0))+t4473*t4491*t4483*t4486*t4499*t4518+t4473*t4491*t4483*t4486*t4525*t4519*(1.0/2.0)+t4473*t4491*t4483*t4486*t4489*t4499*(Cinf*Fg*Pg_1*t4487*t4513*t4516*(7.0/1.2E1)-Cinf*Fg*Pg_1*t4484*t4510*t4515*t4516*(1.0/2.0))+Cinf*Fg*Pg_1*t4490*t4491*t4483*1.0/pow(t4467,7.0/2.0)*t4499*(2.0/3.0)-Cinf*Fg*Pg_1*t4490*t4491*t4483*t4499*t4520*t4518*(1.0/2.0)-Cinf*Fg*Pg_1*t4490*t4491*t4483*t4520*t4525*t4519*(1.0/4.0)+Cinf*Fg*Pg_1*t4490*t4491*t4483*t4487*t4499*t4510*t4520*(1.0/2.0));
            }
            
            // v_rho_b_gamma_aa
            if (deriv >= 2) {
                double t4527 = rho_a+rho_b;
                double t4528 = c*c;
                double t4529 = 1.0/pow(t4527,2.0/3.0);
                double t4530 = 1.0/pow(t4527,1.0/3.0);
                double t4531 = 1.0/t4527;
                double t4532 = rho_a-rho_b;
                double t4533 = 1.0/pow(t4527,4.0/3.0);
                double t4534 = Bg*c*t4531*t4528*1.0E4;
                double t4535 = Cg_4*t4528*t4529;
                double t4536 = Cg_3*c*t4530;
                double t4537 = t4534+t4535+t4536+1.0;
                double t4538 = 1.0/t4537;
                double t4539 = 1.0/pow(t4527,5.0/3.0);
                double t4540 = Bg*t4528*t4529;
                double t4541 = Cg_2*c*t4530;
                double t4542 = Cg_1+t4540+t4541;
                double t4543 = 1.0/two_13;
                double t4544 = 1.0/pow(t4527,7.0/6.0);
                double t4545 = t4542*t4538;
                double t4546 = Cx+t4545;
                double t4547 = 1.0/t4546;
                double t4548 = gamma_ab*2.0;
                double t4549 = gamma_aa+gamma_bb+t4548;
                double t4550 = sqrt(t4549);
                double t4555 = Cinf*Fg*Pg_1*t4550*t4544*t4547;
                double t4551 = exp(-t4555);
                double t4552 = t4531*t4532*(1.0/2.0);
                double t4553 = t4552+1.0/2.0;
                double t4554 = pow(t4553,5.0/3.0);
                double t4556 = -t4552+1.0/2.0;
                double t4557 = pow(t4556,5.0/3.0);
                double t4558 = t4554+t4557;
                double t4559 = 1.0/sqrt(t4558);
                double t4560 = Bg*t4528*t4539*(2.0/3.0);
                double t4561 = Cg_2*c*t4533*(1.0/3.0);
                double t4562 = t4560+t4561;
                double t4563 = t4562*t4538;
                double t4564 = 1.0/(t4527*t4527);
                double t4565 = Bg*c*t4528*t4564*1.0E4;
                double t4566 = Cg_4*t4528*t4539*(2.0/3.0);
                double t4567 = Cg_3*c*t4533*(1.0/3.0);
                double t4568 = t4565+t4566+t4567;
                double t4569 = 1.0/(t4537*t4537);
                double t4575 = t4542*t4568*t4569;
                double t4570 = t4563-t4575;
                double t4571 = t4531*(1.0/2.0);
                double t4572 = t4532*t4564*(1.0/2.0);
                double t4573 = t4571+t4572;
                double t4574 = 1.0/pow(t4527,1.3E1/6.0);
                double t4576 = 1.0/(t4546*t4546);
                double t4577 = 1.0/sqrt(t4549);
                double t4578 = Cinf*Fg*Pg_1*t4550*t4547*t4574*(7.0/6.0);
                double t4579 = t4578-Cinf*Fg*Pg_1*t4550*t4570*t4544*t4576;
                double t4580 = 1.0/pow(t4558,3.0/2.0);
                double t4581 = 1.0/pow(t4527,5.0/2.0);
                double t4582 = pow(t4556,2.0/3.0);
                double t4583 = t4573*t4582*(5.0/3.0);
                double t4584 = pow(t4553,2.0/3.0);
                double t4585 = t4583-t4573*t4584*(5.0/3.0);
                v_rho_b_gamma_aa[Q] += scale * (t4551*t4543*1.0/pow(t4527,7.0/3.0)*t4546*t4559*(-4.0/3.0)-t4533*t4551*t4543*t4570*1.0/sqrt(t4554+pow(t4531*t4532*(-1.0/2.0)+1.0/2.0,5.0/3.0))-t4533*t4551*t4543*t4580*t4546*t4585*(1.0/2.0)+t4533*t4551*t4543*t4546*t4559*t4579+t4533*t4551*t4543*t4546*t4549*t4559*(Cinf*Fg*Pg_1*t4547*t4574*t4577*(7.0/1.2E1)-Cinf*Fg*Pg_1*t4570*t4544*t4576*t4577*(1.0/2.0))+Cinf*Fg*Pg_1*t4550*t4551*t4543*1.0/pow(t4527,7.0/2.0)*t4559*(2.0/3.0)+Cinf*Fg*Pg_1*t4550*t4551*t4543*t4580*t4581*t4585*(1.0/4.0)-Cinf*Fg*Pg_1*t4550*t4551*t4543*t4581*t4559*t4579*(1.0/2.0)+Cinf*Fg*Pg_1*t4550*t4551*t4543*t4570*t4581*t4547*t4559*(1.0/2.0));
            }
            
            // v_rho_b_gamma_ab
            if (deriv >= 2) {
                double t4587 = rho_a+rho_b;
                double t4588 = c*c;
                double t4589 = 1.0/pow(t4587,2.0/3.0);
                double t4590 = 1.0/pow(t4587,1.0/3.0);
                double t4591 = 1.0/t4587;
                double t4592 = rho_a-rho_b;
                double t4593 = 1.0/pow(t4587,4.0/3.0);
                double t4594 = Bg*c*t4591*t4588*1.0E4;
                double t4595 = Cg_4*t4588*t4589;
                double t4596 = Cg_3*c*t4590;
                double t4597 = t4594+t4595+t4596+1.0;
                double t4598 = 1.0/t4597;
                double t4599 = 1.0/pow(t4587,5.0/3.0);
                double t4600 = Bg*t4588*t4589;
                double t4601 = Cg_2*c*t4590;
                double t4602 = Cg_1+t4600+t4601;
                double t4603 = 1.0/two_13;
                double t4604 = 1.0/pow(t4587,7.0/6.0);
                double t4605 = t4598*t4602;
                double t4606 = Cx+t4605;
                double t4607 = 1.0/t4606;
                double t4608 = gamma_ab*2.0;
                double t4609 = gamma_aa+gamma_bb+t4608;
                double t4610 = sqrt(t4609);
                double t4615 = Cinf*Fg*Pg_1*t4610*t4604*t4607;
                double t4611 = exp(-t4615);
                double t4612 = t4591*t4592*(1.0/2.0);
                double t4613 = t4612+1.0/2.0;
                double t4614 = pow(t4613,5.0/3.0);
                double t4616 = -t4612+1.0/2.0;
                double t4617 = pow(t4616,5.0/3.0);
                double t4618 = t4614+t4617;
                double t4619 = 1.0/sqrt(t4618);
                double t4620 = Bg*t4588*t4599*(2.0/3.0);
                double t4621 = Cg_2*c*t4593*(1.0/3.0);
                double t4622 = t4620+t4621;
                double t4623 = t4598*t4622;
                double t4624 = 1.0/(t4587*t4587);
                double t4625 = Bg*c*t4588*t4624*1.0E4;
                double t4626 = Cg_4*t4588*t4599*(2.0/3.0);
                double t4627 = Cg_3*c*t4593*(1.0/3.0);
                double t4628 = t4625+t4626+t4627;
                double t4629 = 1.0/(t4597*t4597);
                double t4635 = t4602*t4628*t4629;
                double t4630 = t4623-t4635;
                double t4631 = t4591*(1.0/2.0);
                double t4632 = t4592*t4624*(1.0/2.0);
                double t4633 = t4631+t4632;
                double t4634 = 1.0/pow(t4587,1.3E1/6.0);
                double t4636 = 1.0/(t4606*t4606);
                double t4637 = 1.0/sqrt(t4609);
                double t4638 = Cinf*Fg*Pg_1*t4610*t4607*t4634*(7.0/6.0);
                double t4639 = t4638-Cinf*Fg*Pg_1*t4610*t4630*t4604*t4636;
                double t4640 = 1.0/pow(t4618,3.0/2.0);
                double t4641 = 1.0/pow(t4587,5.0/2.0);
                double t4642 = pow(t4616,2.0/3.0);
                double t4643 = t4633*t4642*(5.0/3.0);
                double t4644 = pow(t4613,2.0/3.0);
                double t4645 = t4643-t4633*t4644*(5.0/3.0);
                v_rho_b_gamma_ab[Q] += scale * (1.0/pow(t4587,7.0/3.0)*t4611*t4603*t4606*t4619*(-8.0/3.0)-t4593*t4611*t4603*t4630*1.0/sqrt(t4614+pow(t4591*t4592*(-1.0/2.0)+1.0/2.0,5.0/3.0))*2.0-t4593*t4611*t4603*t4640*t4606*t4645+t4593*t4611*t4603*t4606*t4619*t4639*2.0+t4593*t4611*t4603*t4606*t4609*t4619*(Cinf*Fg*Pg_1*t4607*t4634*t4637*(7.0/6.0)-Cinf*Fg*Pg_1*t4630*t4604*t4636*t4637)+Cinf*Fg*Pg_1*1.0/pow(t4587,7.0/2.0)*t4610*t4611*t4603*t4619*(4.0/3.0)+Cinf*Fg*Pg_1*t4610*t4611*t4603*t4640*t4641*t4645*(1.0/2.0)-Cinf*Fg*Pg_1*t4610*t4611*t4603*t4641*t4619*t4639+Cinf*Fg*Pg_1*t4610*t4611*t4603*t4630*t4641*t4607*t4619);
            }
            
            // v_rho_b_gamma_bb
            if (deriv >= 2) {
                double t4647 = rho_a+rho_b;
                double t4648 = c*c;
                double t4649 = 1.0/pow(t4647,2.0/3.0);
                double t4650 = 1.0/pow(t4647,1.0/3.0);
                double t4651 = 1.0/t4647;
                double t4652 = rho_a-rho_b;
                double t4653 = 1.0/pow(t4647,4.0/3.0);
                double t4654 = Bg*c*t4651*t4648*1.0E4;
                double t4655 = Cg_4*t4648*t4649;
                double t4656 = Cg_3*c*t4650;
                double t4657 = t4654+t4655+t4656+1.0;
                double t4658 = 1.0/t4657;
                double t4659 = 1.0/pow(t4647,5.0/3.0);
                double t4660 = Bg*t4648*t4649;
                double t4661 = Cg_2*c*t4650;
                double t4662 = Cg_1+t4660+t4661;
                double t4663 = 1.0/two_13;
                double t4664 = 1.0/pow(t4647,7.0/6.0);
                double t4665 = t4662*t4658;
                double t4666 = Cx+t4665;
                double t4667 = 1.0/t4666;
                double t4668 = gamma_ab*2.0;
                double t4669 = gamma_aa+gamma_bb+t4668;
                double t4670 = sqrt(t4669);
                double t4675 = Cinf*Fg*Pg_1*t4670*t4664*t4667;
                double t4671 = exp(-t4675);
                double t4672 = t4651*t4652*(1.0/2.0);
                double t4673 = t4672+1.0/2.0;
                double t4674 = pow(t4673,5.0/3.0);
                double t4676 = -t4672+1.0/2.0;
                double t4677 = pow(t4676,5.0/3.0);
                double t4678 = t4674+t4677;
                double t4679 = 1.0/sqrt(t4678);
                double t4680 = Bg*t4648*t4659*(2.0/3.0);
                double t4681 = Cg_2*c*t4653*(1.0/3.0);
                double t4682 = t4680+t4681;
                double t4683 = t4682*t4658;
                double t4684 = 1.0/(t4647*t4647);
                double t4685 = Bg*c*t4648*t4684*1.0E4;
                double t4686 = Cg_4*t4648*t4659*(2.0/3.0);
                double t4687 = Cg_3*c*t4653*(1.0/3.0);
                double t4688 = t4685+t4686+t4687;
                double t4689 = 1.0/(t4657*t4657);
                double t4695 = t4662*t4688*t4689;
                double t4690 = t4683-t4695;
                double t4691 = t4651*(1.0/2.0);
                double t4692 = t4652*t4684*(1.0/2.0);
                double t4693 = t4691+t4692;
                double t4694 = 1.0/pow(t4647,1.3E1/6.0);
                double t4696 = 1.0/(t4666*t4666);
                double t4697 = 1.0/sqrt(t4669);
                double t4698 = Cinf*Fg*Pg_1*t4670*t4667*t4694*(7.0/6.0);
                double t4699 = t4698-Cinf*Fg*Pg_1*t4670*t4690*t4664*t4696;
                double t4700 = 1.0/pow(t4678,3.0/2.0);
                double t4701 = 1.0/pow(t4647,5.0/2.0);
                double t4702 = pow(t4676,2.0/3.0);
                double t4703 = t4693*t4702*(5.0/3.0);
                double t4704 = pow(t4673,2.0/3.0);
                double t4705 = t4703-t4693*t4704*(5.0/3.0);
                v_rho_b_gamma_bb[Q] += scale * (t4671*t4663*1.0/pow(t4647,7.0/3.0)*t4666*t4679*(-4.0/3.0)-t4653*t4671*t4663*t4690*1.0/sqrt(t4674+pow(t4651*t4652*(-1.0/2.0)+1.0/2.0,5.0/3.0))+t4653*t4671*t4663*t4666*t4679*t4699-t4653*t4671*t4663*t4666*t4700*t4705*(1.0/2.0)+t4653*t4671*t4663*t4666*t4669*t4679*(Cinf*Fg*Pg_1*t4667*t4694*t4697*(7.0/1.2E1)-Cinf*Fg*Pg_1*t4690*t4664*t4696*t4697*(1.0/2.0))+Cinf*Fg*Pg_1*t4670*t4671*t4663*1.0/pow(t4647,7.0/2.0)*t4679*(2.0/3.0)-Cinf*Fg*Pg_1*t4670*t4671*t4663*t4679*t4699*t4701*(1.0/2.0)+Cinf*Fg*Pg_1*t4670*t4671*t4663*t4700*t4701*t4705*(1.0/4.0)+Cinf*Fg*Pg_1*t4670*t4671*t4663*t4690*t4667*t4679*t4701*(1.0/2.0));
            }
            
        }
    }
}

}
