/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree_ins.c,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

#include <stdio.h>
#include <stdlib.h>

#include "dtree_private.h"

/* 
 * Static protos. 
 */
static bool_t 
dtree_cascade(PSlot_t PSlot, Slot_t Value, Slot_t SftIdx, ulong_t Level);

/* 
 * dtree_ins()
 *
 * Return element if there, insert if not.  Return a NULL if sbrk() fails.
 */
void **
dtree_ins(PPvoid_t PPdt, ulong_t Index)
{
    Slot_t      Slot, SlotNw;   /* The Slot in current Branch */
    ulong_t     Digit;          /* Index into current Branch pointing to Slot */
    ulong_t     SftIdx;         /* Shifted Index to just relevant bits */
    ulong_t     SltId;          /* Least 4 bits of Slot or Pointer */
    ulong_t     Level, ii;      /* Level we are in tree */
    PSlot_t     PSlot, PSlotNw; /* Pointer to next Branch */
    Slot_t     *PSlotToBr;
    ulong_t     Src, Des, FlwrSz;
    PSlot_t    *PPSlot[cDT_ROOT_LEVEL];
    ulong_t     PPSltIx;

    /* printf("\n --dtree_ins(0x%08lx, 0x%08lx), *PPdt = 0x%lx\n", 
     * PPdt, Index, *PPdt);
     */
    
    if (PPdt == NULL)  return((void **)NULL); /* Bug by caller */

ReStart:
    PSlot = (PSlot_t)PPdt;  /* For convienence, cc gens on code */
    SftIdx = Index;         /* For Debuging only, cc gens no code */
    Level = cDT_ROOT_LEVEL; /* Level of root pointer (16=32bit, 32=64bit) */

    /* Slot is the root pointer or null */
    Slot = *PSlot;

    /* Transform the NULL to internal type */
    if (Slot == 0L) Slot = cDT_NULL;
    PPSltIx = Digit = 0L;             /* Root pointer has no Digits */

    for(;;)
    {
        /* printf("Slot = 0x%lx, Level = %lu\n", Slot, Level); */

        PPSlot[PPSltIx++] = (PSlot_t *)PSlot;/* Save all pointers to Branches */

        SltId = Slot & cDT_SLOTIDMSK;        /* Mask =0xF-32bit, =0x1F-64bit */

        if (DT_IS_FLWR(Slot))
        {
            /* Get location of the Pointer (PSlot) to this branch */
            PSlotToBr = PSlot + Digit;

            /* Check if Narrow pointer or cDT_NULL */
            if (SltId == cDT_NARROW_PT)
            {
                if (Slot == cDT_NULL)
                {
                    /* Insert a new Brown Flower, add Index and exit */

                    /* Get some memory for a Brown flower */
                    if ((SlotNw = dtree_malloc(cDT_BR_FLWR_ID, Level)) == 0L) 
                        return((void **)NULL);

                    /* Get pointer to Brown flower */
                    PSlotNw = DT_FLWR_POINTER_TO_POINTER(SlotNw);

                    /* Stuff with Index and zero Value area */
                    *PSlotNw = SftIdx;
                    *(PSlotNw + cDT_BR) = 0L;

                    /* Put new branch in tree */
                    *PSlotToBr = SlotNw;

                    /* printf("Got Brown Flower @ 0x%lx, "
                     *        "now Stuff 0x%x, and RETURN\n",
                     *        PSlotNw, SlotNw);
                     */

                    TotalIndexes++;
                    /* And return pointer to Value area */
                    return((void **)(PSlotNw + cDT_BR));
                }

                /* Check if the Skipped Level bits equal */
                if ((SftIdx & *(PSlot+1)) != *(PSlot+0L))
                {
                    /* No, core dump for now */
                    Slot = *(PSlot_t)-1L;
                }

                /* Form the new SftId and Level */
                SftIdx <<= *(PSlot+2);
                Level -= *(PSlot+2);
                Slot = *(PSlot+3);
                continue;
            }
            break;
        }

        /* Zero Least 4 bits */
        PSlot = (PSlot_t)(Slot ^ SltId);
        Digit = DT_SFTIDX_TO_DIGIT(SftIdx, SltId);

        /* Shift out decoded Index bits */
        DT_SFTIDX_DIGITS(SftIdx, SltId);

        /* Adjust Level */
        Level -= SltId +1L;

        if (Level == 0) 
            return((void **)(PSlot + Digit)); /* to Leaf */

        /* Get Next Level Slot */
        Slot = *(PSlot + Digit);
    }

    /* PSlot   Is pointer to last Branch
     * Digit   Index into current Branch
     * SltId   Is the color of Branch Flower cDT_FLWR_ID
     * SftIdx  Index to decode at this Branch level
     * Slot    Is Flower pointer if not cDT_NULL
     */

    /* printf("PSlot = 0x%08lx, SftIdx = 0x%08lx, Level = %ld\n", 
     *         PSlot, SftIdx, Level);
     */

    /* Get pointer to this branch */
    PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

    /* All flowers are same format, so decode the length from SltId */

    /* Get number of Indexes in Flower */
    FlwrSz = DT_SLOTID_TO_FLWR_SIZE(Slot);

    /* Search Flower for a match */
    for (ii = 0L; ii < FlwrSz; ii++)
    {
        if (SftIdx == *(PSlot+ii))
        {
            /* Found matching Index -- done */
            return((void **)(PSlot+ii+FlwrSz));
        }
    }

    TotalIndexes++;

    /* If to big to grow, then do a cascade */
    if (FlwrSz == cDT_YL)
    {
        if (dtree_cascade(PSlotToBr, 0L, SftIdx, Level) == cDT_TRUE) 
            goto ReStart;

        return((void **)NULL);   /* Out of memory */
    }

    /*printf("Grow to a bigger Flower with FlwrSz = %lu\n", FlwrSz);*/

    /* This perhaps should be a build flower subroutine!!!!!!!!
     * like: yyy = buildflower(Sourceflower, len, Sftidx, Value, level);
     */

    /* Check if we really need a Leaf.  Is there a better way? */
    if (Level == 1L && FlwrSz == cDT_OR)
    {
        /* Allocate new Leaf */
        gDtreeStats.ss_Leaves++;
        if ((SlotNw = dtree_malloc(cDT_LEAF_ID, Level)) == 0L) 
        {
            return((void **)NULL);
        }

        /* Get pointer to newly allocated flower */
        PSlotNw = DT_FLWR_POINTER_TO_POINTER(SlotNw);

        /* Now, Insert the new Index using a sort/merge */
        for (Src = Des = 0L; Des < cDT_YL; Des++)
        {
            if (SftIdx < *(PSlot + Src) || Src == FlwrSz)
            {
                /* Stuff Value into Flower */
                *(PSlotNw+Des) = 0L;
                ii = Des;
                SftIdx = -1L; /* Never to return here */
            }
            else
            {
                *(PSlotNw+Des+0L) = *(PSlot+Src+FlwrSz);
                Src++;
            }
        }
    }
    else
    {
        if ((SlotNw = dtree_malloc(cDT_BR_FLWR_ID + FlwrSz, Level)) == 0L) 
        {
            return((void **)NULL);
        }

        /* Get pointer to newly allocated flower */
        PSlotNw = DT_FLWR_POINTER_TO_POINTER(SlotNw);

        /* Now, Insert the new Index using a sort/merge */
        for (Src = Des = 0L; Des < (FlwrSz+1); Des++)
        {
            if (SftIdx < *(PSlot + Src) || Src == FlwrSz)
            {
                /* Stuff Index into Flower */
                *(PSlotNw+Des) = SftIdx;
                ii = Des+FlwrSz+1;
                *(PSlotNw+ii) = 0L;
                SftIdx = -1L; /* Never to return here */
            }
            else
            {
                *(PSlotNw+Des+0L) = *(PSlot+Src+0L);
                *(PSlotNw+Des+FlwrSz+1) = *(PSlot+Src+FlwrSz);
                Src++;
            }
        }
    }
    
    /* And place new Flower in Branch */
    *PSlotToBr = SlotNw;

    /* We are done with it, so free the old Flower */
    dtree_free_branch(PSlot, SltId, cDT_TRUE);

    /* And return pointer to Value area Flower */
    return((void **)(PSlotNw + ii));
} /* dtree_ins() */


/* Used for Staging for building flowers */
typedef struct _INDEX_VALUE_PAIR
{
    Slot_t Index;
    Slot_t Value;
} ip_t, *Pip_t;


#ifdef DEBUG
void
PrintIndex4(size_t Index)
{
    size_t Digit, ii, Numb;

    Numb = Index;

    for (ii = 0L; ii < 32; ii += 2)
    {
        Digit = (Numb >> (30 - ii)) & 3;
        printf("%d", Digit);
    }
    printf(" ");
}
#endif

/* This routine allocates the Flower and copys the Shifted Indexes
 * and Values from the passed in arrays.
 * This routine will to sort the Indexes
 */
Slot_t
dtree_build_flower(Pip_t PSlotSrc, ulong_t FlwrSz, ulong_t Level)
{
    ulong_t ii;
    Slot_t  Slot;
    PSlot_t PSlot;

#ifdef DEBUG
    for (ii = 0; ii < FlwrSz; ii++) {
        printf("Flwr size = %lu   ", FlwrSz);
        PrintIndex4(PSlotSrc[ii].Index);
        printf(" 0x%08lx ",PSlotSrc[ii].Index);
        printf("\n");
    }
    /* Check if sorted */
    for (ii = FlwrSz-1; ii; ii--) {
        for (jj = 0; jj < ii; jj++) {
            if (PSlotSrc[jj].Index > PSlotSrc[jj+1].Index) {
                printf("Bug data in Flower is not sorted, FlwrSz = %d\n", 
                        FlwrSz);
                PrintIndex4(PSlotSrc[0].Index);
                printf(" ");
                PrintIndex4(PSlotSrc[1].Index);
                printf(" ");
                PrintIndex4(PSlotSrc[2].Index);
                printf(" ");
                PrintIndex4(PSlotSrc[3].Index);
                printf("\n");
            }
        }
    }

    if (FlwrSz == 4) {
        if (Level < 9) {
            printf("Allowcate Yellow Flower at Level = %lu\n", Level);
        }
    }
#endif /* DEBUG */

    if (Level < 0) printf("Panic at Line = %d, Level < 0\n", __LINE__);

    /* Check if we really need a Leaf.  Is there a better way? */
    if (Level == 1L && FlwrSz == cDT_YL)
    {
        /* Allocate new Leaf */
        gDtreeStats.ss_Leaves++;
        if ((Slot = dtree_malloc(cDT_LEAF_ID, Level)) == 0L) return(0L);

        /* Get normal pointer from Leaf Pointer */
        PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

        /* Copy just the data to Leaf */
        for (ii = 0L; ii < FlwrSz; ii++)
        {
            *(PSlot + ii) = PSlotSrc[ii].Value;
        }
    }
    else
    {
        /* Allocate new Flower */
        if ((Slot = dtree_malloc(cDT_BR_FLWR_ID + FlwrSz - 1, Level)) == 0L)
            return(0L);

        /* Get normal pointer from Flower Pointer */
        PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

        /* Copy the data to Flower */
        for (ii = 0L; ii < FlwrSz; ii++)
        {
            *(PSlot + ii) = PSlotSrc[ii].Index;
            *(PSlot + ii + FlwrSz) = PSlotSrc[ii].Value;
        }
    }
    return(Slot);   /* return pointer to Flower or Leaf */

} /* dtree_build_flower() */

/* Routine to take StageA's five sorted Indexes, and make Flowers
 *   PSlot is the branch that will point to the Flowers
 *   Return is cDT_FALSE == out of memory
 *   Return is cDT_TRUE  == Done sucessfully
 */
static ulong_t
dtree_splay_flower(PSlot_t PSlot, ip_t StageA[], ulong_t Level)
{
    ulong_t Digit, FlwrSz, Idx, Flr, NewIdx;
    Slot_t  Slot;

    Digit = DT_SFTIDX_TO_DIGIT(StageA[0].Index,0);

    for (FlwrSz = Idx = Flr = 0L; Idx < (cDT_MAX_FLWR+1); Idx++)
    {
        /* Get the new Index to work with */
        NewIdx = StageA[Idx].Index;

        /* Put it down a level */
        StageA[Idx].Index <<= cDT_BITS_PER_DIGIT;

        if (DT_SFTIDX_TO_DIGIT(NewIdx,0) == Digit)
        {
            FlwrSz++; /* count in current expanse */
        }
        else
        {
            /* New expanse, build and attach flower to new Branch */
            Slot = dtree_build_flower(&StageA[Flr], FlwrSz, Level);

            /* Leave data structure clean (as practical) */
            if (Slot == 0L) return(cDT_FALSE);

            /* Place flower in Branch */
            *(PSlot + Digit) = Slot;

            /* Keep marker for next Flower in StageA */
            Flr = Idx;
            FlwrSz = 1;

            /* Get next Digit to next expanse */
            Digit = DT_SFTIDX_TO_DIGIT(NewIdx,0);
        }
    }

    /* Check if StageA contains the 5 Indexes */
    if (FlwrSz > cDT_YL)
    {
        printf("Panic -- dtree_splay_flower() called with synonyms, "
               "Line = %u\n", __LINE__);
        exit(1);
    }

    /* Build flower, place pointer to it and exit */
    Slot = dtree_build_flower(&StageA[Flr], FlwrSz, Level);

    /* Leave data structure clean (as practical) */
    if (Slot == 0L) return(cDT_FALSE);

    /* Place flower in Branch */
    *(PSlot + Digit) = Slot;

    return(cDT_TRUE);

} /* dtree_splay_flower() */

/* This routine is called to expand a Yellow Flower -- do a cascade. */
static bool_t
dtree_cascade(PSlot_t PSlot, Slot_t Value, Slot_t SftIdx, ulong_t Level)
{
    ulong_t  ii, Digit;
    PSlot_t  Psl;           /* Pointer to next Branch */
    Slot_t   BstIdx;
    Slot_t   Slot;
    ulong_t  Src, Des;
    ip_t     StageA[cDT_YL+1];
    Slot_t   NewIdx, NewVal;
    Slot_t   Expanse;

    /* First lets SORT the five at their new level */
    Psl = DT_FLWR_POINTER_TO_POINTER(*PSlot);

    /* Since the Flowers are always sorted at this level all
     * we have to do here is a merge.
     * Copy/Merge all five Indexes and Values into StageA array
     */

    BstIdx = SftIdx;
    for (Src = Des = 0L; Des < (cDT_YL+1); Des++)
    {
        NewIdx = *(Psl + Src);

        /* Insert the new guy where it fits in */
        if (BstIdx < NewIdx || Src == cDT_YL)
        {
            NewIdx = BstIdx;
            NewVal = Value;
            BstIdx = -1L;   /* Prevent this path again */
        }
        else 
        { 
            /* Get Index and Value from full flower we are going to replace */
            NewVal = *(Psl+Src+cDT_YL);
            Src++;              /* Advance the Psl index */
        }
        StageA[Des].Index = NewIdx;
        StageA[Des].Value = NewVal;
    }

    /* We are done with the old Yellow Flower, so free it */
    dtree_free_branch(Psl, cDT_YL_FLWR_ID, cDT_TRUE);

    /* Get the maximum expanse of the Five Indexes */
    Expanse = StageA[cDT_YL].Index ^ StageA[0].Index;

    /* Calculate how many New levels */
    for (ii = 0L; ; ii++) 
    {
        if (Expanse & (cDT_DGMSK << ((cDT_ROOT_LEVEL-1L) * 
                       cDT_BITS_PER_DIGIT))) break;
        Expanse <<= cDT_BITS_PER_DIGIT;
    }

    /* Shift out to be decoded Index bits, if necessary */
    if (ii != 0L)
    {
        for (Des = 0; Des < (cDT_YL+1); Des++)
        {
            StageA[Des].Index <<= cDT_BITS_PER_DIGIT * ii;
        }
    }
    /* Pslot == location of old yellow flower */

    Level -= ii + 1;

    Digit = 0L;
    do 
    {
        /* Get a Branch */
        if ((Slot = dtree_malloc(cDT_NAR_BRANCH_ID, Level)) == 0L) 
            return(cDT_FALSE);

        /* Fill with dtree Nulls */
        dtree_fill_branch_nulls(Slot);

        /* And put it in place */
        *(PSlot+Digit) = Slot;
        PSlot = DT_FLWR_POINTER_TO_POINTER(Slot);

        /* Get next digit to decode */
        Digit = DT_SFTIDX_TO_DIGIT(SftIdx,0);
        /* Shift out decoded Index bits */
        SftIdx <<= cDT_BITS_PER_DIGIT;

    } while (ii--);

    return(dtree_splay_flower(PSlot, StageA, Level)); 

} /* dtree_cascade() */

