/**
 ** Proll (PROM replacement)
 ** system.c: shared miscallenea.
 ** Copyright 1999 Pete Zaitcev
 ** This code is licensed under GNU General Public License.
 **/
#include <stdarg.h>
#include <asi.h>
#include <crs.h>
#ifndef NULL
#define	NULL ((void*)0)
#endif

#include "pgtsrmmu.h"

#include "vconsole.h"
#include <timer.h>		/* Local copy of 2.2 style include */
#include <general.h>		/* __P() */
#include <net.h>		/* init_net() */
#include <romlib.h>		/* we are a provider for part of this. */
#include <netpriv.h>		/* myipaddr */
#include <arpa.h>
#include <system.h>		/* our own prototypes */

/*
 * We export this.
 */
char idprom[IDPROM_SIZE];

void probe_bank(struct bank *p, unsigned int, unsigned int, unsigned int);
int probe_memory_word(unsigned int addr, unsigned int alias);
int probe_virt_addr(unsigned int address, int write);
int probe_phys_addr(unsigned int address, int write);

/*
 * Poke the hardware and fill bank structure.
 * By necessity this routine knows too much about MrCoffee specifics.
 * Also a lot of black magic happens here because we do not know
 * the right way to probe stuff. For example, leave bank zero empty
 * and put 8MB into bank one. PROM detects 8MB just fine, but we
 * detect 1MB in bank 0 - bullshit alert. XXX
 */
void get_banks_layout(struct banks *p, unsigned int maxbanks)
{
	int x;		/* Fill index for bankv[] */
	int i;
	unsigned int base, bs = 32*1024*1024;

	x = 0;
	base = 0;
	for (i = 0; i < maxbanks; i++) {
		probe_bank(&p->bankv[x], base, 1024*1024, bs);
		if (p->bankv[x].length != 0) x++;
		base += bs;
	}
	p->nbanks = x;
}

void probe_bank(struct bank *p,
    unsigned int base, unsigned int minsize, unsigned int maxsize)
{
	unsigned size;

	/*
	 * Watch for overflows in address arithmetics.
	 */
	size = minsize;
	for (;;) {
		if (size == minsize) {
			if (base != 0) {
				/*
				 * This clause is not really necessary but...
				 */
				if (probe_memory_word(base + size - sizeof(int),
				    0 + size - sizeof(int)) != 0) {
					p->start = base;
					p->length = 0;
					return;
				}
			} else {
				if (probe_memory_word(base + size - sizeof(int),
				    0) != 0) {
					p->start = base;
					p->length = 0;
					return;
				}
			}
		} else {
			if (probe_memory_word(base + size - sizeof(int),
			    base + (size >> 1) - sizeof(int)) != 0) {
				p->start = base;
				p->length = size >> 1;
				return;
			}
		}
		if (size >= maxsize) break;
		size <<= 1;
	}
	p->start = base;
	p->length = size;
	return;
}

/*
 * Read write memory and see if it works.
 * Check for aliasing.
 */
int probe_memory_word(unsigned int addr, unsigned int alias) {
	unsigned save, salias;
	unsigned xval, xalias;

	if (alias != 0) {
		if (probe_phys_addr(addr, 0) != 0) {
			printk("read fault at 0x%x\n", addr);
			return -1;
		}
		save = ld_bypass(addr);
		salias = ld_bypass(alias);
		if (probe_phys_addr(addr, 1) != 0) {
			printk("write fault at 0x%x\n", addr);
			return -1;
		}

		st_bypass(alias, save);
		st_bypass(addr, ~save);
		xalias = ld_bypass(alias);
		xval = ld_bypass(addr);
		st_bypass(addr, save);
		st_bypass(alias, salias);

		if (xval != (~save)) {
			printk("no memory at 0x%x\n", addr);
			return -1;
		}
		if (xalias != save) {
#if 0
			printk("memory at 0x%x aliased to 0x%x\n",
			    addr, alias);
#endif
			return -1;
		}
	} else {
		if (probe_phys_addr(addr, 0) != 0) {
			printk("read fault at 0x%x\n", addr);
			return -1;
		}
		save = ld_bypass(addr);
		if (probe_phys_addr(addr, 1) != 0) {
			printk("write fault at 0x%x\n", addr);
			return -1;
		}
		st_bypass(addr, ~save);
		xval = ld_bypass(addr);
		st_bypass(addr, save);
		if (xval != (~save)) {
			printk("no memory at 0x%x\n", addr);
			return -1;
		}
	}
	/* printk("memory at 0x%x\n", addr); */ /* P3 */
	return 0;
}

#if 0
/*
 * Poke at the given address, return the success/failure.
 * This code cooperates with srmmu_fault in head.S.
 */
int		/* 0 - OK, <>0 - faulted */
probe_virt_addr(unsigned int address, int write)
{
	volatile extern unsigned int ignore_fault;
	volatile extern int fault_ignored;
	int dum;

	printk("probing 0x%x...", address);
	fault_ignored = 0;
	ignore_fault = address;
	if (write) {
		*(volatile unsigned int *)address = 0x00ff7700;
	} else {
		dum = *(volatile unsigned int *)address;
	}
	ignore_fault = 0;
	if (fault_ignored) {
		printk(" fault taken\n");
	} else {
		printk(" no fault\n");
	}
	return fault_ignored;
}
#endif

/*
 * Create an I/O mapping to pa[size].
 * Returns va of the mapping or 0 if unsuccessful.
 */
void *
map_io(unsigned pa, int size)
{
	void *va;
	unsigned int npages;
	unsigned int off;
	unsigned int mva;

	off = pa & (PAGE_SIZE-1);
	npages = (off + size + (PAGE_SIZE-1)) / PAGE_SIZE;
	pa &= ~(PAGE_SIZE-1);

	va = mem_alloc(&cio, npages*PAGE_SIZE, PAGE_SIZE);
	if (va == 0) return va;

	mva = (unsigned int) va;
    /* printk("map_io: va 0x%x pa 0x%x off 0x%x npages %d\n", va, pa, off, npages); */ /* P3 */
	while (npages-- != 0) {
		map_page(pmem.pl1, mva, pa, 1, pmem.pbas);
		mva += PAGE_SIZE;
		pa += PAGE_SIZE;
	}

	return (void *)((unsigned int)va + off);
}

/*
 * Tablewalk routine used for testing.
 * Returns PTP/PTE.
 */
unsigned int
proc_tablewalk(int ctx, unsigned int va)
{
	unsigned int pa1;

	__asm__ __volatile__ ("lda [%1] %2, %0" :
				"=r" (pa1) :
				"r" (AC_M_CTPR), "i" (ASI_M_MMUREGS));
	/* printk(" ctpr %x ctx %x\n", pa1, ctx); */ /* P3 */
	pa1 <<= 4;
	pa1 = ld_bypass(pa1 + (ctx << 2));
	if ((pa1 & 0x03) == 0) goto invalid;
	return mem_tablewalk((pa1 & 0xFFFFFFF0) << 4, va);

invalid:
	printk(" invalid %x\n", pa1);
	return 0;
}

/*
 * Walk the tables in memory, starting at physical address pa.
 */
unsigned int
mem_tablewalk(unsigned int pa, unsigned int va)
{
	unsigned int pa1;

	printk("pa %x va %x", pa, va);
	pa1 = ld_bypass(pa + (((va&0xFF000000)>>24) << 2));
	if ((pa1 & 0x03) == 0) goto invalid;
	printk(" l1 %x", pa1);
	pa1 <<= 4;    pa1 &= 0xFFFFFF00;
	pa1 = ld_bypass(pa1 + (((va&0x00FC0000)>>18) << 2));
	if ((pa1 & 0x03) == 0) goto invalid;
	printk(" l2 %x", pa1);
	pa1 <<= 4;    pa1 &= 0xFFFFFF00;
	pa1 = ld_bypass(pa1 + (((va&0x0003F000)>>12) << 2));
	if ((pa1 & 0x03) == 0) goto invalid;
	printk(" l3 %x", pa1);
	printk(" off %x\n", va&0x00000FFF);
	return pa1;
invalid:
	printk(" invalid %x\n", pa1);
	return 0;
}

/*
 * Make CPU page tables.
 * Returns pointer to context table.
 * Here we ignore memory allocation errors which "should not happen"
 * because we cannot print anything anyways if memory initialization fails.
 */
void makepages(struct phym *t, unsigned int highbase)
{
	unsigned int *ctp, *l1, pte;
	int i;
	unsigned int pa, va;

	ctp = mem_zalloc(&cmem, NCTX_SWIFT*sizeof(int), NCTX_SWIFT*sizeof(int));
	l1 = mem_zalloc(&cmem, 256*sizeof(int), 256*sizeof(int));

	pte = SRMMU_ET_PTD | (((unsigned int)l1 - PROLBASE + highbase) >> 4);
	for (i = 0; i < NCTX_SWIFT; i++) {
		ctp[i] = pte;
	}

	pa = highbase;
	for (va = PROLBASE; va < PROLBASE+PROLSIZE; va += PAGE_SIZE) {
		map_page(l1, va, pa, 0, highbase);
		pa += PAGE_SIZE;
	}

	/* We need to start from LOADBASE, but kernel wants PAGE_SIZE. */
	pa = PAGE_SIZE;
	for (va = PAGE_SIZE; va < LOWMEMSZ; va += PAGE_SIZE) {
		map_page(l1, va, pa, 0, highbase);
		pa += PAGE_SIZE;
	}

	t->pctp = ctp;
	t->pl1 = l1;
	t->pbas = highbase;
}

/*
 * Create a memory mapping from va to epa in page table pgd.
 * highbase is used for v2p translation.
 */
int
map_page(unsigned int *pgd, unsigned int va,
    unsigned int epa, int type, unsigned int highbase)
{
	unsigned int pte;
	unsigned int *p;
	unsigned int pa;

	pte = pgd[((va)>>SRMMU_PGDIR_SHIFT) & (SRMMU_PTRS_PER_PGD-1)];
	if ((pte & SRMMU_ET_MASK) == SRMMU_ET_INVALID) {
		p = mem_zalloc(&cmem, SRMMU_PTRS_PER_PMD*sizeof(int),
		    SRMMU_PTRS_PER_PMD*sizeof(int));
		if (p == 0) goto drop;
		pte = SRMMU_ET_PTD |
		    (((unsigned int)p - PROLBASE + highbase) >> 4);
		pgd[((va)>>SRMMU_PGDIR_SHIFT) & (SRMMU_PTRS_PER_PGD-1)] = pte;
		/* barrier() */
	}

	pa = ((pte & 0xFFFFFFF0) << 4);
	pa += (((va)>>SRMMU_PMD_SHIFT & (SRMMU_PTRS_PER_PMD-1)) << 2);
	pte = ld_bypass(pa);
	if ((pte & SRMMU_ET_MASK) == SRMMU_ET_INVALID) {
		p = mem_zalloc(&cmem, SRMMU_PTRS_PER_PTE*sizeof(int),
		    SRMMU_PTRS_PER_PTE*sizeof(int));
		if (p == 0) goto drop;
		pte = SRMMU_ET_PTD |
		    (((unsigned int)p - PROLBASE + highbase) >> 4);
		st_bypass(pa, pte);
	}

	pa = ((pte & 0xFFFFFFF0) << 4);
	pa += (((va)>>PAGE_SHIFT & (SRMMU_PTRS_PER_PTE-1)) << 2);

	pte = SRMMU_ET_PTE | ((epa & PAGE_MASK) >> 4);
	if (type) {		/* I/O */
		pte |= SRMMU_REF;
		/* SRMMU cannot make Supervisor-only, but not exectutable */
		pte |= SRMMU_PRIV;
	} else {		/* memory */
		pte |= SRMMU_REF|SRMMU_CACHE;
		pte |= SRMMU_PRIV;		/* Supervisor only access */
	}
	st_bypass(pa, pte);
	return 0;

drop:
	return -1;
}

/*
 * Poke at the given physical address, return the success/failure.
 * This technique is sun4m specific and works for lower 4GB only.
 * This code cooperates with srmmu_fault in head.S.
 * Actually bypass access never produces a fault on Swift.
 * Instead it returns a random bus traffic happened before.
 */
int		/* 0 - OK, <>0 - faulted */
probe_phys_addr(unsigned int address, int write)
{
	volatile extern unsigned int ignore_fault;
	volatile extern int fault_ignored;
	int dum;

	fault_ignored = 0;
	ignore_fault = address;
	if (write) {
		st_bypass(address, 0x00ff7700);
	} else {
		dum = ld_bypass(address);
	}
	ignore_fault = 0;
	return fault_ignored;
}

/*
 * Switch page tables.
 */
void
init_mmu_swift(unsigned int ctp_phy)
{
	unsigned int addr;

	/*
	 * Flush cache
	 */
	for (addr = 0; addr < 0x2000; addr += 0x10) {
		__asm__ __volatile__ ("sta %%g0, [%0] %1\n\t" : :
		    "r" (addr), "i" (ASI_M_DATAC_TAG));
		__asm__ __volatile__ ("sta %%g0, [%0] %1\n\t" : :
		    "r" (addr<<1), "i" (ASI_M_TXTC_TAG));
	}

	/*
	 * Switch ctx table
	 */
	ctp_phy >>= 4;
	/* printk("done flushing, switching to %x\n", ctp_phy); */
	__asm__ __volatile__ ("sta %0, [%1] %2\n\t" : :
	    "r" (ctp_phy), "r" (AC_M_CTPR), "i" (ASI_M_MMUREGS));

	/*
	 * Flush old page table references
	 */
	__asm__ __volatile__ ("sta %%g0, [%0] %1\n\t" : :
	    "r" (0x400), "i" (ASI_M_FLUSH_PROBE) : "memory");
}

/*
 * add_timer, del_timer
 * This should go into sched.c, but we have it split for different archs.
 */
struct timer_list_head {
	struct timer_list *head, *tail;
};

static struct timer_list_head timers;		/* Anonymous heap of timers */

void add_timer(struct timer_list *timer) {
	struct timer_list *p;
	if (timer->prev != NULL || timer->next != NULL) {
		printk("bug: kernel timer added twice at 0x%x.\n",
		    __builtin_return_address(0));
		return;
	}
	if ((p = timers.tail) != NULL) {
		timer->prev = p;
		p->next = timer;
		timers.tail = timer;
	} else {
		timers.head = timer;
		timers.tail = timer;
	}
	return;
}

int del_timer(struct timer_list *timer) {
	struct timer_list *p;
	int ret;

	if (timers.head == timer) timers.head = timer->next;
	if (timers.tail == timer) timers.tail = timer->prev;
	if ((p = timer->prev) != NULL) p->next = timer->next;
	if ((p = timer->next) != NULL) p->prev = timer->prev;
	ret = timer->next != 0 || timer->prev != 0;
	timer->next = NULL;
	timer->prev = NULL;
	return ret;
}

void run_timers() {
	struct timer_list *p;

	p = timers.head;
	while (p != NULL) {
		if (p->expires < jiffies) {
			del_timer(p);		/* XXX make nonstatic member */
			(*p->function)(p->data);
			p = timers.head;
		} else {
			p = p->next;
		}
	}
}

/*
 * Allocate memory. This is reusable.
 */
void mem_init(struct mem *t, char *begin, char *limit)
{
	t->start = begin;
	t->uplim = limit;
	t->curp = begin;
}

void mem_fini(struct mem *t)
{
	t->curp = 0;
}

void *mem_alloc(struct mem *t, int size, int align)
{
	char *p;

	p = (char *)((((unsigned int)t->curp) + (align-1)) & ~(align-1));
	if (p >= t->uplim || p + size > t->uplim) return 0;
	t->curp = p + size;
	return p;
}

void *mem_zalloc(struct mem *t, int size, int align)
{
	char *p;

	if ((p = mem_alloc(t, size, align)) != 0) bzero(p, size);
	return p;
}

/*
 * Library functions
 */
void bzero(void *s, int len) {
	while (len--) *((char *)s)++ = 0;
}

void bcopy(void *f, void *t, int len) {
	while (len--) *((char *)t)++ = *((char *)f)++;
}

/* Comparison is 7-bit */
int bcmp(void *s1, void *s2, int len)
{
	int i;
	char ch;

	while (len--) {
		ch = *((char *)s1)++;
		if ((i = ch - *((char *)s2)++) != 0)
			return i;
		if (ch == 0)
			return 0;
	}
	return 0;
}

int strlen(char *s) {
	char *p;
	for (p = s; *p != 0; p++) { }
	return p - s;
}

void printk(char *fmt, ...)
{
	struct prf_fp {
		void *xfp;
		void (*write)(void *, char *, int);
	} prfa;
	extern void prf(struct prf_fp *, char *fmt, va_list adx);
	va_list x1;

	va_start(x1, fmt);
	prfa.xfp = &dp0;
	prfa.write = (void*)vcon_write;
	prf(&prfa, fmt, x1);
	va_end(x1);
}

/* This is taken from x86 to be used in network kernel. Returns 15 bits. */
short int random()
{
	static unsigned int seed = 151;
	seed = (seed + 23968)*0x015A4E35 >> 1;
	return seed & 0x7FFF;
}

void fatal()
{
	printk("fatal.");
loop: goto loop;
}

/*
 * Get the highest bit number from the mask.
 */
int highc(int mask, int size)
{
	int m1;

	m1 = 1 << size;
	while (size != 0) {
		size--;
		m1 >>= 1;
		if (m1 & mask) break;
	}
	return size;
}

/*
 */
unsigned int ld_bp_swap(unsigned int ptr) {
	unsigned int n;
	n = ld_bypass(ptr);
	n = (n>>24 & 0xFF) | (n>>8 & 0xFF00) | ((n&0xFF00) << 8) | (n<<24);
	return n;
}

void st_bp_swap(unsigned int ptr, unsigned int n) {
	n = (n>>24 & 0xFF) | (n>>8 & 0xFF00) | ((n&0xFF00) << 8) | (n<<24);
	st_bypass(ptr, n);
};
