/*-----------------------------------------------------------------------
QMBTagger - Qt Based MusicBrainz MP3/OGG/FLAC Tagger
Copyright (C) 2003,2004 Ewen Cheslack-Postava
 
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------*/

#include "qmbfile.h"

#if HAVE_LIBMUSICBRAINZ

#include <vector>
#include <qstring.h>
#include <qapplication.h>
#include <qdir.h>
#include <musicbrainz/queries.h>

QMBFile::QMBFile()
:bufferlen(0), buffer(0), channels(0), sampleRate(0), songlength(0), queryresults(NULL) {
    //we do nothing here.  we want to allow checking of return value,
    //so we only allow class to function through calling of Extract()
}

QMBFile::~QMBFile() {
    if (buffer != 0) {
        delete [] buffer;
        bufferlen = 0;
    }
    if (queryresults != NULL) {
        delete [] queryresults;
    }
}

//returns:
//  -2: failed Finalizing signature
//  -1: failed GenerateSignature
//  0: success
int QMBFile::CreateID(trm_t &trm) {

    if (!metadata.GetTRMID().isEmpty())
        return 0;

    //set the data we got from Extract()
    trm_SetPCMDataInfo(trm, sampleRate, channels, 16);

    //set the song length (in seconds) found in Extract()
    trm_SetSongLength(trm, (long int)songlength);

    //try to generate the signature
    if(trm_GenerateSignature(trm, buffer, bufferlen) == 0) {
        //failed the GenerateSignature because more data was needed
        qWarning("QMBFile::CreateID(): Failed to generate signature - more data was needed (%s).", fullfilename.local8Bit().data());
        return -1;
    } else {
        //signature successful and was given enough data
        if(trm_FinalizeSignature(trm, signature, NULL) == -1) {
            //finalize signature failed, bail out gracefully
            qWarning("QMBFile::CreateID(): Failed to finalize signature (%s).", fullfilename.local8Bit().data());
            return -2;
        }
    }

    //now just get it into ASCII form
    trm_ConvertSigToASCII(trm, signature, ascii_sig);
    //now copy this into the metadata
    metadata.SetTRMID(ascii_sig);

    if (buffer != 0) {
        delete [] buffer;
        bufferlen = 0;
        buffer = NULL;
    }
    return 0;
}

//returns:
//-1 on error
// otherwise the number of queries returned
int QMBFile::QueryServer(MusicBrainz* mb) {

    //now get ready for the actual query
    vector<std::string> queryargs(6);

    //set the query args
    queryargs[0] = metadata.GetTRMID().latin1();
    queryargs[1] = metadata.GetArtistName().latin1();
    queryargs[2] = metadata.GetAlbumName().latin1();
    queryargs[3] = metadata.GetTrackName().latin1();
    queryargs[4] = metadata.GetTrackNumber().latin1();
    queryargs[5] = metadata.GetDuration().latin1();

    //run the query
    if (!mb->Query(MBQ_TrackInfoFromTRMId,&queryargs)) {
        std::string queryerror; //FIXME: we want this to use QString if possible
        mb->GetQueryError(queryerror);
        qWarning("QMBFile::QueryServer(): MBQ_TrackInfoFromTRMId on file (%s) failed: %s", fullfilename.local8Bit().data(), queryerror.c_str());
        return -1;
    }

    numresults = mb->DataInt(MBE_GetNumTracks);

    if (numresults < 1) {
        qWarning("QMBFile::QueryServer(): MBQ_TrackInfoFromTRMId on file (%s) returned no results, trying MBQ_FileInfoLookup.", fullfilename.local8Bit().data());

        //setup the query args for MBQ_FileInfoLookup
        vector<std::string> filequeryargs(11);
        filequeryargs[0] = metadata.GetTRMID().latin1();
        filequeryargs[1] = metadata.GetArtistName().latin1();
        filequeryargs[2] = metadata.GetAlbumName().latin1();
        filequeryargs[3] = metadata.GetTrackName().latin1();
        filequeryargs[4] = metadata.GetTrackNumber().latin1();
        filequeryargs[5] = metadata.GetDuration().latin1();
        filequeryargs[6] = metadata.GetFileName().latin1();
        filequeryargs[7] = metadata.GetArtistID().latin1();
        filequeryargs[8] = metadata.GetAlbumID().latin1();
        filequeryargs[9] = metadata.GetTrackID().latin1();
        filequeryargs[10] = "10"; //maxresults

        if (!mb->Query(MBQ_FileInfoLookup, &filequeryargs)) {
            std::string queryerror; //FIXME: we want this to use QString if possible
            mb->GetQueryError(queryerror);
            qWarning("QMBFile::QueryServer(): MBQ_FileInfoLookup on file (%s) failed: %s", fullfilename.local8Bit().data(), queryerror.c_str());
            return -1;
        }

        numresults = mb->DataInt(MBE_GetNumTracks);

        if (numresults < 1) {
            qWarning("QMBFile::QueryServer(): Couldn't find any results for file (%s).", fullfilename.local8Bit().data());
            return -1;
        }
    }

    queryresults = new QMBMetadata[numresults];

    for (int selecttrack = 1; selecttrack <= numresults; selecttrack++) {
        mb->Select(MBS_Rewind);
        mb->Select(MBS_SelectTrack, selecttrack);
        queryresults[selecttrack-1].SetTrackName((QString)mb->Data(MBE_TrackGetTrackName).c_str());
        QString trackIdURI = (QString)mb->Data(MBE_TrackGetTrackId).c_str(); //save so we can use the URI to get the trackNum
        QString trackIdCopy = trackIdURI;
        this->StripURL(trackIdCopy);
        queryresults[selecttrack-1].SetTrackID(trackIdCopy);

        queryresults[selecttrack-1].SetDuration((QString)mb->Data(MBE_TrackGetTrackDuration).c_str());
        queryresults[selecttrack-1].SetArtistName((QString)mb->Data(MBE_TrackGetArtistName).c_str());
        QString artistid = (QString)mb->Data(MBE_TrackGetArtistId).c_str();
        this->StripURL(artistid);
        queryresults[selecttrack-1].SetArtistID(artistid);

        queryresults[selecttrack-1].SetRelevance(mb->DataInt(MBE_LookupGetRelevance));

        mb->Select(MBS_SelectTrackAlbum);
        queryresults[selecttrack-1].SetAlbumName((QString)mb->Data(MBE_AlbumGetAlbumName).c_str());
        QString albumid = (QString)mb->Data(MBE_AlbumGetAlbumId).c_str();
        this->StripURL(albumid);
        queryresults[selecttrack-1].SetAlbumID(albumid);
        queryresults[selecttrack-1].SetTrackNumber(QString::number(mb->GetOrdinalFromList(MBE_AlbumGetTrackList, trackIdURI.latin1())));

        //finally just set the trmid from the value we already have
        queryresults[selecttrack-1].SetTRMID(metadata.GetTRMID());
    }

    return numresults;
}

void QMBFile::StripURL(QString &url) {
    int lastslash = url.findRev("/");

    url = url.right(url.length() - lastslash - 1);
}

void QMBFile::setMetadata(TRACK *track) {
    //get the album name
    metadata.SetAlbumName(track->album);
    //get the song title
    metadata.SetTrackName(track->title);
    //get the artist name
    metadata.SetArtistName(track->artist);
    //get the track number
    metadata.SetTrackNumber(QString::number(track->tracknumber));
}

#endif /*HAVE_LIBMUSICBRAINZ*/
