/* $Id: FilteredDataEntryReader.java,v 1.3 2005/06/11 13:13:15 eric Exp $
 *
 * ProGuard -- shrinking, optimization, and obfuscation of Java class files.
 *
 * Copyright (c) 2002-2005 Eric Lafortune (eric@graphics.cornell.edu)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package proguard.io;

import java.io.*;


/**
 * This DataEntryReader delegates to one of two other DataEntryReader instances,
 * depending on whether the data entry passes through a given data entry filter
 * or not.
 *
 * @author Eric Lafortune
 */
public class FilteredDataEntryReader implements DataEntryReader
{
    private DataEntryFilter dataEntryFilter;
    private DataEntryReader acceptedDataEntryReader;
    private DataEntryReader rejectedDataEntryReader;


    /**
     * Creates a new FilteredDataEntryReader with only a reader for accepted
     * data entries.
     * @param dataEntryFilter         the data entry filter.
     * @param acceptedDataEntryReader the DataEntryReader to which the reading
     *                                will be delegated if the filter accepts
     *                                the data entry. May be <code>null</code>.
     */
    public FilteredDataEntryReader(DataEntryFilter dataEntryFilter,
                                   DataEntryReader acceptedDataEntryReader)
    {
        this(dataEntryFilter, acceptedDataEntryReader, null);
    }


    /**
     * Creates a new FilteredDataEntryReader.
     * @param dataEntryFilter         the data entry filter.
     * @param acceptedDataEntryReader the DataEntryReader to which the reading
     *                                will be delegated if the filter accepts
     *                                the data entry. May be <code>null</code>.
     * @param rejectedDataEntryReader the DataEntryReader to which the reading
     *                                will be delegated if the filter does not
     *                                accept the data entry. May be
     *                                <code>null</code>.
     */
    public FilteredDataEntryReader(DataEntryFilter dataEntryFilter,
                                   DataEntryReader acceptedDataEntryReader,
                                   DataEntryReader rejectedDataEntryReader)
    {
        this.dataEntryFilter         = dataEntryFilter;
        this.acceptedDataEntryReader = acceptedDataEntryReader;
        this.rejectedDataEntryReader = rejectedDataEntryReader;
    }


    // Implementations for DataEntryReader.

    public void read(DataEntry dataEntry)
    throws IOException
    {
        if (dataEntryFilter.accepts(dataEntry))
        {
            if (acceptedDataEntryReader != null)
            {
                acceptedDataEntryReader.read(dataEntry);
            }
        }
        else
        {
            if (rejectedDataEntryReader != null)
            {
                rejectedDataEntryReader.read(dataEntry);
            }
        }
    }
}
