//
//  PREqualize.m
//  PRICE
//  Image level equalization
//
//  Created by Riccardo Mottola on Fri Dec 05 2003.
//  Copyright (c) 2003-2004 Carduus. All rights reserved.
//
// This program is free software; you can redistribute it and/or modify it under the terms of the version 2 of the GNU General Public License as published by the Free Software Foundation.
// This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

#import <math.h>
#import <limits.h>

#import "PREqualize.h"


@implementation PREqualize

- (PRImage *)equalizeImage:(PRImage *)srcImage
{
    NSBitmapImageRep *srcImageRep;
    PRImage          *destImage;
    NSBitmapImageRep *destImageRep;
    int              w, h;
    int              x, y;
    int              i;
    unsigned char    *srcData;
    unsigned char    *destData;
    int              bytesPerPixel;
    int              pixNum;
    BOOL             isColor;
    
    /* some trace */
    printf("levels: %d\n", UCHAR_MAX);
    
    /* get source image representation and associated information */
    srcImageRep = [srcImage tiffRep];
    
    w = [srcImageRep pixelsWide];
    h = [srcImageRep pixelsHigh];
    pixNum = h * w;
    printf("pixels: %d\n", pixNum);
    bytesPerPixel = [srcImageRep bitsPerPixel] /8;
    
    /* check bith depth and color/greyscale image */
    if ([srcImageRep hasAlpha])
    {
        if ([srcImageRep samplesPerPixel] == 2)
        {
            printf("Grayscale image\n");
            isColor = NO;
        }
        else
        {
            printf("Color image\n");
            isColor = YES;
        }
    }
    else
    {
        if ([srcImageRep samplesPerPixel] == 1)
        {
            printf("Grayscale image\n");
            isColor = NO;
        }
        else
        {
            printf("Color image\n");
            isColor = YES;
        }
    }
    
    /* allocate destination image and its representation */
    destImage = [[PRImage alloc] initWithSize:NSMakeSize(w, h)];
    if (isColor)
    {
        destImageRep = [[NSBitmapImageRep alloc]
                    initWithBitmapDataPlanes:NULL
                    pixelsWide:w
                    pixelsHigh:h
                    bitsPerSample:8
                    samplesPerPixel:3
                    hasAlpha:NO
                    isPlanar:NO
                    colorSpaceName:NSCalibratedRGBColorSpace
                    bytesPerRow:0
                    bitsPerPixel:0];
    } else
    {
        destImageRep = [[NSBitmapImageRep alloc]
                    initWithBitmapDataPlanes:NULL
                    pixelsWide:w
                    pixelsHigh:h
                    bitsPerSample:8
                    samplesPerPixel:1
                    hasAlpha:NO
                    isPlanar:NO
                    colorSpaceName:NSCalibratedWhiteColorSpace
                    bytesPerRow:0
                    bitsPerPixel:0];
    }
    srcData = [srcImageRep bitmapData];
    destData = [destImageRep bitmapData];
    
    if (isColor)
    {
        unsigned long int histogramDenormR[UCHAR_MAX+1]; /* not normalized pixel count for each level */
        unsigned long int histogramDenormG[UCHAR_MAX+1]; /* not normalized pixel count for each level */
        unsigned long int histogramDenormB[UCHAR_MAX+1]; /* not normalized pixel count for each level */
        float histogramR[UCHAR_MAX+1];                   /* normalized histogram */
        float histogramG[UCHAR_MAX+1];                   /* normalized histogram */
        float histogramB[UCHAR_MAX+1];                   /* normalized histogram */
        float cumulativeHistogramR[UCHAR_MAX+1];         /* cumulative histogram */
        float cumulativeHistogramG[UCHAR_MAX+1];         /* cumulative histogram */
        float cumulativeHistogramB[UCHAR_MAX+1];         /* cumulative histogram */
        
        /* calculate the histogram */
        for (i = 0; i <= UCHAR_MAX; i++)
            histogramDenormR[i] = histogramDenormG[i] = histogramDenormB[i] =  0;
        for (y = 0; y < h; y++)
            for (x = 0; x < w*3; x += 3)
            {
                histogramDenormR[srcData[y*(w*3) + x]]++;
                histogramDenormG[srcData[y*(w*3) + x + 1]]++;
                histogramDenormB[srcData[y*(w*3) + x + 2]]++;
            }
    
        /* normalize histogram */
        for (i = 0; i <= UCHAR_MAX; i++)
        {
            histogramR[i] = (float)histogramDenormR[i] / (float)pixNum;
            histogramG[i] = (float)histogramDenormG[i] / (float)pixNum;
            histogramB[i] = (float)histogramDenormB[i] / (float)pixNum;
        }
        
        /* cumulative histogram */
        cumulativeHistogramR[0] = histogramR[0];
        cumulativeHistogramG[0] = histogramG[0];
        cumulativeHistogramB[0] = histogramB[0];
        for (i = 1; i <= UCHAR_MAX; i++)
        {
            cumulativeHistogramR[i] = cumulativeHistogramR[i-1] + histogramR[i];
            cumulativeHistogramG[i] = cumulativeHistogramG[i-1] + histogramG[i];
            cumulativeHistogramB[i] = cumulativeHistogramB[i-1] + histogramB[i];
        }
        
        /* equalize */
        for (y = 0; y < h; y++)
            for (x = 0; x < w*3; x += 3)
            {
                destData[y*(w*3) + x] = floor((UCHAR_MAX+0.9)*cumulativeHistogramR[srcData[y*(w*3) + x]]);
                destData[y*(w*3) + x + 1] = floor((UCHAR_MAX+0.9)*cumulativeHistogramG[srcData[y*(w*3) + x + 1]]);
                destData[y*(w*3) + x + 2] = floor((UCHAR_MAX+0.9)*cumulativeHistogramB[srcData[y*(w*3) + x + 2]]);
            }
    } else
    {
        unsigned long int histogramDenorm[UCHAR_MAX+1]; /* not normalized pixel count for each level */
        float histogram[UCHAR_MAX+1];                   /* normalized histogram */
        float cumulativeHistogram[UCHAR_MAX+1];         /* cumulative histogram */
        /* calculate the histogram */
        for (i = 0; i <= UCHAR_MAX; i++)
            histogramDenorm[i] = 0;
        for (y = 0; y < h; y++)
            for (x = 0; x < w; x++)
                histogramDenorm[srcData[y*w + x]]++;
    
        /* normalize histogram */
        for (i = 0; i <= UCHAR_MAX; i++)
            histogram[i] = (float)histogramDenorm[i] / (float)pixNum;
        
        /* cumulative histogram */
        cumulativeHistogram[0] = histogram[0];
        for (i = 1; i <= UCHAR_MAX; i++)
            cumulativeHistogram[i] = cumulativeHistogram[i-1] + histogram[i];
        
        /* equalize */
        for (y = 0; y < h; y++)
            for (x = 0; x < w; x++)
                destData[y*w + x] = floor((UCHAR_MAX+0.9)*cumulativeHistogram[srcData[y*w + x]]);
    }
    
    [destImage addRepresentation:destImageRep];
    [destImageRep release];
    [destImage autorelease];
    return destImage;
}


@end
