/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: presentation.cc,v $
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:34  slash
 * Initial release.
 *
 */

#include <stdio.h>

#include <libintl.h>
#define _(String) gettext(String)

#include <string>
#include <fstream.h>

#include <parser.h>

#include "main.h"
#include "commentnode.h"
#include "htmlnode.h"
#include "latexnode.h"
#include "slidenode.h"
#include "sectionnode.h"
#include "parameters.h"
#include "presentation.h"


//---------------------------------------------------------------------------
Presentation::Presentation() : Node()
{
    xml_tree_ = NULL;
    toc_nodes_.clear();
}

//---------------------------------------------------------------------------
Presentation::~Presentation()
{
    toc_nodes_.clear();
}


//---------------------------------------------------------------------------
int Presentation::parseXMLFile()
{
    ifstream xml_file(Parameters_.xml_file_.c_str());

    if (!xml_file) {
        ENDL;
        printf(_("The XML-file \"%s\" doesn't exist"),
               Parameters_.xml_file_.c_str());
        ENDL;
        return -1;
    }

    xml_tree_ = xmlParseFile(Parameters_.xml_file_.c_str());
    if (xml_tree_ == NULL) {
        return -1;
    }

    // Find the XML-node, where the presentation starts
    xmlNodePtr node = xml_tree_->xmlRootNode;
    while ((node->type != XML_ELEMENT_NODE) ||
           strcmp((char*)node->name, "presentation")) {
        node = node->next;
    }

    // Extract the child-nodes.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        if (!strcmp((char*)node->name, "html")) {
            child_nodes_.push_back(new HTMLNode(node));

        } else if (!strcmp((char*)node->name, "latex")) {
            child_nodes_.push_back(new LaTeXNode(node));

        } else if (!strcmp((char*)node->name, "slide")) {
            SlideNode *slide = new SlideNode(node);
            child_nodes_.push_back(slide);
            toc_nodes_.push_back((TOCInterface*)slide);

        } else if (!strcmp((char*)node->name, "section")) {
            SectionNode *section = new SectionNode(node);
            child_nodes_.push_back(section);
            toc_nodes_.push_back((TOCInterface*)section);

        } else if (node->type == XML_COMMENT_NODE) {
            child_nodes_.push_back(new CommentNode(node));

        } else {
            Number_Of_Errors_++;
        }
    }

    SlideNode::initFilenameOfFirstAndLastSlide();

    return 0;
}


//---------------------------------------------------------------------------
void Presentation::writeHTML(int parameter) const
{
    copyThemeFiles();

    // Write the table of contents.
    printf(_("Creating the table of contents ... "));
    writeTOC();
    printf(_("finished."));
    ENDL;

    // Write the slides.
    printf(_("Creating %d slides ... "), SlideNode::All_Slides_.size());
    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeHTML();
    }
    printf(_("finished."));
    ENDL;
}


//---------------------------------------------------------------------------
void Presentation::writeLaTeX() const
{
    // Create the LaTeX-file.
    Output_.open("presentation.tex");

    // Write the LaTeX-header.
    if (Parameters_.isOutlineEnabled()) {
        Output_ << "\\documentclass[12pt]{article}" << endl
                << "\\usepackage{graphicx}" << endl
//              << "\\usepackage[pdftex]{graphicx}" << endl
                << "\\usepackage{tabularx}" << endl
                << "\\usepackage{fancyheadings}" << endl
                << "\\usepackage{latexsym}" << endl
//              << "\\usepackage[pdftex,plainpages=false]{hyperref}" << endl
                << endl;

        Output_ << "\\renewcommand{\\rmdefault}{phv}" << endl
                << "\\renewcommand{\\sfdefault}{phv}" << endl
                << "\\renewcommand{\\ttdefault}{pcr}" << endl << endl;

        Output_ << "\\newcommand{\\PreserveBackslash}[1]{\\let\\temp=\\\\#1\\let\\\\=\\temp}" << endl
                << "\\let\\PBS=\\PreserveBackslash" << endl
<< "\\newcolumntype{L}{>{\\PBS\\raggedright\\hspace{0pt}}l}" << endl
                << "\\newcolumntype{R}{>{\\PBS\\raggedright\\hspace{0pt}}r}" << endl
                << "\\newcolumntype{C}{>{\\PBS\\raggedright\\hspace{0pt}}c}" << endl
                << "\\newcolumntype{Y}{>{\\PBS\\raggedright\\hspace{0pt}}X}" << endl << endl;

        Output_ << "\\textwidth=6.5in" << endl
                << "\\textheight=8.5in " << endl
                << "\\oddsidemargin=-0.00in" << endl
                << "\\evensidemargin=-0.00in" << endl
                << "\\topmargin=-0.25in" << endl
                << "\\parskip=0.15in" << endl
                << "\\parindent=0in" << endl
                << "\\pagenumbering{arabic}" << endl;

    } else {
        Output_ << "\\documentclass{seminar}" << endl
                << "\\usepackage{slidesec}" << endl
                << "\\usepackage{semcolor}" << endl
                << "\\usepackage{graphicx}" << endl
                << "\\usepackage[latin1]{inputenc}" << endl
                << "\\usepackage[T1]{fontenc}" << endl
                << "\\usepackage{latexsym}" << endl
                << "\\usepackage{url}" << endl;
//      if (Parameters_.output_format_ == PDFLATEX) {
//          Output_ << "\\usepackage{ae,aecompl}" << endl;
//      }
    }
    

    Output_ << endl << "\\begin{document}" << endl << endl;

    // Write the content of all slides to the LaTeX-file.
    for (unsigned int i=0; i<child_nodes_.size();i++) {
        child_nodes_[i]->writeLaTeX();
    }

    // Write the LaTeX-footer.
    Output_ << "\\end{document}" << endl;
    Output_.close();
}

//---------------------------------------------------------------------------
void Presentation::writeXML() const
{
    Output_.open("presentation.xml");

    Output_ << "<?xml version=\"1.0\"?>" << endl
            << "<!DOCTYPE presentation SYSTEM \"/usr/local/share/prestimel/prestimel.dtd\">" << endl;

    Output_ << "<presentation>" << endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</presentation>" << endl;

    Output_.close();
}


//---------------------------------------------------------------------------
void Presentation::copyThemeFiles() const
{
    string source, destination;

    // Copy the next-button to the target-directory.
    if (Parameters_.next_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_NEXT;
        destination = BUTTON_NEXT;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the prev-button to the target-directory.
    if (Parameters_.prev_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_PREV;
        destination = BUTTON_PREV;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the first-button to the target-directory.
    if (Parameters_.first_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_FIRST;
        destination = BUTTON_FIRST;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the last-button to the target-directory.
    if (Parameters_.last_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_LAST;
        destination = BUTTON_LAST;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the note-button to the target-directory.
    if (Parameters_.note_) {
        source = Parameters_.full_path_to_theme_ + BUTTON_NOTE;
        destination = BUTTON_NOTE;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the stylesheet to the target-directory.
    source = Parameters_.full_path_to_theme_ + STYLES;
    switch (Parameters_.getResolution()) {
      case 640:  source += "_640"; break;
      case 800:  source += "_800"; break;
      case 1024: source += "_1024"; break;
    }
    source += STYLES_EXT;
    destination = string(STYLES) + STYLES_EXT;
    fcopy(source.c_str(), destination.c_str());

    // Copy the background-image for the slide to the target-directory.
    if (Parameters_.isSlideBackgroundEnabled()) {
        source = Parameters_.full_path_to_theme_ + BG_SLIDE + JPG_EXT;
        destination = string(BG_SLIDE) + JPG_EXT;
        fcopy(source.c_str(), destination.c_str());
    }

    // Copy the background-graphic for the header to the target-directory.
    if (Parameters_.isHeaderBackgroundEnabled()) {
        source = Parameters_.full_path_to_theme_ + BG_HEADER;
        switch (Parameters_.getResolution()) {
          case 640:  source += "_640"; break;
          case 800:  source += "_800"; break;
          case 1024: source += "_1024"; break;
        }
        source += GIF_EXT;
        destination = string(BG_HEADER) + GIF_EXT;
        fcopy(source.c_str(), destination.c_str());
    }
}


//---------------------------------------------------------------------------
void Presentation::writeTOC() const
{
    Output_.open("toc.html");

    // First, write the HTML-header.
    Output_ << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">" << endl
            << "<HTML>" << endl
            << "<HEAD>" << endl
            << "<META"
            << " name=\"Generator\""
            << " content=\"PresTiMeL " << VERSION << "\""
            << ">" << endl
            << "<TITLE>Index of the presentation</TITLE>" << endl
//          << "<LINK"
//          << " href=\"styles.css\""
//          << " rel=\"stylesheet\""
//          << " type=\"text/css\""
//          << ">" << endl
            << "</HEAD>" << endl
            << "<BODY>" << endl
            << "<TABLE"
            << " summary=\"index\""
            << " border=\"0\""
            << ">" << endl;

    // Now loop through all slides and sections.
    for (unsigned int i=0; i<toc_nodes_.size(); i++) {
        Output_ << "<TR><TD>";
        toc_nodes_[i]->writeTOCEntry();
        Output_ << "</TD></TR>" << endl;
    }

    Output_ << "</TABLE>" << endl
            << "</BODY>" << endl
            << "</HTML>" << endl;

    Output_.close();
}
