/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: listnode.h,v $
 * Revision 1.1.1.1  2001/01/30 18:36:48  slash
 * Initial release.
 *
 */

#ifndef LISTNODE_H
#define LISTNODE_H

#include "environmentnode.h"

//---------------------------------------------------------------------------
/**
 * This base-class provides an additional interface for list-environments.
 */
class ListNode : public EnvironmentNode {
  public:
    //-----------------------------------------------------------------------
    /**
     * The constructor initializes the list-node.
     *
     * @param level The level of the list-node.
     * @param with_left_margin Indicates, if the HTML-output should have a
     *                         margin on the left side.
     */
    ListNode(unsigned int level = 0,
             int with_left_margin = 1);

    //-----------------------------------------------------------------------
    /**
     * Destructor.
     */
    ~ListNode();


    //-----------------------------------------------------------------------
    /**
     * This method writes the node to the global output filestream Output_.
     * The format of the output must be HTML.
     *
     * @param number_of_elements The number of items to write.
     *                           If -1, all items will be written NORMAL.
     *                           If -2, all items will be written WEAK.
     */
    void writeHTML(int number_of_elements = -1) const;


  protected:     

    //-----------------------------------------------------------------------
    /**
     * This helper-method writes the HTML-header for the list to the
     * global output filestream Output_.
     * It must be overloaded by all derivated classes.
     */
    virtual void writeHTMLHeader() const = 0;

    //-----------------------------------------------------------------------
    /**
     * This helper-method writes the HTML-footer for the list to the
     * global output filestream Output_.
     * It must be overloaded by all derivated classes.
     */
    virtual void writeHTMLFooter() const = 0;

    //-----------------------------------------------------------------------
    /**
     * This helper-method writes all child-nodes to the global output
     * filestream Output_.
     *
     * @param style The style, how the items should be written.
     */
    void writeHTMLAllItems(int style) const;

    //-----------------------------------------------------------------------
    /**
     * This helper-method writes the given number of items to the global
     * output filestream Output_.
     *
     * @param number_of_items The number of items to write.
     */
    void writeHTMLItems(int number_of_items) const;

    //-----------------------------------------------------------------------
    /**
     * This method returns the number of characters of the longest item.
     *
     * @return The length of the longest item.
     */
    virtual unsigned int getLength() const;


  protected:

    //-----------------------------------------------------------------------
    /**
     * If is_sub_list_[i] is 1, the i'th child-node is a sub-list node.
     */
    vector<int> is_sub_list_;

    //-----------------------------------------------------------------------
    /**
     * Indicates the level of the list.
     */
    unsigned int level_;

    //-----------------------------------------------------------------------
    /**
     * 1, if the HTML-output should have a margin on the left side, else 0.
     */
    int with_left_margin_;

    //-----------------------------------------------------------------------
    /**
     * The maximum of the number of characters of the items.
     */
    unsigned int max_length_;
};

#endif //LISTNODE_H
