/*****
*
* Copyright (C) 2001 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include <libprelude/prelude-log.h>

#include "rules.h"
#include "rules-operations.h"
#include "rules-parsing.h"
#include "rules-default.h"


#define PARSE_ERROR_MAX_LENGTH 500
static char parse_error_str[PARSE_ERROR_MAX_LENGTH + 1];


/*
 * key:value
 */
typedef struct parse_key {
	char *key;
	int set;
        
	union {
		parse_f_t parse;
		parse_set_f_t parse_set;
	} fun;

	struct parse_key *next;
} parse_key_t;



/*
 * post processing function
 */
typedef struct post_processing {
	post_processing_f_t pp_fun;
	struct post_processing *next;
} post_processing_t;

 
static parse_key_t parse_key_set = { NULL, 0, { NULL }, NULL };
static post_processing_t post_processing_set = { NULL, NULL };


int parse_error_locked = 1;



/*
 * parameters
 */
parameters_t *signature_parser_make_parameters(char *str) 
{
	parameters_t *new;

        new = malloc(sizeof(parameters_t));
	if ( ! new )
		return NULL;

	new->str = str;
	new->not = 0;
	new->next = NULL;

	return new;
}



parameters_t *signature_parser_inverse_parameters(parameters_t *parameters) 
{
	parameters_t *param;

        for ( param = parameters; param != NULL; param = param->next )
		param->not = ! param->not;

	return parameters;
}




parameters_t *signature_parser_link_parameters(parameters_t *parameters1, parameters_t *parameters2) 
{
	parameters_t **p_next;

        p_next = &parameters1->next;

        while( *p_next )
		p_next = &( (*p_next)->next );

	*p_next = parameters2;

	return parameters1;
}



void signature_parser_free_parameters(parameters_t *parameters) 
{
        parameters_t *next;
        
	while (parameters) {
		next = parameters->next;
		free(parameters->str);
		free(parameters);
		parameters = next;
	}
}



/*
 * add a post processing function
 */
int signature_parser_add_post_processing(post_processing_f_t pp_fun) 
{
	post_processing_t *pp_tmp;

        pp_tmp = malloc(sizeof(post_processing_t));
	if ( ! pp_tmp ) {
		log(LOG_ERR, "memory exhausted.");
		return -1;
	}

	pp_tmp->pp_fun = pp_fun;
	pp_tmp->next = post_processing_set.next;
	post_processing_set.next = pp_tmp;

	return 0;
}




/*
 * Function to be called to set parsing error.
 */
void signature_parser_set_error(const char *format, ...) 
{
	if ( ! parse_error_locked ) {
		va_list arg;
                
		va_start(arg, format);

		vsnprintf(parse_error_str, PARSE_ERROR_MAX_LENGTH, format, arg);
		parse_error_locked = 1;

		va_end(arg);
	}
}




/*
 * Get the parser error buffer.
 */
const char *signature_parser_get_error_buffer(void) 
{
	return parse_error_str;
}




/*
 * Parse a key and it's parameters.
 */
int signature_parser_parse_key(const char *key, parameters_t *parameters, rules_t **rule) 
{
        rules_t *tmp;
	parse_key_t *keytmp;
        
	*rule = NULL;
        keytmp = parse_key_set.next;

	while( keytmp != NULL && strcasecmp(key, keytmp->key) )
		keytmp = keytmp->next;

	if ( ! keytmp ) {
                signature_parser_set_error( "Unknow key %s", key );
		return -1;
	}
	       
	if ( keytmp->set ) {
                if ( keytmp->fun.parse_set(parameters, rule) < 0 ) {
                        delete_rules(*rule);
                        return -1;
                }

                return 0;
        }        
        
        for ( ; parameters; parameters = parameters->next ) {
                tmp = NULL;
                
                if ( keytmp->fun.parse(parameters->str, &tmp) < 0 ) {                        
                        delete_rules(*rule);
                        return -1;
                }
                
                if ( parameters->not )
                        tmp = rules_not(tmp);
                
                *rule = rules_or(*rule, tmp);
        }

        return 0;
}



/*
 *
 */
static parse_key_t *create_key_generic(const char *key) 
{
        parse_key_t *new;

        new = malloc(sizeof(parse_key_t));
	if (! new ) {
                log(LOG_ERR, "memory exhausted.\n");
		return new;
	}

	new->key = strdup(key);
	new->set = 0;
	new->next = parse_key_set.next;
	parse_key_set.next = new;

        return new;
}




/*
 * Add a key to the known key list.
 */
int signature_parser_add_one_arg_key(const char *key, parse_f_t parse) 
{
	parse_key_t *new;

        new = create_key_generic(key);
        if ( ! new )
                return -1;
        
	new->set = 0;
	new->fun.parse = parse;
        
	return 0;
}



/*
 * Add a key set to the known key list.
 */
int signature_parser_add_multiple_args_key(const char *key, parse_set_f_t parse) 
{
	parse_key_t *new;

        new = create_key_generic(key);
        if ( ! new )
                return -1;
        
	new->set = 1;
	new->fun.parse_set = parse;

	return 0;
}



/*
 * 
 */
int signature_parser_post_processing(rules_t *rules) 
{
        post_processing_t *pp;
	        
        for ( pp = post_processing_set.next; pp != NULL; pp = pp->next ) {

                if ( pp->pp_fun(rules) < 0 ) {
                        delete_rules(rules);
                        signature_parser_set_error("Invalid Rules Post-Processing.");
                        return -1;
                }
                
        }
		
        return 0;
}


#if 0

/*
 * parse a rule and add it to the tree
 */
int signature_parser_add_signature(rules_node_t *root, const char *rule, run_f_t run)
{
        int ret = -1;
        rules_t *rules;

        rules = signature_parser_parse_signature(rule);
        if ( rules ) {
                ret = signature_engine_add_rules(root, rules, run);
                delete_rules(rules);
        }
        
        return ret;
}
#endif
