/*****
*
* Copyright (C) 1998, 1999, 2000, 2002, 2003 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
*****/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <errno.h>


#include "packet.h"

#include <libprelude/list.h>
#include <libprelude/plugin-common.h>
#include <libprelude/plugin-common-prv.h>
#include <libprelude/prelude-log.h>
#include <libprelude/config-engine.h>
#include <libprelude/prelude-io.h>
#include <libprelude/prelude-message.h>
#include <libprelude/prelude-getopt.h>
#include <libprelude/idmef-tree.h>
#include <libprelude/sensor.h>
#include <libprelude/variable.h>
#include <libprelude/prelude-path.h>
#include <libprelude/prelude-list.h>
#include <libprelude/prelude-async.h>
#include <libprelude/daemonize.h>

#include "config.h"
#include "pconfig.h"
#include "capture.h"
#include "tcp-stream.h"


extern Pconfig_t config;


typedef struct {
        struct list_head list;
        int snaplen;
        int promisc;
        const char *bpf;
        const char *device;
        const char *inputfile;
        const char *outputfile;
} devconf_t;


static devconf_t *cur;
static int global_promisc = 1;
static uid_t prelude_user = 0;
static LIST_HEAD(devices_list);
static int global_snaplen = 8192;
static const char *global_bpf = NULL;



static devconf_t *add_device(void) 
{
        devconf_t *new;

        cur = new = calloc(1, sizeof(*new));
        if ( ! new ) {
                log(LOG_ERR, "memory exhausted.\n");
                return NULL;
        }

        new->bpf = global_bpf;
        new->promisc = global_promisc;
        new->snaplen = global_snaplen;
        
        list_add_tail(&new->list, &devices_list);

        return new;
}



static int print_version(prelude_option_t *opt, const char *arg) 
{
        printf("prelude-nids %s.\n", VERSION);
        return prelude_option_success;
}




static int get_version(char *buf, size_t size) 
{
        snprintf(buf, size, "prelude-nids %s\n", VERSION);
        return prelude_option_success;
}



static int print_help(prelude_option_t *opt, const char *arg)
{
        prelude_option_print(CLI_HOOK, 25);
        return prelude_option_end;
}



static int set_capture_from_file(prelude_option_t *opt, const char *arg) 
{
        devconf_t *dev;
        
        dev = add_device();
        if ( ! dev )
                return prelude_option_error;
                
        dev->inputfile = strdup(arg);

        return prelude_option_success;
}


static int set_capture_from_device(prelude_option_t *opt, const char *arg) 
{
        int ret;
        devconf_t *dev;
        char buf[256], vname[256];

        /*
         * This might fail if the interface is not bound to an addr (stealth).
         */
        ret = capture_get_device_address(arg, buf, sizeof(buf));
        if ( ret == 0 ) {
                snprintf(vname, sizeof(vname), "%s_ADDRESS", arg);
                                
                ret = variable_set(strdup(vname), strdup(buf));        
                if ( ret < 0 ) {
                        log(LOG_ERR, "couldn't create variable %s.\n", vname);
                        return prelude_option_error;
                }
        }
        
        dev = add_device();
        if ( ! dev )
                return prelude_option_error;
        
        dev->device = strdup(arg);
        
        return prelude_option_success;
}


static int set_capture_to_file(prelude_option_t *opt, const char *arg) 
{
        if ( ! cur )
                return prelude_option_error;

        if ( cur->inputfile ) {
                log(LOG_INFO, "can't both read and write packet from/to a file.\n");
                return prelude_option_error;
        }
        
        cur->outputfile = strdup(arg);
        
        return prelude_option_success;
}


static int set_bpf_rule(prelude_option_t *opt, const char *arg) 
{
        if ( cur )
                cur->bpf = strdup(arg);
        else
                global_bpf = strdup(arg);
        
        return prelude_option_success;
}



static int set_snaplen(prelude_option_t *opt, const char *arg) 
{
        /* specific to a device */
        if ( cur )
                cur->snaplen = atoi(arg);
        else
                global_snaplen = atoi(arg);
        return prelude_option_success;
}



static int set_promiscuous_mode_off(prelude_option_t *opt, const char *arg) 
{
        /* specific o a device */
        if ( cur )
                cur->promisc = 0;
        else
                global_promisc = 0;

        return prelude_option_success;
}




static int get_report_all(char *buf, size_t size) 
{
        snprintf(buf, size, "%d", config.report_all);
        return prelude_option_success;
}




static int set_report_all(prelude_option_t *opt, const char *arg) 
{
        config.report_all = ! config.report_all;
        return prelude_option_success;
}



static int set_quiet_mode(prelude_option_t *opt, const char *arg) 
{
        prelude_log_use_syslog();
        return prelude_option_success;
}


static int set_daemon_mode(prelude_option_t *opt, const char *arg) 
{        
        prelude_daemonize(config.pidfile);
        prelude_log_use_syslog();

        return prelude_option_success;
}


static int set_pidfile(prelude_option_t *opt, const char *arg) 
{        
        config.pidfile = strdup(arg);
        return prelude_option_success;
}


static int set_prelude_user_id(prelude_option_t *opt, const char *arg) 
{
        struct passwd *p;

        p = getpwnam(arg);
        if ( ! p ) {
                log(LOG_ERR, "couldn't find user %s.\n", arg);
                return prelude_option_error;
        }

        /*
         * we can't apply the new UID now, as opening capture
         * devices require root access. Store it for later use.
         */
        prelude_user = p->pw_uid;

        /*
         * tell the prelude library that every operation should be done as
         * non root.
         */
        prelude_set_program_userid(p->pw_uid);

        return prelude_option_success;
}



static int set_statefull_only(prelude_option_t *opt, const char *arg) 
{
        config.statefull_only = 1;
        return prelude_option_success;
}




static int get_statefull_only(char *buf, size_t size)
{
        snprintf(buf, size, "%d", config.statefull_only);
        return prelude_option_success;
}




static int setup_capture_device(void) 
{
        devconf_t *dev;
        int ret, devnum = 0;
        struct list_head *tmp;
        
        list_for_each(tmp, &devices_list) {

                dev = list_entry(tmp, devconf_t, list);

                if ( dev->inputfile ) 
                        ret = capture_from_file(dev->inputfile, dev->bpf);
                else 
                        ret = capture_from_device(dev->device, dev->bpf, dev->snaplen, dev->promisc, dev->outputfile);
                
                if ( ret < 0 )
                        return -1;

                devnum++;
        }
        
        return devnum;
}




int pconfig_set(int argc, char **argv) 
{
        int ret;
        prelude_option_t *opt;
        
        /*
         * default.
         */
        config.statefull_only = 0;
        config.report_all = 0;
        config.pidfile = NULL;
        
        prelude_option_add(NULL, CLI_HOOK, 'h', "help",
                           "Print this help", no_argument, print_help, NULL);
        
        prelude_option_add(NULL, CLI_HOOK|WIDE_HOOK, 'v', "version",
                           "Print version number", no_argument, print_version,
                           get_version);
        
        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 'q', "quiet",
                           "Quiet mode", no_argument, set_quiet_mode, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 'u', "user",
                           "Run as the specified user", required_argument,
                           set_prelude_user_id, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 'd', "daemon",
                           "Run in daemon mode", no_argument, set_daemon_mode, NULL);
        
        /*
         * we want this option CB to run before -d option cb
         */
        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 'P', "pidfile",
                                 "Write Prelude PID to pidfile", required_argument,
                                 set_pidfile, NULL);
        prelude_option_set_priority(opt, option_run_first);

                
        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK, 'i', "interface",
                           "Capture packet from interface", required_argument,
                           set_capture_from_device, NULL);
        
        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'w', "write-to",
                           "Record packets to a file", required_argument,
                           set_capture_to_file, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'r', "read-from",
                           "Read packets from a file", required_argument,
                           set_capture_from_file, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'b', "bpf",
                           "Specify a BPF rule. You can use a different -b option for each interface.",
                           required_argument, set_bpf_rule, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'p', "promisc",
                           "Disable promiscuous mode. You can use the -p option globally or per interface", 
                           no_argument, set_promiscuous_mode_off, NULL);
        
        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 's', "snaplen",
                           "Specify the maximum number of bytes to capture by packet. "
                           "The value should be set to the biggest MTU of all captured interface. "
                           "This option can be used globaly (before any -i or -r option) or on a per interface basis",
                           required_argument, set_snaplen, NULL);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'a', "report-all",
                           "The default is to return as soon as a signature match. "
                           "Set this option in order to report all matched signature",
                           no_argument, set_report_all, get_report_all);

        prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'z', "statefull-only",
                           "Don't analyses TCP packet not associated with a stream.",
                           no_argument, set_statefull_only, get_statefull_only);
        
        tcp_stream_init_config();
        
	ret = prelude_sensor_init("prelude-nids", PRELUDE_CONF, argc, argv);
	if ( ret == prelude_option_error || ret == prelude_option_end )
                exit(1);

        ret = setup_capture_device();
        if ( ret <= 0 ) {
		fprintf(stderr, "\nNo interface nor file to read the packet from were specified.\n\n");
		return -1;
	}

        if ( prelude_user ) {
                ret = setuid(prelude_user);
                if ( ret < 0 ) {
                        fprintf(stderr, "couldn't change UID to %d.\n", prelude_user);
                        return -1;
                }
        }        

        ret = prelude_async_init();
        if ( ret < 0 ) {
                fprintf(stderr, "error initializing asynchronous subsystem.\n");
                return -1;
        }
        
        return 0;
}

