#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <config.h>
#include <support.h>

#define	PPXPD
#include <xcio.h>
#include <xcmdlist.h>
#include <sysmsg.h>

#include "log.h"
#include "option.h"
#include "chat.h"
#include "console.h"
#include "command.h"
#include "phase.h"
#include "dev/device.h"

extern int devFd;
static pid_t fgPid=0;

static struct exeq_s {
    struct exeq_s *next;
    void (*qhook)();
    char **argv;
    int argc, runmode;
    pid_t pid;
} *eqHead;

void
AllExit()
{
    Logf(LOG_OS, "quit\n");
/*    ConsoleOutf("\n");*/
    if (DevClose) DevClose(-1);
    DnsRestoreConf();
    DnsRelay(NULL);
    AccountLogWrite(FALSE);
    LastLogWrite(FALSE);

    while (eqHead) {
	if (eqHead->runmode & EXEC_BACK) kill(eqHead->pid, SIGQUIT);
	eqHead = eqHead->next;
    }

    BindClose();
    ConsoleCloseAll();
    SysIfClose();
    exit(0);
}

int
CmdConnect(int argc, char *argv[])
{
    int status=0;

    if (pppInfo.l_stat & LSTAT_PPP) return(0);
    if (argc == 0 || (*argv[0] | ' ') == 'c')
	pppInfo.l_stat &= ~LSTAT_TTY;
    else
	pppInfo.l_stat = LSTAT_TTY;
    if (!pppOpt.line || !DevOpen) {
	ConsoleMsg(MS_E_SETLINE, NULL);
	return(-1);
    }
    if (devFd < 0) {
	pppInfo.minfo = (unsigned)-1;
	pppInfo.connect = 0;
	memset(&pppInfo.r, 0, sizeof(struct npkt_s));
	memset(&pppInfo.s, 0, sizeof(struct npkt_s));
	pppInfo.n_stat = 0;
/*	CloseDevice();*/
	TimerPause();
	if ((devFd = DevOpen(pppOpt.line, &status, argc, argv)) < 0) {
	    LogError("CmdConnect DevOpen");
	    return(status);
	}
	TimerStart();
	if (status == OPEN_DELAYED) ConsoleUpdate(TRUE);
	else StartDevice();
    } else pppInfo.l_stat |= LSTAT_TTY;
    return(0);
}

int
CmdDisconnect(int argc, char *argv[])
{
    if (pppInfo.l_stat & (LSTAT_DIAL|LSTAT_CHAT|LSTAT_TTY))
	pppInfo.l_stat &= ~(LSTAT_DIAL|LSTAT_CHAT|LSTAT_TTY);
    PhaseDown();
    return(0);
}

int
CmdEcho(int argc, char *argv[])
{
    char *d;

    d = argc > 1 ? argv[1]: "";
    SendEchoRequest(d, strlen(d));
    return(0);
}

/*
static int
CmdShow(int argc, char *argv[])
{
    return(0);
}
*/

static int
CmdQuit(int argc, char *argv[])
{
    PhaseQuit();
    CmdDisconnect(0, NULL);
    return(0);
}

static void
ExecNext()
{
    struct exeq_s *eqp=eqHead;
    pid_t pid;

    while (eqp && eqp->pid) eqp = eqp->next;
    if (!eqp || ((fgPid > 0) && !(eqp->runmode & EXEC_BACK))) return;
    if ((pid = fork()) == 0) {
	char *path, *env;
	int slen, ulen;

	slen = strlen(sysPPxP);
	ulen = usrPPxP ? strlen(usrPPxP): 0;
	env = Malloc(slen + ulen + sizeof("rc/:") * 2);
	if (!(eqp->runmode & EXEC_SECURE) && usrPPxP)
	    sprintf(env, "%s/rc:%s/rc", usrPPxP, sysPPxP);
	else
	    sprintf(env, "%s/rc", sysPPxP);
	setenv("PATH", env, 1);
	SetPPxPEnvs();
	if (eqp->runmode & EXEC_SECURE) SuperPrivilege(TRUE);
	chdir("/tmp");
	path = Malloc((slen > ulen ? slen: ulen) + strlen(eqp->argv[0])
		      + sizeof("/rc/"));
	strcpy(path, eqp->argv[0]);
	if (*path != '/') {
	    *path = '\0';
	    if (!(eqp->runmode & EXEC_SECURE) && usrPPxP) {
		sprintf(path, "%s/rc/%s", usrPPxP, eqp->argv[0]);
		if (access(path, X_OK)) *path = '\0';
	    }
	    if (!*path) {
		sprintf(path, "%s/rc/%s", sysPPxP, eqp->argv[0]);
		if (access(path, X_OK)) *path = '\0';
	    }
	}
	if (*path) execv(path, eqp->argv);
	LogError(eqp->argv[0]);
	exit(0);
    } else if (pid == -1) {
	LogError("ExecNext fork");
    } else {
	eqp->pid = pid;
	if (!(eqp->runmode & EXEC_BACK)) fgPid = pid;
    }
}

static void
ExecEnqueue(int argc, char *argv[], int runmode)
{
    struct exeq_s *eqp;
    int a;

    eqp = TALLOC(struct exeq_s);
    eqp->qhook = NULL;
    eqp->next = eqHead;
    eqHead = eqp;
    eqp->argv = (char **)Malloc((argc + 1) * sizeof(char *));
    eqp->argc = argc;
    for (a = 0; a < argc; a ++) eqp->argv[a] = Strdup(argv[a]);
    eqp->argv[a] = NULL;
    eqp->runmode = runmode;
    eqp->pid = 0;
    ExecNext();
}

static void
SigChld(int sig)
{
    int ps;
    pid_t pid;
    struct exeq_s *eqp, *eq1;

#ifdef	ONESHOT_SIGNAL
    signal(SIGCHLD, SigChld);
#endif
    pid = wait3(&ps, WNOHANG, 0);
    eq1 = NULL;
    eqp = eqHead;
    while (eqp) {
	if (eqp->pid == pid) {
	    int a;

	    if (eq1) eq1->next = eqp->next;
	    else eqHead = eqp->next;
	    for (a = 0; a < eqp->argc; a ++) Free(eqp->argv[a]);
	    if (eqp->argc) Free(eqp->argv);
	    if (eqp->qhook) eqp->qhook(ps);
	    Free(eqp);
	    if (pid == fgPid) fgPid = 0;
	    ExecNext();
	    return;
	}
	eq1 = eqp;
	eqp = eqp->next;
    }
}

/*
 * stored by RunBuiltInCommand:
 *   argv[0]=command name resolved from 'type'
 *   -> argv[0]=NULL means built-in command not found
 */

int
RunBuiltInCommand(u_int8_t type, int argc, char *argv[], int runmode)
{
    int i;
    struct command_s *csp;
    extern char *EnvToStr();

    if (type >= NUM_XCMD) return(-1);
    csp = &xcmdList[type];
    if (ISLOG(LOG_COMMAND)) {
	Logf(LOG_COMMAND, "<%s>", csp->name);
	for (i = 1; i < argc; i ++) Logf(LOG_COMMAND, "<%s>", argv[i]);
	Logf(LOG_COMMAND, "\n");
    }
    if ((csp->flags & CMD_SECURE) && !(runmode & EXEC_SECURE)) {
	errno = EPERM;
	LogError(csp->name);
	return(-1);
    }
    if (!argv[0]) argv[0] = Strdup(csp->name);
    if (runmode & EXEC_SECURE) SuperPrivilege(TRUE);
    i = csp->cmd(argc, argv);
    SuperPrivilege(FALSE);
    return(i);
}

int
RunCommand(int argc, char *argv[], int runmode)
{
    u_int8_t n;
    int i, len=strlen(argv[0]);
    extern char *ExtractEnvs();

    for (i = 1; i < argc; i ++) {
	if (strstr(argv[i], "$(")) {
	    char *ext;

	    ext = ExtractEnvs(argv[i], 0);
	    Free(argv[i]);
	    argv[i] = Strdup((ext && *ext) ? ext: "");
	}
    }
    for (n = 0; n < NUM_XCMD; n ++) {
	if (xcmdList[n].flags & CMD_FULLNAME) {
	    if (!strcasecmp(argv[0], xcmdList[n].name)) break;
	} else {
	    if (!strncasecmp(argv[0], xcmdList[n].name, len)) break;
	}
    }
    if (n < NUM_XCMD)
	return(RunBuiltInCommand(n, argc, argv, runmode));
    if (!(runmode & EXEC_NOEXEC)) {
	ExecEnqueue(argc, argv, runmode);
	return(0);
    }
    return(-1);
}

int
RunCommandLine(char *buf, int runmode)
{
    int argc=0, ret=-1;
    char *argv[128], *p;

    if ((p = strpbrk(buf, "\n\r")) != NULL) *p = '\0';
    p = buf + strlen(buf) - 1;
    while ((p > buf) && strchr(" \t&", *p)) {
	if (*p == '&') runmode |= EXEC_BACK;
	*p = '\0';
	p --;
    }
    while (*buf) {
	while (*buf && strchr("\t ", *buf)) buf ++;
	if (*buf) {
	    if ((p = strpbrk(buf, "\t ")) != NULL) *p = '\0';
	    argv[argc ++] = Strdup(buf);
	    if (!p) break;
	    buf = p + 1;
	}
    }
    if (argc > 0) {
/*
int a;
for(a=0;a<argc;a++)printf("%s ", argv[a]);
printf("\n");
*/
	argv[argc] = NULL;
	ret = RunCommand(argc, argv, runmode);
    }
    while (argc > 0) if (argv[-- argc]) Free(argv[argc]);
    return(ret);
}

void
CommandSetup()
{
    signal(SIGTERM, AllExit);
    signal(SIGPIPE, SIG_IGN);
    signal(SIGCHLD, SigChld);
}

void
ListupCommand(struct console_s *cp)
{
    struct xcio_s xc;
    u_char *p;
    unsigned int i;

    xc.xid = cp->xid;
    xc.type = XCIO_LISTUP;
    xc.buf[0] = XLABEL_COMMAND;
    for (i = 0; i < NUM_XCMD; i ++) {
	xc.buf[1] = (u_char) i;
	/* xc.buf[2] = help; */
	p = &xc.buf[3];
	p += SprintF(p, xcmdList[i].name);
	*p = '\0';
	xc.len = p - xc.buf + 1;
	if (i == NUM_XCMD - 1) xc.type |= XCIO_LAST;
	XcioWrite(cp->fd, &xc);
    }
}

pid_t
MakeProcess(void (*qhook)())
{
    pid_t pid;

    if ((pid = fork()) == 0) {
    } else if (pid > 0) {
	struct exeq_s *eqp;

	eqp = TALLOC(struct exeq_s);
	eqp->next = eqHead;
	eqHead = eqp;
	eqp->argv = NULL;
	eqp->argc = 0;
	eqp->qhook = qhook;
	eqp->runmode = EXEC_BACK;
	eqp->pid = pid;
    }
    return(pid);
}
