
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stddef.h>
#include <stdint.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/time.h>
#include <event.h>

#include <debug/memory.h>

#include "flow.h"
#include "list.h"

struct flow_data
{
   struct list_head time;	/* flows sorted by time (ascending)		*/
   struct list_head size;	/* flows sorted by size (descending)	*/
   struct list_head free;	/* free list							*/
   time_t active;			/* active timeout						*/
   time_t inactive;			/* inactive timeout						*/
};

static struct flow_data flow_data;

int flow_open (size_t entries,time_t active,time_t inactive)
{
   size_t i;

   memset (&flow_data,0L,sizeof (struct flow_data));
   INIT_LIST_HEAD (&flow_data.time);
   INIT_LIST_HEAD (&flow_data.size);
   INIT_LIST_HEAD (&flow_data.free);
   flow_data.active = active;
   flow_data.inactive = inactive;

   for (i = 0; i < entries; i++)
	 {
		struct flow *flow;

		if ((flow = mem_alloc (sizeof (struct flow))) == NULL)
		  {
			 flow_close ();
			 return (-1);
		  }

		list_add (&flow->time,&flow_data.free);
	 }

   return (0);
}

void flow_close (void)
{
   struct list_head *node,*tmp;
   struct flow *entry;

   list_for_each (node,tmp,&flow_data.time)
	 {
		entry = list_entry (node,struct flow,time);
		list_del (&entry->time);
		list_del (&entry->size);
		evtimer_del (&entry->active);
		evtimer_del (&entry->inactive);
		mem_free (entry);
	 }

   list_for_each (node,tmp,&flow_data.free)
	 {
		entry = list_entry (node,struct flow,time);
		list_del (&entry->time);
		mem_free (entry);
	 }
}

static void flow_timeout (int fd,short event,void *arg)
{
   flow_remove (arg);
}

void flow_insert (const struct flow *flow)
{
   struct flow *entry;
   struct timeval tv;

   if ((entry = flow_find (flow)) != NULL)
	 {
		entry->octets += flow->octets;
		entry->packets += flow->packets;
		entry->flags |= flow->flags;

		/* re-insert flow if this is the first fragment */
		if ((flow->sport || flow->dport) && (flow->flags & FLOW_FRAG))
		  {
			 entry->sport = flow->sport;
			 entry->dport = flow->dport;
			 entry->flags &= ~FLOW_FRAG;

			 if (flow_find (entry) != NULL)
			   {
				  struct flow frag = *entry;

				  flow_remove (entry);
				  flow_insert (&frag);

				  return;
			   }
		  }

		/* reset active timer */
		tv.tv_sec = flow_data.inactive, tv.tv_usec = 0;
		evtimer_del (&entry->inactive);
		evtimer_add (&entry->inactive,&tv);
	 }
   else
	 {
		if (list_empty (&flow_data.free))
		  {
			 entry = list_entry (flow_data.time.prev,struct flow,time);
			 list_del (&entry->size);
			 evtimer_del (&entry->active);
			 evtimer_del (&entry->inactive);
		  }
		else entry = list_entry (flow_data.free.prev,struct flow,time);

		list_del (&entry->time);
		*entry = *flow;
		list_add_tail (&entry->time,&flow_data.time);
		list_add (&entry->size,&flow_data.size);

		/* set active timer (called when flow grows too old) */
		tv.tv_sec = flow_data.active, tv.tv_usec = 0;
		evtimer_set (&entry->active,flow_timeout,entry);
		evtimer_add (&entry->active,&tv);

		/* set inactive timer (called when flow isn't updated) */
		tv.tv_sec = flow_data.inactive, tv.tv_usec = 0;
		evtimer_set (&entry->inactive,flow_timeout,entry);
		evtimer_add (&entry->inactive,&tv);
	 }
}

void flow_remove (struct flow *flow)
{
   evtimer_del (&flow->active);
   evtimer_del (&flow->inactive);
   list_del (&flow->time);
   list_del (&flow->size);
   list_add (&flow->time,&flow_data.free);
}

static __inline__ struct flow *flow_find_normal (const struct flow *flow)
{
   struct list_head *node,*tmp;

   list_for_each (node,tmp,&flow_data.time)
	 {
		struct flow *entry = list_entry (node,struct flow,time);

		if (entry != flow &&
			entry->proto == flow->proto &&
			entry->tos == flow->tos &&
			entry->saddr == flow->saddr &&
			entry->daddr == flow->daddr &&
			entry->sport == flow->sport &&
			entry->dport == flow->dport)
		  return (entry);
	 }

   return (NULL);
}

static __inline__ struct flow *flow_find_fragment (const struct flow *flow)
{
   struct list_head *node,*tmp;

   list_for_each (node,tmp,&flow_data.time)
	 {
		struct flow *entry = list_entry (node,struct flow,time);

		if (entry->proto == flow->proto &&
			entry->tos == flow->tos &&
			entry->saddr == flow->saddr &&
			entry->daddr == flow->daddr &&
			entry->id == flow->id)
		  return (entry);
	 }

   return (NULL);
}

struct flow *flow_find (const struct flow *flow)
{
   return (!(flow->flags & FLOW_FRAG) ?
		   flow_find_normal (flow) :
		   flow_find_fragment (flow));
}

static int flow_compare_size (struct list_head *a,struct list_head *b)
{
   struct flow *flow1 = list_entry (a,struct flow,size);
   struct flow *flow2 = list_entry (b,struct flow,size);

   return (flow1->bitrate < flow2->bitrate);
}

void flow_process (void (*process) (const struct flow *))
{
   struct list_head *node,*tmp;
   time_t now = time (NULL);

   list_for_each (node,tmp,&flow_data.size)
	 {
		struct flow *flow = list_entry (node,struct flow,size);

		flow->bitrate = (double) flow->octets / (double) (now - flow->timestamp);
	 }

   list_sort (&flow_data.size,flow_compare_size);

   list_for_each (node,tmp,&flow_data.size)
	 process (list_entry (node,struct flow,size));
}

