/*
 * Potamus: an audio player
 * Copyright (C) 2004, 2005, 2006, 2007 Adam Sampson <ats@offog.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <bio2jack.h>
#include <time.h>
#include <glib.h>
#include "buffer.h"
#include "format.h"
#include "output.h"
#include "output-jack.h"

// FIXME: rewrite this to use JACK directly rather than bio2jack. We've already
// got an internal buffer, so we might as well use it...
// FIXME: flush the output buffer on a seek.

typedef struct output_jack {
	int dev;
	sample_format fmt;
} output_jack;

static int oj_play(struct output *o, buffer *buf, sample_format *fmt) {
	output_jack *a = (output_jack *) o->data;

	while (buf->used > 0) {
		// Work around a bug in bio2jack 0.7: if you try to write more than
		// will fit in the ringbuffer, it won't let you write anything!
		// (The ringbuffer's 16k, although there doesn't appear to be
		// any way of finding that out other than looking at the code.)
		long amount = buf->used;
		if (amount > 512)
			amount = 512;

		long n = JACK_Write(a->dev, buf->data, amount);
		buffer_remove(buf, n);

		if (n == 0) {
			// The bio2jack buffer is full -- wait a bit before trying again.
			struct timespec ts = {.tv_sec = 0, .tv_nsec = 1000000};
			nanosleep(&ts, NULL);
		}
	}

	return 0;
}

static void oj_release(struct output *o) {
	// Doesn't need to do anything.
}

static void oj_close(struct output *o) {
	output_jack *a = (output_jack *) o->data;

	if (JACK_Close(a->dev) < 0)
		g_error("JACK_Close failed");

	free(a);
	free(o);
}

output *output_new_jack(sample_format **force_fmt) {
	output *o = output_alloc();

	output_jack *a = malloc(sizeof *a);
	if (a == NULL)
		g_error("out of memory");
	o->data = a;

	o->play = oj_play;
	o->release = oj_release;
	o->close = oj_close;

	static int jack_initialised = 0;
	if (!jack_initialised) {
		JACK_Init();

		char name[40];
		snprintf(name, sizeof name, "potamus_%d", getpid());
		JACK_SetClientName(name);

		jack_initialised = 1;
	}

	a->fmt.bits = 16; // FIXME should be 32
	a->fmt.channels = 2; // FIXME: should be configurable?
	a->fmt.byte_format = END_NATIVE;

	unsigned long rate = 48000;
	int rc;
	for (int i = 0; i < 2; i++) {
		// Kludge: we call this once to get the rate (if we haven't
		// guessed it correctly already), then again to open if it
		// didn't work the first time.
		rc = JACK_Open(&a->dev, a->fmt.bits, &rate, a->fmt.channels);
		if (rc == 0)
			break;
	}
	if (rc != 0)
		goto out;
	a->fmt.rate = rate;
	*force_fmt = &a->fmt;

	if (JACK_SetAllVolume(a->dev, 100) != 0)
		g_error("JACK_SetAllVolume failed");

	return o;

out:
	free(a);
	free(o);
	return NULL;
}
