/*      
ClientDB.cc
*/

/*
Use Berkely DB 1.85
*/

#include "ClientDB.h"
extern Language *L;
      
extern bool InverseDBSort;
 
/*
 1. Nombre
 2. Orden
 3. Tipo
 4. MaxLength
 5. DisplayLength (in Display Record)
 6. IndexLength (table percent, in Index Page, 0 for not display)
 7. Label
 8. Default
 9. Could be null
10. Optional 1
11. Optional 2
12. Optional 3
13. Optional 4
*/                
                
ClientDB::ClientDB (const char *adbfname, const char *adbdir, bool ReadOnly, bool aUseIndex)
  {  
  db = NULL;
  dbindex = NULL;
  UseIndex = aUseIndex;
  xstrncpy (dbdir, CMAXBUFFER, adbdir); addLastSlash (dbdir);
  xstrncpy (dbfname, CMAXBUFFER, dbdir); xstrncat (dbfname, CMAXBUFFER, adbfname);
  initStr (ErrorMsg);
  if (ReadOnly == false)
    {
    DefOpenMode = O_RDWR | O_CREAT; 
    }
  else
    {    
    DefOpenMode = O_RDONLY;
    }
  DefCreateMode = 0666;
  DBType = DB_HASH; 
  initStr (dbinfo.dbname);
  initStr (dbinfo.dbstruct);
  initStr (dbinfo.dbpermissions);
  initStr (dbinfo.dbversion);   
  dbinfo.dblastint = 0;
  dbinfo.dbreccount = 0;   
  dbinfo.dbmaxsize = 100000; //bytes
  dbinfo.MatField = new TMatField;
  ModeDB = fMODE_FULL;
  SLSearch.Clear();
  }  

ClientDB::~ClientDB ()
  {
  //
  }

bool ClientDB::DBopen (void)
  {
  initStr (ErrorMsg);
  db = dbopen (dbfname, DefOpenMode, DefCreateMode, DBType, NULL);
  //DEBUG (dbfname);
  if (db == NULL) {xstrncpy (ErrorMsg, CMAXBUFFER, strerror(errno)); return false;}      
  if (UseIndex == true)
    {
    xstrncpy (dbindexfname, CMAXBUFFER, dbfname); xstrncat (dbindexfname, CMAXBUFFER, "."); xstrncat (dbindexfname, CMAXBUFFER, DB_INDEX_EXTENSION);
    dbindex = dbopen (dbindexfname, DefOpenMode, DefCreateMode, DBType, NULL);
    if (dbindex == NULL) 
      {
      if (!generaIndexFromDB (dbfname, dbindexfname, ErrorMsg)) {return false;}
      dbindex = dbopen (dbindexfname, DefOpenMode, DefCreateMode, DBType, NULL);
      if (dbindex == NULL)
        {
        xstrncpy (ErrorMsg, CMAXBUFFER, strerror(errno)); return false;
        }
      }  
    }    
  if (obtainDBInfo() == false) {return false;}
  //displayDBStructure (NULL);
  //displayDBPermissions (NULL);  
  //displayDBInfo (NULL);
  return true;    
  }

bool ClientDB::DBclose (void)
  {
  int ret;
  delete dbinfo.MatField;
  initStr (ErrorMsg);
  Sync();      
  ret = db->close(db);
  db = NULL;
  if (UseIndex == true) 
    {
    ret = dbindex->close(dbindex);
    dbindex = NULL;
    }
  return true;  
  }

int ClientDB::getTotalNumRecords (void)
  {
  int rval, reccount = 0;
  DBT key, data;
  if (db == NULL) return -1;
  if (UseIndex == true) {if (dbindex == NULL) return -1;}
  rval = db->seq (db, &key, &data, R_FIRST);
  while (!rval)
    {
    ++reccount;
    rval = db->seq (db, &key, &data, R_NEXT); 
    }        
  return reccount;
  }

int ClientDB::getNumRecords (void)
  {
  if ((ModeDB == fMODE_SEARCH) && (SLSearch.Count() > 0)) return SLSearch.Count();
  return dbinfo.dbreccount;
  }

bool ClientDB::incRecordCount (const TUniqId uniqid)
  {
  int ires;
  TNumber anum;
  DBT dkey, ddata;
  
  dkey.data = (void *)DB_RECCOUNT; dkey.size = strlen (DB_RECCOUNT);
  xitoa (dbinfo.dbreccount + 1, anum);
  ddata.data = (void *)anum; ddata.size = strlen (anum); 
  ires = db->put (db, &dkey, &ddata, 0);
  if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "incRecordCount: %s", strerror(errno)); return false;}
  
  ++dbinfo.dbreccount;

  if (UseIndex == true)
    {
    xitoa (dbinfo.dbreccount, anum);
    dkey.data = (void *)anum; dkey.size = strlen (anum);
    ddata.data = (void *)uniqid;  ddata.size = strlen (uniqid);
    ires = dbindex->put (dbindex, &dkey, &ddata, 0);
    if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "incRecordCount: %s", strerror(errno)); return false;}  
    }
    
  return true;
  }

bool ClientDB::delRecordCount (void)
  {
  int ires;
  TNumber anum;
  DBT dkey, ddata;
  
  dkey.data = (void *)DB_RECCOUNT; dkey.size = strlen (DB_RECCOUNT);
  xitoa (dbinfo.dbreccount - 1, anum);
  ddata.data = (void *)anum; ddata.size = strlen (anum); 
  ires = db->put (db, &dkey, &ddata, 0);
  if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "incRecordCount: %s", strerror(errno)); return false;}
  
  --dbinfo.dbreccount;

  return true;
  }

bool ClientDB::generaUniqID (TUniqId uniqid)
  {
  int ires;
  TNumber anum;
  DBT dkey, ddata;
    
  memset (uniqid, int('0'), CMAXUNIQID);  
  ++(dbinfo.dblastint);
  xitoa (dbinfo.dblastint, anum);  
  for (int i=0; i<(signed int)strlen(anum); ++i)
    {
    uniqid[CMAXUNIQID - i - 1] = anum[i];
    }  
  uniqid[CMAXUNIQID] = '\0';    
  dkey.data = (void *)DB_LASTINT; dkey.size = strlen(DB_LASTINT);
  ddata.data = (void *)anum; ddata.size = strlen(anum);
  ires = db->put(db, &dkey, &ddata, 0); //equiv: DBM_REPLACE 
  if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "generaUniqID: %s", strerror(errno)); return false;}
  return true;
  }
  
//Return a legal uniqid, not empty necessary  
bool ClientDB::generaIDFromPos (int pos, TUniqId uniqid)
  {
  TNumber anum;
  int L;
    
  memset (uniqid, int('0'), CMAXUNIQID);  
  xitoa (pos, anum);  
  L = strlen(anum);
  for (int i = 0; i < (signed int)L; ++i)
    {
    uniqid[CMAXUNIQID + i - L] = anum[i];
    }  
  uniqid[CMAXUNIQID] = '\0';    
  return existsRecord (uniqid);
  }  
  
bool ClientDB::existsRecord (TUniqId uniqid)
  {
  DBT dkey, ddata;
  int ires;
  TBuffer abuf;
  
  if (db == NULL) return false;  

  xstrncpy (abuf, CMAXBUFFER, uniqid);
  xstrncat (abuf, CMAXBUFFER, "_");
  xstrncat (abuf, CMAXBUFFER, DB_FLAGRECORD_NAME);
    
  dkey.data = (void *)abuf; dkey.size = strlen(abuf);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) 
    {
    return false;
    } 
  else if (ires < 0) 
    {
    return false;
    }
  else
    {
    return true;
    }
  }
  
void ClientDB::displayDBInfo (XSocket *aXS)
  {
  if (aXS == NULL)
    {
    DEBUG ("Information for db file    =%s", dbfname); 
    if (UseIndex == true) {DEBUG ("          db index file    =%s", dbindexfname);}
    DEBUG ("-------------------------------------------------------");
    DEBUG ("              DBVersion    =%s", dbinfo.dbversion);
    DEBUG ("              DBName       =%s", dbinfo.dbname);  
    DEBUG ("              DBPermissions=%s", dbinfo.dbpermissions);  
    DEBUG ("              DBStructure  =%s", dbinfo.dbstruct);  
    DEBUG ("              DBLastInt    =%d", dbinfo.dblastint);    
    DEBUG ("              DBRecCount   =%d", dbinfo.dbreccount);     
    DEBUG ("              DBMaxSize    =%d", dbinfo.dbmaxsize);        
    DEBUG ("         TOTAL Record Count=%d", getTotalNumRecords());  
    DEBUG ("       USEFULL Record Count=%d", getNumRecords());
    DEBUG ("-------------------------------------------------------");
    }
  else
    {
    TBuffer abuf;
    xsnprintf (abuf, CMAXBUFFER, "Information for db file    =%s\n", dbfname); aXS->Write (abuf, strlen(abuf));
    if (UseIndex == true) {xsnprintf (abuf, CMAXBUFFER, "          db index file    =%s\n", dbindexfname); aXS->Write (abuf, strlen(abuf));}
    xsnprintf (abuf, CMAXBUFFER, "-------------------------------------------------------\n"); aXS->Write (abuf, strlen(abuf));
    xsnprintf (abuf, CMAXBUFFER, "              DBVersion    =%s\n", dbinfo.dbversion); aXS->Write (abuf, strlen(abuf));
    xsnprintf (abuf, CMAXBUFFER, "              DBName       =%s\n", dbinfo.dbname); aXS->Write (abuf, strlen(abuf)); 
    xsnprintf (abuf, CMAXBUFFER, "              DBPermissions=%s\n", dbinfo.dbpermissions); aXS->Write (abuf, strlen(abuf)); 
    xsnprintf (abuf, CMAXBUFFER, "              DBStructure  =%s\n", dbinfo.dbstruct); aXS->Write (abuf, strlen(abuf)); 
    xsnprintf (abuf, CMAXBUFFER, "              DBLastInt    =%d\n", dbinfo.dblastint);aXS->Write (abuf, strlen(abuf));    
    xsnprintf (abuf, CMAXBUFFER, "              DBRecCount   =%d\n", dbinfo.dbreccount); aXS->Write (abuf, strlen(abuf));    
    xsnprintf (abuf, CMAXBUFFER, "              DBMaxSize    =%d\n", dbinfo.dbmaxsize); aXS->Write (abuf, strlen(abuf));       
    xsnprintf (abuf, CMAXBUFFER, "         TOTAL Record Count=%d\n", getTotalNumRecords()); aXS->Write (abuf, strlen(abuf)); 
    xsnprintf (abuf, CMAXBUFFER, "       USEFULL Record Count=%d\n", getNumRecords()); aXS->Write (abuf, strlen(abuf));
    xsnprintf (abuf, CMAXBUFFER, "-------------------------------------------------------\n"); aXS->Write (abuf, strlen(abuf));    
    }
  }

const char *ClientDB::getDBName (void)
  {
  return dbinfo.dbname;
  }

int ClientDB::getDBSize (void)
  {
  return FileSize (dbfname);
  }  

bool ClientDB::isDBSearched (void)
  {
  if (ModeDB == fMODE_SEARCH) return true;
  else return false;
  }    

int ClientDB::getFoundRecords (void)
  {
  return SLSearch.Count();
  }
 
bool ClientDB::isRecordReserved (TUniqId uniqid)
  {
  if ((strcmp (uniqid, DB_VERSION) == 0) || 
      (strcmp (uniqid, DB_NAME) == 0) || 
      (strcmp (uniqid, DB_STRUCTURE) == 0) ||
      (strcmp (uniqid, DB_LASTINT) == 0) || 
      (strcmp (uniqid, DB_PERMISSIONS) == 0) ||
      (strcmp (uniqid, DB_MAXSIZE) == 0) ||
      (strcmp (uniqid, DB_RECCOUNT) == 0)) return true;
  return false;    
  }

bool ClientDB::obtainDBInfo (void)
  {
  TNumber anum;
  DBT dkey, ddata;
  int ires, L; 
  
  dkey.data = (void *)DB_VERSION; dkey.size = strlen(DB_VERSION);                     
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXBUFFER, ddata.size); memcpy (dbinfo.dbversion, ddata.data, L); dbinfo.dbversion[L + 1] = '\0';
         
  dkey.data = (void *)DB_NAME; dkey.size = strlen(DB_NAME);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXBUFFER, ddata.size); memcpy (dbinfo.dbname, ddata.data, L); dbinfo.dbname[L] = '\0';         
    
  dkey.data = (void *)DB_LASTINT; dkey.size = strlen(DB_LASTINT);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXNUMBER, ddata.size); memcpy (anum, ddata.data, L); anum[L] = '\0';         
  dbinfo.dblastint = xatoidef (anum, 0);   
  
  dkey.data = (void *)DB_RECCOUNT; dkey.size = strlen(DB_RECCOUNT);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXNUMBER, ddata.size); memcpy (anum, ddata.data, L); anum[L] = '\0';         
  dbinfo.dbreccount = xatoidef (anum, 0);      

  dkey.data = (void *)DB_PERMISSIONS; dkey.size = strlen(DB_PERMISSIONS);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXBUFFER, ddata.size); memcpy (dbinfo.dbpermissions, ddata.data, L); dbinfo.dbpermissions[L + 1] = '\0';
  if (!parseDBPermissions ()) {return false;}

  dkey.data = (void *)DB_STRUCTURE; dkey.size = strlen(DB_STRUCTURE);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXBUFFER, ddata.size); memcpy (dbinfo.dbstruct, ddata.data, L); dbinfo.dbstruct[L + 1] = '\0';
  if (!parseDBStructure ()) {return false;}
  
  dkey.data = (void *)DB_MAXSIZE; dkey.size = strlen(DB_MAXSIZE);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "obtainDBInfo: %s", strerror(errno)); return false;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "Error: %s", strerror(errno)); return false;}
  L = min (CMAXNUMBER, ddata.size); memcpy (anum, ddata.data, L); anum[L] = '\0';         
  dbinfo.dbmaxsize = xatoidef (anum, 0);   
    
  return true;  
  }

const char *ClientDB::getErrorMsg (void)
  {
  return ErrorMsg;
  }

bool ClientDB::parseDBPermissions (void)
  {
  return true;
  }

bool ClientDB::Sync (void)
  {
  int iret1 = 0, iret2 = 0;
  iret1 = db->sync (db, 0);
  if (UseIndex == true) {iret2 = dbindex->sync (dbindex, 0);}
  if ((iret1 != 0) || (iret2 != 0)) return false;
  return true;
  }

void ClientDB::displayDBPermissions (XSocket *aXS)
  {
  if (aXS == NULL)
    {            
    DEBUG ("dbpermissions=%s", dbinfo.dbpermissions);
    }
  else
    {
    TBuffer abuf;
    xsnprintf (abuf, CMAXBUFFER, "dbpermissions='%s'\n", dbinfo.dbpermissions); aXS->Write (abuf, strlen(abuf));      
    }
  }
  
void ClientDB::displayDBStructure (XSocket *aXS)
  {
  TBuffer abuf;
  TDBField DBField;
  for (int i=0; i < dbinfo.MatField->Count(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (aXS == NULL)
      {
      DEBUG ("name='%s'",             DBField.fname);
      DEBUG ("sequence='%d'",         DBField.fsequence);
      DEBUG ("type='%s'",             getStringFromType(DBField.ftype));
      DEBUG ("maxlength='%d'",        DBField.fmaxlength);
      DEBUG ("displaylength='%d'",    DBField.fdisplaylength);
      DEBUG ("findexlength='%d'",     DBField.findexlength);
      DEBUG ("label='%s'",            DBField.flabel);
      DEBUG ("default='%s'",          DBField.fdefault);
      DEBUG ("couldbenull='%d'",      DBField.fcouldbenull);    
      DEBUG ("fopt1='%s'",            DBField.fopt1);    
      DEBUG ("fopt2='%s'",            DBField.fopt2);
      DEBUG ("fopt3='%s'",            DBField.fopt3);    
      DEBUG ("fopt4='%s'",            DBField.fopt4);    
      }
    else
      {      
      xsnprintf (abuf, CMAXBUFFER, "name='%s'\n",         DBField.fname); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "sequence='%d'\n",     DBField.fsequence);  aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "type='%s'\n",         getStringFromType(DBField.ftype)); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "maxlength='%d'\n",    DBField.fmaxlength);  aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "displaylength='%d'\n",DBField.fdisplaylength); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "findexlength='%d'\n", DBField.findexlength);  aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "label='%s'\n",        DBField.flabel); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "default='%s'\n",      DBField.fdefault); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "couldbenull='%d'\n",  DBField.fcouldbenull); aXS->Write (abuf, strlen(abuf));     
      xsnprintf (abuf, CMAXBUFFER, "fopt1='%s'\n",        DBField.fopt1); aXS->Write (abuf, strlen(abuf));     
      xsnprintf (abuf, CMAXBUFFER, "fopt2='%s'\n",        DBField.fopt2); aXS->Write (abuf, strlen(abuf));  
      xsnprintf (abuf, CMAXBUFFER, "fopt3='%s'\n",        DBField.fopt3); aXS->Write (abuf, strlen(abuf));     
      xsnprintf (abuf, CMAXBUFFER, "fopt4='%s'\n",        DBField.fopt4); aXS->Write (abuf, strlen(abuf));     
      }
    }
  }

char *ClientDB::getDefaultValueField (TType atype, TBuffer adefault)
  {
  if ((atype = fDATETIME) && (strcasecmp ("NOW", adefault) == 0))
    {
    TBuffer abuf;
    time_t now; 
    struct tm *atm;

    now = time (NULL);
    atm = localtime((time_t *)&now);
    xsnprintf (abuf, CMAXBUFFER, "%.2d/%.2d/%.4d %.2d:%.2d:%.2d",
               atm->tm_mday, atm->tm_mon + 1, atm->tm_year + 1900, 
               atm->tm_hour, atm->tm_min, atm->tm_sec);
    return xstrdup (abuf);
    }
  return adefault;  
  }

int ClientDB::getNumFields (void)
  {
  return dbinfo.MatField->Count();
  }

//void ClientDB::getMatField (const TMatField *amatfield)
const TMatField *ClientDB::getMatField (void)
  {
  return dbinfo.MatField;
  }

bool ClientDB::isFieldName (TFieldName fieldname)
  {
  TDBField DBField;
  for (int i=0; i < dbinfo.MatField->Count(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (strcmp (DBField.fname, fieldname) == 0) return true;
    }
  return false;
  }

const char *ClientDB::getStringFromType (TType atype)
  {
  switch (atype)
    {
    case fTEXT:     return "TEXT"; break;
    case fVARCHAR:  return "VARCHAR"; break;
    case fDATETIME: return "DATETIME"; break;   
    case fURL:      return "URL"; break;   
    case fINT:      return "INT"; break;   
    case fREAL:     return "REAL"; break; 
    case fRESERVED: return "RESERVED"; break;              
    default:        return "TEXT"; break;
    }
  }    
  
TType ClientDB::getTypeFromString (const char *stringtype)
  {               
  enum TType thetype = fTEXT;           
  TBuffer uplinea;
  xstrncpy (uplinea, CMAXBUFFER, stringtype); xucase (uplinea);      
       if (strcmp (uplinea, "TEXT") == 0)     return fTEXT;
  else if (strcmp (uplinea, "VARCHAR") == 0)  return fVARCHAR;         
  else if (strcmp (uplinea, "DATETIME") == 0) return fDATETIME;
  else if (strcmp (uplinea, "URL") == 0)      return fURL;
  else if (strcmp (uplinea, "INT") == 0)      return fINT;            
  else if (strcmp (uplinea, "REAL") == 0)     return fREAL;   
  else if (strcmp (uplinea, "RESERVED") == 0) return fRESERVED;                
  return thetype;
  }        

bool ClientDB::isMainField (TFieldName fieldname)
  {
  TDBField DBField;
  for (int i=0; i < dbinfo.MatField->Count(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (strcmp (fieldname, DBField.fname) == 0) 
      {
      if (DBField.fsequence == 1) return true;
      }
    }  
  return false;  
  }  

bool ClientDB::parseDBStructure (void)
  {
  const char FIELD_SEP = '!';  
  TFieldName token;
  int m, ll, numfield = -1, tipotoken = -1, L;
  char c, letra[2];
  TDBField DBField;
  
  initStr (ErrorMsg);  
  letra[1] = '\0';
  dbinfo.MatField->Clear();
  initStr (token);
  tipotoken = 0;
  L = strlen(dbinfo.dbstruct);
  for (int i=0; i<(signed int)L; ++i)
    {                     
    c = dbinfo.dbstruct[i];
    letra[0] = c;
    if (c == '(')
      {
      tipotoken = -1;
      initStr (token); initStr (DBField.fname); DBField.fsequence = -1; initStr (DBField.fdefault); 
      initStr (DBField.flabel);
      DBField.ftype = fTEXT;
      DBField.fmaxlength = -1; 
      DBField.fdisplaylength = -1;
      DBField.findexlength = 0; 
      DBField.fcouldbenull = false;
      }
    else if (c == ')')
      {
      dbinfo.MatField->Add (DBField);
      ++numfield;
      }
    else if (c == FIELD_SEP)
      {  
      ++tipotoken;
      if (token == NULL) ll = 0; else ll = strlen(token);
      m = imin (CMAXFIELDNAME, ll) + 1;      
      switch (tipotoken)
        {
        case  0: if (m == 0) strcpy(DBField.fname, "defname"); else xstrncpy (DBField.fname, m, token); break;
        case  1: DBField.fsequence = xatoidef (token, 0); break;
        case  2: DBField.ftype = getTypeFromString (token); break; 
        case  3: DBField.fmaxlength = xatoidef (token, 100); break;                          
        case  4: DBField.fdisplaylength = xatoidef (token, 20); break; 
        case  5: DBField.findexlength = xatoidef (token, 0); break; 
        case  6: if (m == 0) strcpy(DBField.flabel, DBField.fname); else xstrncpy (DBField.flabel, m, token); break;
        case  7: if (m == 0) initStr(DBField.fdefault); else xstrncpy (DBField.fdefault, m, token); break;                                    
        case  8: DBField.fcouldbenull = ((token == NULL) ? false : xatobdef (token, false)); break;           
        case  9: if (m == 0) initStr(DBField.fopt1); else xstrncpy (DBField.fopt1, m, token); break;                     
        case 10: if (m == 0) initStr(DBField.fopt2); else xstrncpy (DBField.fopt2, m, token); break;                     
        case 11: if (m == 0) initStr(DBField.fopt3); else xstrncpy (DBField.fopt3, m, token); break;                     
        case 12: if (m == 0) initStr(DBField.fopt4); else xstrncpy (DBField.fopt4, m, token); break;                                    
        }
      initStr (token);  
      }
    else
      {
      strcat (token, letra);
      }  
    }  
                     
  //RESERVED FIELDS TO ALL RECORDS
  //    FLAG FIELD
  DBField.ftype          = fRESERVED;
  strcpy (DBField.fname, DB_FLAGRECORD_NAME);
  initStr(DBField.flabel);
  strcpy (DBField.fdefault, DB_FLAGRECORD_DEFAULT);
  DBField.fmaxlength     = 10;
  initStr(DBField.fopt1);
  initStr(DBField.fopt2);
  initStr(DBField.fopt3);
  initStr(DBField.fopt4);
  DBField.fsequence      = -1; 
  DBField.fdisplaylength = -1;
  DBField.findexlength   = -1;
  DBField.fcouldbenull   = 0;  
  dbinfo.MatField->Add (DBField);
  
  return true;
  }

char *ClientDB::getUniqIDFromIndexPos (int pos, TUniqId uniqid)
  {
  TNumber anum;
  DBT dkey, ddata;
  int ires, L;
 
  if (UseIndex == false) return NULL;
  
  if (dbindex == NULL) return NULL;  
  //if ((pos <= 0) || (pos > getNumRecords())) {xsnprintf (ErrorMsg, CMAXBUFFER, "getUniqIDFromIndexPos: Error: Invalid pos"); return NULL;}

  if ((ModeDB == fMODE_SEARCH) && (SLSearch.Count() > 0)) 
    {
    xstrncpy (uniqid, CMAXUNIQID + 1, SLSearch.getString(pos).cstr());
    }
  else
    {
    xitoa (pos, anum);
    dkey.data = (void *)anum; dkey.size = strlen(anum);
    ires = dbindex->get (dbindex, &dkey, &ddata, 0);
    if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "getUniqIDFromIndexPos: %s", strerror(errno)); return NULL;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "getUniqIDFromIndexPos: %s", strerror(errno)); return NULL;}
    L = ddata.size;
    memcpy (uniqid, ddata.data, ddata.size);
    uniqid[L] = '\0';  
    }    
  return uniqid;
  }

char *ClientDB::getField_Text_ByID (TUniqId uniqid, TFieldName fieldname, char *data, int *len)
  {
  TBuffer id;
  DBT dkey, ddata;
  int ires; 

  *len = -1; if (db == NULL) return NULL;  if (UseIndex == true) if (dbindex == NULL) return NULL;  
  if (!isFieldName(fieldname)) {xsnprintf (ErrorMsg, CMAXBUFFER, "getField_Text_ByID: Error: No field %s", fieldname); return NULL;}

  xsnprintf (id, CMAXBUFFER, "%s_%s", uniqid, fieldname);  
  if (isRecordReserved (id) == true) {xsnprintf (ErrorMsg, CMAXBUFFER, "getField_Text_ByID: Error: Record is reserved"); return NULL;}
  dkey.data = (void *)id; dkey.size = strlen(id);
  ires = db->get (db, &dkey, &ddata, 0);
  if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "getField_Text_ByID: %s", strerror(errno)); return NULL;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "getField_Text_ByID: %s", strerror(errno)); return NULL;}
  *len = ddata.size;
  data = (char *)malloc ((sizeof(char) * (*len)) + 1);
  if (data == NULL) {xsnprintf (ErrorMsg, CMAXBUFFER, "getField_Text_ByID: %s", strerror(errno)); return NULL;}
  memcpy (data, ddata.data, ddata.size);
  data[*len] = '\0';  
  return data;  
  }

char *ClientDB::displayField_Text_ByID (TUniqId uniqid, TFieldName fieldname, TType fieldtype, char *data, int *len)
  {
  TBuffer id, abuf;
  DBT dkey, ddata;
  int ires, lendata; 
  char *dataname;

  *len = -1; if (db == NULL) return NULL;  if (UseIndex == true) if (dbindex == NULL) return NULL;  
  if (!isFieldName(fieldname)) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: Error: No field %s", fieldname); return NULL;}

  if (fieldtype == fURL)
    {
    xsnprintf (id, CMAXBUFFER, "%s_%s", uniqid, fieldname);  
    if (isRecordReserved (id) == true) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: Error: Record is reserved"); return NULL;}
    dkey.data = (void *)id; dkey.size = strlen(id);
    ires = db->get (db, &dkey, &ddata, 0);
    if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;}
    *len = ddata.size;
    data = (char *)malloc ((sizeof(char) * (*len)) + 1);
    if (data == NULL) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;}
    memcpy (data, ddata.data, ddata.size);
    data[*len] = '\0';      
                         
    xsnprintf (id, CMAXBUFFER, "%s_name", uniqid, fieldname);  
    if (isRecordReserved (id) == true) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: Error: Record is reserved"); return NULL;}
    dkey.data = (void *)id; dkey.size = strlen(id);
    ires = db->get (db, &dkey, &ddata, 0);
    dataname = NULL;
    if (ires == 0)
      {
      lendata = ddata.size;
      dataname = (char *)malloc ((sizeof(char) * (lendata)) + 1);
      if (dataname == NULL) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;}
      memcpy (dataname, ddata.data, ddata.size);
      dataname[lendata] = '\0';
      }    
    xsnprintf (abuf, CMAXBUFFER, "<A HREF=\"%s\">%s</A>", data, dataname);
    data = xstrdup (abuf);    
    }
  else
    {
    xsnprintf (id, CMAXBUFFER, "%s_%s", uniqid, fieldname);  
    if (isRecordReserved (id) == true) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: Error: Record is reserved"); return NULL;}
    dkey.data = (void *)id; dkey.size = strlen(id);
    ires = db->get (db, &dkey, &ddata, 0);
    if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;}
    *len = ddata.size;
    data = (char *)malloc ((sizeof(char) * (*len)) + 1);
    if (data == NULL) {xsnprintf (ErrorMsg, CMAXBUFFER, "displayField_Text_ByID: %s", strerror(errno)); return NULL;}
    memcpy (data, ddata.data, ddata.size);
    data[*len] = '\0';  
    }
  return data;  
  }

bool ClientDB::putField_Text_ByID (const TUniqId uniqid, TFieldName fieldname, char *data, int len)
  {
  TBuffer id;
  DBT dkey, ddata;
  int ires; 
  if (db == NULL) return false;  if (UseIndex == true) if (dbindex == NULL) return false;  
  if (!isFieldName(fieldname)) {xsnprintf (ErrorMsg, CMAXBUFFER, "putField_Text_ByID: Error: No field %s", fieldname); return false;}
  xsnprintf (id, CMAXBUFFER, "%s_%s", uniqid, fieldname);  
  if (isRecordReserved (id) == true) {xsnprintf (ErrorMsg, CMAXBUFFER, "putField_Text_ByID: Error: Record is reserved"); return false;}
  dkey.data = (void *)id; dkey.size = strlen(id);
  ddata.data = (void *)data; ddata.size = len;
  ires = db->put (db, &dkey, &ddata, 0);
  if (ires != 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "putField_Text_ByID: %s", strerror(errno)); return false;}
  return true;  
  }
        
int ClientDB::copyRecordsByID (StringList *SLUniqIDItems)
  {
  DBT dkey, ddata, ddatacopied;  
  TUniqId uniqid;
  TDBField DBField;
  TBuffer af, afnew;
  int ret = 0, ires;
  
  for (int i = 0; i < SLUniqIDItems->Count(); ++i)
    {
    if (!generaUniqID (uniqid)) {xsnprintf (ErrorMsg, CMAXBUFFER, "copyRecordsByID: Error: Generating UniqId"); return -1;}
    for (int j = 0; j < getNumFields(); ++j)
      {
      DBField = dbinfo.MatField->elementAt(j);
      xsnprintf (af, CMAXBUFFER, "%s_%s", SLUniqIDItems->getString(i).cstr(), DBField.fname);
      xsnprintf (afnew, CMAXBUFFER, "%s_%s", uniqid, DBField.fname);      
      dkey.data  = (void *)af; dkey.size = strlen(af);
      ires = db->get (db, &dkey, &ddata, 0);
      //if (ires == 1) {xsnprintf (ErrorMsg, CMAXBUFFER, "copyRecordsByID: %s", strerror(errno)); return -1;} else if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "copyRecordsByID: %s", strerror(errno)); return -1;}
      if (ires == 0)
        {
        dkey.data  = (void *)afnew; dkey.size = strlen(afnew);      
        ddatacopied = ddata;                      
        ires = db->put (db, &dkey, &ddatacopied, 0);
        if (ires != 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "copyRecordsByID: %s", strerror(errno)); return -1;}
        }
      }
    if (!incRecordCount (uniqid)) {return -1;}      
    changeFlagById (uniqid, POS_FLAG_COPIED, true);              
    ++ret;
    }
  return ret;  
  }
    
int ClientDB::deleteIDsFromIndex (StringList *SLIDItems)
  {
  int rval;
  DBT dkey, ddata;
  TBuffer abuf, abuf2;
  StringList SL;
  
  if (UseIndex == false) return 0;
  
  SL.Clear();
  rval = dbindex->seq (dbindex, &dkey, &ddata, R_FIRST);
  while (!rval)
    {
    memcpy (abuf, ddata.data, ddata.size); abuf [ddata.size] = '\0';            
    if (SLIDItems->indexOf (abuf) >= 0) 
      {
      memcpy (abuf2, dkey.data, dkey.size); abuf2 [dkey.size] = '\0';
      SL.Add (abuf2);
      }     
    rval = dbindex->seq (dbindex, &dkey, &ddata, R_NEXT); 
    }                      
  for (int i = 0; i < SL.Count(); ++i)
    {
    xstrncpy (abuf, CMAXBUFFER, SL.getString(i).cstr());
    dkey.data  = (void *)abuf; dkey.size = strlen(abuf);    
    rval = dbindex->del (dbindex, &dkey, 0);
    if (rval != 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "deleteIDsFromIndex: %s", strerror(errno)); return -1;}
    }
  return SL.Count();  
  }

bool ClientDB::pasteItems (StringList *SLPapeleraCopy, StringList *SLPapeleraCut)
  {
  if ((SLPapeleraCopy != NULL) && (SLPapeleraCopy->Count() > 0))
    {
    if (copyRecordsByID (SLPapeleraCopy) <= -1) return false;
    return true;
    }
  else if ((SLPapeleraCut != NULL) && (SLPapeleraCut->Count() > 0))
    {                         
    if (copyRecordsByID (SLPapeleraCut) <= -1) return false;
    return true;
    }
  else
    {
    xsnprintf (ErrorMsg, CMAXBUFFER, "%s", L->get(ERR_CLIPBOARD_EMPTY));
    return false;
    }    
  }

bool ClientDB::openIndex (void)
  {
  if (UseIndex == false) return false;  
  dbindex = dbopen (dbindexfname, DefOpenMode, DefCreateMode, DBType, NULL);
  if (dbindex == NULL) {xstrncpy (ErrorMsg, CMAXBUFFER, strerror(errno)); return false;}    
  return true;
  }

bool ClientDB::closeIndex (void)
  {
  if (UseIndex == false) return false;
  dbindex->close(dbindex);
  dbindex = NULL;
  return true;
  }

int CDB_CompareEntriesByKeyPos (const void *e1, const void *e2)
  {
  int res;
  res = PDBNodo (e1)->key - PDBNodo(e2)->key;
  return res;  
  }  

void CDB_FlipEntries (const void *e1, const void *e2)
  {
  TDBNodo temp;
  temp.key    = PDBNodo (e1)->key;
  temp.uniqid = PDBNodo (e1)->uniqid;
  temp.value  = PDBNodo (e1)->value;
  PDBNodo (e1)->key    = PDBNodo (e2)->key;    PDBNodo (e2)->key   = temp.key;
  PDBNodo (e1)->uniqid = PDBNodo (e2)->uniqid; PDBNodo (e2)->uniqid = temp.uniqid;    
  PDBNodo (e1)->value  = PDBNodo (e2)->value;  PDBNodo (e2)->value = temp.value;  
  }

bool ClientDB::compactIndex (void)
  {
  TNumber anum;
  TBuffer abuf1, abuf2;
  int ires, rval, recpos = 0;
  DBT dkey, ddata;
  TDBNodo dbnodo;
  
  if (UseIndex == false) return false;
    
  if (db == NULL) return false; if (dbindex == NULL) return false;  

  dbnodolist.Clear();
  rval = dbindex->seq (dbindex, &dkey, &ddata, R_FIRST);
  while (!rval)
    {
    ++recpos;
    memcpy (abuf1, dkey.data, dkey.size); abuf1 [dkey.size] = '\0';
    memcpy (abuf2, ddata.data, ddata.size); abuf2 [ddata.size] = '\0';
    dbnodo.key = xatoidef (abuf1, 0);
    dbnodo.uniqid = abuf2;
    initStr (dbnodo.value);
    dbnodolist.Add (dbnodo);
    rval = dbindex->seq (dbindex, &dkey, &ddata, R_NEXT); 
    }
  closeIndex ();
  
  dbnodolist.SORT (CDB_CompareEntriesByKeyPos, CDB_FlipEntries);  
  
  if (!DeleteFile (dbindexfname)) {xstrncpy (ErrorMsg, CMAXBUFFER, "Can not delete the index file for compact."); return false;}
  if (!openIndex()) {xstrncpy (ErrorMsg, CMAXBUFFER, "Can not open the index file."); return false;}
  
  for (int i = 0; i < dbnodolist.Count(); ++i)
    {    
    xstrncpy (abuf1, CMAXBUFFER, xitoa (i + 1, anum));
    xstrncpy (abuf2, CMAXBUFFER, dbnodolist.elementAt(i).uniqid);    
    dkey.data  = (void *)abuf1; dkey.size = strlen (abuf1);
    ddata.data = (void *)abuf2; ddata.size = strlen (abuf2); 
    ires = dbindex->put (dbindex, &dkey, &ddata, 0);
    if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "compactIndex: %s", strerror(errno)); return false;}
    
    }  
  return true;  
  }

/*
FALTA chequear algunos tipos
*/
bool ClientDB::checkValidRecord (const char *envfile)
  {
  int L;
  TDBField DBField;    
  char *pc = NULL;
  for (int i = 0; i < getNumFields(); ++i)
    {            
    pc = NULL;
    DBField = dbinfo.MatField->elementAt(i);
    (const char *)pc = cgienv_GetComplexVar (envfile, DBField.fname);
    if ((DBField.fcouldbenull == false) && (IsEmpty(pc) == true) && (DBField.ftype != fRESERVED)) 
      {
      xsnprintf (ErrorMsg, CMAXBUFFER, "checkValidRecord: Field '%s' with label '%s' can not be null.", DBField.fname, DBField.flabel); free (pc); return false;
      }    
    if ((DBField.ftype == fRESERVED) && (IsEmpty(pc) == false)) 
      {
      xsnprintf (ErrorMsg, CMAXBUFFER, "checkValidRecord: Field '%s' is RESERVED.", DBField.fname); free (pc); return false;
      }          
    switch (DBField.ftype)
      {
      case fTEXT: {
                  L = strlen (pc);
                  if (L > DBField.fmaxlength) {xsnprintf (ErrorMsg, CMAXBUFFER, "checkValidRecord: String too long."); free (pc); return false;}  
                  break;
                  }
      case fVARCHAR: 
                  {
                  L = strlen (pc);
                  if (L > DBField.fmaxlength) {xsnprintf (ErrorMsg, CMAXBUFFER, "checkValidRecord: String too long."); free (pc); return false;}  
                  break;      
                  }
      case fDATETIME: { 
                  break;      
                  }
      case fURL:  {
                  break;
                  }
      case fINT:  {
                  break;
                  }
      case fREAL: {
                  break;
                  }
      case fRESERVED: 
                 {
                 //Never must go here.
                 break;
                 } 
      }  
    free(pc);
    } 
  return true;     
  } 

bool ClientDB::SaveRecordFromCGI (const char *envfile, char *recordid)    
  {
  TDBField DBField;  
  TBuffer abuf;
  TUniqId uniqid;
  char *pc;

  if (!checkValidRecord (envfile)) {return false;};
  if (strcmp (recordid, "-1") == 0) //NEW BLANK RECORD
    {   
    generaUniqID (uniqid);
    for (int i = 0; i < getNumFields(); ++i)
      {
      DBField = dbinfo.MatField->elementAt(i);
      if (DBField.ftype == fTEXT)
        {
        (const char *)pc = cgienv_GetComplexVar (envfile, DBField.fname);
        if (!putField_Text_ByID (uniqid, DBField.fname, pc, strlen(pc))) return false;
        }
      else if (DBField.ftype == fRESERVED)
        {  
        strcpy (abuf, DBField.fdefault);
        if (!putField_Text_ByID (uniqid, DBField.fname, abuf, strlen(abuf))) return false;
        }        
      else
        {  
        cgienv_GetSimpleVarDef (envfile, DBField.fname, abuf, "");
        if (!putField_Text_ByID (uniqid, DBField.fname, abuf, strlen(abuf))) return false;
        }
      }        
    incRecordCount (uniqid);
    strcpy (recordid, uniqid);
    return true;
    }
  else                              //UPDATING EXISTING RECORD
    {  
    for (int i = 0; i < getNumFields(); ++i)
      {
      DBField = dbinfo.MatField->elementAt(i);
      if (DBField.ftype == fTEXT)
        {
        (const char *)pc = cgienv_GetComplexVar (envfile, DBField.fname);
        if (!putField_Text_ByID (recordid, DBField.fname, pc, strlen(pc))) return false;
        }
      else if (DBField.ftype == fRESERVED)
        {  
        /*NOTHING*/
        }        
      else
        {  
        cgienv_GetSimpleVarDef (envfile, DBField.fname, abuf, "");
        if (!putField_Text_ByID (recordid, DBField.fname, abuf, strlen(abuf))) return false;
        }
      }    
    }    
  return true; 
  }  

bool ClientDB::SaveRecordFixedIdFromCGI (const char *envfile, const char *recordid)    
  {
  TDBField DBField;  
  TBuffer abuf;
  char *pc;

  if (!checkValidRecord (envfile)) {return false;};
  for (int i = 0; i < getNumFields(); ++i)
    {
    //DEBUG ("DBField.fname=--%s--", DBField.fname);
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype == fTEXT)
      {
      (const char *)pc = cgienv_GetComplexVar (envfile, DBField.fname);
      if (!putField_Text_ByID (recordid, DBField.fname, pc, strlen(pc))) return false;
      }
    else if (DBField.ftype == fRESERVED)
      {  
      strcpy (abuf, DBField.fdefault);
      if (!putField_Text_ByID (recordid, DBField.fname, abuf, strlen(abuf))) return false;                
      }        
    else
      {  
      cgienv_GetSimpleVarDef (envfile, DBField.fname, abuf, "");
      if (!putField_Text_ByID (recordid, DBField.fname, abuf, strlen(abuf))) return false;
      }
    }
  incRecordCount (recordid);  
  Sync();  
  return true; 
  } 

bool ClientDB::generaHTMLFormEditNewRecord (StringList *SLHTML)
  {
  TDBField DBField;  
  TBuffer abuf;
  
  SLHTML->Clear();
  xsnprintf (abuf, CMAXBUFFER, "<CENTER>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "<TABLE width=\"100%%\" BORDER=\"0\">\n"); SLHTML->Add (abuf);    
  for (int i=0; i < dbinfo.MatField->Count(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype != fRESERVED)
      {
      xsnprintf (abuf, CMAXBUFFER, "<TR valign=\"top\">\n"); SLHTML->Add (abuf);
      xsnprintf (abuf, CMAXBUFFER, "<TD align=\"right\"><B>%s:</B></TD>\n", DBField.flabel); SLHTML->Add (abuf);
      switch (DBField.ftype)
        {            
        case fTEXT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><TEXTAREA NAME=\"%s\" ROWS=\"%d\" COLS=\"%d\" WRAP=\"HARD\">%s</TEXTAREA></TD>\n", 
                     DBField.fname, 10, 55, DBField.fdefault);
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;
          }
        case fVARCHAR:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, getDefaultValueField (DBField.ftype, DBField.fdefault));
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;                                                                                                             
          }
        case fDATETIME:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, getDefaultValueField (DBField.ftype, DBField.fdefault));
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;
          }
        case fURL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, getDefaultValueField (DBField.ftype, DBField.fdefault));
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;
          }
        case fINT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, getDefaultValueField (DBField.ftype, DBField.fdefault));
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;
          }
        case fREAL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, getDefaultValueField (DBField.ftype, DBField.fdefault));
          SLHTML->Add (abuf);
          SLHTML->Add ("</TR>\n");
          break;
          }
        case fRESERVED: 
          {
          //Never must go here
          break;
          }
        }   
      }
    }  
  xsnprintf (abuf, CMAXBUFFER, "</TR>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</TABLE>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</CENTER>\n"); SLHTML->Add (abuf);
  return true;
  }

bool ClientDB::generaHTMLFormEditRecordByID (TUniqId uniqid, StringList *SLHTML)
  {
  TDBField DBField;  
  TBuffer abuf;
  int len;
  char *pc = NULL;
  
  SLHTML->Clear();
  xsnprintf (abuf, CMAXBUFFER, "<CENTER>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "<TABLE width=\"100%%\" BORDER=\"0\">\n"); SLHTML->Add (abuf);  
  
  for (int i = 0; i < getNumFields(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype != fRESERVED)
      {              
      pc = getField_Text_ByID (uniqid, DBField.fname, pc, &len);
      xsnprintf (abuf, CMAXBUFFER, "<TR valign=\"top\">\n"); SLHTML->Add (abuf);
      xsnprintf (abuf, CMAXBUFFER, "<TD align=\"right\"><B>%s:</B></TD>\n", DBField.flabel); SLHTML->Add (abuf);
      switch (DBField.ftype)
        {            
        case fTEXT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><TEXTAREA NAME=\"%s\" ROWS=\"%d\" COLS=\"%d\" WRAP=\"HARD\">%s</TEXTAREA></TD>\n", 
                     DBField.fname, 10, 55, (pc != NULL) ? pc : "");
          break;
          }
        case fVARCHAR:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, (pc != NULL) ? pc : "");
          break;                                                                                                             
          }
        case fDATETIME:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, (pc != NULL) ? pc : "");
          break;
          }
        case fURL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, (pc != NULL) ? pc : "");
          break;
          }
        case fINT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, (pc != NULL) ? pc : "");
          break;
          }
        case fREAL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\"><INPUT TYPE=TEXT NAME=\"%s\" SIZE=\"%d\" MAXLENGTH=\"%d\" VALUE=\"%s\"></TD>", 
                     DBField.fname, DBField.fdisplaylength, DBField.fmaxlength, (pc != NULL) ? pc : "");
          break;
          }
        case fRESERVED: 
          {
          //Never must go here
          break;
          } 
        }
      free (pc);
      SLHTML->Add (abuf);
      SLHTML->Add ("</TR>\n");
      }
    }  
  xsnprintf (abuf, CMAXBUFFER, "</TR>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</TABLE>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</CENTER>\n"); SLHTML->Add (abuf);
  return true;  
  }
  
bool ClientDB::generaHTMLFormDisplayRecordByID (TUniqId uniqid, StringList *SLHTML)
  {
  TDBField DBField;  
  TBuffer abuf;
  int len;
  char *pc = NULL;
  
  SLHTML->Clear();
  xsnprintf (abuf, CMAXBUFFER, "<PRE>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "<CENTER>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "<TABLE width=\"100%%\" BORDER=\"0\">\n"); SLHTML->Add (abuf);  
  
  for (int i = 0; i < getNumFields(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype != fRESERVED)
      {              
      pc = getField_Text_ByID (uniqid, DBField.fname, pc, &len);
      xsnprintf (abuf, CMAXBUFFER, "<TR valign=\"top\">\n"); SLHTML->Add (abuf);
      xsnprintf (abuf, CMAXBUFFER, "<TD align=\"right\"><B>%s:</B></TD>\n", DBField.flabel); SLHTML->Add (abuf);
      switch (DBField.ftype)
        {            
        case fTEXT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>\n", (pc != NULL) ? pc : "");
          break;
          }
        case fVARCHAR:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>", (pc != NULL) ? pc : "");
          break;                                                                                                             
          }
        case fDATETIME:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>", (pc != NULL) ? pc : "");
          break;
          }
        case fURL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>", (pc != NULL) ? pc : "");
          break;
          }
        case fINT:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>", (pc != NULL) ? pc : "");
          break;
          }
        case fREAL:
          {
          xsnprintf (abuf, CMAXBUFFER, "<TD align=\"left\">%s</TD>", (pc != NULL) ? pc : "");
          break;
          }
        case fRESERVED: 
          {
          //Never must go here
          break;
          } 
        }
      free (pc);
      SLHTML->Add (abuf);
      SLHTML->Add ("</TR>\n");
      }
    }  
  xsnprintf (abuf, CMAXBUFFER, "</TR>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</TABLE>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</CENTER>\n"); SLHTML->Add (abuf);
  xsnprintf (abuf, CMAXBUFFER, "</PRE>\n"); SLHTML->Add (abuf);  
  return true;  
  }
      
bool ClientDB::generaHTMLIndexPage (const char *SortUrl, StringList *SLHTML)
  {
  TDBField DBField;  
  TBuffer abufurlsort, abuf;
  
  SLHTML->Clear();
  for (int i = 0; i < getNumFields(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.findexlength > 0)
      {
      xsnprintf (abufurlsort, CMAXBUFFER, SortUrl, DBField.fname, DBField.flabel);
      xsnprintf (abuf, CMAXBUFFER, "<TD width=\"%d%%\" align=\"left\"><B>%s</B></TD>\n", 
                 DBField.findexlength, abufurlsort);
      SLHTML->Add (abuf);
      }    
    } 
  return true;
  }

int CDB_CompareEntriesBySTRCASE (const void *e1, const void *e2)
  {
  int res;
  res = strcasecmp (PDBNodo (e1)->value, PDBNodo(e2)->value);
  if (InverseDBSort == true) {res = (-1) * res;}
  return res;  
  }  
  
int CDB_CompareEntriesByNUMERIC (const void *e1, const void *e2)
  {
  int res;
  double fres, val1, val2;

  val1 = strtod (PDBNodo (e1)->value, NULL);
  val2 = strtod (PDBNodo (e2)->value, NULL);
  fres = val1 - val2;
  if (InverseDBSort == true) {fres = (-1.0) * fres;}
  if (fres < 0) res = -1; else res = 1;
  return res;  
  }  

bool ClientDB::Sort (TFieldName fieldname)
  {
  int rval, len, ires;
  DBT dkey, ddata;
  TBuffer abuf1, abuf2;
  char *pc = NULL;
  TDBNodo dbnodo;
  TType fieldtype;
  TNumber anum;

  if (UseIndex == false) return false;    

  fieldtype = getTypeFromString (fieldname);
  if (InverseDBSort == true) InverseDBSort = false; else InverseDBSort = true;
  dbnodolist.Clear();
  if (db == NULL) return false;
  if (UseIndex == true) if (dbindex == NULL) return false;  
  rval = dbindex->seq (dbindex, &dkey, &ddata, R_FIRST);
  while (!rval)
    {
    memcpy (abuf1, dkey.data, dkey.size); abuf1 [dkey.size] = '\0';
    memcpy (abuf2, ddata.data, ddata.size); abuf2 [ddata.size] = '\0';    
    dbnodo.key = xatoidef (abuf1, 0);
    dbnodo.uniqid = abuf2;    
    xstrncpy (dbnodo.value, CMAXBUFFER, getField_Text_ByID (dbnodo.uniqid, fieldname, pc, &len));    
    dbnodolist.Add (dbnodo);
    rval = dbindex->seq (dbindex, &dkey, &ddata, R_NEXT); 
    }          
  closeIndex ();
  switch (fieldtype)
    {
    case fTEXT:     
    case fVARCHAR:  
    case fDATETIME: 
    case fURL:      
      {
      dbnodolist.SORT (CDB_CompareEntriesBySTRCASE, CDB_FlipEntries);
      break;
      }
    case fINT:      
    case fREAL:     
      {
      dbnodolist.SORT (CDB_CompareEntriesByNUMERIC, CDB_FlipEntries);
      break;
      }
    default:        
      {
      dbnodolist.SORT (CDB_CompareEntriesBySTRCASE, CDB_FlipEntries);
      break;
      }
    }

  if (!DeleteFile (dbindexfname)) {xstrncpy (ErrorMsg, CMAXBUFFER, "Can not delete the index file for compact."); return false;}
  if (!openIndex()) {xstrncpy (ErrorMsg, CMAXBUFFER, "Can not open the index file."); return false;}
  
  for (int i = 0; i < dbnodolist.Count(); ++i)
    {    
    dbnodo = dbnodolist.elementAt(i);
    xstrncpy (abuf1, CMAXBUFFER, xitoa (i + 1, anum));
    xstrncpy (abuf2, CMAXBUFFER, dbnodo.uniqid);    
    dkey.data  = (void *)abuf1; dkey.size = strlen (abuf1);
    ddata.data = (void *)abuf2; ddata.size = strlen (abuf2); 
    ires = dbindex->put (dbindex, &dkey, &ddata, 0);
    if (ires < 0) {xsnprintf (ErrorMsg, CMAXBUFFER, "compactIndex: %s", strerror(errno)); return false;}    
    }  
  dbnodolist.Clear();    
  return true;
  }

void ClientDB::getAllFieldNames (StringList *SLFieldNames)
  {
  TDBField DBField;    
  SLFieldNames->Clear();
  for (int i = 0; i < getNumFields(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype != fRESERVED) SLFieldNames->Add (DBField.fname);
    }          
  }  

void ClientDB::getAllFieldLabels (StringList *SLFieldLabels)
  {
  TDBField DBField;    
  SLFieldLabels->Clear();
  for (int i = 0; i < getNumFields(); ++i)
    {
    DBField = dbinfo.MatField->elementAt(i);
    if (DBField.ftype != fRESERVED) SLFieldLabels->Add (DBField.flabel);
    }          
  }
    
bool ClientDB::getFlagsById (TUniqId uniqid, TFlags flags)
  {
  char *pc = NULL;
  int len; 
  pc = getField_Text_ByID (uniqid, DB_FLAGRECORD_NAME, pc, &len);
  xstrncpy (flags, CMAXFLAGS + 1, pc);
  free (pc);
  return true;
  } 

bool ClientDB::putFlagsById (TUniqId uniqid, TFlags flags)
  {
  return putField_Text_ByID (uniqid, DB_FLAGRECORD_NAME, flags, strlen(flags));
  }      

bool ClientDB::isFlagById (TUniqId uniqid, int queflag)
  {     
  TFlags flags;
  getFlagsById (uniqid, flags);
  if (IsEmpty(flags)) {return false;}
  if (strlen(flags) != CMAXFLAGS) {return false;}
  if (flags[queflag] != FEMPTY) return true; else return false;
  }

bool ClientDB::changeFlagById (TUniqId uniqid, int queflag, bool value)
  {     
  TFlags flags;
  getFlagsById (uniqid, flags);
  if (IsEmpty(flags)) {return false;}
  if (strlen(flags) != CMAXFLAGS) {return false;}
  if (queflag == POS_FLAG_COPIED)
    {
    if (value == false) {flags[POS_FLAG_COPIED] = FEMPTY;} else {flags[POS_FLAG_COPIED] = FCOPIED;}  
    }
  else if (queflag == POS_FLAG_SEARCHED)
    {
    if (value == false) {flags[POS_FLAG_SEARCHED] = FEMPTY;} else {flags[POS_FLAG_SEARCHED] = FSEARCHED;}  
    }
  else if (queflag == POS_FLAG_DELETED)
    {
    if (value == false) {flags[POS_FLAG_DELETED] = FEMPTY;} else {flags[POS_FLAG_DELETED] = FDELETED;}  
    }    
  else 
    {
    return false;        
    }
  return putFlagsById (uniqid, flags);
  }
  
int ClientDB::deleteRecordsByID (StringList *SLUniqIDItems)
  {
  DBT dkey;
  TDBField DBField;
  TBuffer af;
  int ret = 0, ires;
  
  for (int i = 0; i < SLUniqIDItems->Count(); ++i)
    {
    for (int j = 0; j < getNumFields(); ++j)
      {
      DBField = dbinfo.MatField->elementAt(j);
      xsnprintf (af, CMAXBUFFER, "%s_%s", SLUniqIDItems->getString(i).cstr(), DBField.fname);
      //DEBUG ("i=%d,j=%d,af=%s", i, j, af);
      dkey.data  = (void *)af; dkey.size = strlen(af);
      ires = db->del (db, &dkey, 0);
      }
    if (!delRecordCount ()) {return -1;}      
    ++ret;
    }
  Sync();  
  deleteIDsFromIndex (SLUniqIDItems);  
  if (UseIndex == true) {if (!compactIndex ()) {return -1;}}
  return ret;  
  }              

int ClientDB::markRecordsByID (StringList *SLUniqIDItems, int queflag, bool value)
  {
  TUniqId uniqid;
  int cont = 0, NUMRECS;  
  if (queflag == POS_FLAG_SEARCHED)  //SPECIAL CASE. I MUST TO CLEAN THE FLAGS
    {
    NUMRECS = getNumRecords();
    for (int recordpos = 1; recordpos <= NUMRECS; ++recordpos)
      {
      if (getUniqIDFromIndexPos (recordpos, uniqid) != NULL) 
        {
        if (SLUniqIDItems->indexOf(uniqid) >= 0)
          {
          changeFlagById (uniqid, POS_FLAG_SEARCHED, true);
          ++cont;
          }
        else
          {
          changeFlagById (uniqid, POS_FLAG_SEARCHED, false); 
          }  
        }
      }
    }
  else
    {  
    for (int i = 0; i < SLUniqIDItems->Count(); ++i)
      {
      xstrncpy (uniqid, CMAXUNIQID + 1, SLUniqIDItems->getString(i).cstr());
      if (changeFlagById (uniqid, queflag, value)) {++cont;}
      }
    }
  return cont;  
  }               

int ClientDB::Expunge (void)
  {      
  TUniqId uniqid;
  int NUMRECS;
  StringList SLMarkedDeleted;
  
  NUMRECS = getNumRecords();  
  for (int recordpos = 1; recordpos <= NUMRECS; ++recordpos)
    {
    if (getUniqIDFromIndexPos (recordpos, uniqid) != NULL) 
      {
      if (isFlagById (uniqid, POS_FLAG_DELETED) == true)
        {
        SLMarkedDeleted.Add (uniqid);
        }
      }
    }    
  return deleteRecordsByID (&SLMarkedDeleted);
  }

void ClientDB::deleteSearch (void)
  {
  ModeDB = fMODE_FULL;
  SLSearch.Clear();      
  }  

bool ClientDB::Search (TFieldName fieldname, const char *whatsearch, StringList *SLRes)
  {
  TDBField DBField;
  TUniqId uniqid;
  char *pc = NULL;
  int NUMRECS, len, i;
  bool encontrado = false;

  ModeDB = fMODE_FULL;
  SLSearch.Clear();
  SLRes->Clear();
  NUMRECS = getNumRecords();
  if (IsEmpty(whatsearch)) {return true;}
  if (!isFieldName(fieldname) && (strcmp (fieldname, "fd_all") != 0)) {xsnprintf (ErrorMsg, CMAXBUFFER, "Search: Error: No field %s", fieldname); return false;}
  for (int recordpos = 1; recordpos <= NUMRECS; ++recordpos)
    {
    if (strcmp (fieldname, "fd_all") == 0)
      {
      if (getUniqIDFromIndexPos (recordpos, uniqid) != NULL) 
        {
        i = 0;
        encontrado = false;
        while ((i < getNumFields()) && (encontrado == false))
          {     
          DBField = dbinfo.MatField->elementAt(i);              
          pc = getField_Text_ByID (uniqid, DBField.fname, pc, &len);
          if (Match (pc, whatsearch) == true) {encontrado=true; SLSearch.Add (uniqid); SLRes->Add (uniqid);}    
          free (pc);
          ++i;
          }
        }
      }
    else
      {        
      if (getUniqIDFromIndexPos (recordpos, uniqid) == NULL) {return false;}
      pc = getField_Text_ByID (uniqid, fieldname, pc, &len);
      if (Match (pc, whatsearch) == true) {SLSearch.Add (uniqid); SLRes->Add (uniqid);}    
      free (pc);
      }
    } 
  if (SLSearch.Count () > 0) 
    {
    ModeDB = fMODE_SEARCH;
    }
  else
    {
    ModeDB = fMODE_FULL;
    }     
  return true;
  }
 
bool ClientDB::isRightIndexPos (int pos, TUniqId uniqid)
  {
  TNumber anum;
  DBT dkey, ddata;
  int ires, L;
  
  if (UseIndex == false) return false;    
  
  initStr (uniqid);
  if ((db == NULL) || (dbindex == NULL)) return false;  

  xitoa (pos, anum);
  dkey.data = (void *)anum; dkey.size = strlen(anum);
  ires = dbindex->get (dbindex, &dkey, &ddata, 0);
  if (ires == 1) 
    {
    return false;
    } 
  else if (ires < 0) 
    {
    return false;
    }
  L = ddata.size;
  memcpy (uniqid, ddata.data, ddata.size);
  uniqid[L] = '\0';  
    
  return true;
  }

bool Match (const char *pc, const char *whatsearch)
  {
  if (IsEmpty(whatsearch)) return false;
#ifdef USE_REGEX
  int errcode, status;
  regex_t re;
  TBuffer pattern, abuferr;
 
  if (whatsearch[0] != '\'') 
    {
    ProcessPattern (whatsearch, pattern);
    }
  else
    {
    //QUITO EL ''''
    memcpy (pattern, whatsearch + 1, CMAXBUFFER);
    }  
  errcode = regcomp (&re, pattern, REG_EXTENDED | REG_NOSUB | REG_ICASE);
  if (errcode != 0) 
    {        
    regerror (errcode, (const regex_t *)&re, abuferr, CMAXBUFFER);
    DEBUG ("%s", abuferr);
    return false;   
    }
  status = regexec ((const regex_t *)&re, pc, (size_t) 0, NULL, 0);
  regfree (&re);
  if (status == 0) 
    {         
    return true;   
    }
  else if (status == REG_NOMATCH)
    {
    return false;
    }  
  else
    {
    regerror (status, (const regex_t *)&re, abuferr, CMAXBUFFER);
    DEBUG ("%s", abuferr);
    return false;   
    }
#else
   if (strstr (pc, whatsearch) != NULL) return true; else return false;
#endif
  }
 
void ProcessPattern (const char *whatsearch, TBuffer pattern)
  {
  char c, c2;
  int cont = 0;
  while ((cont < (signed)strlen (whatsearch)) && (cont < CMAXBUFFER))
    {
    c = whatsearch[cont];
    switch (c)
      {
      case '': case '': case '': case '': {c2 = 'a'; break;}
      case '': case '': case '': case '': {c2 = 'A'; break;}      
      case '': case '': case '': case '': {c2 = 'e'; break;}      
      case '': case '': case '': case '': {c2 = 'E'; break;}            
      case '': case '': case '': case '': {c2 = 'i'; break;}      
      case '': case '': case '': case '': {c2 = 'I'; break;}            
      case '': case '': case '': case '': {c2 = 'o'; break;}
      case '': case '': case '': case '': {c2 = 'O'; break;}      
      case '': case '': case '': case '': {c2 = 'u'; break;}
      case '': case '': case '': case '': {c2 = 'U'; break;}      
      default: {c2 = c; break;} 
      }
    pattern[cont] = c2;  
    ++cont;
    }
  pattern[cont] = '\0';  
  }
       
bool getDBNameNoClass (const char *dbfilename, TBuffer dbname)
  {
  DB *dbread;
  DBT dkey, ddata;
  int ires, L; 
  
  initStr (dbname);
  
  dbread = dbopen (dbfilename, O_RDONLY, 0, DB_HASH, NULL);
  if (dbread == NULL) {return false;}      
      
  dkey.data = (void *)DB_NAME; dkey.size = strlen(DB_NAME);
  ires = dbread->get (dbread, &dkey, &ddata, 0);
  if (ires == 1) {return false;} else if (ires < 0) {return false;}
  L = min (CMAXBUFFER, ddata.size); memcpy (dbname, ddata.data, L); dbname[L] = '\0';         
    
  dbread->close(dbread);
  return true;
  }

int getTotalDatabases (const char *dbdir, StringList *SLFilenames, StringList *SLNames)
  {
  int filetipo;
  TBuffer buf, ext, dbname;
  struct dirent *ent;
  struct stat statbuf;
  DIR *dirp;

  SLFilenames->Clear();
  SLNames->Clear();  
  if ((dirp = opendir (dbdir)) == NULL) {return -1;}
  while ((ent = readdir(dirp)) != NULL)
    {
    if ((strcmp (ent->d_name, ".") != 0) && (strcmp (ent->d_name, "..") != 0))
      {
      xstrncpy (buf, CMAXBUFFER, dbdir); addLastSlash (buf); xstrncat (buf, CMAXBUFFER, ent->d_name);
      lstat (buf, &statbuf);      
      filetipo = statbuf.st_mode & S_IFMT; 
      if (filetipo == S_IFREG)
         {
         xstrncpy (ext, CMAXBUFFER, getExtensionFromFile (buf));
         if (strcmp (ext, DB_EXTENSION) == 0)
           {
           SLFilenames->Add (ent->d_name);
           getDBNameNoClass ((const char *)buf, dbname);
           SLNames->Add (dbname);
           }
         }
      }
    }
  closedir(dirp); 
  return SLFilenames->Count();   
  }

bool creaDatabase (const char *user, const char *dbtemplate, const char *dbuserpath, const char *name, TBuffer sterror)
  {
  DB *dbwrite;
  DBT dkey, ddata;
  TBuffer oldname, newname, oldidxname, newidxname;
  int p = 1, ires;
  
  initStr(sterror);
  xsnprintf (oldname, CMAXBUFFER, "%s/%s", DATABASES_PATH, dbtemplate);
  xsnprintf (newname, CMAXBUFFER, "%s/%s_%d.db", dbuserpath, user, p);
  while (FileExists (newname) == true)
    {
    ++p;
    xsnprintf (newname, CMAXBUFFER, "%s/%s_%d.db", dbuserpath, user, p);
    }
  if (!CopyFileOverWrite (oldname, newname)) {xstrncpy(sterror, CMAXBUFFER, "Error: creaDatabase: CopyFileOverWrite"); return false;}
  xstrncpy(oldidxname, CMAXBUFFER, oldname); xstrncat (oldidxname, CMAXBUFFER, ".index");
  xstrncpy(newidxname, CMAXBUFFER, newname); xstrncat (newidxname, CMAXBUFFER, ".index");
  if (!CopyFileOverWrite (oldidxname, newidxname)) {xstrncpy(sterror, CMAXBUFFER, "Error: creaDatabase: CopyFileOverWrite"); return false;} 
  dbwrite = dbopen (newname, O_RDWR, 0, DB_HASH, NULL);
  if (dbwrite == NULL) {xsnprintf (sterror, CMAXBUFFER, "Error: creaDatabase: dbopen"); return false;}      
  dkey.data = (void *)DB_NAME; dkey.size = strlen(DB_NAME);
  ddata.data = (void *)name; ddata.size = strlen (name);
  ires = dbwrite->put (dbwrite, &dkey, &ddata, 0);
  if (ires < 0) {xsnprintf (sterror, CMAXBUFFER, "Error: creaDatabase: put"); return false;}
  dbwrite->close(dbwrite); 
  return true;
  }    

bool renameDatabase (const char *dbuserpath, const char *dbfn, const char *newname, TBuffer sterror)
  {
  DB *dbwrite;
  DBT dkey, ddata;
  TBuffer dbfilename;
  int ires;
  
  initStr(sterror);
  xsnprintf (dbfilename, CMAXBUFFER, "%s/%s", dbuserpath, dbfn);
  dbwrite = dbopen (dbfilename, O_RDWR, 0, DB_HASH, NULL);
  if (dbwrite == NULL) {xsnprintf (sterror, CMAXBUFFER, "Error: renameDatabase: dbopen"); return false;}      
  dkey.data = (void *)DB_NAME; dkey.size = strlen(DB_NAME);
  ddata.data = (void *)newname; ddata.size = strlen (newname);
  ires = dbwrite->put (dbwrite, &dkey, &ddata, 0);
  if (ires < 0) {xsnprintf (sterror, CMAXBUFFER, "Error: renameDatabase: put"); return false;}
  dbwrite->close(dbwrite); 
  return true;
  }    

bool deleteDatabase (const char *dbuserpath, const char *dbfn, TBuffer sterror)
  {
  TBuffer dbfilename, dbidxfilename;
  initStr(sterror);
  xsnprintf (dbfilename, CMAXBUFFER, "%s/%s", dbuserpath, dbfn);
  if (!DeleteFile (dbfilename)) {xsnprintf (sterror, CMAXBUFFER, "Error: deleteDatabase: DeleteFile"); return false;}
  xstrncpy (dbidxfilename, CMAXBUFFER, dbfilename); xstrncat (dbidxfilename, CMAXBUFFER, ".index");
  DeleteFile (dbidxfilename);
  return true;
  }

bool dumpDatabase (const char *dbuserpath, const char *dbfn, TBuffer sterror, XSocket *aXS)
  {
  ClientDB *acdb;
  int NUMRECS;
  TUniqId uniqid;
  TDBField DBField;  
  TBuffer abuf;
  int len;
  char *pc = NULL;
  TMatField *amf = NULL;
    
  initStr(sterror);

  acdb = new ClientDB (dbfn, dbuserpath, true, true);
  acdb->DBopen();
  acdb->displayDBInfo (aXS);
  acdb->displayDBStructure (aXS);
  acdb->displayDBPermissions (aXS);
  
  NUMRECS = acdb->getNumRecords();  
  for (int recordpos = 1; recordpos <= NUMRECS; ++recordpos)
    {
    if (acdb->getUniqIDFromIndexPos (recordpos, uniqid) != NULL) 
      {
      xsnprintf (abuf, CMAXBUFFER, "Record %d\n", recordpos); aXS->Write (abuf, strlen(abuf));
      xsnprintf (abuf, CMAXBUFFER, "-------------------------------------------------------\n"); aXS->Write (abuf, strlen(abuf));
      for (int i = 0; i < acdb->getNumFields(); ++i)
        {              
        (const TMatField *)amf = acdb->getMatField ();        
        DBField = amf->elementAt(i);
        if (DBField.ftype != fRESERVED)
          {              
          xsnprintf (abuf, CMAXBUFFER, "%s\n", DBField.flabel); aXS->Write (abuf, strlen(abuf));          
          pc = acdb->getField_Text_ByID (uniqid, DBField.fname, pc, &len);
          aXS->Write (pc, len);
          aXS->Write ("\n", 1);
          free (pc);
          }
        xsnprintf (abuf, CMAXBUFFER, "--------------\n"); aXS->Write (abuf, strlen(abuf));
        }  
      }
    }  

  delete acdb;  
  return true;
  }
  
int getTotalFields (StringList *SLFields)
  {
  SLFields->Clear();
  SLFields->Add ("TEXT");
  SLFields->Add ("VARCHAR");  
  SLFields->Add ("DATETIME");  
  SLFields->Add ("URL");  
  SLFields->Add ("INT");
  SLFields->Add ("REAL");
  return SLFields->Count();
  }

bool generaIndexFromDB (const char *dbfname, const char *dbindexfname, TBuffer strerr)
  {
  TNumber anum;
  TBuffer abuf;
  DB *adbindex, *adb;
  DBT dkey, ddata;
  StringList SL;
  int L, rval, reccount = 1, ires;
 
  SL.Clear();
  initStr (strerr);
  adb = dbopen (dbfname, O_RDONLY, 0666, DB_HASH, NULL);  
  rval = adb->seq (adb, &dkey, &ddata, R_FIRST);
  while (!rval)
    {
    L = min (CMAXUNIQID, dkey.size); 
    memcpy (abuf, dkey.data, L); abuf[L] = '\0';     
    if (abuf[0] != '_')
      { 
      if (SL.indexOf (abuf) < 0)
        {
        SL.Add (abuf);
        }
      }    
    ++reccount;
    rval = adb->seq (adb, &dkey, &ddata, R_NEXT); 
    } 
  adb->close(adb);     
   
  DeleteFile (dbindexfname); 
  adbindex = dbopen (dbindexfname, O_RDWR | O_CREAT, 0666, DB_HASH, NULL);
  for (int i = 0; i < SL.Count(); ++i)
    {
    xitoa (i + 1, anum);  
    dkey.data = (void *)anum; dkey.size = strlen (anum);
    xstrncpy (abuf, CMAXUNIQID + 1, SL.getString(i).cstr());
    ddata.data = (void *)abuf; ddata.size = strlen (abuf); 
    ires = adbindex->put (adbindex, &dkey, &ddata, 0);
    if (ires < 0) {xsnprintf (strerr, CMAXBUFFER, "generaIndexFromDB: Put"); return false;}
    }   
  adbindex->close(adbindex);                
  
  return true;
  }  
  
