/* tcp.c  TCP communications routines for HERMES */

#ifdef TERM
#include <termnet.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#include "tcp.h"
#include "postit.h"

#define DONT_HOG_CPU


#ifdef DONT_HOG_CPU
#ifdef usleep
#define NAP usleep(100000)
#else
#define NAP sleep(1)
#endif
#else
#define NAP
#endif


#ifdef DEBUG
void dump(unsigned char *what, int howmany)
{
  int i;

  for (i=0; i<howmany; i++) printf("%2.2X ",what[i]);
  printf("\n");
}
#endif

/* Open a socket and connect to remote host on a given port. Return value
is the socket handle if positive or an error if negative. */
int tcp_open_connection(char *hostname, unsigned short port)
{
  struct protoent *proto;
  struct hostent *host;
  struct sockaddr_in server;
  int socknr;
  unsigned char ip[4];
  int i;
  unsigned char *c;

  proto=getprotobyname("tcp");
  if (!proto) return -1;	/* no such protocol */
  socknr=socket(AF_INET,SOCK_STREAM,proto->p_proto);
  if (socknr==-1) return -2; /* no socket available */
  /* We have a socket on our machine; now let's find out how to
     connect to the remote host */
  if (isdigit(*hostname)) {
    /* We are given a string with four decimal numbers, separated
       by dots */
    c=hostname;
    for (i=0; i<4; i++) {
      ip[i]=0;
      if (!*c) return -3;	/* bad hostname */
      while (*c!='.'&&*c) {
	ip[i]=ip[i]*10+(*c&15);
	c++;
      }
      if (*c=='.') c++;
    }
    host=(struct hostent *)gethostbyaddr(ip,4,AF_INET);
  }
  else	{
    /* it's a real hostname */
    host=(struct hostent *)gethostbyname(hostname);
  }
  if (!host) return -4;	/* unknown host */
  bcopy(host->h_addr,ip,4);	
  bzero((char *)&server,sizeof(server));
  bcopy(host->h_addr,(char *)&server.sin_addr,host->h_length);
  server.sin_family=host->h_addrtype;
  server.sin_port=htons(port);	/* note that this is high-byte first!! */
  /* OK, now let's connect! */
  if (verbosity>=3) printf("Connecting to %u.%u.%u.%u, port %hu on socket %d...\n",
			   ip[0],ip[1],ip[2],ip[3],port,socknr);
  if (connect(socknr,(void *)&server,sizeof(server))==-1) return -6;	/* connection refused */
  return socknr;
}

/* This function closes a connection on a given socket for both rx and tx.
Returns 0 on success, -1 on error */
int tcp_close_connection(int socknr)
{
  return shutdown(socknr,2);
}

/* Read a line from the socket. If no data available, return 0, else return
number of characters read (including linefeed) */
int tcp_receive_line(int socknr, char *line, int maxsize)
{
  char *c;
  int flags;
  int x;

  c=line; *c=0; maxsize--;
  flags=fcntl(socknr,F_GETFL);	/* remember previous state */
  fcntl(socknr,F_SETFL,(long)(flags|O_NONBLOCK));	/* make sure read returns if no data available */
  while (1) {
    x=read(socknr,c,1);
    if (x==-1) {
      if (errno==EAGAIN) {
	if (c==line) break;
	continue;
      }
      break;
    }
    if (x==0) continue;
    c+=x;
    if (*(c-1)=='\n'||--maxsize<=0) {
      *c=0;
      break;
    }
  }
  fcntl(socknr,F_SETFL,flags);
  return c-line;
}

static char timeoutflag;
static void sigalarm(int x)
{
  timeoutflag=1;
#ifdef DEBUG
  printf("Timeout");
#endif
}

/* Read a line from the socket with timeout. If no data available, return 0, else return
number of characters read (including linefeed) */
int tcp_receive_line_to(int socknr, char *line, int maxsize, long timeout)
{
  char *c;
  int flags;
  int x;

  c=line; *c=0; maxsize--;
  flags=fcntl(socknr,F_GETFL);	/* remember previous state */
  fcntl(socknr,F_SETFL,(long)(flags|O_NONBLOCK));	/* make sure read returns if no data available */
  signal(SIGALRM,sigalarm);
  timeoutflag=0;
  alarm(timeout);
  while (1) {
    if (timeoutflag) {
      *c=0;
      break;
    }
    x=read(socknr,c,1);
    if (x==0) {
      NAP;
      continue;
    }
    if (x==-1) {
      if (errno==EAGAIN) {
	continue;
      }
      break;
    }
    c+=x;
    if (*(c-1)=='\n'||--maxsize<=0) {
      *c=0;
      break;
    }
  }
  fcntl(socknr,F_SETFL,flags);
  return c-line;
}

/* Read a block from the socket with timeout. If no data available, return 0, else return
number of characters read */
int tcp_receive_block_to(int socknr, char *block, int maxsize, long timeout)
{
  char *c;
  int flags;
  int x;

  c=block; *c=0;
  flags=fcntl(socknr,F_GETFL);	/* remember previous state */
  fcntl(socknr,F_SETFL,(long)(flags|O_NONBLOCK));	/* make sure read returns if no data available */
  signal(SIGALRM,sigalarm);
  timeoutflag=0;
  alarm(timeout);
  while (1) {
    if (timeoutflag) {
      *c=0;
      break;
    }
    x=read(socknr,c,maxsize);
    if (!x) {
      NAP;
      continue;
    }
    if (x==-1) {
      if (errno==EAGAIN) {
	continue;
      }
      break;
    }
    c+=x; maxsize-=x;
    break;
  }
  fcntl(socknr,F_SETFL,flags);
  return c-block;
}

/* Write a null-terminated string to a socket. Returns: # of chars written or
-1 on error */
int tcp_send_line(int socknr, char *line)
{
  return write(socknr,line,strlen(line));
}

/* Write a block to a socket. Returns # of bytes written or -1 on error */
int tcp_send_block(int socknr, char *block, int size)
{
  return write(socknr,block,size);
}
