/*
 * chat.c        This is a generic chat handler
 *               for initialization of the modems.
 *
 * Version:      (@)#chat.c  1.00  12-Sep-1995  MvS.
 *
 */
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/poll.h>

#include "../pslave_cfg.h"
#include "server.h"

enum special_code
{
  SPECIAL_NONE = -1,
  SPECIAL_TIMEOUT = 0,	/* change the timeout for the next expect string */
  SPECIAL_WAIT,		/* wait for carrier detect, parm must be DCD */
  SPECIAL_STATUS,	/* sets status in utmp, two params. */
  SPECIAL_ABORT,	/* sets abort string, one param. */
  SPECIAL_SETVAR	/* sets a string variable after receiving certain text,
			   two params. */
};

struct special_chat
{
  const char *word;
  enum special_code code;
  int req_words; /* params + 1 */
};

struct special_chat sw [] = {
  {"TIMEOUT", SPECIAL_TIMEOUT, 2},
  {"WAIT", SPECIAL_WAIT, 2},
  {"STATUS", SPECIAL_STATUS, 3},
  {"ABORT", SPECIAL_ABORT, 2},
  {"SETVAR", SPECIAL_SETVAR, 2},
  {NULL, SPECIAL_NONE, 0}
};

/*
 *  Send a string to the remote side. Process any
 *  left-over escape sequences.
 */
static void chat_send(int fd, const char *p)
{
  char *s, *start, *end;
  bool addcr = true; /* whether to add a '\r' to the end of the string */
  int flags, val;
  char c;

  nsyslog (LOG_DEBUG, "chat_send(%s)", p);

  /* Translate escape characters, pass 1 */
  s = xstrdup(p);
  tstr(s, true, s);

  /* Delay a bit and then flush the input. */
  xusleep(GetChatSendDelay() * 100000);
  tcflush(0, TCIFLUSH);

  /* Now send the string. */
  start = end = s;
  while(1)
  {
    /* Collect characters till backslash or end of string. */
    while(*end != '\\' && *end)
      end++;
    if(end != start) write(fd, start, end - start);
    if(*end == 0)
    {
      if(addcr)
      {
        c = '\r';
        write(fd, &c, 1);
      }
      free(s);
      return;
    }

    /* Special escape sequence. */
    val = -1;
    switch (*++end)
    {
      case 'd': /* delay */
        xsleep(1);
      break;
      case 'p': /* pause */
        xusleep(100000);
      break;
      case 'l': /* lower DTR */
        ioctl(fd, TIOCMGET, &flags);
        flags &= ~TIOCM_DTR;
        ioctl(fd, TIOCMSET, &flags);
        xsleep(1);
        flags |= TIOCM_DTR;
        ioctl(fd, TIOCMSET, &flags);
      break;
      case 'c': /* Don't add CR */
        addcr = false;
      break;
      case 'K': /* Send a break */
        tcsendbreak(fd, 0);
      break;
      default: /* Just fill in character. */
        val = *end;
      break;
    }
    /* Write character if needed. */
    if(val >= 0)
    {
      c = val;
      write(fd, &c, 1);
    }

    /* And continue after escape character. */
    if (*end) end++;
    start = end;
  }
}

/*
 * Wait until fd is readable or timeout/error occurs. If fd is readable, try
 * to read 1 char.
 */

static int read_char(char *c_ptr, int fd, bool *timed_out)
{
  struct pollfd read_fd;
  time_t poll_timeout;
  struct itimerval left;
  int ret;

  /*
   * We don't enclose all poll code in if(GetChatTimeout()) {...},
   * because we want to be able to specify infinite timeouts on
   * non-blocking sockets too.
   */

  if(GetChatTimeout())
  {
    if(timed_out && *timed_out)
      return (-1);
    if(getitimer (ITIMER_REAL, &left))
    {
      nsyslog(LOG_DEBUG, "read_char: getitimer: %d(%m)", errno);
      return -1;
    }
    if((! left.it_value.tv_sec) && (! left.it_value.tv_usec))
    {
      nsyslog(LOG_DEBUG, "read_char: timed out outside of poll");
      if(timed_out)
        *timed_out = true;
      return -1;
    }
    poll_timeout = left.it_value.tv_sec*1000+(left.it_value.tv_usec ? 1 : 0);
  }
  else
  {
    poll_timeout = -1;
  }
  read_fd.fd = fd;
  read_fd.events = POLLIN;
  read_fd.revents = 0;
  switch (ret = poll (&read_fd, 1, poll_timeout))
  {
    case -1:
      nsyslog(LOG_DEBUG, "read_char: poll: %d(%m)", errno);
      return -1;
    case 0:
      if(GetChatTimeout())
      {
        nsyslog(LOG_DEBUG, "read_char: timed out in poll");
        if(timed_out)
          *timed_out = true;
      }
      else
      {
        nsyslog(LOG_DEBUG, "read_char: bad thing"
          "happened: timed out in poll after"
          "infinite timeout had been specified");
      }
      return -1;
    case 1:
    break;
    default:
      nsyslog(LOG_DEBUG, "read_char: poll returned: %d", ret);
      return -1;
  }
  if((read_fd.revents & POLLIN) == 0)
  {
    nsyslog(LOG_DEBUG, "read_char: bad poll mask: %d", read_fd.revents);
    return -1;
  }
  switch (ret = read (fd, c_ptr, 1))
  {
    case -1:
      nsyslog(LOG_DEBUG, "read_char: read: %d(%m)", errno);
      return -1;
    case 1:
    break;
    default:
      nsyslog(LOG_DEBUG, "read_char: read %d chars", ret);
      return -1;
  }
  return 0;
}

struct abort
{
  char *str;
  int len;
};

#define MAX_ABORT 16
static struct abort abort_arr[MAX_ABORT];

static const char *check_abort(const char *str, int len)
{
  int i;
  for(i = 0; i < MAX_ABORT && abort_arr[i].str; i++)
  {
    if(abort_arr[i].len < len)
    {
      const char *tmp_str = str + len - 1 - abort_arr[i].len;
      if(!strncmp(tmp_str, abort_arr[i].str, abort_arr[i].len))
        return abort_arr[i].str;
    }
  }
  return NULL;
}

/*
 * Add an abort string to the list.  A string of "CLEAN" will remove all
 * entries.
 */
static void special_chat_abort(const char *str)
{
  int i;
  if(!strcmp(str, "CLEAN"))
  {
    for(i = 0; i < MAX_ABORT; i++)
    {
      if(abort_arr[i].str)
        free(abort_arr[i].str);
      memset(&abort_arr[i], 0, sizeof(struct abort));
    }
    nsyslog(LOG_DEBUG, "Cleaned abort list.");
    return;
  }
  for(i = 0; i < MAX_ABORT; i++)
  {
    if(abort_arr[i].str == NULL)
    {
      abort_arr[i].str = xstrdup(str);
      abort_arr[i].len = strlen(str);
      nsyslog(LOG_DEBUG, "Added abort string \"%s\" at %d.", str, i);
      return;
    }
  }
  nsyslog(LOG_ERR, "No space for abort string \"%s\"", str);
}

typedef enum
{
  eAssign = '?',
  eOverRule = '=',
  eAppend = '+'
} VAR_TYPE;

struct setvar
{
  char *str;
  int len;
  char *var_buf;
  int var_len;
  VAR_TYPE type;
};

#define MAX_SETSTR 16
static struct setvar setvar_arr[MAX_ABORT];

static unsigned int abort_setvar_max_len()
{
  unsigned int len = 0;
  int i;
  for(i = 0; i < MAX_ABORT && abort_arr[i].str; i++)
    len = MAX(len, strlen(abort_arr[i].str));
  for(i = 0; i < MAX_SETSTR && setvar_arr[i].str; i++)
    len = MAX(len, strlen(setvar_arr[i].str));
  return len + 1;
}

static void check_setvar(const char *str, int str_len, int fd)
{
  int i;
  for(i = 0; i < MAX_SETSTR && setvar_arr[i].str; i++)
  {
    if(setvar_arr[i].len < str_len)
    {
      const char *tmp_str = str + str_len - 1 - setvar_arr[i].len;
      if(!strncmp(tmp_str, setvar_arr[i].str, setvar_arr[i].len))
      {
        int len = 0;
        bool start = true;
        char c;
        if(setvar_arr[i].var_buf[0] && setvar_arr[i].type == eAssign)
        {
          nsyslog(LOG_DEBUG
                , "Match on setvar string \"%s\", but already set to \"%s\"."
                , setvar_arr[i].str, setvar_arr[i].var_buf);
          return;
        }
        nsyslog(LOG_DEBUG, "Match on setvar string \"%s\".", setvar_arr[i].str);
        if(setvar_arr[i].type == eAppend)
          len = strlen(setvar_arr[i].var_buf);
        while(1)
        {
          if(read_char(&c, fd, NULL))
            break;
          if(start && c == ' ')
            continue;
          start = false;
          if(c == '\r' || c == '\n')
            break;
          if(len < setvar_arr[i].var_len - 1)
            setvar_arr[i].var_buf[len++] = c;
        }
        setvar_arr[i].var_buf[len] = '\0';
      }
    }
  }
}

/* macro to set the variable for the setvar operation */
#define SET_VAR_BUF(XX) \
{ \
  setvar_arr[i].var_buf = ai->XX; \
  setvar_arr[i].var_len = sizeof(ai->XX); \
}

/*
 * Add a setvar string to the list.  A string of "CLEAN" will remove all
 * entries.
 */
static void special_chat_setvar(const char *str, struct auth *ai)
{
  int i;
  if(!strcmp(str, "CLEAN"))
  {
    for(i = 0; i < MAX_SETSTR; i++)
    {
      if(setvar_arr[i].str)
        free(setvar_arr[i].str);
      memset(&setvar_arr[i], 0, sizeof(struct setvar));
    }
    nsyslog(LOG_DEBUG, "Cleaned setvar list.");
    return;
  }
  if((str[1] != eAssign && str[1] != eOverRule && str[1] != eAppend) || !str[2])
  {
    nsyslog(LOG_ERR, "Setvar string \"%s\" unparsable.", str);
    return;
  }
  for(i = 0; i < MAX_ABORT; i++)
  {
    if(setvar_arr[i].str == NULL)
    {
      switch(str[0])
      {
      case 'C':
        SET_VAR_BUF(conn_info);
      break;
      case 'S':
        SET_VAR_BUF(cli_src);
      break;
      case 'D':
        SET_VAR_BUF(cli_dst);
      break;
      default:
        nsyslog(LOG_ERR, "Setvar string \"%s\" unparsable.", str);
        return;
      }
      setvar_arr[i].str = xstrdup(str + 2);
      setvar_arr[i].len = strlen(setvar_arr[i].str);
      setvar_arr[i].type = str[1];
      nsyslog(LOG_DEBUG, "Added setvar string \"%s\" at %d.", setvar_arr[i].str, i);
      return;
    }
  }
  nsyslog(LOG_ERR, "No space for setvar string \"%s\"", str);
}

/*
 * Expect a string.
 */
static CHAT_ERR real_chat_expect(int fd, char *buf, unsigned int buf_len
    , const char *expect)
{
/* buf: buffer for storing last buf_len - 2 bytes of data read, it is
 *      checked for a match with expect after every byte.
 * timed_out: whether a timeout has occurred
 * expect: the string we look for
 *
 */
  unsigned int expect_len;
  char c;
  int retries = 0;
  bool timed_out = false;

  while(1)
  {
    nsyslog(LOG_DEBUG, "chat_expect(%s, %d)", expect, GetChatTimeout());
    expect_len = strlen(expect);
    if(expect_len > buf_len - 1)
      expect_len = buf_len - 1;

    /* Empty expects always succeed. */
    if(*expect == 0)
    {
      nsyslog(LOG_DEBUG, "chat_expect - got it");
      return eNoErr;
    }

    /* Wait for string to arrive, or timeout. */

    if(GetChatTimeout())
    {
      signal (SIGALRM, SIG_IGN);
      timed_out = false;
      alarm(GetChatTimeout());
    }
    memset(buf, 0, buf_len);

    /* Put every char in a buffer and shift. */
    while(1)
    {
      if(read_char(&c, fd, &timed_out))
      {
        if(timed_out)
          break;
        nsyslog(LOG_DEBUG, "chat_expect (%s) - got (%s) with error",
          expect, (buf + buf_len - 1 - expect_len));
        if(retries++ >= 3)
          break;
        continue;
      }
/* Insert character at the end of the string (buf_len - 2, buf_len - 1 is '\0')
 * and move everything left. */
      memmove(buf, buf + 1, buf_len - 2);
      buf[buf_len - 2] = c;
      /* first see if we got a setvar string */
      check_setvar(buf, buf_len, fd);
      /* See if we got it. */
      if(strncmp(expect, buf + buf_len - 1 - expect_len, expect_len) == 0)
      {
        nsyslog(LOG_DEBUG, "chat_expect - got it");
        return eNoErr;
      }
      else
      {
        const char *abort_msg = check_abort(buf, buf_len);
        if(abort_msg)
        {
          nsyslog(LOG_ERR, "initchat failed, ABORT \"%s\".", abort_msg);
          return eAbort;
        }
      }
    } /* end while */

    if(!timed_out)
    {
      nsyslog(LOG_DEBUG, "chat_expect(%s): interrupted", expect);
      return eInterrupt;
    }
    nsyslog(LOG_DEBUG, "chat_expect(%s): timeout (retry)", expect);
    return eTimeOut;
  }
  /*NOTREACHED*/
  return eNoErr;
}

static char *sub_expect_tok(char *buf, const char *end)
{
  char *str = buf;
  if(buf > end)
    return NULL;
  while(str < end && *str)
  {
    if(*str == '\\' && *(str + 1))
    {
      str += 2;
    }
    else
    {
      if(*str == '-')
        *str = '\0';
      else
        str++;
    }
  }
  return buf;
}

/* buf: temporary buffer for real_chat_expect() to use, it's easier to clean
 *      up memory in one place here than in multiple places in real_chat_expect
 * str: the string passed to real_chat_expect()
 */
static CHAT_ERR chat_expect(int fd, const char *expect)
{
  char *buf;
  char *str = xstrdup(expect);
  char *end = str + strlen(str);
  char *exp;
  CHAT_ERR rc;
  unsigned int buf_len;

  tstr(str, false, str);

  buf_len = MAX((strlen(expect) + 1), abort_setvar_max_len());
  buf = xmalloc(buf_len);
  exp = sub_expect_tok(str, end);
  rc = real_chat_expect(fd, buf, buf_len, exp);
  if(rc == eTimeOut)
  {
    exp = sub_expect_tok(exp + strlen(exp) + 1, end);
    if(exp)
    {
      chat_send(fd, exp);
      exp = sub_expect_tok(exp + strlen(exp) + 1, end);
      if(exp)
        rc = real_chat_expect(fd, buf, buf_len, exp);
    }
  }
  free(str);
  free(buf);
  alarm(0);
  return rc;
}

/*
 * Return its index in sw array of special chat word or -1 if not special
 */
static int get_special_index(const char *word)
{
  int i;

  for(i = 0; sw[i].word; ++i)
  {
    if (!strcmp(word, sw[i].word))
      return i;
  }
  return -1;
}

/*
 *  Given a string, converts it to integer and sets chat timeout
 */

static int set_chat_timeout(const char *str)
{
  char *end_conv;
  long int val;

  val = strtol(str, &end_conv, 10);
  if(*end_conv != 0)
  {
    nsyslog (LOG_DEBUG, "set_chat_timeout: bad integer %s", str);
    return -1;
  }
  if(val < 0)
  {
    nsyslog (LOG_DEBUG, "set_chat_timeout: negative timeout %ld", val);
    return -1;
  }
  SetChatTimeout((int)val);
  return 0;
}

/*
 * Implements special chat keyword "WAIT"
 */

static int special_chat_wait(const char *str)
{
  struct itimerval left;
  int TIO_flags;

  if(strcmp (str, "DCD"))
  {
    nsyslog (LOG_DEBUG, "special_chat_wait: bad wait parameter: %s", str);
    return (-1);
  }
  if(GetChatTimeout() == 0)
  {
    if(ioctl(0, TIOCMIWAIT, TIOCM_CD) == -1)
    {
      nsyslog(LOG_DEBUG, "special_chat_wait: ioctl"
        "(0, TIOCMIWAIT, TIOCM_CD): %d(%m)", errno);
      return (-1);
    }
  }
  else
  {
    signal (SIGALRM, SIG_IGN);
    alarm(GetChatTimeout());
    while(1)
    {
      if(getitimer(ITIMER_REAL, &left))
      {
        nsyslog(LOG_DEBUG, "special_chat_wait: getitimer: %d(%m)", errno);
        alarm(0);
        return -1;
      }
      if(ioctl(0, TIOCMGET, &TIO_flags))
      {
        nsyslog (LOG_DEBUG, "special_chat_wait:"
          "ioctl(0, TIOCMGET, ...): %d(%m)", errno);
        alarm(0);
        return -1;
      }
      if(TIO_flags & TIOCM_CAR)
        break;
      if((left.it_value.tv_sec == 0) && (left.it_value.tv_usec == 0))
      {
        nsyslog(LOG_DEBUG, "special_chat_wait: timed out");
        return -1;
      }
      xsleep(1);
    }
  }
  nsyslog(LOG_DEBUG, "Carrier detected");
  alarm(0);
  return 0;
}

/*
 *  This is the main chat loop;
 *  you have to feed it with an argument
 *  vector and a count.
 */
static int chatarray(int fd, int argc, const char * const * const argv, struct auth *ai)
{
/* ind: the special chat index, val is SPECIAL_NONE(-1), SPECIAL_TIMEOUT,
 *      SPECIAL_WAIT, SPECIAL_STATUS, or SPECIAL_ABORT.
 * c: the index to the argv[] array
 */
  int c;
  int ind;

  nsyslog(LOG_DEBUG, "chatarray: %d words", argc);

  special_chat_abort("CLEAN");
  special_chat_setvar("CLEAN", NULL);

  /*
   * Now do alternative expect-sends. Before each expect check if current
   * expect string is a special word.
   */
  for(c = 0; c != argc; )
  {
    ind = get_special_index(argv[c]);
    if(ind == -1)
    {
      CHAT_ERR rc;
      rc = chat_expect(fd, argv[c++]);
      if(rc != eNoErr)
        return rc;
      if(c == argc)
        break;
      chat_send(fd, argv[c++]);
    }
    else
    {
      nsyslog(LOG_DEBUG, "chatarray: special word: %s, "
        "index: %d, code: %d, req. words: %d",
        sw[ind].word, ind, sw[ind].code, sw[ind].req_words);
      if((argc - c) < sw[ind].req_words)
      {
        nsyslog(LOG_DEBUG, "Special chat word %s is missing parameter(s)",
          argv[c]);
        return -1;
      }
      switch (ind)
      {
        case SPECIAL_TIMEOUT:
          if(set_chat_timeout(argv[c+1]))
            return -1;
        break;
        case SPECIAL_WAIT:
          if(special_chat_wait(argv[c+1]))
            return -1;
        break;
        case SPECIAL_STATUS:
          update_utmp(argv[c+1], argv[c+2], ai, 0);
        break;
        case SPECIAL_ABORT:
          special_chat_abort(argv[c+1]);
        break;
        case SPECIAL_SETVAR:
          special_chat_setvar(argv[c+1], ai);
        break;
        default:
        break;
      }
      c += sw[ind].req_words;
    }
  }
  return 0;
}

/*
 *  This is a generic chat; you can
 *  just pass a string to it.
 *      The string will be split into send/expect pairs, if it becomes more
 *      than NUM_ARGS-2 parts then the rest will be ignored.
 */
#define NUM_ARGS 128
CHAT_ERR chat(int fd, const char *str, struct auth *ai)
{
/* args: pointers to an array of expect/send strings for the chat
 * s: copy of the input string str that is split and has args[] point to it
 * i: index to args[]
 * rc: return code from the real work
 */
  const char *args[NUM_ARGS];
  char *strbuf;
  char *strptr;
  int i = 0;
  CHAT_ERR rc;

  memset(args, 0, sizeof(args));
  if(str == NULL || str[0] == '\0')
    return eNoErr;

  strbuf = xstrdup(str);

  strptr = strbuf;

  /* Split string up in expect-send pairs. */
  while(*strptr && (i + 1) < NUM_ARGS)
  {
    if(*strptr == ' ' || *strptr == '\t' || i == 0)
    {
      if(i)
      {
        /* End of a string. */
        *strptr++ = '\0';
      }
      /* Skip until next non-space. */
      while(*strptr == ' ' || *strptr == '\t')
        strptr++;
      if(*strptr == '\0')
        continue;
      args[i++] = strptr;
    }
    if(*strptr == '"')
    {
      if(strptr != args[i - 1])
      {
        nsyslog(LOG_ERR, "Quote not at begin of string in chat.");
        return eStupidAdmin;
      }
      args[i - 1]++;  /* Don't include the quote in the string */
      strptr++;
      while(*strptr && *strptr != '"')
      {
        if(*strptr == '\\' && *(strptr+1))
          strptr++;
        strptr++;
      }
      if(*strptr != '"')
      {
        nsyslog(LOG_ERR, "Unterminated quote in chat.");
        return eStupidAdmin;
      }
      *strptr = '\0';
      strptr++;
    }
    else
    {
      if(*strptr)
        strptr++;
    }
  }
  args[i] = NULL;

  /* Now call the real routine. */
  rc = chatarray(fd, i, args, ai);
  free(strbuf);
  return rc;
}

