/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxmass-ui-seqed-widget-mnm-modif.h"
#include "polyxmass-ui-seqed-widget.h"
#include "polyxedit-ui-masses-display-wnd.h"
#include "polyxmass-pixbuf-rendering.h"



/************** HIGHL LEVEL GUI MONOMER MODIFICATION FUNCTIONS ***************/

GtkWidget *
polyxmass_seqed_widget_mnm_modif_wnd_setup (GtkWidget *seqed_widget)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeSelection* selection = NULL; 
  
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  PxmEditCtxt *editctxt = NULL;
    
  PxmMonomer *monomer = NULL;
  
  GladeXML *xml = NULL;

  gint idx = -1;
  gint start = -1;
  gint end = -1;
  

  gchar *gui_file = NULL;
  gchar *help = NULL;

  g_assert (seqed_widget != NULL);

  polchemdefctxt = PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxedit-monomer-modif.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "monomer_modif_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "monomer_modif_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the monomer "
		    "modification window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);

  /* IMMEDIATELY set the pointer to the seqed_widget as a datum to the
     window. We will need this in function that only get the pointer
     to the window.
   */
  g_object_set_data (G_OBJECT (window), "seqed_widget", seqed_widget);
  

  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polseq_prop_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_prop_vbox", widget);
  

  
  /* Are we setting up this window for a polymer sequence window, in which case we 
     should have a edictxt pointer datum available:
  */
  editctxt = g_object_get_data (G_OBJECT (seqed_widget), "editctxt");
  
  if (editctxt != NULL)
    {
      /* Set the name of the polymer sequence to its correspondent GtkEntry.
       */
      widget = glade_xml_get_widget (xml, "polseq_name_entry"); 
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window), 
			 "polseq_name_entry", widget);
      
      g_assert (editctxt->polymer->plminfo->name != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), 
			  editctxt->polymer->plminfo->name);
      
      /* Set the code of the polymer sequence to its correspondent GtkEntry.
       */
      widget = glade_xml_get_widget (xml, "polseq_code_entry"); 
      g_assert (widget != NULL);
      g_object_set_data (G_OBJECT (window), 
			 "polseq_code_entry", widget);
      
      g_assert (editctxt->polymer->plminfo->code != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), 
			  editctxt->polymer->plminfo->code);
      
      
      /* Set the identity number (actually the seqed_widget pointer) to 
       * its GtkEntry (this is useful when cleavages are made to identify
       * unambiguously the polymer from which results are displayed).
       */
      widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
      g_object_set_data (G_OBJECT (window), 
			 "polseq_id_number_entry",
			 widget);
      help = g_strdup_printf ("%p", seqed_widget);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);
    }
  else
    {
      /* We just hide the stuff there.
       */
      gtk_widget_hide (widget);
    }
  
  
  /* Now deal systematically with the widgets that we will use later.
   */
  widget = glade_xml_get_widget (xml, 
				 "selected_monomer_code_entry");
  g_object_set_data (G_OBJECT (window), 
		     "selected_monomer_code_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "selected_monomer_position_entry");
  g_object_set_data (G_OBJECT (window), 
		     "selected_monomer_position_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "available_modifications_treeview");
  g_object_set_data (G_OBJECT (window), 
		     "available_modifications_treeview",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_selected_monomer_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_selected_monomer_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_same_code_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_same_code_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_from_list_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_from_list_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_all_monomers_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_all_monomers_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_spec_pos_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_spec_pos_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_spec_pos_entry");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_spec_pos_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_treeview");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_treeview",
		     widget);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
  
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

  /* The textview where errors are to be displayed.
   */
  widget = glade_xml_get_widget (xml, 
				 "monomer_modification_errors_textview");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_errors_textview",
		     widget);
  
  /* The buttons are connected through signals to related functions.
   */
  widget = 
    glade_xml_get_widget (xml, "monomer_modification_modify_button");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_modify_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxmass_seqed_widget_mnm_modif_wnd_modify_button), 
		    window);

  widget = 
    glade_xml_get_widget (xml, "monomer_modification_un_modify_button");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_un_modify_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxmass_seqed_widget_mnm_modif_wnd_un_modify_button),
		    window);


  gtk_widget_show_all (GTK_WIDGET (window));
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* And now that we know all the window is set up, we can fill the
     data in it: the list of available modifications, the list of the
     monomers in the polymer chemistry definition context, the
     position of the possibly selected monomer...
  */

  /* First the index of the currently selected monomer. Function returns
     a pointer to a monomer if only one is selected, and the index
     of the selected monomer into idx.
   */
  idx = -1;
  monomer = polyxmass_seqed_widget_get_selected_monomer (seqed_widget, &idx);
  
  if (monomer != NULL)
    {
      /* If a single monomer is selected, then its idx cannot be -1.
       */
      g_assert (idx != -1);
      
      /* We can print the relevant info in their entries.
       */
      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_code_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), monomer->code);
      
      /* Print the position of the selected monomer, not its index.
       */
      help = g_strdup_printf ("%d", idx + 1);

      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_position_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), help);

      g_free (help);
    }
  else
    {
      /* We can print the relevant explanations the entry.
       */
      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_code_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget),
			  _("Not a single monomer selected"));

      /* Get the indices of the selection if there is one.
       */
      if (FALSE != 
	  polyxmass_seqed_widget_get_selection_indices (seqed_widget,
							&start, &end))
	{
	  /* Remember that due to the handling of the polymer sequence
	     displayed in the sequence editor, the end value that is 
	     returned is indeed corresponding to the real index of the last
	     selected monomer + 1, so care should be taken not to outbound
	     the polymer sequence array.
	  */
	  
	  /* We show the (start + 1) value because we want to show
	     a position and not an index.
	  */
	  help = 
	    g_strdup_printf ("[%d..%d]", start + 1, end /* NOT + 1 */);
	  
	  /* Print the position of the selection, not its index.
	   */
	  widget = g_object_get_data (G_OBJECT (window), 
				      "selected_monomer_position_entry");
	  g_assert (widget != NULL);
	  
	  gtk_entry_set_text (GTK_ENTRY (widget), help);
	  
	  g_free (help);
	}
      else
	{
	  /* Print that there is no selection.
	   */
	  widget = g_object_get_data (G_OBJECT (window), 
				      "selected_monomer_position_entry");
	  g_assert (widget != NULL);
	  
	  gtk_entry_set_text (GTK_ENTRY (widget), _("No selection"));
	  
	}

      /* Note that if there is no single monomer selected, it makes no
	 sense to have the radio button "target_selected_monomer_radiobutton"
	 selected. We set it to the other one, where the user selects the
	 positions with regex-like strings.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "target_monomers_spec_pos_radiobutton");
      g_assert (widget != NULL);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  /* Second fill the treeview of modifications avaialable to the
     polymer chemistry definition context of that specific polymer
     sequence.
  */
  treeview = 
    GTK_TREE_VIEW (g_object_get_data (G_OBJECT (window), 
				      "available_modifications_treeview"));

  if (FALSE == 
      polyxmass_seqed_widget_mnm_modif_wnd_setup_modifs_treeview (treeview,
								  polchemdef))
    {
      g_critical (_("%s@%d: failed to fill the modifications list\n"),
		  __FILE__, __LINE__);
    }
  
  /* Third fill the treeview of all the monomer codes present in the
     polymer chemistry definition context of that specific polymer
     sequence.
  */
  treeview = 
    GTK_TREE_VIEW (g_object_get_data (G_OBJECT (window), 
				      "target_monomers_treeview"));
  
  if (FALSE == 
      polyxmass_seqed_widget_mnm_modif_wnd_setup_monomcodes_treeview (treeview,
								      polchemdef))
    {
      g_critical (_("%s@%d: failed to fill the monomer codes list\n"),
		  __FILE__, __LINE__);
    }
  
  /* The signal of the window itself.
   */
  /* Signal / callback connections.
   */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxmass_seqed_widget_mnm_modif_wnd_delete_event), 
     seqed_widget);

  g_signal_connect 
    (G_OBJECT (window),
     "destroy_event",
     G_CALLBACK (polyxmass_seqed_widget_mnm_modif_wnd_destroy_event), 
     seqed_widget);
  
  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one monomer modification window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("monomer_modif_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (seqed_widget),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxmass_seqed_widget_mnm_modif_wnd_really_close);
  
  g_free (help);
 
  return window;
}


gboolean
polyxmass_seqed_widget_mnm_modif_wnd_setup_modifs_treeview (GtkTreeView *treeview,
							    PxmPolchemdef *polchemdef)
{
  GPtrArray *GPA = NULL;

  GtkCellRenderer *renderer = NULL;
  GtkListStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;

  PxmModif *modif;


  g_assert (polchemdef != NULL);
  GPA = polchemdef->modifGPA;
  
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MDF_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  modif = g_ptr_array_index (GPA, iter);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MDF_NAME, 
			      modif->name,
			      
			      COLUMN_MDF_ACTFORM, 
			      modif->actform,
			      
			      COLUMN_MDF_EDITABLE, FALSE,
			      -1);
	}
    }
  
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (model));  

  g_object_unref (G_OBJECT (model));

  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  

  /* Modif Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  /*
    GtkTreeViewColumn *column = NULL;
    
    column = 
    gtk_tree_view_column_new_with_attributes (_("Name"), 
    renderer,
    "text", COLUMN_MDF_NAME,
    NULL); 
    
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  */
  
  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MDF_NAME,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_NAME);


  /* Modif Actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Actform"), renderer,
     "text", COLUMN_MDF_ACTFORM,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_ACTFORM);

  return TRUE;
}


gboolean
polyxmass_seqed_widget_mnm_modif_wnd_setup_monomcodes_treeview (GtkTreeView *treeview,
						      PxmPolchemdef *polchemdef)
{
  GPtrArray *GPA = NULL;

  GtkCellRenderer *renderer = NULL;
  GtkListStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;

  PxmMonomer *monomer;



  g_assert (polchemdef != NULL);
  GPA = polchemdef->monomerGPA;
  
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING, /* monomer name */
			      G_TYPE_STRING, /* monomer code */
			      G_TYPE_STRING /* monomer formula */,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      COLUMN_MNM_CODE, 
			      monomer->code,
			      
 			      COLUMN_MNM_FORMULA, 
			      monomer->formula,

			      COLUMN_MNM_EDITABLE, FALSE,
			      -1);
	}
    }

  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (model));  

  g_object_unref (G_OBJECT (model));

  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);
  

  /* Monomer Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  /*
    GtkTreeViewColumn *column = NULL;
    
    column = 
    gtk_tree_view_column_new_with_attributes (_("Name"), 
    renderer,
    "text", COLUMN_MNM_NAME,
    NULL); 
    
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  */
  
  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MNM_NAME,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_NAME);


  /* Monomer Code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Code"), renderer,
     "text", COLUMN_MNM_CODE,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_CODE);


  /* Monomer Formula column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Formula"), renderer,
     "text", COLUMN_MNM_FORMULA,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_FORMULA);
  
  return TRUE;
}


void
polyxmass_seqed_widget_mnm_modif_wnd_modify_button (GtkWidget *button,
						    gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;
  
  gint pos_count = -1;
  
  gint full_success = 0;
  gint render_failure = 0;
  gint full_failure = 0;
  gint chem_failure = 0;
  
  GtkTreeView * treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection* selection = NULL; 
  GtkTreeModel *model = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter;
  GtkTextBuffer *buffer = NULL;

  GArray *GA = NULL;

  gchar *modif_name = NULL;
  gchar *errors = NULL;
  gchar *help = NULL;
  


  g_assert (window != NULL);
  
  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  
  treeview = g_object_get_data (G_OBJECT (window), 
				"available_modifications_treeview");
  
  /* Ensure right away that there is one modification selected.
   */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  selection = gtk_tree_view_get_selection (treeview);
  
  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&treeiter))
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) window,
	 _("Select a modification to apply"),
	 POLYXMASS_NORM_MSG_TIMEOUT);
    
      return; /*no selection, just return. */
    }
  

  /* At this point treeiter points to the selected item in the treeview.
     This means that we must get the name out of the item.
   */
  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the treeiter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_MDF_NAME, &modif_name,
		      -1);

  g_assert (modif_name != NULL); /* Free it later.*/
  
  /* We now have the name of the modif that is selected in the
     treeview. We can go on and allocate the GArray that will hold
     all the positions where the modification has to occur. These
     positions will be analyzed in another function.
   */
  GA = g_array_new (TRUE, TRUE, (guint) sizeof (gint));

  pos_count = 
    polyxmass_seqed_widget_mnm_modif_wnd_prepare_indices_array (window,
								GA);
  
  if (-1 == pos_count)
    return;

  /*
    debug_printf (("number of positions to modify: %d\n", pos_count));
  */
  
  /* At this stage we have an array of integers that represent all the
     indices at which the monomers of the polymer sequence should be
     modified using the modification that we got previously in the
     modif_name variable (see above). Time has come to perform the
     modification proper.
   */
  polyxmass_seqed_widget_mnm_modif_modify_monomer_GA (GA,
						      modif_name,
						      seqed_widget, 
						      &errors,
						      &full_success,
						      &full_failure,
						      &chem_failure,
						      &render_failure);
  
  g_free (modif_name);
  
  /* If at least one chemical modification was performed, irrespective
     of the graphical rendering, the polymer sequence is modified.
  */
  if (render_failure != 0 || chem_failure != 0 || full_success != 0)
    {
      polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						       TRUE);
    }
  
  help = g_strdup_printf (_("Full success = %d; Full failure = %d; "
			    "Chem failure = %d; Render failure = %d"),
			  full_success, full_failure, 
			  chem_failure, render_failure);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					 help, 
					 POLYXMASS_LONG_MSG_TIMEOUT);
  
  g_free (help);
  

  /* If we got a string back in errors of length > 0, display its
     contents in the errors textview.
  */
  if (errors != NULL)
    {
      textview = g_object_get_data (G_OBJECT (window),
				    "monomer_modification_errors_textview");
      g_assert (textview != NULL);
      
      buffer = gtk_text_view_get_buffer (textview);
      
      /* Get start of buffer.
       */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter, 0);
      
      gtk_text_buffer_insert (buffer, &text_iter, errors, -1);
      
      g_free (errors);
    }
  
  /* Finally free the whole indices' GArray.
   */
  g_array_free (GA, TRUE);

  
  /* The selection polygon (if there was any around monomer(s) which
     have been modified) may have been partically "overwritten".
  */
  polyxmass_seqed_widget_update_sel_polygon (seqed_widget);
    

  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (seqed_widget), 
						      PXM_MASSCALC_TARGET_BOTH);
  
  return;
}

void
polyxmass_seqed_widget_mnm_modif_wnd_un_modify_button (GtkWidget *button,
						       gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *seqed_widget = NULL;

  gint pos_count = -1;
  
  gint full_success = 0;
  gint render_failure = 0;
  gint full_failure = 0;
  gint chem_failure = 0;
  
  GtkTreeView * treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection* selection = NULL; 
  GtkTreeModel *model = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter;
  GtkTextBuffer *buffer = NULL;

  GArray *GA = NULL;

  gchar *modif_name = NULL;
  gchar *errors = NULL;
  gchar *help = NULL;
  


  g_assert (window != NULL);
  
  seqed_widget = g_object_get_data (G_OBJECT (window), "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), 
				      "available_modifications_treeview");
  
  /* Check if there is one modification selected. Indeed, the
     processing here is a bit different than for the modification:
     there can be no modification selected. If such is the case, that
     means that the un-modification processing will be performed
     whatever the current modification of the target monomers as
     determined later by the call to
     polyxmass_seqed_widget_mnm_modif_wnd_prepare_indices_array ().  Instead, if
     a modification is selected, then the un-modification will only
     effectively be performed on the monomers from the indices GArray
     that are modified with this modification precisely.
   */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  selection = gtk_tree_view_get_selection (treeview);
  
  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&treeiter))
    {
      /* No modification is selected. The un-modification will be
	 performed on absolutely all the monomers at the indices
	 obtained by calling
	 polyxmass_seqed_widget_mnm_modif_wnd_prepare_indices_array ().
      */
    }
  else
    {
      /* At this point treeiter points to the selected item in the treeview.
	 This means that we must get the name out of the item.
      */
      /* Make sure you terminate calls to gtk_tree_model_get()
	 with a '-1' value. We can do this call because we have the treeiter
	 pointing to the selected node!
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_MDF_NAME, &modif_name,
			  -1);
      
      g_assert (modif_name != NULL); /* Free it later.*/
    }
  
  /* Now modif_name is either NULL or points to an allocated string
     containing the modification name.
  */

  /* We now have the name of the modif that is selected in the
     treeview. We can go on and allocate the GArray that will hold
     all the positions where the modification has to occur. These
     positions will be analyzed in another function.
   */
  GA = g_array_new (TRUE, TRUE, (guint) sizeof (gint));

  pos_count = 
    polyxmass_seqed_widget_mnm_modif_wnd_prepare_indices_array (window,
								GA);
  
  if (-1 == pos_count)
    return;

  /*
    debug_printf (("number of positions to un-modify: %d\n", pos_count));
  */

  
  /* At this stage we have an array of integers that represent all the
    indices at which the monomers of the polymer sequence should be
    modified using the modification that we got previously in the
    modif_name variable (see above). Time has come to perform the
    modification proper.
   */
  polyxmass_seqed_widget_mnm_modif_un_modify_monomer_GA (GA,
							 modif_name,
							 seqed_widget, 
							 &errors,
							 &full_success,
							 &full_failure,
							 &chem_failure,
							 &render_failure);
  
  if (modif_name != NULL)
    g_free (modif_name);
  /* If at least one chemical modification was performed, irrespective
     of the graphical rendering, the polymer sequence is modified.
  */
  if (render_failure != 0 || chem_failure != 0 || full_success != 0)
    polyxmass_seqed_widget_signal_sequence_modified (PXM_SEQED_WIDGET (seqed_widget),
						     TRUE);
  
    /*
      old 
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
    */

  
  help = g_strdup_printf (_("Full success = %d; Full failure = %d; "
			    "Chem failure = %d; Render failure = %d"),
			  full_success, full_failure, 
			  chem_failure, render_failure);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				    help, 
				    POLYXMASS_NORM_MSG_TIMEOUT);
  
  g_free (help);
  

  /* If we got a string back in errors of length > 0, display its
     contents in the errors textview.
  */
  if (errors != NULL)
    {
      textview = g_object_get_data (G_OBJECT (window),
				    "monomer_modification_errors_textview");
      g_assert (textview != NULL);
      
      buffer = gtk_text_view_get_buffer (textview);
      
      /* Get start of buffer.
       */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter, 0);
      
      gtk_text_buffer_insert (buffer, &text_iter, errors, -1);
      
      g_free (errors);
    }
  
  /* Finally free the whole indices' GArray.
   */
  g_array_free (GA, TRUE);

  
  /* The selection polygon (if there was any around monomer(s) which
     have been modified) may have been partically "overwritten".
  */
  polyxmass_seqed_widget_update_sel_polygon (seqed_widget);
    

  /* The sequence and the selection.
   */
  polyxmass_seqed_widget_signal_mass_update_required (PXM_SEQED_WIDGET (seqed_widget), 
						      PXM_MASSCALC_TARGET_BOTH);
  return;
}



gint
polyxmass_seqed_widget_mnm_modif_wnd_prepare_indices_array (GtkWidget *window,
						  GArray *GA)
{
  gint iter = 0;
  gint max = 0;
  gint target = -1;
  
  gboolean valid = FALSE;
  gboolean selected = FALSE;

  gchar *code = NULL;
  gchar *text = NULL;
  
  GtkWidget *entry = NULL;

  GtkTreeView * treeview = NULL;
  GtkTreeSelection* selection = NULL; 
  GtkTreeIter treeiter;
  GtkTreeModel *model = NULL;

  GPtrArray *monomerGPA = NULL;

  PxmMonomer *monomer_iter = NULL;
  PxmMonomer *monomer_sel = NULL;

  PxmSeqedWidget *seqed_widget = NULL;


  g_assert (window != NULL);
  g_assert (GA != NULL);

  
  seqed_widget = g_object_get_data (G_OBJECT (window), 
				    "seqed_widget");
  g_assert (seqed_widget != NULL);
  
  monomerGPA = PXM_SEQED_WIDGET (seqed_widget)->polymer->monomerGPA;
  g_assert (monomerGPA != NULL);
  
  
  /* Get the target of the modification operation.
   */
  target = polyxmass_seqed_widget_mnm_modif_wnd_get_target (window);

  if (target == MODIF_TARGET_ALL_MONOMERS)
    {
      /* Each single monomer in the sequence must be modified! Thus
	 put in the array of all the monomer indices for which
	 modification should occur all the indices of the polymer! That
	 is so easy!
      */
      for (iter = 0; iter < monomerGPA->len; iter++)
	GA = g_array_append_val (GA, iter);
    }

  else if (target == MODIF_TARGET_SEL_MONOMER
	   || target == MODIF_TARGET_ALL_MONOMERS_CODE)
    {
      /* Modification should be applied on the selected monomer or on
	 all the monomers of the sequence that have the same code as
	 the selected one. Which means for both cases only one monomer
	 should currently be selected in the sequence editor. Reuse
	 the iter variable to get the index of the putatively selected
	 monomer: that's recycling!
       */
      iter = -1;
      monomer_sel = polyxmass_seqed_widget_get_selected_monomer (GTK_WIDGET (seqed_widget),
							      &iter);

      if (monomer_sel == NULL)
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
						 _("Select a single monomer"),
						 POLYXMASS_NORM_MSG_TIMEOUT);
	  return -1;
	}
      
      /* A selected monomer may not have an index = -1. Good sanity
	 check.
       */
      g_assert (iter != -1);
      
      /* We can handle the modification right away.
       */
      if (target == MODIF_TARGET_SEL_MONOMER)
	{
	  /* Only the monomer that is currently selected should be
	     modified. And we just finished getting its index, so that
	     we can put it in the array of the indices (actually one
	     only in our present case) of the monomers to be modified.
	   */
	  GA = g_array_append_val (GA, iter);
	}
      
      /* If it is not SEL_MONOMER, then it is ALL_MONOMERS_CODE!
       */
      else 
	{
	  /* All monomers that have the same code as the one which is
	     currently selected should be modified. We have the code
	     right away because we have the monomer!
	   */

	  /* Iterate in the polymer sequence and for each monomer
	     check if it has the same code as the one we have at
	     hand. If so, just add the index to the array of positions
	     that should be modified.
	  */
	  for (iter = 0; iter < monomerGPA->len; iter++)
	    {
	      monomer_iter = g_ptr_array_index (monomerGPA, iter);
	      
	      if (0 == strcmp (monomer_sel->code, monomer_iter->code))
		GA = g_array_append_val (GA, iter);
	    }
	} 
    }
  /* end of
     else if (target == MODIF_TARGET_SEL_MONOMER
     || target == MODIF_TARGET_ALL_MONOMERS_CODE)
  */

  else if (target == MODIF_TARGET_ALL_MONOMERS_LIST)
    {
      /* The code(s) of the monomers for which the modification 
       * is to be applied is(are) currently selected in the 
       * codes' treeview. We'll have to check that the list actually 
       * contains selected items:
       */
      
      /* Get the treeview widget in which the user may select no/one/more
	 monomer to be modified.
      */
      treeview = g_object_get_data (G_OBJECT (window), 
					    "target_monomers_treeview");
      
      model = gtk_tree_view_get_model (treeview);
      g_assert (model != NULL);
      
      /* Remember that we have a multiple-selection treeview here,
	 conversely to what was for the modification treeview. This is
	 why we cannot ask for the selection as we did in the modif
	 treeview code above.
      */
      selection = gtk_tree_view_get_selection (treeview);
      g_assert (selection != NULL);
      
      /* Get the first treeiter in the list.
       */
      valid = gtk_tree_model_get_iter_first (model, &treeiter);
      while (valid)
	{
	  /* Walk through the treeveiw list of monomers , reading each
	     row, only if actually selected.
	   */
	  selected = gtk_tree_selection_iter_is_selected (selection, 
							  &treeiter);
      
	  if (selected != TRUE)
	    {
	      /* Go the the next treeview item.
	       */
	      valid = gtk_tree_model_iter_next (model, &treeiter);
	      
	      continue;
	    }
	  	  
	  /* The row is selected, thus we can ask that the contents of
	     the CODE column be allocated in 'code'. Make sure to
	     terminate the call to gtk_tree_model_get() * with a '-1'
	     value.
	  */
	  gtk_tree_model_get (model, &treeiter, 
			      COLUMN_MNM_CODE, &code,
			      -1);
	  
	  /* We have a monomer code allocated in code, that we should
	     compare with the monomer code of each monomer iterated in
	     the polymer's monomerGPA array. If there is a match
	     that means that the iter value should be stored in the
	     array of all the positions where the modification should
	     occur.
	  */
	  for (iter = 0; iter < monomerGPA->len; iter++)
	    {
	      monomer_iter = g_ptr_array_index (monomerGPA, iter);
	      
	      if (0 == strcmp (monomer_iter->code, code))
		GA = g_array_append_val (GA, iter);
	      
	    }

	  /* Finished doing the comparison work with the currently 
	     iterated list's monomer code 'code'. We have to free it
	     because the gtk_tree_model_get () call above allocated it.
	  */
	  g_free (code);
	  
	  /* And now go the the next treeview item.
	   */
	  valid = gtk_tree_model_iter_next (model, &treeiter);
	}
    }
  /* end of
     else if (target == MODIF_TARGET_ALL_MONOMERS_LIST)
  */
  
  else if (target == MODIF_TARGET_SPEC_POS_MONOMERS)
    {
      /* The user should have written some positions in the entry.
       */
      entry = g_object_get_data (G_OBJECT (window), 
				 "target_monomers_spec_pos_entry");
      g_assert (entry != NULL);
      
      text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
      g_assert (text != NULL);
      
      /* We should make sure that there is no single
	 space/tab/whatever character in the string that we got.
       */
      text = libpolyxmass_globals_unspacify_string (&text);

      if (strlen (text) <= 0)
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					     _("Enter valid position(s)"),
					     POLYXMASS_NORM_MSG_TIMEOUT);

	  g_free (text);
	  
	  return -1;
	}
      
      max = PXM_SEQED_WIDGET (seqed_widget)->polymer->monomerGPA->len - 1;
      
      if (-1 == libpolyxmass_globals_fill_int_array_match_exp (text, GA, max))
	g_critical (_("%s@%d: failed to process string: '%s'\n"),
	       __FILE__, __LINE__, text);
      
      g_free (text);
    }
  /* end of 
     else if (target == MODIF_TARGET_SPEC_POS_MONOMERS)
  */

  return GA->len;
}


gint
polyxmass_seqed_widget_mnm_modif_wnd_get_target (GtkWidget *window)
{
  GSList *group = NULL;
  GtkWidget *radio = NULL;
  GSList *iter = NULL;
  
  
  g_assert (window != NULL);
  
  radio = g_object_get_data (G_OBJECT (window), 
			     "target_selected_monomer_radiobutton");
  g_assert (radio != NULL);
  
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  
  iter = group;
  
  
  while (iter != NULL)
    {
      radio = iter->data;
      
      if (TRUE == 
	  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)))
	{
	  if (0 == strcmp ("target_selected_monomer_radiobutton", 
		      gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_SEL_MONOMER;
	  
	  else if  (0 == strcmp ("target_monomers_same_code_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS_CODE;

	  else if  (0 == strcmp ("target_monomers_from_list_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS_LIST;

	  else if  (0 == strcmp ("target_all_monomers_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS;

	  else if  (0 == strcmp ("target_monomers_spec_pos_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_SPEC_POS_MONOMERS;

	  else
	    g_warning (_("%s@%d: failed to get the radiobutton widget\n"),
		   __FILE__, __LINE__);
	}

      iter = g_slist_next (iter);
    }
  return -1;
}



/************** LOW LEVEL NON-GUI MONOMER MODIFICATION FUNCTIONS ***************/

PxmModifRes
polyxmass_seqed_widget_mnm_modif_modify_monomer (PxmMonomer *mnm,
						 gint idx,
						 gchar *modif,
						 GtkWidget *seqed_widget)
{
  GdkPixbuf *pixbuf = NULL;

  PxmMonomer *monomer = NULL;

  PxmPolymer *polymer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  PxmMonicon *monicon_old = NULL;
  PxmMonicon *monicon_new = NULL;




  g_assert (modif != NULL);

  g_assert (seqed_widget != NULL);

  polchemdefctxt = PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);
  

  /* Modifying a monomer is a two-step process: first the monomer must be 
     chemically modified, and next, if the modification is successful,
     the graphical rendering must be performed so that the modification
     can be visualized in the sequence editor.
  */

  /* Sanity checks pertaining to the size of the polymer sequence.
   */
  g_assert (polymer->monomerGPA->len > 0);
  g_assert (idx >= 0);
  g_assert (idx <= polymer->monomerGPA->len - 1);

  /* Make sure we can correctly access the monomer to be modified.
   */
  if (mnm == NULL)
    monomer = g_ptr_array_index (polymer->monomerGPA, idx);
  else
    monomer = mnm;
  
  g_assert (monomer != NULL);
  
  /* Make sure the modif object is known to the polchemdef.
   */
  if (-1 == pxmchem_modif_get_index_by_name (modif, polchemdef->modifGPA))
    {
      g_critical (_("%s@%d: modif is not known to the polchemdef: '%s'\n"),
	     __FILE__, __LINE__, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* If the monomer is already modified somehow, the function below
     cleanly unmodifies it prior to modifying it with 'modif'. Note
     that the function below only does non-graphics stuff. We'll have
     to manage the rendering below.
  */
  if (FALSE == pxmchem_monomer_modify (monomer, modif))
    {
      g_critical (_("%s@%d: failed to modify monomer at index: '%d' "
	       "with modif: '%s'\n"),
	     __FILE__, __LINE__, idx, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* At this point the chemical modification went ok, so we can manage
     the rendering process. In a prudent way of doing things, we start
     by making the new rendered monicon, and if everything goes well
     we make the replacement, otherwise we leave the current monicon
     in place but issue a critical message.
  */

  /* Render a new pixbuf corresponding to the modified monomer.
   */
  pixbuf = 
    polyxmass_pixbuf_rendering_render_modif (monomer->code,
					     modif,
					     PXM_SEQED_WIDGET (seqed_widget)->monicon_size,
					     &(PXM_SEQED_WIDGET (seqed_widget)->pixbufGData),
					     polchemdefctxt);
  
  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed to render monicon for monomer: '%s' "
	       "at index: '%d' with modif: '%s'. Chemical modification "
	       "was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx, modif);
	  
      return MODIF_CHEM_Y_RENDER_N;
    }

  /* Now that we have the pixbuf, we can allocate the new monicon so that
     a new canvas_item is prepared later.
  */
  monicon_new = polyxmass_monicon_new ();

  if (FALSE == polyxmass_pixbuf_rendering_monicon_make_canvas_item (monicon_new, 
								    PXM_SEQED_WIDGET (seqed_widget)->
								    monicon_size,
								    pixbuf,
								    idx,
								    seqed_widget))
    {
      g_critical (_("%s@%d: failed to make canvas_item for monomer code: '%s' "
		    "at index: '%d', with modif: '%s'. Chemical modification "
		    "was performed ok, however.\n"),
		  __FILE__, __LINE__, monomer->code, idx, modif);
      
      return MODIF_CHEM_Y_RENDER_N ;
    }


  /* At this point, the pixbuf that we got in the first place, with
     the call to polyxmass_pixbuf_rendering_render_no_modif() got
     referenced by the gnome_canvas_item_new() that makes the canvas
     item in the polyxmass_pixbuf_rendering_monicon_make_canvas_item()
     call. Thus, when we call the pxm_monicon_free() function on the
     monicon, it should just 

     gtk_object_destroy(GTK_OBJECT(monicon->canvas_item)); 

     which will unref the pixbuf contained in it.

     So, the story goes : 

     1. we get a pixbuf (either pre-existing or allocated anew) with
     no reference count incrementing.
	
     2. we construct a canvas_item by using it, which increments its
     reference count by 1.

     3. we put that pixbuf into a monicon object that will contain it
     up to its freeing.

     4. some day we free the monicon, which will gtk_object_destroy
     the canvas_item, which will decrement the reference count by
     1. At this stage we get a reference count identical to the
     situation in point 1.
  */


  
  /* Apparently we succeeded in all our attempts, so get the pointer
     to the monicon that currently renders the monomer. Remove it from
     the array of monicons and free it.
  */
  monicon_old = 
    g_ptr_array_remove_index (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA, idx);
  g_assert (monicon_old != NULL);

  polyxmass_monicon_free (monicon_old);


  /* At this point we have a fully qualified monicon. Depending on the
     value of idx, with respect to the size of the moniconGPA we
     either insert the monicon in the sequence array or we append it.
     We cannot use the polymer->monomerGPA->len value, because it does
     not change, while moniconGPA->len has changed as the effect of
     removing the old monicon from it. I was having crashes due using
     the polymer->monomerGPA->len value here. Which was wrong of
     course.
   */
  if (idx == PXM_SEQED_WIDGET (seqed_widget)->moniconGPA->len)
    g_ptr_array_add (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA, 
		     monicon_new);
  else
    g_ptr_array_insert_val (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA, 
			    idx, monicon_new);

  return MODIF_CHEM_Y_RENDER_Y;
}


/* It is the responsibility of the caller to set the parameters to
   values that she will use to establish if the function was 
   successful or not.
*/
void
polyxmass_seqed_widget_mnm_modif_modify_monomer_GA (GArray *GA,
						    gchar *modif,
						    GtkWidget *seqed_widget,
						    gchar **errors,
						    gint *full_success,
						    gint *full_failure,
						    gint *chem_failure,
						    gint *render_failure)
{
  gint iter = 0;
  gint idx = 0;

  GString *gs = NULL;
  
  PxmModifRes res = MODIF_CHEM_N_RENDER_N;
  
  GPtrArray *monomerGPA = NULL;

  PxmPolymer *polymer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

 


  /* We have a GArray (GA) of indices corresponding to indices in the
     polymer sequence where the modification (its name is in modif)
     should occur.
   */


  g_assert (GA != NULL);
  /* If no monomer is to be modified just return 0 since we do not
     modify any monomer !
   */
  if (GA->len <= 0)
    return ;
    
  g_assert (seqed_widget != NULL);

  polchemdefctxt = PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  monomerGPA = polymer->monomerGPA;
  g_assert (monomerGPA != NULL);
  /* If the polymer sequence is empty, no modif can occur!
   */
  if (monomerGPA->len <= 0)
    return ;
  
  g_assert (modif != NULL);
  g_assert (-1 != pxmchem_modif_get_index_by_name (modif,
						   polchemdef->modifGPA));
  
  /* If errors is set, we want errors to be a pointer to a NULL gchar
     pointer. We will allocate the string ourselves and let the caller 
     free it after.
   */
  if (errors != NULL)
    {
      g_assert (*errors == NULL);
      
      /* Prepare the GString that will store the errors, but only if 
	 error reporting is asked.
      */
      gs = g_string_new ("");
    }
    
  /* We now should iterate in the array of indices of the monomers to
     be modified and take proper actions each time.
  */
  for (iter = 0; iter < GA->len; iter++)
    {
      idx = g_array_index (GA, gint, iter);
      
      /* Check that we are not going to access a non-existent
	 monomer... especially if the user edits the polymer sequence
	 by removing monomers from it.
      */
      if (idx > monomerGPA->len)
	{
	  g_critical (_("%s@%d: a position marked to be modified is greater "
		   "than the size of the polymer! "
		   "Do not edit the sequence!\n"),
		 __FILE__, __LINE__);
	  
	  return ;
	}
      
      /* Do the modification work proper.
       */
      res = polyxmass_seqed_widget_mnm_modif_modify_monomer (NULL, idx, modif,
							     seqed_widget);
      
      /* Now, a number of possibilities need to be envisaged, and errors
	 will have to be logged. Since the first operation is to 
	 make the chemical modification, if it goes wrong the function
	 returns immediately with MODIF_CHEM_N_RENDER_N.
      */
      switch (res)
	{
	case MODIF_CHEM_Y_RENDER_Y:
	  if (full_success != NULL)
	    (*full_success)++;
      	  
	  break;

	case MODIF_CHEM_N_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not modified - no visual\n"),
	       iter + 1);

	  if (full_failure != NULL)
	    (*full_failure)++;
	  
	  break;

	case MODIF_CHEM_Y_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - modified - no visual\n"),
	       iter + 1);
	  
	  if (render_failure != NULL)
	    (*render_failure)++;
	  	  
	  break;

	case MODIF_CHEM_N_RENDER_Y: /* added for symmetry, for the moment.*/
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not modified - visual\n"),
	       iter + 1);
	  
	  if (chem_failure != NULL)
	    (*chem_failure)++;
	  	  
	  break;
	}
    }
  
  /* Now that we have finished doing the work, if the user wanted to
     get errors, then we have to give them.
   */
  if (errors != NULL)
    {
      *errors = gs->str;
      /* We know we have allocated this GString above, because errors
	 was not NULL!
      */
      g_string_free (gs, FALSE);
    }

  return;
}


PxmModifRes
polyxmass_seqed_widget_mnm_modif_un_modify_monomer (PxmMonomer *mnm,
						    gint idx,
						    gchar *modif,
						    GtkWidget *seqed_widget)
{
  gint res = UN_MODIF_FAILURE;

  GdkPixbuf *pixbuf = NULL;

  PxmMonomer *monomer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  PxmPolymer *polymer = NULL;
  
  PxmMonicon *monicon_old = NULL;
  PxmMonicon *monicon_new = NULL;


  g_assert (seqed_widget != NULL);

  polchemdefctxt = PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);



  /* Un-modifying a monomer is a two-step process: first the monomer
     must be chemically un-modified (if its current modif is the same
     as the modif param, if the latter is non-NULL), and next, if the
     un-modification is successful, the graphical rendering must be
     performed so that the modification is eliminated from the monicon
     of the no-more-modified monomer.
  */

  /* Sanity checks pertaining to the size of the polymer sequence.
   */
  g_assert (polymer->monomerGPA->len > 0);
  g_assert (idx >= 0);
  g_assert (idx <= polymer->monomerGPA->len - 1);

  /* Make sure we can correctly access the monomer to be un-modified.
   */
  if (mnm == NULL)
    monomer = g_ptr_array_index (polymer->monomerGPA, idx);
  else
    {
      monomer = mnm ;
      
      /* Sanity check.
       */
      g_assert (monomer == g_ptr_array_index (polymer->monomerGPA, idx));
    }
  
  g_assert (monomer != NULL);

  /* Make sure the modif object is known to the polchemdef, if modif
     passed as param is non-NULL.
   */
  if (modif != NULL && 
      -1 == pxmchem_modif_get_index_by_name (modif, polchemdef->modifGPA))
    {
      g_critical (_("%s@%d: modif is not known to the polchemdef: '%s'\n"),
	     __FILE__, __LINE__, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* The call below may return any of these values:
     UN_MODIF_SUCCESS, if the unmodification was successful
     UN_MODIF_NO_MATCH, if the monomer was modified != 'modif'
     UN_MODIF_NO_MODIF, if the monomer was not modified
     UN_MODIF_FAILURE, if something went wrong.
  */
  res = pxmchem_monomer_un_modify (monomer, modif);
  

  if (res == UN_MODIF_FAILURE)
    {
      if (modif != NULL)
	{
	  g_critical (_("%s@%d: failed to un-modify monomer "
		   "at index: '%d' with modif: '%s'\n"),
		 __FILE__, __LINE__, idx, modif);
	}
      
      else
	{
	  g_critical (_("%s@%d: failed to or did not un-modify monomer "
		   "at index: '%d'\n"),
		 __FILE__, __LINE__, idx);
	}
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* The situation is tricky here: we do not want to modify the appearance
     of a monomer if its internal chemical state did not change. Which
     is why we have to check to return value precisely. In fact,
     we only go on with the graphical rendering of the monicon if
     res == UN_MODIF_SUCCESS.
  */
  if (res != UN_MODIF_SUCCESS)
    return MODIF_CHEM_N_RENDER_N;
  
  
  /* At this point the chemical un-modification went ok, so we can
     manage the rendering process. In a prudent way of doing things,
     we start by making the new rendered monicon, and if everything
     goes well we make the replacement, otherwise we leave the current
     monicon in place but issue a critical message.
  */

  /* Render a new pixbuf corresponding to the un-modified monomer.
   */
  pixbuf = polyxmass_pixbuf_rendering_render_no_modif 
    (monomer->code,
     PXM_SEQED_WIDGET (seqed_widget)->monicon_size,
     &(PXM_SEQED_WIDGET (seqed_widget)->pixbufGData),
     polchemdefctxt);
  
  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed to render monicon for un-modified "
	       "monomer: '%s' at index: '%d'. Chemical un-modification "
	       "was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx);
	  
      return MODIF_CHEM_Y_RENDER_N;
    }

  /* Now that we have the pixbuf, we can allocate the new monicon so that
     a new canvas_item is prepared later.
  */
  monicon_new = polyxmass_monicon_new ();

  if (FALSE == polyxmass_pixbuf_rendering_monicon_make_canvas_item 
      (monicon_new, 
       PXM_SEQED_WIDGET (seqed_widget)->monicon_size,
       pixbuf,
       idx,
       seqed_widget))
    {
      g_critical (_("%s@%d: failed to make canvas_item for un-modified "
		    "monomer, code: '%s' at index: '%d'. Chemical "
		    "un-modification was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx);
	  
      return MODIF_CHEM_Y_RENDER_N ;
    }


  /* At this point, the pixbuf that we got in the first place, with
     the call to polyxmass_pixbuf_rendering_render_no_modif() got
     referenced by the gnome_canvas_item_new() that makes the canvas
     item in the polyxmass_pixbuf_rendering_monicon_make_canvas_item()
     call. Thus, when we call the pxm_monicon_free() function on the
     monicon, it should just 

     gtk_object_destroy(GTK_OBJECT(monicon->canvas_item)); 

     which will unref the pixbuf contained in it.

     So, the story goes : 

     1. we get a pixbuf (either pre-existing or allocated anew) with
     no reference count incrementing.
	
     2. we construct a canvas_item by using it, which increments its
     reference count by 1.

     3. we put that pixbuf into a monicon object that will contain it
     up to its freeing.

     4. some day we free the monicon, which will gtk_object_destroy
     the canvas_item, which will decrement the reference count by
     1. At this stage we get a reference count identical to the
     situation in point 1.
  */


  
  /* Apparently we succeeded in all our attempts, so get the pointer
     to the monicon that currently renders the monomer. Remove it from
     the array of monicons and free it.
  */
  monicon_old = 
    g_ptr_array_remove_index (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA, idx);
  g_assert (monicon_old != NULL);
  polyxmass_monicon_free (monicon_old);


  /* At this point we have a fully qualified monicon. Depending on the
     value of idx, with respect to the size of the moniconGPA we
     either insert the monicon in the sequence array or we append it.
     We cannot use the polymer->monomerGPA->len value, because it does
     not change, while moniconGPA->len has changed as the effect of
     removing the old monicon from it. I was having crashes due using
     the polymer->monomerGPA->len value here. Which was wrong of
     course.
   */
  if (idx == PXM_SEQED_WIDGET (seqed_widget)->moniconGPA->len)
    g_ptr_array_add (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA,
		     monicon_new);
  else
    g_ptr_array_insert_val (PXM_SEQED_WIDGET (seqed_widget)->moniconGPA,
			    idx, monicon_new);

  return MODIF_CHEM_Y_RENDER_Y;
}


void
polyxmass_seqed_widget_mnm_modif_un_modify_monomer_GA (GArray *GA,
						       gchar *modif,
						       GtkWidget *seqed_widget,
						       gchar **errors,
						       gint *full_success,
						       gint *full_failure,
						       gint *chem_failure,
						       gint *render_failure)
{
  gint iter = 0;
  gint idx = 0;

  GString *gs = NULL;
  
  PxmModifRes res = MODIF_CHEM_N_RENDER_N;
  
  PxmPolymer *polymer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;

  GPtrArray *monomerGPA = NULL;


  /* We have a GArray (GA) of indices corresponding to indices in the
     polymer sequence where the un-modification should occur. Note
     that modif may be NULL, in which case the un-modification will
     happen whatever the modification with which the target monomer is
     modified. Instead, if modif is non-NULL, the target monomer is
     un-modified only if currently modified with a modification of the
     same name as the string passed as parameter 'modif'.
   */


  g_assert (GA != NULL);
  /* If no monomer is to be un-modified just return 0 since we do not
     modify any monomer !
   */
  if (GA->len <= 0)
    return ;
    
  g_assert (seqed_widget != NULL);

  polchemdefctxt = PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);
  
  monomerGPA = polymer->monomerGPA;
  g_assert (monomerGPA != NULL);

  /* If the polymer sequence is empty, no un-modif can occur!
   */
  if (monomerGPA->len <= 0)
    return ;
  
  /* Sanity check: if modif is non-NULL it must be known to the
     polymer type!
  */
  if (modif != NULL)
    g_assert (-1 != pxmchem_modif_get_index_by_name (modif,
						     polchemdef->modifGPA));
  
  /* If errors is set, we want errors to be a pointer to a NULL gchar
     pointer. We will allocate the string ourselves and let the caller 
     free it after.
   */
  if (errors != NULL)
    {
      g_assert (*errors == NULL);
      
      /* Prepare the GString that will store the errors, but only if 
	 error reporting is asked.
      */
      gs = g_string_new ("");
    }
    
  /* We now should iterate in the array of indices of the monomers to
     be un-modified and take proper actions each time.
  */
  for (iter = 0; iter < GA->len; iter++)
    {
      idx = g_array_index (GA, gint, iter);
      
      /* Check that we are not going to access a non-existent
	 monomer... especially if the user edits the polymer sequence
	 by removing monomers from it.
      */
      if (idx > monomerGPA->len)
	{
	  g_critical (_("%s@%d: a position marked to be modified is greater "
		   "than the size of the polymer! "
		   "Do not edit the sequence!\n"),
		 __FILE__, __LINE__);
	  
	  return ;
	}
      
      /* Do the un-modification work proper.
       */
      res = polyxmass_seqed_widget_mnm_modif_un_modify_monomer (NULL, idx, modif,
								seqed_widget);
      
      /* Now, a number of possibilities need to be envisaged, and errors
	 will have to be logged. Since the first operation is to 
	 make the chemical modification, if it goes wrong the function
	 returns immediately with MODIF_CHEM_N_RENDER_N.
      */
      switch (res)
	{
	case MODIF_CHEM_Y_RENDER_Y:
	  if (full_success != NULL)
	    (*full_success)++;
      	  
	  break;

	case MODIF_CHEM_N_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not un-modified - no visual\n"),
	       iter + 1);

	  if (full_failure != NULL)
	    (*full_failure)++;
	  
	  break;

	case MODIF_CHEM_Y_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - un-modified - no visual\n"),
	       iter + 1);
	  
	  if (render_failure != NULL)
	    (*render_failure)++;
	  	  
	  break;

	case MODIF_CHEM_N_RENDER_Y: /* added for symmetry, for the moment.*/
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not un-modified - visual\n"),
	       iter + 1);
	  
	  if (chem_failure != NULL)
	    (*chem_failure)++;
	  	  
	  break;
	}
    }
  
  /* Now that we have finished doing the work, if the user wanted to
     get errors, then we have to give them.
   */
  if (errors != NULL)
    {
      *errors = gs->str;
      /* We know we have allocated this GString above, because errors
	 was not NULL!
      */
      g_string_free (gs, FALSE);
    }

  return;
}








/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxmass_seqed_widget_mnm_modif_wnd_really_close (GtkWidget *window)
{
  gint count = 0;

  
  g_assert (window != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
}


gboolean
polyxmass_seqed_widget_mnm_modif_wnd_delete_event (GtkWidget *window, 
					     GdkEventAny *event, 
					     gpointer data)
{
  GtkWidget *seqed_widget = data;
  
  gchar *help = NULL;
  
  
  g_assert (window != NULL);
  g_assert (seqed_widget != NULL);
  
  
  help = g_strdup_printf ("monomer_modif_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called, thus
     destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT (seqed_widget), 
		     help, NULL);
  g_free (help);
    
  return TRUE;
}

  
gboolean
polyxmass_seqed_widget_mnm_modif_wnd_destroy_event (GtkWidget *window, 
					      GdkEventAny *event, 
					      gpointer data)
{
  return FALSE;
}
