/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxdef-globals.h"

#include "polyxdef-ui-polchemdef.h"

#include "polyxdef-ui-syntaxcheck.h"

#include "polyxdef-ui-atomdef.h"
#include "polyxdef-ui-monomers.h"
#include "polyxdef-ui-modifs.h"
#include "polyxdef-ui-cleavespecs.h"
#include "polyxdef-ui-fragspecs.h"




GtkWidget *
polyxdef_ui_polchemdef_wnd_setup_window (PxmDefCtxt *defctxt)
{
  GtkWidget *widget = NULL;
  GladeXML *xml = NULL;

  gchar *help = NULL;
  gchar *gui_file = NULL;
  
  /* We are setting up the polymer chemistry definition window for the
     defctxt. The definition window is referenced in the defctxt with the
     definition_wnd member...
  */
  g_assert (defctxt != NULL);

  /* Basically, we don't want to call this function twice for the same
     definition context.
  */
  g_assert (defctxt->definition_wnd == NULL);


  /* Even if we are working on a brand new polymer chemistry
     definition, one was allocated, which has allocated arrays for
     chemical data, but a number of NULL pointers also. So we'll have
     to check these latter each time we want to access them during the
     setting up of this definition window.
  */
  g_assert (defctxt->polchemdef != NULL);

  g_assert (defctxt->polchemdef->atomGPA != NULL);
  
  /* Create the window into which a polymer chemistry definition is
   * going to be handled, be it new or loaded from an xml file.
   */  
  gui_file = 
    g_strdup_printf ("%s/polyxdef.glade", userspec->gladedir);

  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "polyxdef_polchemdef_wnd", 
		       PACKAGE);
  g_free (gui_file);

  if (xml == NULL)
    {
      g_critical (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  defctxt->definition_wnd = glade_xml_get_widget (xml, 
						  "polyxdef_polchemdef_wnd");
  
  if (defctxt->definition_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to create the polymer chemistry "
	       "definition window\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  /* Store the pointers of all the widgets that we may need to access
     later...
   */

  /* 
     There is a GtkEntry in the window that is commonly used to display
     messages. We need this one set early in the interface creation
     process so that we can display errors...

   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "messages_entry", widget);


  widget = glade_xml_get_widget (xml, "polchemdef_type_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_type_entry", widget);
    

  /* If the defctxt->polchemdef->type is non-NULL, set it to the entry.
   */
  if (defctxt->polchemdef->type != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), defctxt->polchemdef->type);

  g_signal_connect 
    (G_OBJECT (widget),
     "changed",
     G_CALLBACK (polyxdef_ui_polchemdef_wnd_editable_changed),
     defctxt);
  
  
  widget = glade_xml_get_widget (xml, "polchemdef_status_label");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_status_label", widget);
  
  /* The file is not modified !
   */
  pxmchem_polchemdef_set_modified (defctxt->polchemdef, FALSE);
  defctxt->is_modified = FALSE;
  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);


  /* Atom definition stuff....
   */
  widget = glade_xml_get_widget (xml, 
				 "atom_def_frame");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atom_def_frame", widget);

  widget = glade_xml_get_widget (xml, 
				 "atom_def_filename_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atom_def_filename_entry", widget);


  widget = glade_xml_get_widget (xml, "atom_def_read_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atom_def_read_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxdef_ui_polchemdef_wnd_atom_definition_read_button),
     defctxt);
  

  widget = glade_xml_get_widget (xml, "atom_def_locate_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atom_def_locate_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxdef_ui_polchemdef_wnd_atom_definition_locate_button),
     defctxt);



  /* The combo/entry where the user will select the atom definition
     suitable for use with the polymer chemistry definition.
  */
  widget = glade_xml_get_widget (xml, "atom_def_vbox");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "atom_def_vbox", widget);

  if (-1 == 
      polyxdef_ui_polchemdef_wnd_fill_atom_defs_combo_list (defctxt))
    g_critical (_("%s@%d: failed to set the atom definitions to combo list\n"),
	   __FILE__, __LINE__);
  

  /* Now that we have the list of all the atom definitions available
     on the current system, we can try finding what's the one that is
     appropriate for the polymer chemistry definition we are handling
     in this dfinition context, so that we can set its name to the
     combobox of available atom definitions.

     Note, however, that if the definition_wnd was setup for a new
     polymer chemistry definition (through New menu item in the main
     program window, then, of course, the polymer chemistry definition
     pointed to by defctxt->polchemdef is totally incomplete... the
     user just wants to start defining it. In this case, it makes no
     sense to try to find the proper atom definition with respect to
     the current defctxt->polchemdef->type... Thus we skip that
     search.
  */
  if (defctxt->polchemdef->type != NULL)
    {
      if (FALSE == 
	  polyxdef_ui_polchemdef_wnd_set_atomdef_name_to_combo_list (defctxt))
	{
	  g_critical (_("%s@%d: failed to set the proper atom definition name "
		   "for current polymer chemistry definition to combo list\n"),
		 __FILE__, __LINE__);
	  
	  return NULL;
	}
      
      /* Now that we know what atom definition file should be loaded
	 for use with the current polymer chemistry definition, we
	 should load it ! 

	 Note that contrary to what happens with the loading of an
	 atom definition file in the atom definition window, we are
	 not populating the defctxt->atomGPA array, but
	 the polymer chemistry definition one
	 (defctxt->polchemdef->atomGPA), because the workings of the
	 polymer chemistry definition are so that a PxmPolchemDef
	 object thinks atoms are defined in polchemdef->atomGPA
	 (validation functions, for example).
      */
      if (FALSE == 
	  polyxdef_ui_polchemdef_wnd_atom_definition_read_button (NULL, 
								  defctxt))
	{
	  g_critical (_("%s@%d: failed to read the atom definition file\n"),
		 __FILE__, __LINE__);
	  
	  return NULL;
	}
    } 
  /* End of 
     if (defctxt->polchemdef->type != NULL)
  */
  
  widget = glade_xml_get_widget (xml, 
				 "polchemdef_left_cap_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_left_cap_entry", widget);

  if (defctxt->polchemdef->leftcap != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), defctxt->polchemdef->leftcap);
  
  g_signal_connect (G_OBJECT (widget),
		    "changed",
		    G_CALLBACK (polyxdef_ui_polchemdef_wnd_editable_changed),
		    defctxt);


  widget = glade_xml_get_widget (xml, 
				 "polchemdef_right_cap_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_right_cap_entry", widget);

  if (defctxt->polchemdef->rightcap != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), defctxt->polchemdef->rightcap);

  g_signal_connect (G_OBJECT (widget),
		    "changed",
		    G_CALLBACK (polyxdef_ui_polchemdef_wnd_editable_changed),
		    defctxt);


  widget = glade_xml_get_widget (xml, 
				 "polchemdef_ionizerule_actform_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_ionizerule_actform_entry", widget);


  if (defctxt->polchemdef->ionizerule->actform != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (widget), 
			  defctxt->polchemdef->ionizerule->actform);
    }
  
  g_signal_connect (G_OBJECT (widget),
		    "changed",
		    G_CALLBACK (polyxdef_ui_polchemdef_wnd_editable_changed),
		    defctxt);
  

  widget = glade_xml_get_widget (xml, 
				 "polchemdef_ionizerule_charge_entry");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_ionizerule_charge_entry", widget);


  /* Note that the ionizerule is allocated upon newing of the polchemdef.
   */
  help = g_strdup_printf ("%d", defctxt->polchemdef->ionizerule->charge);
  g_assert (help != NULL);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  g_signal_connect (G_OBJECT (widget),
		    "changed",
		    G_CALLBACK (polyxdef_ui_polchemdef_wnd_editable_changed),
		    defctxt);

    
  /* Notebook widgets.
   */
  widget = glade_xml_get_widget (xml, "polchemdef_notebook");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_notebook", widget);


  /* The monomers' notebook page must be correctly set up:
   */
  if (FALSE ==
      polyxdef_ui_polchemdef_wnd_setup_monomers_notebook_page (defctxt,
							       xml))
    {
      g_critical (_("%s@%d: failed to set up the monomers' tab.\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  
  /* The modifs' notebook page must be correctly set up:
   */
  if (FALSE ==
      polyxdef_ui_polchemdef_wnd_setup_modifs_notebook_page (defctxt,
							     xml))
    {
      g_critical (_("%s@%d: failed to set up the modifs' tab.\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  

  
  /* The cleavespecs' notebook page must be correctly set up:
   */
  if (FALSE ==
      polyxdef_ui_polchemdef_wnd_setup_cleavespecs_notebook_page (defctxt,
								  xml))
    {
      g_critical (_("%s@%d: failed to set up the cleavespecs' tab.\n"),
	     __FILE__, __LINE__);
      
      return NULL;
    }
  
  

  /* The fragspecs' notebook page must be correctly set up:
   */
  if (FALSE ==
      polyxdef_ui_polchemdef_wnd_setup_fragspecs_notebook_page (defctxt,
								xml))
    {
      g_critical (_("%s@%d: failed to set up the fragspecs' tab.\n"),
	     __FILE__, __LINE__);
      
      return NULL;
    }
  
  

  /* Syntax checking stuff...
   */  
  widget = glade_xml_get_widget (xml, 
				 "polchemdef_check_syntax_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_check_syntax_button", widget);

  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK 
     (polyxdef_ui_polchemdef_wnd_polchemdef_check_syntax_button),
     defctxt);
  
  widget = glade_xml_get_widget (xml, 
				 "polchemdef_save_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_save_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_polchemdef_save_button),
		    defctxt);

  widget = glade_xml_get_widget (xml, 
				 "polchemdef_saveas_button");
  g_object_set_data (G_OBJECT (defctxt->definition_wnd),
		     "polchemdef_saveas_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_polchemdef_saveas_button),
		    defctxt);

  

  /* We don't need the GladeXML object any more, so unref it
   * to save some memory 
   */
  g_object_unref (G_OBJECT (xml));


  /* Connect some important signals for the window proper.
   */
  g_signal_connect (G_OBJECT (defctxt->definition_wnd),
		    "delete_event",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_polchemdef_wnd_delete_event),
		    defctxt);
  
  g_signal_connect (G_OBJECT (defctxt->definition_wnd),
		    "destroy_event",
		    G_CALLBACK 
		    (polyxdef_ui_polchemdef_wnd_polchemdef_wnd_destroy_event),
		    defctxt);
  

  return defctxt->definition_wnd;
}


void
polyxdef_ui_polchemdef_wnd_editable_changed (GtkWidget *widget,
					     gpointer data)
{
  PxmDefCtxt *defctxt = data;
  
  gint test = 0;

  gchar *help = NULL;
  
    
  g_assert (widget != NULL);
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);
  

  /* When an item is modified in one of the entries, we must 
     have the polchemdef object reflect thoses changes immediately, because
     the user may ask for a save.
  */


  if (0 == strcmp (gtk_widget_get_name (widget), 
		   "polchemdef_type_entry"))
    {
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help !=NULL);
      
      /* The name of the polchemdef has changed.
       */
      pxmchem_polchemdef_set_type (defctxt->polchemdef, help);
      g_free (help);
    }
  
  else if (0 == strcmp (gtk_widget_get_name (widget), 
			"polchemdef_left_cap_entry"))
    {
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help !=NULL);
      
      /* The leftcap of the polchemdef has changed.
       */
      pxmchem_polchemdef_set_leftcap (defctxt->polchemdef, help);
      g_free (help);
    }
  
  else if (0 == strcmp (gtk_widget_get_name (widget), 
			"polchemdef_right_cap_entry"))
    {
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help !=NULL);
      
      /* The rightcap of the polchemdef has changed.
       */
      pxmchem_polchemdef_set_rightcap (defctxt->polchemdef, help);
      g_free (help);
    }
  
  else if (0 == strcmp (gtk_widget_get_name (widget), 
			"polchemdef_ionizerule_actform_entry"))
    {
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help !=NULL);
      
      /* The ionizerule_actform of the polchemdef has changed.
       */
      pxmchem_ionizerule_set_actform (defctxt->polchemdef->ionizerule, help);
      g_free (help);
    }
  
  else if (0 == strcmp (gtk_widget_get_name (widget), 
			"polchemdef_ionizerule_charge_entry"))
    {
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help !=NULL);
      
      /* The ionizerule_charge of the polchemdef has changed.
       */
      if (FALSE == libpolyxmass_globals_strtoi (help, &test, 10))
	{
	  polyxmass_timeoutmsg_message_set 
	    ((GtkWindow *) defctxt->definition_wnd,
	     _("The ionizerule charge " 
	       "is invalid"), 
	     POLYXMASS_MEDI_MSG_TIMEOUT);
	  
	  g_free (help);
	  
	  return;
	}

      pxmchem_ionizerule_set_charge (defctxt->polchemdef->ionizerule, test);
      g_free (help);
    }
  
  pxmchem_polchemdef_set_modified (defctxt->polchemdef, TRUE);
  defctxt->is_modified = TRUE;
  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);
  
  return ;
}

void
polyxdef_ui_polchemdef_wnd_atomdef_combobox_changed (GtkWidget *widget,
						     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *entry;

  gint index = 0;
  
  PxmAtomSpec *as = NULL;

  g_assert (widget != NULL);
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  
  /* The items that are located in the combo box list have been initially
     read from the global array of the available atom definitions,
     atom_defsGPA. So we just have to get the item in that array that
     has exactly the same index as the one in the list.
  */

  index = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));
  /*
    debug_printf (("index is %d\n", index));
  */
  if (index == -1)
    return;
  
  as = g_ptr_array_index (polyxmass_atomdefsGPA, index);
  g_assert (as != NULL);
  
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_def_filename_entry");
  g_assert (entry != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (entry), as->file);
}


gint
polyxdef_ui_polchemdef_wnd_fill_atom_defs_combo_list (PxmDefCtxt *defctxt)
{
  gint iter = 0;
  gint count = 0;
  
  GtkWidget *atom_def_vbox = NULL;
  GtkWidget *combo_box = NULL;
  
  PxmAtomSpec *as = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  

  /* We have the global polyxmass_atom_defsGPA array that holds a
     description of all the atom definitions available on the system
     (array of PxmAtomSpec instances. We just want that the files of
     the atom definitions be listed in the combo list.
  */
  g_assert (polyxmass_atomdefsGPA != NULL);

  if (polyxmass_atomdefsGPA->len <= 0)
    return 0;
  
  atom_def_vbox = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
				     "atom_def_vbox");
  g_assert (atom_def_vbox != NULL);
  
  combo_box = gtk_combo_box_entry_new_text ();

  for (iter = 0; iter < polyxmass_atomdefsGPA->len ; iter++)
    {
      as = g_ptr_array_index (polyxmass_atomdefsGPA, iter);
      g_assert (as != NULL);
      
      gtk_combo_box_append_text (GTK_COMBO_BOX (combo_box), as->atomdef);

      count++;
    }
  
  g_signal_connect 
    (G_OBJECT (combo_box),
     "changed",
     G_CALLBACK (polyxdef_ui_polchemdef_wnd_atomdef_combobox_changed),
     defctxt);

  /* Finally we can pack the combo bo to the vertical box in the frame.
   */
  gtk_box_pack_start (GTK_BOX (atom_def_vbox), combo_box, 
		      TRUE, TRUE, 0);


  gtk_widget_show_all (GTK_WIDGET (atom_def_vbox));
  

  /* The combo has to be before the "Edit Atom Definition" button.
   */
  gtk_box_reorder_child (GTK_BOX (atom_def_vbox), combo_box, 0);
  
  
  /* Set some data to the window.
   */
  g_object_set_data (G_OBJECT (defctxt->definition_wnd), 
		     "atom_def_combo", combo_box);

  return count;
}


gboolean
polyxdef_ui_polchemdef_wnd_set_atomdef_name_to_combo_list (PxmDefCtxt *defctxt)
{
  PxmAtomSpec *as = NULL;

  GtkWidget *atom_def_combo = NULL;

  gint idx = 0;
    

  /* The window passed as parameter contains has a pointer to its
     polymer chemistry definition window, "polchemdef". We want to get
     the type of that polymer so that we can find the proper atom
     definition for it as described in the dictionary of equivalences
     between polymer definitions and atom definition names... Once we
     have found the proper atom definition name, we can set the
     corresponding item in the combo box and thus the corresponding
     file name in the entry.
  */
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);

  as = polyxmass_globals_get_matching_polchemdef_atomdef (defctxt->
							  polchemdef->type,
							  NULL,
							  NULL,
							  &idx);
  if (as != NULL)
    {
      /* We found an item in the polyxmass_atomdefsGPA global array of 
	 atom definitions. Thus 'idx' now contains the index of the 
	 found PxmAtomSpec instance.
	 
	 But free immediately the instance returned since it was
	 allocated as a convenience we do not use now.
      */
      libpolyxmass_atomspec_free (as);
      
      /* We finally have an index. We should set the selection of 
	 the combo_box to the item at index 'idx'.
      */
      atom_def_combo = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
					  "atom_def_combo");
      g_assert (atom_def_combo != NULL);
      
      /* This should automatically put the corresponding atom
	 definition file name to the corresponding GtkEntry.
      */
      gtk_combo_box_set_active (GTK_COMBO_BOX (atom_def_combo), 
				idx);
      
      return TRUE;
    }
  
  return FALSE;
}



gboolean
polyxdef_ui_polchemdef_wnd_atom_definition_read_button (GtkWidget *widget, 
							gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *entry = NULL;

  gchar *filename = NULL;
  gchar *message = NULL;
  

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  g_assert (defctxt->polchemdef != NULL);
  g_assert (defctxt->polchemdef->atomGPA != NULL);
  

  /* We are asked to read the file that is set to the combo box entry.
   */
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
			     "atom_def_filename_entry");
  g_assert (entry != NULL);

  filename = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (filename != NULL);
  
  /* Make sure that the file actually exists !
   */
  if (FALSE == g_file_test (filename, G_FILE_TEST_EXISTS))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) defctxt->definition_wnd,
					_("File not found"),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      g_free (filename);
      
      return FALSE;
    }

  /* We should now parse the file and see if we can successfully read
     the atom definition. But first make sure that the array is empty,
     if no empty it forcibly.
  */
  if (defctxt->polchemdef->atomGPA->len > 0)
    pxmchem_atom_GPA_empty (defctxt->polchemdef->atomGPA);
  
  if (-1 == pxmchem_atom_render_xml_file (filename, 
					  defctxt->polchemdef->atomGPA))
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd,
	 _("Failed to read the atom definition file"),
	 POLYXMASS_LONG_MSG_TIMEOUT);

      g_free (filename);

      return FALSE;
    }

  /* 
     At this point we know that the rendering of the atom definition
     is OK.
  */
  message = 
    g_strdup_printf (_("Succeeded to read the atom definition file; " 
		       "Atoms parsed: '%d'"), 
		     defctxt->polchemdef->atomGPA->len);
  
  polyxmass_timeoutmsg_message_set 
    ((GtkWindow *) defctxt->definition_wnd, message, 
     POLYXMASS_NORM_MSG_TIMEOUT);
  
  g_free (filename);
  g_free (message);
  
  return TRUE;
}


void
polyxdef_ui_polchemdef_wnd_atom_definition_locate_button (GtkWidget *widget,
							  gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *dialog;
  GtkWidget *entry;

  
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);


  /* We are asked to let the user find an atom definition file on
     disk. This is done by allowing him to browse the disk with a
     GtkFileChooser dialog.
   */

  dialog = gtk_file_chooser_dialog_new (_("Find Atom Definition File"),
					GTK_WINDOW (defctxt->definition_wnd),
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
  
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *filename;
      
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

      /* Set that file to the GtkEntry.
       */
      entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd),
				 "atom_def_filename_entry");
      g_assert (entry != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (entry), filename);
      
      g_free (filename);
    }
  
  gtk_widget_destroy (dialog);
  
  return;
}



/* SYNTAX ERROR CHECKING FUNCTIONS
 */
gboolean
polyxdef_ui_polchemdef_wnd_polchemdef_check_syntax_button (GtkWidget *widget, 
							   gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *entry = NULL;

  gboolean valid_result = TRUE;

  gint test = 0;
    
  gchar *valid = NULL;
  gchar *help = NULL;
  
  
  /* This function might be called by clicking onto the Check button
     or programmatically.
  */

  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  /* All we want to do is get the polymer chemistry definition out of
   * the window and next call the validating function. This function
   * returns an allocated string that we can display later.
   */
    
  /* There are some fieds that need reading manually here:
   */
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "polchemdef_type_entry");
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);
  pxmchem_polchemdef_set_type (defctxt->polchemdef, help);
  g_free (help);
  
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "polchemdef_left_cap_entry");
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);
  pxmchem_polchemdef_set_leftcap (defctxt->polchemdef, help);
  g_free (help);
  
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "polchemdef_right_cap_entry");
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);
  pxmchem_polchemdef_set_rightcap (defctxt->polchemdef, help);
  g_free (help);
  
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "polchemdef_ionizerule_actform_entry");
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);
  pxmchem_ionizerule_set_actform (defctxt->polchemdef->ionizerule, help);
  g_free (help);
  
  entry = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			     "polchemdef_ionizerule_charge_entry");
  help = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
  g_assert (help != NULL);

  if (FALSE == libpolyxmass_globals_strtoi (help, &test, 10))
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd,
	 _("The ionizerule charge " 
	   "is invalid"), 
	 POLYXMASS_MEDI_MSG_TIMEOUT);
      
      g_free (help);
      
      return FALSE;
    }
  
  pxmchem_ionizerule_set_charge (defctxt->polchemdef->ionizerule, test);
  g_free (help);
  
  valid_result = pxmchem_polchemdef_validate_all (defctxt->polchemdef, 
						  &valid);
  
  if (valid_result == FALSE)
    {
      /* There was at least one error, which means that the string in
	 valid has been allocated. We should show that string and
	 later free it.
      */
      g_assert (valid != NULL);

      if (defctxt->syntax_check_wnd != NULL)
	{
	  polyxdef_ui_syntaxcheck_wnd_reset (defctxt);
	}
      else
	{
	  /* We have to setup the window first.
	   */
	  defctxt->syntax_check_wnd = 
	    polyxdef_ui_syntaxcheck_wnd_setup_window (defctxt, NULL);
	  
	  if (defctxt->syntax_check_wnd == NULL)
	    {
	      g_message (_("%s@%d: failed to set up the syntax-checking "
		       "window for the following errors: '%s'\n\n"),
		     __FILE__, __LINE__, valid);
	      
	      g_free (valid);
	      
	      return FALSE;
	    }
	}

      /* We now have an empty textview where to set the errors' string.
       */
      polyxdef_ui_syntaxcheck_wnd_set_errors (defctxt, valid);

      g_free (valid);
      
      return FALSE;
    }
  else if (valid_result == TRUE)
    {
      /* Great, the validation succeeded, just inform the user.
       */
      g_message (_("%s@%d: the validation succeeded\n"),
	     __FILE__, __LINE__);
      
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) defctxt->definition_wnd, 
	 _("The validation succeeded"), 
	 POLYXMASS_NORM_MSG_TIMEOUT);

      return TRUE;
    }
  else
    g_assert_not_reached ();

  return FALSE;
}



/* SAVE BUTTONS
 */
void
polyxdef_ui_polchemdef_wnd_polchemdef_save_button (GtkWidget *widget, 
						   gpointer data)
{
  PxmDefCtxt *defctxt = data;

  gint result = -1;

  gchar *xml = NULL;
      
  FILE *filep = NULL;
 
  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);

  
  /* We do not save a polymer chemistry definition which is not valid.
   */
  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_polchemdef_check_syntax_button (NULL, 
								 defctxt))
    return ;


  /* Now, if the defctxt->polchemdef->file member is non-NULL, then
     that means that the polchemdef was actually saved at least once
     or was read from a disk file (in other words, it is not
     "new"). Then we can save the polymer chemistry definition to that
     file immediately. Conversely, we need to get a file name from the
     user.
  */
  if (defctxt->polchemdef->file == NULL)
    {
      return polyxdef_ui_polchemdef_wnd_polchemdef_saveas_button (NULL, 
								  defctxt);
    }
 
  /* First of all make sure we can open the file to write to it.
   */
  filep = fopen (defctxt->polchemdef->file, "w");

  if (filep == NULL)
    {
      g_critical (_("%s@%d: failed to open file: '%s'\n"),
	     __FILE__, __LINE__, defctxt->polchemdef->file);
      
      return;
    }
  
   /* Construct a string with all the xml-formatted data pertaining to
      the polymer chemistry definition 'polchemdef'.
   */
  xml = 
    pxmchem_polchemdef_format_xml_string_polchemdefdata (defctxt->polchemdef,
							 "  ", 0);
  g_assert (xml != NULL);

  /* Copy the xml data string to the file 'file'.
   */
  result = fputs (xml, filep);
  
  fclose (filep);
  
  g_free (xml);
  
  if (result == EOF || result < 0)
    {
      g_critical (_("%s@%d: failed to save polchemdef to file: '%s'\n"),
	     __FILE__, __LINE__, defctxt->polchemdef->file);
      
      return;
    }

  pxmchem_polchemdef_set_modified (defctxt->polchemdef, FALSE);
  defctxt->is_modified = FALSE;
  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

  return;
}


void
polyxdef_ui_polchemdef_wnd_polchemdef_saveas_button (GtkWidget *widget,
						     gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *dialog = NULL;

  gint result = -1;

  gchar *file = NULL;
  gchar *xml = NULL;
  gchar *help = NULL;
  
  FILE *filep = NULL;
 

  g_assert (defctxt != NULL);
  g_assert (defctxt->polchemdef != NULL);
  g_assert (defctxt->definition_wnd != NULL);

  
  /* We do not save a polymer chemistry definition which is not valid.
   */
  if (FALSE == 
      polyxdef_ui_polchemdef_wnd_polchemdef_check_syntax_button (NULL, 
								 defctxt))
    return ;


  dialog = 
    gtk_file_chooser_dialog_new (_("Save Polymer Chemistry Definition File"),
				 GTK_WINDOW (defctxt->definition_wnd),
				 GTK_FILE_CHOOSER_ACTION_SAVE,
				 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				 GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				 NULL);
  
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      
      if (TRUE == g_file_test (file, G_FILE_TEST_EXISTS))
	{
	  g_critical ("%s@%d: file does exist already: '%s'\n",
		 __FILE__, __LINE__, file);
	  
	  g_free (file);
	  
	  /* We can destroy the file chooser dialog.
	   */
	  gtk_widget_destroy (dialog);

	  return;
	}
    }
  else
    {
      g_free (file);
      
      /* We can destroy the file chooser dialog.
       */
      gtk_widget_destroy (dialog);
      
      return;
    }
  
  /* Ok, we have a file to which the polymer chemistry definition can
     be performed.
  */

  /* We can destroy the file chooser dialog.
   */
  gtk_widget_destroy (dialog);
  
  /* First of all make sure we can open the file to write to it.
   */
  filep = fopen (file, "w");
  
  if (filep == NULL)
    {
      g_critical (_("%s@%d: failed to open file: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      g_free (file);
      
      return;
    }

  /* Construct a string with all the xml-formatted data pertaining to
   * the polymer chemistry definition 'polchemdef'.
   */
  xml = 
    pxmchem_polchemdef_format_xml_string_polchemdefdata (defctxt->
							 polchemdef,
							 "  ", 0);
  g_assert (xml != NULL);

  /* Copy the xml data string to the file 'file'.
   */
  result = fputs (xml, filep);

  fclose (filep);
  
  g_free (xml);
  
  if (result == EOF || result < 0)
    {
      g_critical (_("%s@%d: failed to save polchemdef to file: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return;
    }

  /* Now that the polymer chemistry definition is in a new file, we
     have to set its member datum to this new file name:
   */
  pxmchem_polchemdef_set_file (defctxt->polchemdef, file);

  pxmchem_polchemdef_set_modified (defctxt->polchemdef, FALSE);
  defctxt->is_modified = FALSE;
  polyxdef_ui_polchemdef_wnd_update_status_label (defctxt);

  /* Now we should change the window title so that it reads the new
     filename.
   */
  help = g_strdup_printf (_("Polymer Chemistry Definition: %s"), 
			  defctxt->polchemdef->file);

  gtk_window_set_title (GTK_WINDOW (defctxt->definition_wnd), help);
  g_free (help);

  g_free (file);
  
  return;
}


/* HELPERS
 */
void
polyxdef_ui_polchemdef_wnd_update_status_label (PxmDefCtxt *defctxt)
{
  GtkWidget *widget = NULL;
  
  g_assert (defctxt->definition_wnd != NULL);
  

  widget = g_object_get_data (G_OBJECT (defctxt->definition_wnd), 
			      "polchemdef_status_label");
  g_assert (widget != NULL);
  

  gtk_label_set_text (GTK_LABEL (widget), 
		      defctxt->is_modified == TRUE ? 
		      _("Modified") : _("Unmodified"));

  return;
}




/* WINDOW CLOSING FUNCTIONS
 */
gboolean
polyxdef_ui_polchemdef_wnd_polchemdef_wnd_delete_event (GtkWidget *window,
							GdkEvent *event,
							gpointer data)
{
  PxmDefCtxt *defctxt = data;

  GtkWidget *dialog = NULL;
  
  gint dialog_result = 0;
  
  gchar *help = NULL;
    


  g_assert (defctxt != NULL);
  g_assert (defctxt->definition_wnd != NULL);
  g_assert (defctxt->polchemdef != NULL);


  /* We only destroy the window if its corresponding polchemdef
     instance is not modified.
  */
  if (FALSE == defctxt->is_modified)
    {
      /* Remove the window from the array of polyxdef contexts.
       */
      if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
				       defctxt))
	{
	  g_critical (_("%s@%d: failed to remove context from array.\n"),
		 __FILE__, __LINE__);
	}

      polyxdef_defctxt_free (defctxt);  

      return TRUE;
    }

  /* The polyxdef context associated with this widget is modified, we
   * should ask if it should be saved before.
   */
  help = g_strdup_printf (_("The polymer chemistry definition of "
			  "type: '%s' is modified, save it?\n"),
			  defctxt->polchemdef->type);
  
  dialog = gtk_message_dialog_new (GTK_WINDOW (defctxt->definition_wnd),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_QUESTION,
				   GTK_BUTTONS_YES_NO,
				   help);
  g_free (help);
  
  /* Add the CANCEL button, so that the user can simply cancel 
   * the operation.
   */
  gtk_dialog_add_button (GTK_DIALOG (dialog),
			 GTK_STOCK_CANCEL,
			 GTK_RESPONSE_CANCEL);

  dialog_result = gtk_dialog_run (GTK_DIALOG (dialog));

  switch (dialog_result)
    {
    case GTK_RESPONSE_YES:

      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);

      /* Next ask for a save of the polchemdef.
       */
      polyxdef_ui_polchemdef_wnd_polchemdef_save_button (NULL, defctxt);

      if (FALSE == defctxt->is_modified) 
	{
	  /* File was successfully saved.
	   */
	  /* Remove the window from the array of opened windows.
	   */
	  if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
					   defctxt))
	    {
	      g_critical (_("%s@%d: failed to remove context from array.\n"),
		     __FILE__, __LINE__);
	    }
	  
	  polyxdef_defctxt_free (defctxt);  
	  
	  return TRUE;
	}
      
    case GTK_RESPONSE_NO:
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      /* Remove the window from the array of opened windows.
       */
      if (FALSE == g_ptr_array_remove (polyxdef_defctxtGPA, 
				       defctxt))
	{
	  g_critical (_("%s@%d: failed to remove context from array.\n"),
		 __FILE__, __LINE__);
	}
      
      polyxdef_defctxt_free (defctxt);  
      
      return TRUE;
      
      
    case GTK_RESPONSE_CANCEL:
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      return TRUE;
      
    default: 
      
      /* First destroy the message window.
       */
      gtk_widget_destroy (dialog);
      
      break;
    }

 return TRUE;
}


gboolean
polyxdef_ui_polchemdef_wnd_polchemdef_wnd_destroy_event (GtkWidget *window,
				  GdkEvent *event,
				  gpointer data)
{
  return FALSE;
}


