/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef PXMCHEM_OLIGOMER_H
#define PXMCHEM_OLIGOMER_H

#include "libpolyxmass-globals.h"
#include "pxmchem-polymer.h"
#include "pxmchem-calcopt.h"
#include "pxmchem-polchemdef.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


  /* typedef's
   */
  typedef struct _PxmOligomer PxmOligomer;

  /* An oligomer struct contains enough data to characterize  
   * a portion of a polymer. In this program, an oligomer is 
   * always considered to be a portion of an existing polymer sequence. 
   * Since the polymer sequence has been in existence when the oligomer 
   * has been constructed, the characterization of the oligomer can be 
   * done by simply giving two gint values : the start_idx (index, in the  
   * polymer sequence, of the first monomer in the oligomer) and  
   * end_idx (index, in the polymer sequence, of the last monomer in 
   * the oligomer). In a word, an oligomer is a delimited stretch of a  
   * polymer sequence. Also note, that if the start_idx is 0 and 
   * the end_idx is the (polymer length - 1), then the oligomer is nothing 
   * else than the full polymer.
   *
   * Taking the following sequence (numbers are indexes):
   * 0-ATGCHGIKLPRDVCS-14 
   * we can define three oligomers : 
   * 1. (0,5) = ATGCHG 
   * 2. (4,10) = HGIKLPR 
   * 3. (0,14) = ATGCHGIKLPRDVCS  (full polymer sequence !).
   *
   * The characterization of the oligomer can be extended using the 
   * prop instance mechanism. For example, upon construction of 
   * the oligomers above, we may desire that each oligomer be 
   * aware of its own monomer sequence (because we know that the oligomer 
   * will survive after the polymer has been destroyed, for example).  
   * This is easily achieved while constructing the oligomer, by creating 
   * a prop object with a name string "SEQUENCE" and a data string 
   * "ATGCHG". By this mechanism it is possible to characterize in the 
   * finest detail a given oligomer.
   */
  struct _PxmOligomer
  {
    /* The pointer to the polymer to which this oligomer refers. This
       is not a pointer obtained by allocating a PxmPolymer. It must not
       be freed, it is only a reference.
    */
    PxmPolymer *polymer;
  
    /* The name of the oligomer, which is a convenience, can be NULL
     * during oligomer operation.
     */
    gchar *name;
    /* Index of the first monomer in the polymer sequence that 
     * will be the left end of the oligomer. Must be >= 0 and less 
     * than (polymer length - 1).
     */
    gint start_idx;
    /* Index of the last monomer in the polymer sequence that will be
       the right end of the oligomer. Must be >= 0 and less than
       (polymer length - 1).
     */
    gint end_idx;

    /* The masses of the oligomer, with no specific way to know how
       these were calculated. So use prop objects to add informations if
       necessary.
    */
    PxmMasspair *masspair;
    
    GPtrArray *propGPA;
  };



  /* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
   */
  PxmOligomer *
  pxmchem_oligomer_new (void);

  PxmOligomer *
  pxmchem_oligomer_dup (const PxmOligomer *oligomer, PxmHowDup how_dup);

  PxmProp *
  pxmchem_oligomer_prop_dup (const PxmProp *prop, PxmHowDup how_dup);

  PxmOligomer *
  pxmchem_oligomer_with_options_new (PxmPolymer *polymer,
				     gint start_idx,
				     gint end_idx,
				     PxmCalcOpt *calcopt,
				     PxmIonizerule *ionizerule,
				     PxmPlmChement plm_chement,
				     PxmPolchemdef *polchemdef);

  gboolean
  pxmchem_oligomer_set_name (PxmOligomer *oligomer, gchar *name);

  gboolean
  pxmchem_oligomer_set_mono (PxmOligomer *oligomer, gdouble mass);

  gboolean
  pxmchem_oligomer_set_avg (PxmOligomer *oligomer, gdouble mass);


  /* COMPARISON FUNCTIONS
   */

  gint
  pxmchem_oligomer_cmp (PxmOligomer *oligomer1, PxmOligomer *oligomer2, 
			PxmHowCmp how_cmp);
  
  gint
  pxmchem_oligomer_prop_cmp (PxmProp *prop1, PxmProp *prop2, PxmHowCmp how_cmp);
  


  /* INTEGRITY CHECKING FUNCTIONS
   */
  gboolean
  pxmchem_oligomer_check_boundaries (PxmOligomer *oligomer,
				     PxmPolymer *polymer);


  /*  LOCATING FUNCTIONS
   */
  PxmOligomer *
  pxmchem_oligomer_get_ptr_by_name (GPtrArray *GPA, gchar *name);


  PxmOligomer *
  pxmchem_oligomer_find_that_encompasses_idx (gint monomer_idx,
					      GPtrArray *GPA,
					      gint *item_idx);

  PxmProp *
  pxmchem_oligomer_find_prop (PxmOligomer *oligomer,
			      PxmProp *prop,
			      PxmHowCmp how_cmp,
			      gint *idx);
  
  PxmProp *
  pxmchem_oligomer_find_prop_in_monomers (PxmOligomer *oligomer,
					  PxmProp *prop,
					  PxmHowCmp how_cmp,
					  gint *idx);
  





  /* UTILITY FUNCTIONS
   */

  


  /* XML-format TRANSACTIONS
   */
  gchar *
  pxmchem_oligomer_format_xml_string_olm (PxmOligomer *oligomer, 
					  gchar *indent, gint offset);


  PxmOligomer *
  pxmchem_oligomer_render_xml_node_olm (xmlDocPtr xml_doc,
					xmlNodePtr xml_node,
					gpointer user_data);


  /* FREE'ING FUNCTIONS
   */
  gboolean
  pxmchem_oligomer_free (PxmOligomer *oligomer);


  /* GPtrArray-RELATED FUNCTIONS
   */
  gint
  pxmchem_oligomer_GPA_free (GPtrArray *GPA);


  /* Frees an array of arrays of oligomers.
   */
  gint
  pxmchem_oligomer_GPA_of_GPA_free (GPtrArray *GPA);
  



#ifdef __cplusplus
}
#endif /* __cplusplus */



#endif /* PXMCHEM_OLIGOMER_H */
