/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "pxmchem-findmass.h"



gint
pxmchem_findmass_oligomer_GPA_of_GPA (GPtrArray *haystackGPA,
				      GPtrArray *GPA,
				      PxmFindmassOpt *fmopt)
{
  gint count = 0;
  gint iter = 0;
  gint jter = 0;
  
  gdouble min = 0;
  gdouble max = 0;

  GPtrArray *oligsGPA = NULL;
  
  PxmOligomer *oligomer_test = NULL;
  PxmOligomer *oligomer = NULL;

  PxmProp *prop = NULL;
  

  /* We get a pointer to an array of oligomer arrays. In this array of
     arrays, we are going to find for oligomers which have masses that
     match the mass passed in the PxmFindmassOpt * parameter. The
     oligomer arrays contained in haystackGPA are going to be iterated
     and for each oligomer its mass (MONO or AVG) will be checked
     against the one in fmopt. Each oligomer found to comply with the 
     finding parameters will be put in GPA.
  
     The GPA array is an array where all the found oligomers (the ones
     the masses of which comply with the mass being found for) are
     to be stored.
  */

  g_assert (haystackGPA != NULL);
  g_assert (GPA != NULL);

  g_assert (fmopt != NULL);
  
  /* We have to calculate the min and max values. The calculation
     depends on the value of the fmopt->tolerance_type variable.
   */
  if (fmopt->tolerance_type == PXM_MASS_TOLERANCE_AMU)
    {
      /* atomic mass units 
       */
      min = fmopt->mass - fmopt->tolerance;
      max = fmopt->mass + fmopt->tolerance;
    }
  else if (fmopt->tolerance_type == PXM_MASS_TOLERANCE_PCT)
    {
      /* percentage
       */
      min = fmopt->mass - (fmopt->tolerance * (fmopt->mass /100));
      max = fmopt->mass + (fmopt->tolerance * (fmopt->mass /100));
    }
  else if (fmopt->tolerance_type == PXM_MASS_TOLERANCE_PPM)
    {
      /* part per million */
      min = fmopt->mass - (fmopt->tolerance * (fmopt->mass /1000000));
      max = fmopt->mass + (fmopt->tolerance * (fmopt->mass /1000000));
    }
  else
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: the value of the tolerance type is unknown: '%d'\n"),
	     __FILE__, __LINE__, fmopt->tolerance_type);
      
      return -1;
    }

  /* Iterate in the array of oligomer arrays.
   */
  for (iter = 0; iter < haystackGPA->len; iter++)
    {
      oligsGPA = g_ptr_array_index (haystackGPA, iter);
      g_assert (oligsGPA != NULL);
      
      for (jter = 0; jter < oligsGPA->len; jter++)
	{
	  /* Get the iterated oligomer.
	   */
	  oligomer_test = g_ptr_array_index (oligsGPA, jter);
	  g_assert (oligomer_test != NULL);
	  
	  /* At this point we have the masses of the currently tested
	     oligomer. We must compared these to the one that is found
	     for.
	  */
	  if (fmopt->mass_type == PXM_MASS_MONO)
	    {
	      if (oligomer_test->masspair->mono > max)
		{
		  continue;
		}
	      
	      if (oligomer_test->masspair->mono < min)
		{
		  continue;
		}
	    }
	  else if (fmopt->mass_type == PXM_MASS_AVG)
	    {
	      if (oligomer_test->masspair->avg > max)
		{
		  continue;
		}
	      
	      if (oligomer_test->masspair->avg < min)
		{
		  continue;
		}
	    }
	  else
	    {
	      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		     _("%s@%d: the mass type is incorrect: '%d'\n"),
		     __FILE__, __LINE__, fmopt->mass_type);
	      
	      return -1;
	    }
	  
	      
	  /* Here, we know that the oligomer has masses complying
	     with the [min--max] mass that was initially found
	     for.
	  */
	  oligomer = pxmchem_oligomer_dup (oligomer_test,
					   PXM_DUP_DEEP);
	  
	  /* Make a property with the fmopt struct after duplication
	     so that the oligomer will be totally self-contained.
	  */
	  prop = libpolyxmass_prop_new ();
	  libpolyxmass_prop_set_name (prop, "FINDMASS_OPT");
	      
	  prop->data = pxmchem_findmassopt_dup (fmopt, 
						PXM_DUP_NO_DEEP);
	      	      
	  /* Set the pointer to the free'ing function that will be called
	     when the prop object is freed.
	  */
	  prop->custom_free = pxmchem_findmassopt_prop_free;
	  prop->custom_dup = pxmchem_findmassopt_prop_dup;
	      
	  g_ptr_array_add (oligomer->propGPA, prop);

	  /* Finally add the oligomer to the array of oligomers
	     and increment effectively the count of added oligomers.
	  */
	  g_ptr_array_add (GPA, oligomer);
	      
	  count++;
	}
      /* end of 
	 for (jter = 0; jter < calcopt->end_idx + 1 - iter; jter++)
      */
    }

  return count;
}






