/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public Licen'se and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxmass-ui-mz-ratio.h"
#include "polyxedit-ui-seqed-wnd.h"



GtkWidget *
polyxmass_mz_ratio_wnd_setup (GPtrArray *atomGPA,
			      PxmIonizerule *ionizerule,
			      GtkWindow *parent_wnd)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;

  GPtrArray *masspairGPA = NULL;

  PxmMasspair *initial_masspair = NULL;

  PxmIonizerule *initial_ionizerule = NULL;
  PxmIonizerule *dest_ionizerule = NULL;
  
  g_assert (atomGPA != NULL);
  g_assert (ionizerule != NULL);
  g_assert (parent_wnd != NULL);



  gui_file = 
    g_strdup_printf ("%s/polyxmass-mz-ratio.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "mz_ratio_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed loading the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "mz_ratio_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed creating the m-to-z ratio calculator "
	       "config window\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }

  /* Immediately set to the window a pointer to the parent_wnd passed
     as parameter.
  */
  g_object_set_data (G_OBJECT (window), "parent_wnd", parent_wnd);
  
  /* Immediately set to the window a pointer to the atomGPA:
   */
  g_object_set_data (G_OBJECT (window), "atomGPA", atomGPA);

  /* Allocate the array where the different masspair objects,
     corresponding to the various m/z ratios computed for different
     charge levels are going to be stored.
  */
  masspairGPA = g_ptr_array_new ();
  g_object_set_data (G_OBJECT (window), "masspairGPA", masspairGPA);

  /* Immediately allocate the masspair object that well use
     continuously.  doubles that we will feed continuously when
     performing calculations...
  */
  initial_masspair = libpolyxmass_masspair_new ();
  g_object_set_data (G_OBJECT (window), 
		     "initial_masspair", initial_masspair);
  
  initial_ionizerule = pxmchem_ionizerule_dup (ionizerule);
  g_object_set_data (G_OBJECT (window), 
		     "initial_ionizerule", initial_ionizerule);
   
  dest_ionizerule = pxmchem_ionizerule_dup (ionizerule);
  g_object_set_data (G_OBJECT (window), "dest_ionizerule", dest_ionizerule);


  /* We may need to display timed-out messages in here:
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);
  

  widget = glade_xml_get_widget (xml, "initial_mono_mz_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "initial_mono_mz_entry", widget);

  widget = glade_xml_get_widget (xml, "initial_avg_mz_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "initial_avg_mz_entry", widget);

  widget = glade_xml_get_widget (xml, "initial_ionization_actform_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "initial_ionization_actform_entry", widget);

  gtk_entry_set_text (GTK_ENTRY (widget), initial_ionizerule->actform);
  
  widget = 
    glade_xml_get_widget (xml, 
			  "initial_ionization_unitary_charge_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "initial_ionization_unitary_charge_spinbutton", widget);
  
  help = g_strdup_printf ("%d", initial_ionizerule->charge);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  widget = glade_xml_get_widget (xml, "initial_ionization_level_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "initial_ionization_level_spinbutton", widget);

  help = g_strdup_printf ("%d", initial_ionizerule->level);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  

  widget = glade_xml_get_widget (xml, "asked_ionization_actform_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "asked_ionization_actform_entry", widget);

  gtk_entry_set_text (GTK_ENTRY (widget), initial_ionizerule->actform);


  widget = 
    glade_xml_get_widget (xml, 
			  "asked_ionization_unitary_charge_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "asked_ionization_unitary_charge_spinbutton", widget);
  
  help = g_strdup_printf ("%d", initial_ionizerule->charge);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  widget =
    glade_xml_get_widget (xml, 
			  "asked_starting_ionization_level_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "asked_starting_ionization_level_spinbutton", widget);
  
  help = g_strdup_printf ("%d", initial_ionizerule->level);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  widget =
    glade_xml_get_widget (xml, 
			  "asked_ending_ionization_level_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "asked_ending_ionization_level_spinbutton", widget);

  help = g_strdup_printf ("%d", initial_ionizerule->level);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  widget = glade_xml_get_widget (xml, "compute_net_charge_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "compute_net_charge_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxmass_mz_ratio_wnd_compute_mz_ratio_button), 
		    window);


  /*  We now have to set up the treeview where the m/z ratios are to
      be displayed.
  */
  widget = glade_xml_get_widget (xml, "treeview_vbox");
  g_assert (widget != NULL);

  g_object_set_data (G_OBJECT (window), "treeview_vbox", widget);
  
  polyxmass_mz_ratio_wnd_setup_mz_ratios_treeview (window, widget);
  


  gtk_widget_show_all (GTK_WIDGET (window));


  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxmass_mz_ratio_wnd_delete_event), 
		    parent_wnd);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxmass_mz_ratio_wnd_destroy_event), 
		    parent_wnd);
  


  /* We have finished setting up the window, and so also using the xml
     data, unref them.
   */
  g_object_unref (G_OBJECT (xml));
  
  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one monomer modification window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("mz_ratio_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (parent_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxmass_mz_ratio_wnd_really_close);
  
  g_free (help);
  
  return window;
}


/* Returns -1 if an error occurs. Returns the number of pka that were
   processed.
 */
gint
polyxmass_mz_ratio_wnd_compute_mz_ratio_button (GtkWidget *button,
						gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  gboolean result = FALSE;

  gint iter = 0;  
  gint start_mz = 0;
  gint end_mz = 0;
  gint moment_mz = 0;
  
  GPtrArray *atomGPA = NULL;
  GPtrArray *masspairGPA = NULL;

  PxmIonizerule *initial_ionizerule = NULL;
  PxmIonizerule *dest_ionizerule = NULL;
  PxmMasspair *initial_masspair = NULL;
  PxmMasspair *masspair = NULL;
  

  g_assert (window != NULL);
  
  atomGPA = g_object_get_data (G_OBJECT (window), "atomGPA");
  g_assert (atomGPA != NULL);
  
  masspairGPA = g_object_get_data (G_OBJECT (window), "masspairGPA");
  g_assert (masspairGPA != NULL);

  /* We want to empty the array because it might have been populated
     already during a previous call.
  */
  libpolyxmass_masspair_GPA_empty (masspairGPA);
  
  /* 
     First get the current ionization data and the asked one.  Also
     get the initial m/z values.
     
     We'll use the current ionization data to revert from initial m/z
     values to the true molecular mass of the analyte.

     When we have the molecular mass of the analyte, then we can start
     computing the new asked m/z values for the z in the
     [start_mz-end_mz] range.
   */  
  result = polyxmass_mz_ratio_wnd_update_initial_ionizerule (window);

  if (FALSE == result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed getting the initial "
					"ionization data",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return -1;
    }


  result = polyxmass_mz_ratio_wnd_update_dest_ionizerule (window);

  if (FALSE == result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed getting the requested"
					"ionization data",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return -1;
    }

  
  result = polyxmass_mz_ratio_wnd_update_initial_masspair (window);
  
  if (FALSE == result)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed getting the initial "
					"mono/avg m/z values",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return -1;
    }

  /* Now that we have update all the data required for the calculations,
     we ought to get a pointer to all these objects, so that we can 
     actually make use of them.
  */

  /* Get the initial m/z values in the initial_masspair object.
   */
  initial_masspair =
    g_object_get_data (G_OBJECT (window), "initial_masspair");
  g_assert (initial_masspair != NULL);

  /* Get the initial_ionizerule and ionizerule objects that were setup
     with data above in its corresponding function call.
  */
  initial_ionizerule = 
    g_object_get_data (G_OBJECT (window), "initial_ionizerule");
  g_assert (initial_ionizerule != NULL);
  
  dest_ionizerule = g_object_get_data (G_OBJECT (window), "dest_ionizerule");
  g_assert (dest_ionizerule != NULL);


  /* Now we should compute the molecular mass of the analyte, based on
     the provided m/z values and on the provided initial_ionizerule
     data. Once we have the molecular mass of the analyte, we can
     start computing new m/z ratios according to the ionization level
     range asked by the user.
  */
  if (FALSE == pxmchem_masscalc_decount_ionizerule (initial_ionizerule,
						    initial_masspair,
						    atomGPA))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed de-ionizing the analyte",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return -1;
    }
  
  /*
    debug_printf (("Molecular mass of mono = %.5f and avg = %.5f\n",
    initial_masspair->mono, initial_masspair->avg));
  */
  
  /* At this point we should get the z interval asked by the user.
  */
  widget = g_object_get_data (G_OBJECT (window), 
			      "asked_starting_ionization_level_spinbutton");
  g_assert (widget != NULL);
  start_mz = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
  
  widget = g_object_get_data (G_OBJECT (window), 
			      "asked_ending_ionization_level_spinbutton");
  g_assert (widget != NULL);
  end_mz = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
  
  if (start_mz > end_mz)
    {
      moment_mz = start_mz;
      start_mz = end_mz;
      end_mz = moment_mz;
    }

  /* At this point we should perform the actual calculations requested
     by the user. Note that we will always keep 'initial_masspair'
     untouched, as we'll use its data to initialize the 'masspair'
     object in which the computations are performed.
  */
  
  for (iter = start_mz; iter <= end_mz; iter++)
    {
      /* Allocate a masspair object and set the right mono/avg data in
	 it.  Later, we will set the different charge/level data to
	 its dest_ionizerule member.
      */
      masspair = libpolyxmass_masspair_new ();
      
      masspair->mono = initial_masspair->mono;
      masspair->avg = initial_masspair->avg;

      masspair->ionizerule->actform = g_strdup (dest_ionizerule->actform);
      masspair->ionizerule->charge = dest_ionizerule->charge;
      masspair->ionizerule->level = iter;
      
      /* Go on with the ionization proper.
       */
      if (FALSE == pxmchem_masscalc_account_ionizerule (masspair->ionizerule,
							masspair,
							atomGPA))
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					    "Failed ionizing the analyte",
					    POLYXMASS_MEDI_MSG_TIMEOUT);

	  libpolyxmass_masspair_free (masspair);
	  
	  return -1;
	}
      
      g_ptr_array_add (masspairGPA, masspair);
      
      /*
	debug_printf (("added masspair m/z mono = %.5f and m/z avg = %.5f\n",
	masspair->mono, masspair->avg));
      */
    }
  /* End of for (iter = start_mz; iter <= end_mz; iter++)
   */

  /* At this point we have to update the data displayed in the 
     treeview...
  */
  polyxmass_mz_ratio_wnd_update_mz_ratios_treeview (window);
  
     
  return 1;
}



gboolean
polyxmass_mz_ratio_wnd_update_initial_ionizerule (GtkWidget *window)
{
  GtkWidget *widget = NULL;
  
  PxmIonizerule *initial_ionizerule = NULL;

  GPtrArray *atomGPA = NULL;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);

  atomGPA = g_object_get_data (G_OBJECT (window), "atomGPA");
  g_assert (atomGPA != NULL);
  
  initial_ionizerule = g_object_get_data (G_OBJECT (window), 
					  "initial_ionizerule");
  g_assert (initial_ionizerule != NULL);
  

  /* Get the ionization actform !
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "initial_ionization_actform_entry");
  g_assert (widget != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (widget));

  /* We want to make sure we can parse that formula OK.
   */
  if (FALSE == pxmchem_actform_check (help, atomGPA))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed checking the initial"
					"ionization actform",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return FALSE;
    }
  
  pxmchem_ionizerule_set_actform (initial_ionizerule, help);
  

  /* Get the ionization unitary charge !
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "initial_ionization_unitary_charge_spinbutton");
  g_assert (widget != NULL);
  
  initial_ionizerule->charge = 
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
  
  /* Get the ionization level !
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "initial_ionization_level_spinbutton");
  g_assert (widget != NULL);
  
  initial_ionizerule->level = 
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
  
  /* GREAT, we have finished qualifying the initial ionizerule object
     which describes most precisely the m/z values that are displayed on top
     of the window. We can return TRUE proudly.
  */
  return TRUE;
}


gboolean
polyxmass_mz_ratio_wnd_update_dest_ionizerule (GtkWidget *window)
{
  GtkWidget *widget = NULL;
  
  PxmIonizerule *dest_ionizerule = NULL;

  GPtrArray *atomGPA = NULL;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);

  atomGPA = g_object_get_data (G_OBJECT (window), "atomGPA");
  g_assert (atomGPA != NULL);

  dest_ionizerule = g_object_get_data (G_OBJECT (window), 
				       "dest_ionizerule");
  g_assert (dest_ionizerule != NULL);
  

  /* Get the ionization actform !
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "asked_ionization_actform_entry");
  g_assert (widget != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (widget));

  /* We want to make sure we can parse that formula OK.
   */
  if (FALSE == pxmchem_actform_check (help, atomGPA))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Failed checking the asked"
					"ionization actform",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return FALSE;
    }
  
  pxmchem_ionizerule_set_actform (dest_ionizerule, help);
  

  /* Get the ionization unitary charge !
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "asked_ionization_unitary_charge_spinbutton");
  g_assert (widget != NULL);
  
  dest_ionizerule->charge = 
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));

  /* Here the situation is a bit tricky as to what pertains to the
     ionization level, as the user asks for a range of levels, from a
     starting ionization level to an ending ionization level. m/z
     ratios will have to be computed for all the levels in the range,
     borders comprised.

     So we do not do that work here, we keep it for another place.
  */
  
  dest_ionizerule->level = 
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (widget));
  
  /* GREAT, we have finished qualifying the ionizerule object which
     describes most precisely how the new m/z values are to be
     computed with the ionization levels displayed at the bottom of
     the window. We can return TRUE proudly.
  */
  return TRUE;
}


gboolean
polyxmass_mz_ratio_wnd_update_initial_masspair (GtkWidget *window)
{
  GtkWidget *widget = NULL;
  
  PxmMasspair *initial_masspair = NULL;

  gchar *help = NULL;
  

  g_assert (window != NULL);

  
  initial_masspair = g_object_get_data (G_OBJECT (window), 
					"initial_masspair");
  g_assert (initial_masspair != NULL);
  
  
  widget = g_object_get_data (G_OBJECT (window), 
			      "initial_mono_mz_entry");
  g_assert (widget != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (widget));


  if (FALSE == libpolyxmass_globals_strtod (help, &initial_masspair->mono))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Bad initial mono m/z value",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return FALSE;
    }


  widget = g_object_get_data (G_OBJECT (window), 
			      "initial_avg_mz_entry");
  g_assert (widget != NULL);

  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (widget));


  if (FALSE == libpolyxmass_globals_strtod (help, &initial_masspair->avg))
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					"Bad initial avg m/z value",
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return FALSE;
    }
  
  /* At this point, we have finished doing the job. 
   */
  polyxmass_mz_ratio_wnd_update_mz_ratios_treeview (window);
  
  return TRUE;
}



GtkTreeModel *
polyxmass_mz_ratio_wnd_create_treeview_model (GPtrArray *masspairGPA)
{
  GtkTreeStore *model;
  GtkTreeIter treeiter;
  
  gint iter = 0;
  gint *count = NULL;
    
  PxmMasspair *masspair = NULL;
  

  g_assert (masspairGPA != NULL);
  
  model = gtk_tree_store_new (COLUMN_COLUMN_COUNT,
			      /* Charge */
			      G_TYPE_INT,
			      /* Level */
			      G_TYPE_INT,
			      /* Mono mass */
			      G_TYPE_DOUBLE,
			      /* Avg Mass */
			      G_TYPE_DOUBLE,
			      /* Pointer to the masspair */
			      G_TYPE_POINTER);
  
  count = g_malloc0 (sizeof (gint));

  for (iter = 0; iter < masspairGPA->len; iter++)
    {
      masspair = g_ptr_array_index (masspairGPA, iter);
      g_assert (masspair != NULL);

      gtk_tree_store_append (model, &treeiter, NULL);

      gtk_tree_store_set 
	(model, &treeiter,
	 
	 COLUMN_CHARGE, masspair->ionizerule->charge,
	 
	 COLUMN_LEVEL, masspair->ionizerule->level,
	 
	 COLUMN_MONO, masspair->mono,
	 
	 COLUMN_AVG, masspair->avg,
	 
	 COLUMN_MASSPAIR_POINTER, masspair,
	 -1);
      
      (*count)++;
    }
  /* end of 
     for (iter = 0; iter < masspairGPA->len; iter++)
  */

  /* Before returning the model set to it the number of items we have
     set to it. The integer that was allocated will be automatically
     freed upon destruction of the model.
  */
  g_object_set_data_full (G_OBJECT (model), "total_m/z", count,
			  (GDestroyNotify) g_free);
  
  return GTK_TREE_MODEL (model);
}


gboolean
polyxmass_mz_ratio_wnd_setup_mz_ratios_treeview (GtkWidget *window, 
						 GtkWidget *vbox)
{
  GPtrArray *masspairGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;


  g_assert (window != NULL);
  g_assert (vbox != NULL);
  

  masspairGPA = g_object_get_data (G_OBJECT (window), "masspairGPA");
  g_assert (masspairGPA != NULL);
  

  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxmass_mz_ratio_wnd_create_treeview_model (masspairGPA);

  g_object_set_data (G_OBJECT (model), "masspairGPA", masspairGPA);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), "treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  
  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);


  /* The charge column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Unit. Charge",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_CHARGE,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_CHARGE);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Level column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Ioniz. Level",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_LEVEL,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_LEVEL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Mono mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Mono Mass",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_MONO,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_MONO);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Avg mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes ("Avg Mass",
					      
					      renderer, 
					      
					      "text",
					      COLUMN_AVG,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_AVG);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Masspair pointer column.
   */
  renderer = gtk_cell_renderer_text_new ();
  
  column = 
    gtk_tree_view_column_new_with_attributes ("Masspair Pointer",
					      
					      renderer,
					      
					      "text", 
					      COLUMN_MASSPAIR_POINTER,
					      
					      NULL);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_visible (column, FALSE),
  g_object_set (G_OBJECT (renderer), "visible", FALSE, NULL);

  /* Finally add the treeview to the scrooll window.
   */
  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}


gint
polyxmass_mz_ratio_wnd_empty_mz_ratios_treeview_store (GtkWidget *window)
{
  GtkTreeModel *model = NULL;
  GtkTreeView *treeview = NULL;
  
  

  g_assert (window != NULL);
  
  treeview = (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
						"treeview");
  g_assert (treeview != NULL);
  
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  gtk_tree_store_clear (GTK_TREE_STORE (model));
  
  return 1;
}


gint
polyxmass_mz_ratio_wnd_update_mz_ratios_treeview (GtkWidget *window)
{
  gint count = 0;
  gint iter = 0;

  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeView *treeview = NULL;
  GtkTreePath *path = NULL;

  PxmMasspair *masspair = NULL;
  
  GPtrArray *masspairGPA = NULL;

  
  g_assert (window != NULL);

  masspairGPA = g_object_get_data (G_OBJECT (window), "masspairGPA");
  g_assert (masspairGPA != NULL);

  
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (window), "treeview");
  
  model = gtk_tree_view_get_model (treeview);
  
  path = gtk_tree_path_new_first ();  

  polyxmass_mz_ratio_wnd_empty_mz_ratios_treeview_store (window);
    
  /* Now that we have emptied the store, we can append the items that 
   * are in the GPA.
   */
  for (iter = 0 ; iter < masspairGPA->len ; iter++)
    {
      /* Acquire an iterator.
       */
      gtk_tree_store_append (GTK_TREE_STORE (model), &treeiter, NULL);
      
      masspair = g_ptr_array_index (masspairGPA, iter);
      g_assert (masspair != NULL);
      
      /* We now have a pointer to the masspair object.
       */
      gtk_tree_store_set 
	(GTK_TREE_STORE (model), &treeiter,
	 
	 COLUMN_CHARGE, masspair->ionizerule->charge,
	 
	 COLUMN_LEVEL, masspair->ionizerule->level,
	 
	 COLUMN_MONO, masspair->mono,
	 
	 COLUMN_AVG, masspair->avg,
	 
	 COLUMN_MASSPAIR_POINTER, masspair,
	 -1);
 
      count++;
    }
  
  return count;
}

















/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxmass_mz_ratio_wnd_really_close (GtkWidget *window)
{
  gint count = 0;

  PxmIonizerule *initial_ionizerule = NULL;
  PxmMasspair *initial_masspair = NULL;
  PxmIonizerule *dest_ionizerule = NULL;
  
  GPtrArray *masspairGPA = NULL;

  
  g_assert (window != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
   
  /* And that both the masspair and ionizerule objects get freed !
   */
  initial_masspair = g_object_get_data (G_OBJECT (window), 
					"initial_masspair");
  g_assert (initial_masspair != NULL);
  libpolyxmass_masspair_free (initial_masspair);
  /*
    debug_printf (("freed initial_masspair\n"));
  */

  initial_ionizerule = g_object_get_data (G_OBJECT (window), 
					  "initial_ionizerule");
  g_assert (initial_ionizerule != NULL);
  pxmchem_ionizerule_free (initial_ionizerule);
  /*
    debug_printf (("freed initial_ionizerule\n"));
  */

  dest_ionizerule = g_object_get_data (G_OBJECT (window), "dest_ionizerule");
  g_assert (dest_ionizerule != NULL);
  pxmchem_ionizerule_free (dest_ionizerule);
  /*
    debug_printf (("freed dest_ionizerule\n"));
  */

  masspairGPA = g_object_get_data (G_OBJECT (window), "masspairGPA");
  g_assert (masspairGPA != NULL);

  libpolyxmass_masspair_GPA_free (masspairGPA);
  /*
    debug_printf (("freed masspairGPA\n"));
  */


  gtk_widget_destroy (window);
}


gint
polyxmass_mz_ratio_wnd_delete_event  (GtkWidget *window, 
				      GdkEventAny *event, 
				      gpointer data)
{
  GtkWindow *parent_wnd = data;
  
  gchar *help = NULL;
  
  g_assert (window != NULL);
  g_assert (parent_wnd != NULL);
  

  help = g_strdup_printf ("mz_ratio_wnd-%p", window);
  

  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT (parent_wnd), help, NULL);
  g_free (help);
    
  return TRUE;
}


gint
polyxmass_mz_ratio_wnd_destroy_event  (GtkWidget *window, 
					 GdkEventAny *event, 
					 gpointer data)
{
  return FALSE;
}

