/*
 * $Id: prefsform.c,v 1.31 2001/09/22 11:13:50 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "os.h"
#include "prefsdata.h"
#include "prefsform.h"
#include "resourceids.h"
#include "screen.h"
#include "util.h"


static void PrefsFormInit( void ) PLKRDB_SECTION;
static void ReleaseLists( void ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define NUMBER_OF_FONTS     5
#define NUMBER_OF_OS2_FONTS 2


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static FontModeType     fontMode;
static ScrollbarType    scrollbar;
static MemHandle        depthHandle;
static MemHandle        fontHandle;
static UInt16           screenDepth;
static UInt16           toolbarButton;
static Char             fontList[ 60 ];
static Char             depthList[ 20 ];



/* Initialize the preference form */
static void PrefsFormInit( void )
{
    ListType*       list;
    ControlType*    ctl;
    FormType*       prefsForm;
    UInt16          selection;
    UInt16          i;
    UInt16          entries;
    Char            depth[ 3 ];
    Char*           temp;
    Char**          fonts;
    Char**          depths;
    UInt32          supportedDepths;

    prefsForm = FrmGetFormPtr( frmPrefs );

    FrmDrawForm( prefsForm );

    CtlSetValue( (ControlType*) GetObjectPtr( frmPrefsStrikethrough ), Prefs()->strikethrough );
    CtlSetValue( (ControlType*) GetObjectPtr( frmPrefsUnderline ), Prefs()->underlineMode );
    CtlSetValue( (ControlType*) GetObjectPtr( frmPrefsLinkClick ), Prefs()->linkClick );

    list        = (ListType*) GetObjectPtr( frmPrefsScrollbarList );
    ctl         = (ControlType*) GetObjectPtr( frmPrefsScrollbarPopup );
    scrollbar   = Prefs()->scrollbar;
    selection   = Prefs()->scrollbar;

    LstSetSelection( list, selection );
    CtlSetLabel( ctl, LstGetSelectionText( list, selection ) );

    screenDepth = Prefs()->screenDepth;

    if ( sysGetROMVerMajor( GetOSVersion() ) == 0x02 )
        supportedDepths = 1;
    else
        WinScreenMode( winScreenModeGetSupportedDepths, NULL, NULL, &supportedDepths, NULL );

    MemSet( (void*) &depthList, sizeof( depthList ), 0 );
    entries     = 0;
    selection   = 0;
    temp        = depthList;
    for ( i = 1; supportedDepths; i++ ) {
        if ( ( supportedDepths & 0x01 ) == 0x01 ) {
            StrIToA( depth, i );
            StrCat( temp, depth );
            temp += StrLen( depth ) + 1;
            if ( screenDepth == i )
                selection = entries;
            entries++;
        }
        supportedDepths >>= 1;
    }

    list        = (ListType*) GetObjectPtr( frmPrefsScreenDepthList );
    ctl         = (ControlType*) GetObjectPtr( frmPrefsScreenDepthPopup );
    depthHandle = SysFormPointerArrayToStrings( depthList, entries );
    depths      = (Char**) MemHandleLock( depthHandle );

    LstSetListChoices( list, depths, entries );
    LstSetHeight( list, entries );
    LstSetSelection( list, selection );
    CtlSetLabel( ctl, LstGetSelectionText( list, selection ) );

    list        = (ListType*) GetObjectPtr( frmPrefsMenuToolbarList );
    ctl         = (ControlType*) GetObjectPtr( frmPrefsMenuToolbarPopup );
    selection   = toolbarButton = Prefs()->toolbarButton;

    LstSetSelection( list, selection );
    CtlSetLabel( ctl, LstGetSelectionText( list, selection ) );

    list        = (ListType*) GetObjectPtr( frmPrefsFontModeList );
    ctl         = (ControlType*) GetObjectPtr( frmPrefsFontModePopup );
    selection   = fontMode = Prefs()->fontMode;

    MemSet( (void*) &fontList, sizeof( fontList ), 0 );
    temp    = fontList;
    SysCopyStringResource( temp, strPrefsStandardFont );
    temp   += StrLen( temp ) + 1;
    SysCopyStringResource( temp, strPrefsBoldFont );
    temp   += StrLen( temp ) + 1;
    SysCopyStringResource( temp, strPrefsLargeFont );
    temp   += StrLen( temp ) + 1;
    SysCopyStringResource( temp, strPrefsLargeBoldFont );
    temp   += StrLen( temp ) + 1;
    SysCopyStringResource( temp, strPrefsNarrowFont );

    if ( sysGetROMVerMajor( GetOSVersion() ) == 0x02 )
        entries = NUMBER_OF_OS2_FONTS;
    else
        entries = NUMBER_OF_FONTS;

    fontHandle  = SysFormPointerArrayToStrings( fontList, entries );
    fonts       = (Char**) MemHandleLock( fontHandle );

    LstSetListChoices( list, fonts, entries );
    LstSetHeight( list, entries );
    LstSetSelection( list, selection );
    CtlSetLabel( ctl, LstGetSelectionText( list, selection ) );
}



/* Unlock list handles */
static void ReleaseLists( void )
{
    if ( depthHandle != NULL ) {
        MemHandleUnlock( depthHandle );
        MemHandleFree( depthHandle );
        depthHandle = NULL;
    }
    if ( fontHandle != NULL ) {
        MemHandleUnlock( fontHandle );
        MemHandleFree( fontHandle );
        fontHandle = NULL;
    }
}



/* Event handler for the preference form */
Boolean PrefsFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;
    Boolean updateDoc;

    SET_A4_FROM_A5 

    handled   = false;
    updateDoc = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmPrefsOK ) {
                updateDoc = true;

                Prefs()->strikethrough  = CtlGetValue( (ControlType*) GetObjectPtr( frmPrefsStrikethrough ) );
                Prefs()->underlineMode  = CtlGetValue( (ControlType*) GetObjectPtr( frmPrefsUnderline ) );
                Prefs()->linkClick      = CtlGetValue( (ControlType*) GetObjectPtr( frmPrefsLinkClick ) );
                Prefs()->toolbarButton  = toolbarButton;

                if ( scrollbar != Prefs()->scrollbar ) {
                    Prefs()->scrollbar = scrollbar;
                    ResetHeight();
                }
                if ( fontMode != Prefs()->fontMode ) {
                    Prefs()->fontMode = fontMode;
                    ResetHeight();
                    SetFontStyles();
                }
                if ( screenDepth != Prefs()->screenDepth ) {
                    Prefs()->screenDepth = screenDepth;
                }
            }
            else if ( event->data.ctlEnter.controlID != frmPrefsCancel )
                break;

            ReleaseLists();
            FrmReturnToForm( Prefs()->lastForm );
            if ( updateDoc )
                FrmUpdateForm( Prefs()->lastForm, frmRedrawUpdateCode );
            handled = true;
            break;

        case popSelectEvent:
        {
            Int16       selection;

            selection = event->data.popSelect.selection;
            if ( selection != noListSelection ) {
                ControlType*    ctl;
                ListType*       list;
                Char*           label;
                UInt16          controlID;

                list        = event->data.popSelect.listP;
                controlID   = event->data.popSelect.controlID;
                ctl         = (ControlType*) GetObjectPtr( controlID );
                label       = LstGetSelectionText( list, selection );

                CtlSetLabel( ctl, label );
                LstSetSelection( list, selection );

                switch ( controlID ) {
                    case frmPrefsScrollbarPopup:
                        scrollbar = (ScrollbarType) selection;
                        break;

                    case frmPrefsScreenDepthPopup:
                        screenDepth = StrAToI( label );
                        break;

                    case frmPrefsMenuToolbarPopup:
                        toolbarButton = selection;
                        break;

                    case frmPrefsFontModePopup:
                        fontMode = (FontModeType) selection;
                        break;

                    default:
                        break;
                }
                handled = true;
            }
            break;
        }

        case frmOpenEvent:
            PrefsFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            ReleaseLists();
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
