/*
 * $Id: mainform.c,v 1.41 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "bookmark.h"
#include "bookmarkform.h"
#include "const.h"
#include "control.h"
#include "debug.h"
#include "document.h"
#include "emailform.h"
#include "history.h"
#include "image.h"
#include "link.h"
#include "mainform.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "prefsform.h"
#include "resourceids.h"
#include "resultform.h"
#include "search.h"
#include "util.h"
#include "screen.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define TOP     0
#define BOTTOM  10

const UInt16 ToolbarButtonBitmaps[]     = { 0, bmpBookmark, BarInfoBitmap,
                                            bmpSettings, bmpBtnAction,
                                            bmpTapAction, bmpTop, bmpBottom };

const UInt16 ToolbarButtonCommands[]    = { 0, mGoAddBookmark, mOptionsDetails,
                                            mOptionsPref, mOptionsButton,
                                            mOptionsControl, mGoTop, mGoBottom };



/* Handle common menu items */
Boolean HandleCommonMenuItems
    (
    UInt16 itemID   /* ID for selected menu item */
    )
{
    Boolean     handled;

    handled = false;
    switch ( itemID ) {
        case mOptionsAbout:
        {
            FormType* form;

            form = FrmInitForm( frmAbout );
            FrmDoDialog( form );
            FrmDeleteForm( form );
            handled = true;
            break;
        }

        case mOptionsContact:
            SetMailto( 0 );
            FrmGotoForm( frmEmail );
            handled = true;
            break;

        case mOptionsPref:
            FrmPopupForm( frmPrefs );
            handled = true;
            break;

        case mOptionsButton:
            FrmPopupForm( frmButton );
            handled = true;
            break;

        case mOptionsControl:
            FrmPopupForm( frmControl );
            handled = true;
            break;

        case mOptionsGesture:
            FrmPopupForm( frmGesture );
            handled = true;
            break;
            
        case mOptionsScl:
            FrmPopupForm( frmScl );
            handled = true;
            break;

        default:
            break;
    }
    return handled;
}



/* Handle a menu event */
static Boolean HandleMenuEvent
    (
    UInt16 itemID   /* ID for selected menu item */
    )
{
    Boolean handled;

    handled = false;
    if ( HandleCommonMenuItems( itemID ) )
        return true;
    else {
        switch ( itemID ) {
            case mGoHome:
            case mGoBack:
            case mGoForward:
                DoControlAction( mGoHome - itemID - 1 );
                handled = true;
                break;

            case mGoTop:
                GotoLocation( TOP );
                handled = true;
                break;

            case mGoBottom:
                GotoLocation( BOTTOM );
                handled = true;
                break;

            case mGoSearch:
                FrmPopupForm( frmSearch );
                handled = true;
                break;

            case mGoSearchAgain:
                SearchAgain();
                handled = true;
                break;

            case mGoAddBookmark:
                FrmPopupForm( frmAddBookmark );
                handled = true;
                break;

            case mGoBookmarks:
                FrmPopupForm( frmBookmarks );
                handled = true;
                break;

            case mViewTopToolbar:
                Prefs()->toolbar = frmMainTop;
                FrmGotoForm( frmMainTop );
                handled = true;
                break;

            case mViewBottomToolbar:
                Prefs()->toolbar = frmMainBottom;
                FrmGotoForm( frmMainBottom );
                handled = true;
                break;

            case mViewNoToolbar:
            {
                MetaRecord* meta;
                Int16       adjustment;
     
                meta = (MetaRecord*) MemHandleLock( GetMetaRecord() );
                ErrFatalDisplayIf( meta == NULL, "MemHandleLock failed" );
                if ( meta->height <= TopLeftY() + ExtentY() )
                    adjustment = -meta->verticalOffset;
                else
                    adjustment = 0;

                MemHandleUnlock( GetMetaRecord() );
     
                DoPageMove( adjustment );
                Prefs()->toolbar = frmMainNone;
                FrmGotoForm( frmMainNone );
                handled = true;
                break;
            }
            
            case mViewAutoscrollStart:
            	DoAutoscrollToggle( AUTOSCROLL_ON );
            	break;
            	
            case mViewAutoscrollStop:
                DoAutoscrollToggle( AUTOSCROLL_OFF );
                break;
                
            case mViewAutoscrollIncr:
                DoAutoscrollIncr();
                break;
                
            case mViewAutoscrollDecr:
                DoAutoscrollDecr();
                break;

            case mOptionsLibrary:
                FrmGotoForm( frmLibrary );
                handled = true;
                break;

            case mOptionsDetails:
                FrmPopupForm( frmDetails );
                handled = true;
                break;

            default:
                break;
        }
    }
    return handled;
}



/* Initialize the main form */
void MainFormInit( void )
{
    FormType*       mainForm;
    ScrollBarType*  scrollbar;
    UInt16          scrollbarIndex;

    mainForm        = FrmGetFormPtr( Prefs()->toolbar );
    scrollbarIndex  = FrmGetObjectIndex( mainForm, frmMainScrollBar );
    scrollbar       = (ScrollBarType*) FrmGetObjectPtr( mainForm, scrollbarIndex );

    if ( Prefs()->scrollbar == SCROLLBAR_LEFT )
        FrmSetObjectPosition( mainForm, scrollbarIndex, 0, TopLeftY() + 2 );
    else
        FrmSetObjectPosition( mainForm, scrollbarIndex, TopLeftX() + ExtentX(), TopLeftY() + 2 );

    FrmDrawForm( mainForm );

    if ( Prefs()->toolbar != frmMainNone ) {
        GetControlBounds( mainForm, bmpMenu, MENUCONTROL );
        GetControlBounds( mainForm, bmpBookmark, BOOKMARKCONTROL );
        GetControlBounds( mainForm, frmMainPercentPopup, OFFSETCONTROL );
        GetControlBounds( mainForm, bmpDbase, LIBRARYCONTROL );
        GetControlBounds( mainForm, bmpFind, FINDCONTROL );
        GetControlBounds( mainForm, bmpAgain, AGAINCONTROL );
        GetControlBounds( mainForm, bmpLeft, LEFTCONTROL );
        GetControlBounds( mainForm, bmpHome, HOMECONTROL );
        GetControlBounds( mainForm, bmpRight, RIGHTCONTROL );
        GetControlBounds( mainForm, bmpAutoscrollStart, AUTOSCROLLSTARTCONTROL );
        GetControlBounds( mainForm, bmpAutoscrollStop, AUTOSCROLLSTOPCONTROL );
        GetControlBounds( mainForm, bmpAutoscrollIncr, AUTOSCROLLINCRCONTROL );
        GetControlBounds( mainForm, bmpAutoscrollDecr, AUTOSCROLLDECRCONTROL );
        
        CtlSetLabel( (ControlType*) GetObjectPtr( frmMainPercentPopup ), "0%" );
    }
    else {
        ClearControlBounds();
    }
    Prefs()->lastForm = Prefs()->toolbar;
    DoAutoscrollToggle( AUTOSCROLL_OFF );
}


/* Event handler for the main form. */
Boolean MainFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean successful;
    Boolean handled;

    SET_A4_FROM_A5 

    successful = false;
    handled = false;
    
    switch ( event->eType ) {
        case penDownEvent:
            MSG( _( "pen down at ( %d,%d )\n", event->screenX, event->screenY ) );

            /* Tap in text area ( i.e. not in header or on scrollbar ) or 
               tap in active area of header ( i.e. title or graphical buttons ) */
            if ( ! LargeImage() && TopLeftY() < event->screenY && event->screenY < ( TopLeftY() + ExtentY() ) ) {
                switch ( Prefs()->scrollbar ) {
                    case SCROLLBAR_LEFT:
                        if ( TopLeftX() + 7 < event->screenX ) {
                            HandlePenDown( event );
                            handled = true;
                        } else {
                            DoAutoscrollToggle( AUTOSCROLL_OFF );
                        }
                        break;

                    default:
                        if ( event->screenX < ExtentX() ) {
                            HandlePenDown( event );
                            handled = true;
                        } else {
                            DoAutoscrollToggle( AUTOSCROLL_OFF );  
                        }
                        break;
                }
            }
            else {
                HandlePenDown( event );
                handled = true;
            }
            break;

        case penMoveEvent:
            if ( Prefs()->controlMode == MODE3 || LargeImage() )
                HandlePenMove( event );
                handled = true;
            break;

        case penUpEvent:
            MSG( _( "pen up at ( %d,%d )\n", event->screenX, event->screenY ) );
            HandlePenUp( event );
            handled = true;
            break;

        case keyDownEvent:
            switch ( event->data.keyDown.chr ) {
                case pageUpChr:
                    DoPageMove( GetScrollValue() );
                    handled = true;
                    break;

                case pageDownChr:
                    DoPageMove( -GetScrollValue() );
                    handled = true;
                    break;

                case spaceChr:
                    if ( Prefs()->gestMode[ GESTURES_RIGHT ] != SELECT_NONE && Prefs()->gestures )
                        DoSelectTypeAction( Prefs()->gestMode[ GESTURES_RIGHT ] );
                    handled = true;
                    break;

                case 'i':
                case 'I':
                case '1':
                    if ( Prefs()->gestMode[ GESTURES_DOWN ] != SELECT_NONE && Prefs()->gestures )
                        DoSelectTypeAction( Prefs()->gestMode[ GESTURES_DOWN ] );
                    handled = true;
                    break;

                case backspaceChr:
                    if ( Prefs()->gestMode[ GESTURES_LEFT ] != SELECT_NONE && Prefs()->gestures )
                        DoSelectTypeAction( Prefs()->gestMode[ GESTURES_LEFT ] );
                    handled = true;
                    break;

                default:
                    break;
            }
            break;

        case sclRepeatEvent:
            DoPageMove( event->data.sclRepeat.value - event->data.sclRepeat.newValue );
            break;

        case menuCmdBarOpenEvent:
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpRight, 
                menuCmdBarResultMenuItem, mGoForward, NULL );
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpHome, 
                menuCmdBarResultMenuItem, mGoHome, NULL );
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpLeft, 
                menuCmdBarResultMenuItem, mGoBack, NULL );
            if ( Prefs()->toolbarButton != 0 ) {
                MenuCmdBarAddButton( menuCmdBarOnLeft, 
                    ToolbarButtonBitmaps[ Prefs()->toolbarButton ], 
                    menuCmdBarResultMenuItem, 
                    ToolbarButtonCommands[ Prefs()->toolbarButton ], NULL );
            }
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpDbase, 
                menuCmdBarResultMenuItem, mOptionsLibrary, NULL );

            handled = false;
            break;

        case menuEvent:
            /* Refresh display to get rid of command text in bottom left corner */
            MenuEraseStatus( NULL );
            handled = HandleMenuEvent( event->data.menu.itemID );
            break;

        case frmGotoEvent:
            SetFindPatternData( event->data.frmGoto.matchPos, 
                event->data.frmGoto.matchLen );
            JumpToRecord( event->data.frmGoto.recordNum, NO_OFFSET );
            DoPageMove( GetSearchAdjustment() );

            ClearFindPatternData();

            handled = true;
            break;

        case frmUpdateEvent:
            SetScreenMode();
            if ( event->data.frmUpdate.updateCode == frmRedrawUpdateCode ) {
                FrmEraseForm( FrmGetFormPtr( Prefs()->toolbar ) );
                MainFormInit();

                successful = ViewRecord( GetHistoryCurrent(), ! GetAddHistory(), 
                                NO_OFFSET );
                if ( ! successful ) {
                    FrmAlert( warnInsufficientMemory );
                    FrmGotoForm( frmLibrary );
                }
                handled = true;
            }
            else if ( event->data.frmUpdate.updateCode == frmShowResult ) {
                DoPageMove( GetSearchAdjustment() );

                ViewRecord( GetHistoryCurrent(), ! GetAddHistory(), NO_OFFSET );
                handled = true;
            }
            else if ( event->data.frmUpdate.updateCode == frmViewRecord ) {
                Int16 reference;

                reference = GetBookmarkRecordID();
                SetVisitedLink( reference );
                successful = ViewRecord( reference, ! GetAddHistory(), NO_OFFSET );
                if ( ! successful ) {
                    FrmAlert( warnInsufficientMemory );
                    FrmGotoForm( frmLibrary );
                }
                handled = true;
            }
            break;

        case frmOpenEvent:
        {
            Int16 reference;

            MainFormInit();
            SetScreenMode();

            /* Start with last visited page or home page at the very first activation */
            reference = GetHistoryCurrent();
            SetVisitedLink( reference );
            successful = ViewRecord( reference, ! GetAddHistory(), NO_OFFSET );

            if ( ! successful ) {
                FrmAlert( warnInsufficientMemory );
                FrmGotoForm( frmLibrary );
            }
            handled = true;
            break;
        }

        case frmCloseEvent:
            DeleteOffScreenWindow();
            SetDefaultScreenMode( false );
            handled = false;
            break;

        default:
            if ( event->eType == nilEvent && Prefs()->autoscrollEnabled ) {
                
                MetaRecord* meta;
                Int16 jump;
                
                meta = (MetaRecord*) MemHandleLock( GetMetaRecord() );
                ErrFatalDisplayIf( meta == NULL, "Autoscrolling nilEvent: MemHandleLock failed" );
                jump = Prefs()->autoscrollJump;
                
                switch ( Prefs()->autoscrollDir ) {
                    case AUTOSCROLL_DOWN:
                        jump = -jump;
                        if ( -meta->verticalOffset + ExtentY() >= meta->height )
                            DoAutoscrollToggle( AUTOSCROLL_OFF );
                        break;
                 
                    case AUTOSCROLL_UP:
                        if ( meta->verticalOffset == 0 )
                            DoAutoscrollToggle( AUTOSCROLL_OFF );
                        break;
                }
                
                MemHandleUnlock( GetMetaRecord() );
                
                if ( Prefs()->autoscrollStayOn ) {
                    EvtResetAutoOffTimer();
                }
                
                switch ( Prefs()->autoscrollMode ) {
                    case AUTOSCROLL_PIXELS:
                        DoPageMove ( jump );
                        break;
                    
                    case AUTOSCROLL_PAGES:
                        DoPageMove( GetScrollValue() * jump );
                        break;
                }
                
                Prefs()->autoscrollLastScrollTime = TimGetTicks(); 

            }    
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
