/*
 * $Id: image.c,v 1.31 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "control.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "image.h"
#include "mainform.h"
#include "os.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "screen.h"
#include "uncompress.h"
#include "util.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define MAX_IMAGE_SIZE 480000   /* 60k * 8 */


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef enum ShowImagesEnum {
    IMAGE_LIMBO = 0,
    ALL_IMAGES  = 1,
    NO_IMAGES   = 2
} ShowImagesType;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Int16            lastImageID = 0;
static ShowImagesType   imageStatus = IMAGE_LIMBO;
static WinHandle        osWindow    = NULL;
static Boolean          largeImage  = false;
static UInt32           oldDepth    = 0;



/* Clear the static image data */
void ResetImageData( void )
{
    lastImageID = 0;
    imageStatus = IMAGE_LIMBO;
}



/* Determine if Palm OS image compression can be used by the device */
Boolean NotSupportedImageCompression
    (
    BitmapType* imagePtr    /* pointer to image structure */
    )
{
    UInt32 osVersion = GetOSVersion();

    if ( imageStatus != IMAGE_LIMBO )
        return ( imageStatus == NO_IMAGES );

    if ( imagePtr->flags.compressed ) {
        if ( ( sysGetROMVerMajor( osVersion ) == 0x03 ) && 
             ( 2 <= sysGetROMVerMinor( osVersion ) ) ) {
            if ( ( imagePtr->compressionType == BitmapCompressionTypeScanLine ) || 
                 ( ( imagePtr->compressionType == BitmapCompressionTypeRLE ) && 
                   ( 5 <= sysGetROMVerMinor( osVersion ) ) ) ) {
                imageStatus = ALL_IMAGES;
            }
            else {
                imageStatus = NO_IMAGES;
            }
        }
        else if ( sysGetROMVerMajor( osVersion ) == 0x04 ) {
            imageStatus = ALL_IMAGES;
        }
        else {
            imageStatus = NO_IMAGES;
        }
    }
    else {
        imageStatus = ALL_IMAGES;
    }
    return ( imageStatus == NO_IMAGES );
}



/* Determine if images should not be rendered  */
Boolean ShowNoImages( void )
{
    return ( imageStatus == NO_IMAGES );
}



/* Draw an inlined bitmap */
void DrawInlineImage
    (
    const Int16         reference,  /* bitmap's record ID */
    const TextContext*  tContext,   /* pointer to text context */
    Int16*              width       /* upon return, set to the width of the image */
    )
{
    MemHandle   imageHandle;
    MemHandle   uncompressHandle;
    Header*     imageRecord;
    BitmapType* imagePtr;
    Char        imageText[ 12 ];

    if ( imageStatus == NO_IMAGES || ( ! ShowImages( GetHistoryCurrent() ) && 
                                       tContext->writeMode ) ) {
        StrCopy( imageText, "[img]" );
        *width = FntCharsWidth( imageText, StrLen( imageText ) );
        WinDrawChars( imageText, StrLen( imageText ), tContext->cursorX, 
            tContext->cursorY - FntCharHeight() );
        return;
    }
    imageHandle = GetRecordHandle( reference );
    imageRecord = (Header*) MemHandleLock( imageHandle );
    ErrFatalDisplayIf( imageRecord == NULL, "DrawInlineImage: MemHandleLock failed" );

    uncompressHandle = NULL;
    switch ( imageRecord->type ) {
        case DATATYPE_TBMP_COMPRESSED:
            if ( reference == lastImageID )
                uncompressHandle = GetUncompressImageHandle();
            else {
                uncompressHandle    = Uncompress( imageRecord );
                lastImageID         = reference;
            }
            if ( uncompressHandle == NULL ) {
                if ( tContext->writeMode ) {
                    StrCopy( imageText, "[img]" );
                    *width = FntCharsWidth( imageText, StrLen( imageText ) );

                    WinDrawChars( imageText, StrLen( imageText ), 
                        tContext->cursorX, tContext->cursorY - FntCharHeight() );
                }
                MemHandleUnlock( imageHandle );
                return;
            }
            imagePtr = (BitmapType*) MemHandleLock( uncompressHandle );
            break;

        case DATATYPE_TBMP:
            imagePtr = (BitmapType*) ( imageRecord + 1 );
            break;

        default:
            MemHandleUnlock( imageHandle );
            return;
    }

    *width = imagePtr->width;

    if ( tContext->writeMode ) {
        if ( TooHighBitDepth( imagePtr->pixelSize ) ||
             ! ShowImages( GetHistoryCurrent() ) ) {
            StrPrintF( imageText, "[img %dbpp]", imagePtr->pixelSize );
            *width = FntCharsWidth( imageText, StrLen( imageText ) );

            WinDrawChars( imageText, StrLen( imageText ), tContext->cursorX, 
                tContext->cursorY - FntCharHeight() );
        }
        else {
            WinDrawBitmap( imagePtr, tContext->cursorX, 
                tContext->cursorY - imagePtr->height );
        }
    }

    if ( imageRecord->type == DATATYPE_TBMP_COMPRESSED )
        MemHandleUnlock( uncompressHandle );

    MSG( _( "image #%d\n", reference ) );

    MemHandleUnlock( imageHandle );
}



/* Get the height and width of the image */
void GetImageMetrics
    (
    const Int16 reference,  /* bitmap's record number */
    Int16*      width,      /* upon return, set to the width of the image */
    Int16*      height      /* upon return, set to the height of the image */
    )
{
    MemHandle   imageHandle;
    MemHandle   uncompressHandle;
    Header*     imageRecord;
    BitmapType* imagePtr;
    Char        imageText[ 12 ];

    ErrFatalDisplayIf( width == NULL || height == NULL, "GetImageMetrics: NULL argument" );

    if ( imageStatus == NO_IMAGES || ! ShowImages( GetHistoryCurrent() ) ) {
        StrCopy( imageText, "[img]" );

        *width  = FntCharsWidth( imageText, StrLen( imageText ) );
        *height = FntCharHeight();

        return;
    }
    imageHandle = GetRecordHandle( reference );
    imageRecord = (Header*) MemHandleLock( imageHandle );
    ErrFatalDisplayIf( imageRecord == NULL, "GetImageMetrics: MemHandleLock failed" );

    uncompressHandle = NULL;
    switch ( imageRecord->type ) {
        case DATATYPE_TBMP_COMPRESSED:
            if ( reference == lastImageID )
                uncompressHandle = GetUncompressImageHandle();
            else {
                uncompressHandle    = Uncompress( imageRecord );
                lastImageID         = reference;
            }
            if ( uncompressHandle == NULL ) {
                StrCopy( imageText, "[img]" );

                *width  = FntCharsWidth( imageText, StrLen( imageText ) );
                *height = FntCharHeight();

                MemHandleUnlock( imageHandle );
                return;
            }
            imagePtr = (BitmapType*) MemHandleLock( uncompressHandle );
            break;

        case DATATYPE_TBMP:
            imagePtr = (BitmapType*) ( imageRecord + 1 );
            break;

        default:
            MemHandleUnlock( imageHandle );
            return;
    }

    if ( NotSupportedImageCompression( imagePtr ) ) {
        FrmAlert( warnImageCompression );
        StrCopy( imageText, "[img]" );

        *width  = FntCharsWidth( imageText, StrLen( imageText ) );
        *height = FntCharHeight();
    }
    else if ( TooHighBitDepth( imagePtr->pixelSize ) || 
              ! ShowImages( GetHistoryCurrent() ) ) {
        StrPrintF( imageText, "[img %dbpp]", imagePtr->pixelSize );

        *width  = FntCharsWidth( imageText, StrLen( imageText ) );
        *height = FntCharHeight();
    }
    else {
        *width  = imagePtr->width;
        *height = imagePtr->height;
    }

    if ( imageRecord->type == DATATYPE_TBMP_COMPRESSED )
        MemHandleUnlock( uncompressHandle );

    MSG( _( "width = %d, height = %d\n", *width, *height ) );

    MemHandleUnlock( imageHandle );
}



/* Determine if images should be rendered for this record */
Boolean ShowImages
    (
    Int16 reference /* record number */
    )
{
    return ! GetBitStatus( SHOW_IMAGES_ID, reference );
}



/* Turn on image rendering for this record */
void ShowImagesOn
    (
    Int16 reference /* record number */
    )
{
    SetBitStatus( SHOW_IMAGES_ID, reference, false );
}



/* Turn off image rendering for this record */
void ShowImagesOff
    (
    Int16 reference /* record number */
    )
{
    SetBitStatus( SHOW_IMAGES_ID, reference, true );
}



/* View bitmap. */
Boolean ViewTbmp
    (
    Header*         record, /* pointer to record */
    const Boolean   newPage /* true if the page is from the history */
    )
{
    Err             error;
    UInt32          reqMem;
    BitmapType*     bitmap;
    WinHandle       screenWindow;
    MemHandle       uncompressHandle;

    RectangleType   imageBounds = { { 0, 0 }, { MaxExtentX(), MaxExtentY() } };
    RectangleType   screenArea  = { { 0, 0 }, { MaxExtentX(), MaxExtentY() } };

    screenWindow = WinGetDrawWindow();

    ResetImageData();

    uncompressHandle    = NULL;
    if ( record->type == DATATYPE_TBMP )
        bitmap = (BitmapType*) ( record + 1 );
    else {
        uncompressHandle = Uncompress( record );

        if ( uncompressHandle == NULL )
            return false;

        bitmap = ( BitmapType * ) MemHandleLock( uncompressHandle );
        ErrFatalDisplayIf( bitmap == NULL, "ViewTbmp: MemHandleLock failed" );
    }

    if ( NotSupportedImageCompression( bitmap ) ) {
        FrmAlert( warnImageCompression );
        if ( uncompressHandle != NULL )
            MemHandleUnlock( uncompressHandle );
        return true;
    }
    /* Make sure the image is not too big for the used screen depth */
    reqMem = (UInt32) bitmap->width * (UInt32) bitmap->height * bitmap->pixelSize;
    if ( MAX_IMAGE_SIZE < reqMem ) {
        FrmAlert( warnLowImageMem );
        if ( uncompressHandle != NULL )
            MemHandleUnlock( uncompressHandle );
        return true;
    }
    if ( Prefs()->screenDepth != bitmap->pixelSize ) {
        oldDepth                = Prefs()->screenDepth;
        Prefs()->screenDepth    = bitmap->pixelSize;
        SetScreenMode();
    }
    if ( TooHighBitDepth( bitmap->pixelSize ) ) {
        FrmAlert( infoTooHighBitDepth );

        if ( uncompressHandle != NULL )
            MemHandleUnlock( uncompressHandle );

        if ( record->uid == HOME_PAGE_ID )
            return false;
        else
            return true;
    }

    WinEraseRectangle( &screenArea, 0 );

    osWindow = WinCreateOffscreenWindow( bitmap->width, bitmap->height, screenFormat, &error );

    if ( osWindow == NULL ) {
        FrmAlert( warnLowImageMem );
        if ( uncompressHandle != NULL )
            MemHandleUnlock( uncompressHandle );
        return true;
    }
    WinSetDrawWindow( osWindow );
    WinDrawBitmap( bitmap, 0, 0 );

    InitializeTbmpData( bitmap->width, bitmap->height );
    largeImage = true;

    if ( uncompressHandle != NULL )
        MemHandleUnlock( uncompressHandle );

    WinSetDrawWindow( screenWindow );
    WinCopyRectangle( osWindow, screenWindow, &imageBounds, 0, 0, winPaint );

    return true;
}



/* Move image up/down/left/right */
void DoImageMove
    (
    Int16 x,
    Int16 y
    )
{
    WinHandle screenWindow;
    RectangleType imageBounds = { { x, y }, { MaxExtentX(), MaxExtentY() } };

    screenWindow = WinGetDrawWindow();

    WinCopyRectangle( osWindow, screenWindow, &imageBounds, 0, 0, winPaint );
}



/* Check if a large image is displayed */
Boolean LargeImage( void )
{
    return largeImage;
}



/* Delete offscreen window */
void DeleteOffScreenWindow( void )
{
    if ( osWindow ) {
        RectangleType imageBounds = { { 0, 0 }, { MaxExtentX(), MaxExtentX() } };

        WinEraseRectangle( &imageBounds, 0 );

        WinDeleteWindow( osWindow, false );
        osWindow = NULL;

        FrmEraseForm( FrmGetFormPtr( Prefs()->toolbar ) );
        MainFormInit();
    }
    if ( oldDepth ) {
        Prefs()->screenDepth = oldDepth;
        SetScreenMode();
        oldDepth = 0;
    }
    largeImage = false;
}
