/*
 * $Id: beam.c,v 1.5 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "util.h"
#include "os.h"
#include "AutoCF.h"
#include "ffslib.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define BUFSIZE         32767
#define FILENAMESIZE    255

#define SUFFIX          ".pdb"
#define SUFFIX_LEN      ( sizeof( SUFFIX ) - 1 )




/* Callback function for ExgDBWrite that writes out the Plucker document.
   Return error code from ExgSend, or 0 if there is no error */
static Err WritePluckerDocument
    (
    const void* dataP,      /* pointer to buffer containing the data, placed 
                               there by ExgDBWrite */
    UInt32*     sizeP,      /* the number of bytes placed in dataP by ExgDBWrite. 
                               If not all data was sent in this chunk, on exit, 
                               sizeP will contain the number of bytes that was sent */
    void*       userDataP   /* pointer to the socket */
    )
{
    Err err;

    *sizeP = ExgSend( (ExgSocketPtr) userDataP, (void*) dataP, *sizeP, &err );

    return err;
}



/* Beam Plucker document in RAM, return error code, or 0 if there is no error */
static Err BeamRAMDocument
    (
    Char*   dbName, /* database name */
    LocalID dbID,   /* database ID */
    UInt16  cardNo  /* card the database is located on */
    )
{
    ExgSocketType s;
    Err           err;
    Char          filename[ dmDBNameLength + SUFFIX_LEN ];

    StrNCopy( filename, dbName, dmDBNameLength );
    StrCat( filename, SUFFIX );

    MemSet( &s, sizeof( ExgSocketType ), 0 );

    s.name        = filename;
    s.type        = "application/x-pluckerdb";
    s.description = dbName;

    err = ExgPut( &s );
    if ( err == errNone ) {
        err = ExgDBWrite( WritePluckerDocument, &s, filename, dbID, cardNo );
        err = ExgDisconnect( &s, err );
    }
    return err;
}



/* Send stream of data, return error code, or 0 if there is no error */
static Err BeamStream
    (
    ExgSocketType*  s,          /* pointer to socket */
    Char*           filename    /* filename ( including path ) */
    )
{
    Err       err;
    UInt32    sent;
    UInt16    ffsRef;
    Int16     bytes;
    Int16     fd;
    Char*     buf;
    Char*     temp;

    buf = (Char*) MemPtrNew( BUFSIZE );
    if ( buf == NULL )
        return 1;

    ffsRef = GetFsfRef();

    fd = FfsOpen( ffsRef, filename, O_RDONLY, 0 );
    if ( fd == -1 ) {
        MemPtrFree( buf );
        return 1;
    }

    err = errNone;
    while ( err == errNone && ! FfsEof( ffsRef, fd ) ) {
        temp = buf;
        bytes = FfsRead( ffsRef, fd, buf, BUFSIZE );
        while ( err == errNone && 0 < bytes ) {
            sent    = ExgSend( s, buf, bytes, &err );
            bytes  -= sent;
            buf    += sent;
        }
        buf = temp;
    }
    FfsClose( ffsRef, fd );
    MemPtrFree( buf );
    return err;
}



/* Beam Plucker document located on CF card, return error code, or 0 if there is no error */
static Err BeamCFDocument
    (
    Char* dbName    /* database name */
    )
{
    ExgSocketType s;
    Err           err;
    Char          filename[ FILENAMESIZE ];

    AcfGetPath( filename );
    StrNCat( filename, dbName, FILENAMESIZE );
    StrNCat( filename, SUFFIX, FILENAMESIZE );

    MemSet( &s, sizeof( ExgSocketType ), 0 );

    s.name        = filename;
    s.type        = "application/x-pluckerdb";
    s.description = dbName;

    err = ExgPut( &s );
    if ( err == errNone ) {
        err = BeamStream( &s, filename );
        err = ExgDisconnect( &s, err );
    }
    return err;
}



/* Beam Plucker document, return error code, or 0 if there is no error */
Err BeamDocument
    (
    Char*   dbName, /* database name */
    LocalID dbID,   /* database ID */
    UInt16  cardNo  /* card the database is located on */
    )
{
    if ( DatabaseInCF( dbID ) )
        return BeamCFDocument( dbName );
    else
        return BeamRAMDocument( dbName, dbID, cardNo );
}
