/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  segment_pro.c:  segment framework operations
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
load_cs_pro(vm_t *vm, selector_t selector, descriptor_cache_t *cache,
        unsigned cpl)
{
  if (!cache->valid)
    monpanic(vm, "load_cs_pro: valid=0\n");
  selector.fields.rpl = cpl;
  /* For now, CPL is a separate flag. */
  G_CPL(vm) = cpl;
  vm->guest_cpu.selector[SRegCS] = selector;
  vm->guest_cpu.desc_cache[SRegCS] = *cache;
  monSegmentUpdated(vm, SRegCS);
}

  void
load_ss_pro(vm_t *vm, selector_t selector, descriptor_cache_t *cache,
        unsigned cpl)
{
  if (!cache->valid)
    monpanic(vm, "load_ss_pro: valid=0\n");
  selector.fields.rpl = cpl;
  vm->guest_cpu.selector[SRegSS] = selector;
  vm->guest_cpu.desc_cache[SRegSS] = *cache;
  monSegmentUpdated(vm, SRegSS);
}

  void
load_sreg_pro(vm_t *vm, unsigned sreg, selector_t selector,
              descriptor_cache_t *cache)
{
  if (!cache->valid)
    monpanic(vm, "load_sreg_pro: valid=0\n");
  vm->guest_cpu.selector[sreg] = selector;
  vm->guest_cpu.desc_cache[sreg] = *cache;
  monSegmentUpdated(vm, sreg);
}

  void
invalidate_sreg_pro(vm_t *vm, unsigned sreg, selector_t selector)
{
  vm->guest_cpu.desc_cache[sreg].valid = 0;
  vm->guest_cpu.selector[sreg] = selector;
  monSegmentUpdated(vm, sreg);
}


  void
descriptor2cache(vm_t *vm, descriptor_cache_t *cache)
{
  Bit32u limit;

  cache->valid = 0; /* start out invalid */

  /* I'm not sure all of these types should be handled in */
  /* here.  Could probably prune out some system seg types. */

  if (cache->desc.type & 0x10) { /* data/code segment descriptors */
    cache->base = (cache->desc.base_low) |
                  (cache->desc.base_med << 16) |
                  (cache->desc.base_high << 24);
    limit = (cache->desc.limit_high<<16) | cache->desc.limit_low;
    if (cache->desc.g) {
      if ( (cache->desc.type & 0xc) == 0x4 ) /* data/expand-down */
        cache->limit_scaled = (limit << 12);
      else /* normal segment */
        cache->limit_scaled = (limit << 12) | 0x0fff;
      }
    else
      cache->limit_scaled = limit;

    cache->valid    = 1;
    }
  else { /* system & gate segment descriptors */
    switch ( cache->desc.type ) {
      case  0: /* reserved */
      case  8: /* reserved */
      case 10: /* reserved */
      case 13: /* reserved */
        break;
      case 1: /* 16-bit TSS (available) */
      case 3: /* 16-bit TSS (busy) */
        cache->base = (cache->desc.base_low) |
                      (cache->desc.base_med << 16);
        cache->limit_scaled = cache->desc.limit_low;
        cache->valid    = 1;
        break;
      case 2: /* LDT descriptor */
        cache->base = (cache->desc.base_low) |
                      (cache->desc.base_med << 16) |
                      (cache->desc.base_high << 24);
        cache->limit_scaled = cache->desc.limit_low;
        cache->valid    = 1;
        break;
      case 4: /* 16-bit call gate */
      case 6: /* 16-bit interrupt gate */
      case 7: /* 16-bit trap gate */
        cache->valid = 1;
        break;
      case 5: /* 16-bit/32-bit task gate */
        cache->valid = 1;
        break;

      case 9:  /* 32-bit TSS (available) */
      case 11: /* 32-bit TSS (busy) */
        cache->base = (cache->desc.base_low) |
                      (cache->desc.base_med << 16) |
                      (cache->desc.base_high << 24);
        limit = (cache->desc.limit_high<<16) | cache->desc.limit_low;
        if (cache->desc.g)
          cache->limit_scaled = (limit << 12) | 0x0fff;
        else
          cache->limit_scaled = limit;
        cache->valid = 1;
        break;

      case 12: /* 32-bit call gate */
      case 14: /* 32-bit interrupt gate */
      case 15: /* 32-bit trap gate */
        cache->valid = 1;
        break;

      default:
        monpanic(vm, "descriptor2cache: case %u\n",
                 (unsigned) cache->desc.type);
      }
    }
}

  void
fetch_raw_descriptor(vm_t *vm, selector_t selector, descriptor_t *desc,
                     unsigned exception_no)
{
  if (selector.fields.ti == 0) { /* GDT */
    if ((selector.fields.index*8 + 7) > vm->guest_cpu.gdtr.limit)
      goto limit_error;
    access_linear(vm, vm->guest_cpu.gdtr.base + selector.fields.index*8,
                  8, 0, OP_READ, desc);
    }
  else { /* LDT */
    if (vm->guest_cpu.ldtr_cache.valid==0)
      monpanic(vm, "fetch_raw_desc: LDT.valid=0\n");
    if ( (selector.fields.index*8 + 7) >
         vm->guest_cpu.ldtr_cache.limit_scaled )
      goto limit_error;
    access_linear(vm, vm->guest_cpu.ldtr_cache.base + selector.fields.index*8,
                  8, 0, OP_READ, desc);
    }
  return;

limit_error:
  monprint(vm, "fetch_raw_desc: sel index out of desc tbl bounds\n");
  monprint(vm, "index:%u ti:%u rpl:%u\n",
           selector.fields.index,
           selector.fields.ti,
           selector.fields.rpl);
  if (selector.fields.ti)
    monprint(vm, "LDT.limit = %u\n", vm->guest_cpu.ldtr_cache.limit_scaled);
  else
    monprint(vm, "GDT.limit = %u\n", vm->guest_cpu.gdtr.limit);
  exception(vm, exception_no, selector.raw & 0xfffc);
}

  unsigned
fetch_descriptor2(vm_t *vm, selector_t selector, descriptor_t *desc)
{
  if (selector.fields.ti == 0) { /* GDT */
    if ((selector.fields.index*8 + 7) > vm->guest_cpu.gdtr.limit)
      return 0;
    access_linear(vm, vm->guest_cpu.gdtr.base + selector.fields.index*8,
                  8, 0, OP_READ, desc);
    return 1;
    }
  else { /* LDT */
    if (vm->guest_cpu.ldtr_cache.valid==0)
      monpanic(vm, "fetch_desc2: LDT.valid=0\n");
    if ( (selector.fields.index*8 + 7) >
         vm->guest_cpu.ldtr_cache.limit_scaled )
      return 0;
    access_linear(vm, vm->guest_cpu.ldtr_cache.base + selector.fields.index*8,
                  8, 0, OP_READ, desc);
    return 1;
    }
}


  void
load_seg_reg(vm_t *vm, unsigned sreg, Bit16u selector_raw)
{
  /* Load a data segment register using selector.  This function */
  /* is not for loading the code segment CS. */
  if (ProtectedMode(vm)) {
    selector_t   selector;
    descriptor_cache_t cache;

    selector.raw = selector_raw;

    if (sreg == SRegSS) {

      if (IsNullSelector(selector)) {
        monpanic(vm, "load_seg_reg: SS: null\n");
        exception(vm, ExceptionGP, 0);
        }

      fetch_raw_descriptor(vm, selector, &cache.desc, ExceptionGP);

      /* examine AR byte of destination selector for legal values: */

      /* selector's RPL must = CPL, else #GP(selector) */
      if (selector.fields.rpl != G_CPL(vm)) {
        monpanic(vm, "load_seg_reg(): rpl != CPL\n");
        goto exception_gp_selector;
        }

      descriptor2cache(vm, &cache);

      if (cache.valid==0) {
        monpanic(vm, "load_seg_reg: not valid descriptor\n");
        goto exception_gp_selector;
        }

      /* AR byte must indicate a writable data segment else #GP(selector) */
      if ( (cache.desc.type & 0x1a) != 0x12 ) {
        monprint(vm, "load_seg_reg: not writable data segment\n");
        goto exception_gp_selector;
        }

      /* DPL in the AR byte must equal CPL else #GP(selector) */
      if (cache.desc.dpl != G_CPL(vm)) {
        monpanic(vm, "load_seg_reg: dpl != CPL\n");
        goto exception_gp_selector;
        }

      /* segment must be marked PRESENT else #SS(selector) */
      if (cache.desc.p == 0) {
        monpanic(vm, "load_seg_reg: NP\n");
        exception(vm, ExceptionSS, selector.raw & 0xfffc);
        }

      goto commit_load;
      }
    else { /* ES,DS,FS,GS */

      if (IsNullSelector(selector)) {
        /* +++ should we load with 0 or given selector? */
        invalidate_sreg_pro(vm, sreg, selector);
        return;
        }

      fetch_raw_descriptor(vm, selector, &cache.desc, ExceptionGP);

      descriptor2cache(vm, &cache);

      if (cache.valid==0) {
        monpanic(vm, "load_seg_reg: not valid descriptor\n");
        goto exception_gp_selector;
        }

      /* AR byte must indicate data or readable code segment else #GP(selector) */
      if ( !(cache.desc.type & 0x10) ||
           ((cache.desc.type & 0x0a)==0x08) ) {
monprint(vm, "load_seg_reg: peip=%x eip=%x\n",
  vm->guest_cpu.prev_eip, vm->guest.addr.guest_context->eip);
monprint(vm, "sreg=%u, sel=0x%x\n", sreg, selector_raw);
        monpanic(vm, "load_seg_reg: (0x%x) not data or readable code\n",
          cache.desc.type);
        goto exception_gp_selector;
        }

      /* If data or non-conforming code, then both the RPL and the CPL
       * must be less than or equal to DPL in AR byte else #GP(selector) */
      if ( !(cache.desc.type & 0x08) ||
           !(cache.desc.type & 0x04) ) {
        if ((selector.fields.rpl > cache.desc.dpl) || (G_CPL(vm) > cache.desc.dpl)) {
          monpanic(vm, "load_seg_reg: RPL & CPL must be <= DPL\n");
          goto exception_gp_selector;
          }
        }

      /* segment must be marked PRESENT else #NP(selector) */
      if (cache.desc.p == 0) {
        monprint(vm, "load_seg_reg: data seg NP\n");
        exception(vm, ExceptionNP, selector.raw & 0xfffc);
        }

      goto commit_load;
      }

commit_load:
    /* All checks passed.  At this point, we can commit the */
    /* selector/descriptors to the segment register and descriptor cache. */
    load_sreg_pro(vm, sreg, selector, &cache);

/* +++ should this be done regardless of commit/exception? */

    /* Now set accessed bit in descriptor, if it isn't already */
    if ( !(cache.desc.type & 1) ) {
      cache.desc.type |= 1; /* Set accessed bit */
      if (selector.fields.ti == 0) {
        /* GDT */
        access_linear(vm,
          vm->guest_cpu.gdtr.base +selector.fields.index*8 + 5,
          1, 0, OP_WRITE, ((Bit8u *)&cache.desc)+5);
        }
      else {
        /* LDT */
        access_linear(vm,
          vm->guest_cpu.ldtr_cache.base + selector.fields.index*8 + 5,
          1, 0, OP_WRITE, ((Bit8u *)&cache.desc)+5);
        }
      }
    return;

/* +++ use RW to access_linear since read for fetch, then */
/*     write for A bit update? */
/* +++ what is the size of the data written back out to set the */
/*     A bit, by a real processor? */

exception_gp_selector:
    exception(vm, ExceptionGP, selector.raw & 0xfffc);
    }

  else if (RealMode(vm)) {
    Bit32u base, limit;

    /* Make cache up-to-date first.  This caches the selector too, so
     * must do this before setting selector.
     */
    cache_sreg(vm, sreg);

    /* Load selector */
    vm->guest_cpu.selector[sreg].raw = selector_raw;

    /* Load descriptor cache */
    base = selector_raw << 4;
    limit = 0xffff;
    /* Since we're only partially reloading the descriptor, get
     * the old values first.
     */
    if (sreg == SRegCS) {
      /* SET_DESCRIPTOR(vm->guest_cpu.desc_cache[sreg].desc,
       *   base, limit, 0, 0, 0, 1, dpl, D_CODE | D_WRITE | D_ACCESSED) */
      vm->guest_cpu.desc_cache[sreg].limit_scaled = limit;
      vm->guest_cpu.desc_cache[sreg].desc.limit_low  = limit;
      vm->guest_cpu.desc_cache[sreg].desc.limit_high = 0;
      }
    else {
      /* SET_DESCRIPTOR(vm->guest_cpu.desc_cache[sreg].desc,
       *   base, limit, 0, 0, 0, 1, dpl, D_DATA | D_WRITE | D_ACCESSED) */
      vm->guest_cpu.desc_cache[sreg].desc.p = 1;
      vm->guest_cpu.desc_cache[sreg].desc.type |= 1; /* Accessed */
      /* +++ what other fields should a RM load effect? */
      }
    vm->guest_cpu.desc_cache[sreg].valid = 1;
    vm->guest_cpu.desc_cache[sreg].base = base;
    vm->guest_cpu.desc_cache[sreg].desc.base_low = base;
    vm->guest_cpu.desc_cache[sreg].desc.base_med = base>>16;
    vm->guest_cpu.desc_cache[sreg].desc.base_high = base>>24;
    /* limit_scaled not modified */

    G_CPL(vm) = 0; /* RM is effectively CPL0 */
    /* Let monitor know about selector/cache updates */
    monSegmentUpdated(vm, sreg);
    }

  else { /* V8086Mode(vm) */
    Bit32u base, limit;

    /* Load selector */
    vm->guest_cpu.selector[sreg].raw = selector_raw;

    /* Load descriptor cache */
    base = selector_raw << 4;
    limit = 0xffff;
    SET_DESCRIPTOR(vm->guest_cpu.desc_cache[sreg].desc,
      base, limit, 0, 0, 0, 1, 3, D_S | D_WRITE | D_ACCESSED)
    if (sreg==SRegCS)
      vm->guest_cpu.desc_cache[sreg].desc.type |= D_EXECUTE;

    vm->guest_cpu.desc_cache[sreg].valid = 1;
    vm->guest_cpu.desc_cache[sreg].base = base;
    vm->guest_cpu.desc_cache[sreg].limit_scaled = limit;

    G_CPL(vm) = 3; /* v86 is effectively CPL3 */
    /* Let monitor know about selector/cache updates */
    monSegmentUpdated(vm, sreg);
    }
}
