/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.resource;

import java.util.List;

import junit.framework.TestCase;

/**
 * テスト・クラスです。
 * <p>
 * @author cypher256
 */
public class TranslationStringTest extends TestCase {

	/**
	 * リスト出力
	 * @param list
	 */
	private void print(List<TranslationString> list) {
		System.out.println();
		for (TranslationString s : list) {
			System.out.println(s);
		}
	}

	//-------------------------------------------------------------------------

	/** テスト */
	public void testTrim() {

		TranslationString t = null;

		// 英語 trim []
		t = new TranslationString("[ All configurations ]");
		System.out.println(t.trim());
		assertEquals("[ All configurations ]", t.toString());
		assertEquals("All configurations", t.trim());

		// 英語 trim -
		t = new TranslationString("-- End of Stream Output --");
		System.out.println(t.trim());
		assertEquals("-- End of Stream Output --", t.toString());
		assertEquals("End of Stream Output", t.trim());

		// 英語 trim <!--
		t = new TranslationString("<!-- End of Stream Output -->");
		System.out.println(t.trim());
		assertEquals("<!-- End of Stream Output -->", t.toString());
		assertEquals("End of Stream Output", t.trim());
	}

	/** テスト */
	public void testSplit() {

		TranslationString t = null;

		// 日本語 1 つ
		t = new TranslationString("あいうえお。");
		List<TranslationString> list = t.split();
		System.out.println(list);
		assertEquals(null, list);

		// 日本語 複数
		t = new TranslationString("あいうえお。  かきくけこ。");
		list = t.split();
		System.out.println(list);
		assertEquals("あいうえお。  ", list.get(0).toString());
		assertEquals("かきくけこ。", list.get(1).toString());

		// 英語 複数、- の前後が数値、非数値
		t = new TranslationString("aaa bbb. 1.23 abc - xxx yyy. 1 - 2.");
		t.trimForce(); // 末尾を削除してみる
		list = t.split();
		System.out.println(list);
		assertEquals("aaa bbb. ", list.get(0).toString());
		assertEquals("1.23 abc - ", list.get(1).toString());
		assertEquals("1.23 abc", list.get(1).trim());
		assertEquals("xxx yyy. ", list.get(2).toString());
		assertEquals("1 - 2.", list.get(3).toString());

		// 英語 整形済みテキスト
		t = new TranslationString("Connector to submit code to the R-GUI (Windows).\n Note:\n - The Connector requires .NET-Framework\n - For some operations the connector uses the system-clipboard (previous content will be deleted).");
		list = t.split();
		assertNull(list);

		// 英語 () 分割
		t = new TranslationString("aaa (Incubation)");
		list = t.split();
		System.out.println(list);
		assertEquals("aaa ", list.get(0).toString());
		assertEquals("(Incubation)", list.get(1).toString());
		assertEquals("Incubation", list.get(1).trim());

		// 英語 () 分割 + . -> 復元
		t = new TranslationString("aaa (label).");
		list = t.split();
		System.out.println(list);
		assertEquals("aaa ", list.get(0).toString());
		assertEquals("(label).", list.get(1).toString());
		assertEquals("label", list.get(1).trim());
		assertEquals("(ラベル)。", list.get(1).revert("ラベル"));

		// 英語 省略を示す .
		t = new TranslationString("aaa Inc. xxx.");
		list = t.split();
		System.out.println(list);
		assertEquals(null, list);

		// 英語 Eclipse エラーコード
		t = new TranslationString("IWAE0005E IWAJ0131I Cannot add:");
		list = t.split();
		System.out.println(list);
		assertEquals(null, list);
		assertEquals("IWAE0005E IWAJ0131I Cannot add:", t.toString());
		assertEquals("IWAJ0131I Cannot add:", t.trim());

		// 英語 ... の前に空白
		t = new TranslationString(
			"MyISAM uses special tree-like cache to make bulk inserts (that is, INSERT ... SELECT, INSERT ... VALUES (...), (...), ..., and LOAD DATA INFILE) faster. " +
			"This variable limits the size of the cache tree in bytes per thread. " +
			"Setting it to 0 will disable this optimisation. " +
			"Note: " +
			"this cache is only used when adding data to non-empty table. " +
			"Default value is 8 MB.");
		list = t.split();
		System.out.println(list);
		assertEquals(6, list.size());

		// 日本語 :
		t = new TranslationString("これは a: b です。");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());

		// 日本語 : が () で囲まれている
		t = new TranslationString("これは (a: b) です。");
		list = t.split();
		System.out.println(list);
		assertEquals(null, list);

		// 日本語 : の後に {n} がなく、かつ 句読点以外の文字がある
		t = new TranslationString("表示中の id: エラーです。");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());

		// 英語 句点分割 :
		t = new TranslationString("IWAE0036E URI Name: {0}; File name: {1}");
		list = t.split();
		System.out.println(list);
		assertEquals("URI Name: {0}; ", list.get(0).toString());
		assertEquals("File name: ", list.get(1).toString());
		assertEquals("{1}", list.get(2).toString());

		// 日本語 句点分割 :
		t = new TranslationString("IWAE0036E URI 名: {0}、ファイル名: {1}");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("URI 名: {0}、ファイル名: ", list.get(0).toString());
		assertEquals("{1}", list.get(1).toString());

		// 日本語 句点分割
		t = new TranslationString("''{0}'' という名前の型は ''{1}'' 内にインポート (単一の型のインポート) されています (コンパイル単位は同じ名前で型をインポートおよび宣言できません)");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("''{0}'' という名前の型は ''{1}'' 内にインポート (単一の型のインポート) されています ", list.get(0).toString());
		assertEquals("(コンパイル単位は同じ名前で型をインポートおよび宣言できません)", list.get(1).toString());

		// 英語 句点分割 () 入れ子
		t = new TranslationString(
			"Specify filters to control the resource copy process. " +
			"(&lt;name&gt; is a file name pattern (only * wild-cards allowed) or the name of a folder which ends with '/')");
		list = t.split();
		print(list);
		assertEquals(2, list.size());
		assertEquals("Specify filters to control the resource copy process. ",
			list.get(0).toString());
		assertEquals("(&lt;name&gt; is a file name pattern (only * wild-cards allowed) or the name of a folder which ends with '/')",
			list.get(1).toString());

		// 日本語 句点分割 (改行があるため句点分割されない)
		t = new TranslationString(
			"使用可能にすると、メソッド/コンストラクターの呼び出しに渡されるときに可変引数のキャストが必要な場合に、コンパイラーはエラーまたは警告を出します。\n" +
			"(例えば、引数 (\"foo\", null) で呼び出される Class.getMethod(String name, Class ... args )。)");
		list = t.split();
		assertNull(list);

		// 英語 句点分割 (ヘルプ) spllit & trim
		t = new TranslationString(
			"JDT Core options control the behavior of core features such as the Java compiler, code formatter, code assist, and other core behaviors.&nbsp; " +
			"The APIs for accessing the options are defined in <a href=\"../reference/api/org/eclipse/jdt/core/JavaCore.html\"><b>JavaCore</b></a>.&nbsp; " +
			"Options can be accessed as a group as follows: ");
		list = t.split();
		print(list);
		assertEquals("JDT Core options control the behavior of core features such as the Java compiler, code formatter, code assist, and other core behaviors.&nbsp; ",
			list.get(0).toString());
		assertEquals("The APIs for accessing the options are defined in <a href=\"../reference/api/org/eclipse/jdt/core/JavaCore.html\"><b>JavaCore</b></a>.&nbsp; ",
			list.get(1).toString());
		assertEquals("Options can be accessed as a group as follows:",
			list.get(2).toString());

		assertEquals("JDT Core options control the behavior of core features such as the Java compiler, code formatter, code assist, and other core behaviors.",
			list.get(0).trim());
		assertEquals("The APIs for accessing the options are defined in <a href=\"../reference/api/org/eclipse/jdt/core/JavaCore.html\"><b>JavaCore</b></a>.",
			list.get(1).trim());
		assertEquals("Options can be accessed as a group as follows:",
			list.get(2).trim());

		// 日本語 句点分割 (ヘルプ)
		t = new TranslationString(
			"JDT コア・オプションは、Java コンパイラー、コード・フォーマッター、コード・アシスト、 およびその他のコアの振る舞いなど、コア・フィーチャーの振る舞いを制御します。 " +
			"オプションにアクセスするための API は <a href=\"../reference/api/org/eclipse/jdt/core/JavaCore.html\"><b>JavaCore</b></a> で定義されます。 " +
			"オプションには、以下のグループとしてアクセスできます。");
		list = t.split();
		print(list);
		assertEquals("JDT コア・オプションは、Java コンパイラー、コード・フォーマッター、コード・アシスト、 およびその他のコアの振る舞いなど、コア・フィーチャーの振る舞いを制御します。 ",
			list.get(0).toString());
		assertEquals("オプションにアクセスするための API は <a href=\"../reference/api/org/eclipse/jdt/core/JavaCore.html\"><b>JavaCore</b></a> で定義されます。 ",
			list.get(1).toString());
		assertEquals("オプションには、以下のグループとしてアクセスできます。",
			list.get(2).toString());

		// 日本語 句点分割 末尾 -
		t = new TranslationString("バンドルのシンボル名に正しくない文字が含まれています。 正しい文字は A-Z a-z 0-9 . _ - です");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("バンドルのシンボル名に正しくない文字が含まれています。 ", list.get(0).toString());
		assertEquals("正しい文字は A-Z a-z 0-9 . _ - です", list.get(1).toString());

		// 英語 句点分割 末尾 -
		t = new TranslationString("Bundle symbolic name contains illegal characters. Legal characters are A-Z a-z 0-9 . _ -");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("Bundle symbolic name contains illegal characters. ", list.get(0).toString());
		assertEquals("Legal characters are A-Z a-z 0-9 . _ -", list.get(1).toString());
		assertEquals("Legal characters are A-Z a-z 0-9 . _ -", list.get(1).trim());

		// 日本語 句点分割 -
		t = new TranslationString("Agent Controller 利用不可 - Agent Controller が実行中で接続受け入れ状態であることを確認してください");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("Agent Controller 利用不可 - ", list.get(0).toString());
		assertEquals("Agent Controller 利用不可",    list.get(0).trim());
		assertEquals("Agent Controller が実行中で接続受け入れ状態であることを確認してください", list.get(1).toString());

		// 英語 句点分割 -
		t = new TranslationString("Agent Controller Unavailable - Please ensure the Agent Controller is running and accepting connections");
		list = t.split();
		System.out.println(list);
		assertEquals(2, list.size());
		assertEquals("Agent Controller Unavailable - ", list.get(0).toString());
		assertEquals("Agent Controller Unavailable",    list.get(0).trim());
		assertEquals("Please ensure the Agent Controller is running and accepting connections", list.get(1).toString());

		// 英語 句点分割 括弧
		t = new TranslationString("Use default SDK (App Engine - 1.2.0)");
		list = t.split();
		System.out.println(list);
		assertEquals(3, list.size());
		assertEquals("Use default SDK ", list.get(0).toString());
		assertEquals("(App Engine - ", list.get(1).toString());
		assertEquals("App Engine", list.get(1).trim());
		assertEquals("1.2.0)", list.get(2).toString());
		assertEquals("1.2.0", list.get(2).trim());
	}
}
