/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect;

import java.io.File;
import java.io.IOException;
import java.lang.instrument.Instrumentation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtMethod;
import javassist.NotFoundException;
import jp.sourceforge.mergedoc.pleiades.Pleiades;
import jp.sourceforge.mergedoc.pleiades.PleiadesOption;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.AspectMapping;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.DynamicTranslationDictionary;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.ExcludePackageProperties;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.ExcludesClassNameCache;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.RegexDictionary;
import jp.sourceforge.mergedoc.pleiades.aspect.resource.TransformedClassCache;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.Files;
import jp.sourceforge.mergedoc.pleiades.resource.PropertySet;

/**
 * Eclipse 起動時のバイトコード変換を行うトランスフォーマーです。
 * <p>
 * @author cypher256
 */
public class LauncherTransformer extends AbstractTransformer {

	/** ロガー */
	private static final Logger log = Logger.getLogger(LauncherTransformer.class);

	/** このクラス名 (AOP static 呼び出し用) */
	private static final String CLASS_NAME = LauncherTransformer.class.getName();

	/** このクラスのインスタンス */
	private static LauncherTransformer thisInstance;

	/** 変換済みクラス・キャッシュ */
	private static volatile TransformedClassCache transformedClassCache;

	/** 始動時のワークスペース・ダイアログ表示有無 */
	private static volatile boolean showWorkspaceSelectionDialog = true;

	/** metadata */
	private static Metadata metadata = new Metadata();

	/**
	 * 起動トランスフォーマーを構築します。
	 */
	public LauncherTransformer() {
		thisInstance = this;
		load();
	}

	/**
	 * プロパティーをロードします。
	 */
	private void load() {

		long start = System.nanoTime();

		// アスペクト定義をロード
		// 注) 別スレッドで処理すると SAXParser でデッドロックする
		AspectMapping.getInstance();

		// -clean に依存しないものをロード (非同期実行)
		// -clean に依存するもは startTranslationTransformer でロード
		Asyncs.execute(new Runnable() {
			public void run() {

				// 翻訳除外パッケージ・プロパティーをロード
				ExcludePackageProperties.getInstance();

				// 正規表現翻訳プロパティーをロード
				RegexDictionary.getInstance();

				// 始動時のワークスペース・ダイアログ表示有無を取得
				File prefs = Pleiades.getResourceFile("../.settings/org.eclipse.ui.ide.prefs");
				if (prefs.exists()) {
					PropertySet p = new PropertySet(prefs);
					showWorkspaceSelectionDialog = Boolean.valueOf(p.get("SHOW_WORKSPACE_SELECTION_DIALOG"));
				}
			}
		});

		Analyses.end(LauncherTransformer.class, "load", start);
	}

	/**
	 * バイトコード変換を行います。
	 */
	@Override
	protected byte[] transform(ClassLoader loader, String className, ProtectionDomain protectionDomain, byte[] bytecode)
			throws CannotCompileException, NotFoundException, IOException {

		long start = System.nanoTime();

		try {
			if (!className.startsWith("org.eclipse.")) {
				return null;
			}
			byte[] transformedBytecode = null;

			// -----------------------------------------------------------
			// Main クラスの変換
			// -> 翻訳トランスフォーマーの開始処理を追加
			// -----------------------------------------------------------
			// 3.2 以前： org.eclipse.core.launcher.Main
			// 3.3 以降： org.eclipse.equinox.launcher.Main
			if (className.endsWith(".launcher.Main")) {

				try {
					// Eclipse Main クラスに開始、終了処理を追加
					CtClass clazz = createCtClass(bytecode, protectionDomain);
					CtMethod basicRun = clazz.getMethod("basicRun", "([Ljava/lang/String;)V");
					basicRun.insertBefore("$1 = " + CLASS_NAME + ".startTranslationTransformer($$);");
					// basicRun.insertAfter(CLASS_NAME + ".shutdown();", true);
					// ↑終了しないうちに起動してしまうのを防ぐため、Workbench クラスに移動

					// Main だけはキャッシュしない
					// キャッシュのロードに -clean の判定が必要だが、この時点では -clean 判定できない
					return clazz.toBytecode();

				} catch (NotFoundException e) {

					// Eclipse 3.3 以降、Eclipse アプリケーションとして起動した場合、
					// org.eclipse.core.launcher.Main は main メソッドしかなく、このクラスから
					// org.eclipse.equinox.launcher.Main#main が呼び出される。
					// 最初の Main でのメソッドなしは無視
					if (className.equals("org.eclipse.core.launcher.Main")) {
						return null;
					}
					throw e;
				}
			}
			// -----------------------------------------------------------
			// IDEWorkbenchAdvisor クラスの変換
			// -> 新規ワークスペース作成時の metadeta コピー処理を追加
			// -----------------------------------------------------------
			else if (className.equals("org.eclipse.ui.internal.ide.application.IDEWorkbenchAdvisor")) {

				CtClass clazz = createCtClass(bytecode, protectionDomain);
				clazz.getDeclaredConstructor(null).insertBefore(
						"org.eclipse.osgi.service.datalocation.Location loc = null;"
								+ "loc = org.eclipse.core.runtime.Platform.getInstanceLocation();"
								+ "java.net.URL workspace = null;" + "if (loc != null) workspace = loc.getURL();"
								+ CLASS_NAME + ".copyMetadata(workspace);");
				transformedBytecode = clazz.toBytecode();
			}
			// -----------------------------------------------------------
			// Workbench クラスの変換
			// -> 終了時のシャットダウン処理を追加
			// -----------------------------------------------------------
			else if (className.equals("org.eclipse.ui.internal.Workbench")) {

				// 終了時の shutdown
				CtClass clazz = createCtClass(bytecode, protectionDomain);
				CtMethod shutdown = clazz.getMethod("shutdown", "()V");
				shutdown.insertBefore(CLASS_NAME + ".shutdown();");

				// イベントループ開始前の処理
				CtMethod runEventLoop = clazz.getMethod("runEventLoop",
						"(Lorg/eclipse/jface/window/Window$IExceptionHandler;Lorg/eclipse/swt/widgets/Display;)V");
				runEventLoop.insertBefore(CLASS_NAME + ".beforeEventLoop($0);");
				transformedBytecode = clazz.toBytecode();
			}
			// -----------------------------------------------------------
			// WorkbenchWindows クラスの変換
			// -> preload の場合はウィンドウを座標外に表示
			// -----------------------------------------------------------
			else if (className.equals("org.eclipse.ui.internal.WorkbenchWindow")) {

				CtClass clazz = createCtClass(bytecode, protectionDomain);
				CtMethod restore = clazz
						.getMethod("restoreState",
								"(Lorg/eclipse/ui/IMemento;Lorg/eclipse/ui/IPerspectiveDescriptor;)Lorg/eclipse/core/runtime/IStatus;");
				restore.insertBefore(CLASS_NAME + ".setWindowPosition($1);");
				transformedBytecode = clazz.toBytecode();
			}
			// -----------------------------------------------------------
			// WorkbenchPreferenceInitializer クラスの変換
			// -> 新規ワークスペース作成時のエンコーディング設定を追加
			// -----------------------------------------------------------
			else if (className.equals("org.eclipse.ui.internal.WorkbenchPreferenceInitializer")) {

				// このトランスフォーマーをエージェントから削除
				Pleiades.getInstrumentation().removeTransformer(this);

				CtClass clazz = createCtClass(bytecode, protectionDomain);
				CtConstructor cons = clazz.getConstructors()[0];
				try {
					cons
							.insertBefore("String enc = "
									+ CLASS_NAME
									+ ".getNewWorkspaceEncoding();"
									+ "if (enc != null) "
									+ "org.eclipse.core.resources.ResourcesPlugin.getPlugin().getPluginPreferences().setValue(\"encoding\", enc);");
				} catch (CannotCompileException e) {
					// Eclipse 3.1 では ResourcesPlugin がロードされていないためエラー
					// Eclipse 3.2 以降のみサポート
					log.warn("自動エンコーディング設定不可。" + e.getMessage());
				}
				transformedBytecode = clazz.toBytecode();
			}

			// 次回起動用にキャッシュ。
			// このメソッドは AOP のみにし、実行時の判定処理があってはいけないことに注意。
			if (transformedBytecode != null && transformedClassCache != null) {
				transformedClassCache.putNextLaunch(className, transformedBytecode);
			}
			return transformedBytecode;

		} finally {

			Analyses.end(LauncherTransformer.class, "transform", start);
		}
	}

	// ------------------------------------------------------------------------
	// 以下、Eclipse に埋め込んだ AOP により呼び出される public static メソッド

	/**
	 * 翻訳トランスフォーマーを開始します。
	 * <p>
	 * @param args Eclipse 起動オプション配列
	 * @return 起動オプション配列
	 */
	public static String[] startTranslationTransformer(String... args) {

		long start = System.nanoTime();

		try {
			// スプラッシュ画像パスを変更
			PleiadesOption agentOption = Pleiades.getPleiadesOption();
			if (!agentOption.isDefaultSplash()) {
				String splashLocation = getSplashLocation();
				if (splashLocation != null) {
					log.debug("スプラッシュ・ロケーション: " + splashLocation);
					System.setProperty("osgi.splashLocation", splashLocation);
				}
			}

			// 起動引数の -clean を取得
			List<String> argList = new LinkedList<String>(Arrays.asList(args));
			PleiadesOption option = Pleiades.getPleiadesOption();
			option.setClean(argList.contains("-clean"));
			log.info("Eclipse の起動を開始しました。-clean:" + option.isClean());

			// キャッシュが無い場合は強制的に -claen を指定
			if (!option.isClean()) {

				File excludeList = Pleiades.getResourceFile(EXCLUDE_CLASS_LIST);
				if (!excludeList.exists()) {

					log.info("変換除外クラス名キャッシュが存在しないため、" + "強制的に -clean モードで起動します。");
					argList.add("-clean");
					option.setClean(true);
				}
			}

			// -clean でない場合はこのトランスフォーマーをエージェントから削除
			// Main クラス以外は変換済みキャッシュを使用する
			if (!option.isClean()) {
				Pleiades.getInstrumentation().removeTransformer(thisInstance);
			}

			// プロセス排他ロック
			Locks.lock();

			Asyncs.execute(new Runnable() {
				public void run() {

					try {
						// 変換除外クラス名キャッシュをロード
						ExcludesClassNameCache.getInstance();

						// 変換済みクラス・キャッシュをロード
						transformedClassCache = TransformedClassCache.getInstance();

						// 翻訳辞書をロード
						DynamicTranslationDictionary.getInstance();

					} catch (Throwable e) {

						log.fatal(e, "リソースのロードに失敗しました。");

					} finally {

						// プロセス排他ロック解除
						Locks.release();
					}
				}
			});

			// 翻訳トランスフォーマーを開始
			Instrumentation inst = Pleiades.getInstrumentation();
			inst.addTransformer(new TranslationTransformer());

			return argList.toArray(new String[argList.size()]);

		} catch (Throwable e) {

			String msg = "翻訳トランスフォーマーの開始に失敗しました。";
			log.fatal(msg);
			throw new IllegalStateException(msg, e);

		} finally {

			Analyses.end(LauncherTransformer.class, "startTranslationTransformer", start);
		}
	}

	/**
	 * metadata をコピーします。
	 * <p>
	 * @param workspace ワークスペース・ロケーション
	 * @throws Exception 例外が発生した場合
	 */
	public static void copyMetadata(final URL workspace) throws Exception {

		if (workspace == null) {
			log.error("ワークスペース・ロケーションを取得できませんでした。");
			return;
		}
		metadata.createNewWorkspaceMetadata(new File(workspace.getFile()));
	}

	/**
	 * 新規ワークスペースの Pleiades 自動設定エンコーディングを取得します。
	 * <p>
	 * @throws Exception 例外が発生した場合
	 * @return エンコーディング。新規ワークスペースでない場合は null。
	 */
	public static String getNewWorkspaceEncoding() throws Exception {

		String newWorkspaceEncoding = metadata.getNewWorkspaceEncoding();
		return newWorkspaceEncoding;
	}

	/**
	 * Eclipse イベントループ開始前の処理です。
	 * <p>
	 * @param workbench org.eclipse.ui.internal.Workbench インスタンス
	 * @throws Exception 例外が発生した場合
	 */
	public static void beforeEventLoop(final Object workbench) throws Exception {

		// Eclipse 始動計測時間をログに出力
		long startTime = Long.valueOf(System.getProperty("eclipse.startTime"));
		long curTime = System.currentTimeMillis();
		final double startupTime = (curTime - startTime) / 1e+3;
		Analyses.flashLog("Eclipse 始動完了。始動時間: %.3f 秒。-clean:%s", startupTime, Pleiades.getPleiadesOption().isClean());

		// Eclipse 始動計測時間をステータスバーに出力
		if (!showWorkspaceSelectionDialog) {
			try {
				Method getActivatedWindow = workbench.getClass().getDeclaredMethod("getActivatedWindow");
				getActivatedWindow.setAccessible(true);
				Object workbenchWindow = getActivatedWindow.invoke(workbench);
				Method setStatus = workbenchWindow.getClass().getMethod("setStatus", String.class);
				setStatus.invoke(workbenchWindow, String.format("Eclipse 始動完了 - 始動時間: %.3f 秒", startupTime));
			} catch (InvocationTargetException e) {
				log.info("始動時間をステータスバーにセットできませんでした。" + e.getTargetException());
			} catch (Exception e) {
				log.info("始動時間をステータスバーにセットできませんでした。" + e);
			}
		}

		// システム・プロパティーをログとして出力
		if (log.isDebugEnabled()) {
			Asyncs.execute(new Runnable() {
				public void run() {
					PropertySet systemProp = new PropertySet(System.getProperties());
					File file = Pleiades.getResourceFile("system.properties");
					systemProp.store(file, "実行時システム・プロパティー");
				}
			});
		}

		// プリロード用の自動終了
		if (Pleiades.getPleiadesOption().isPreload()) {
			try {
				Method close = workbench.getClass().getMethod("close");
				close.invoke(workbench);
			} catch (Throwable e) {
				log.error(e, "プリロード自動終了呼び出しでエラーが発生しました。");
			}
		}

		// 見た目のための GC 予約
		Asyncs.execute(new Runnable() {
			public void run() {
				try {
					long wait = (long) startupTime * 1000;
					Thread.sleep(wait);
					log.debug("GC " + (wait / 1000));
					for (int i = 0; i < 5; i++) {
						System.gc();
					}
				} catch (InterruptedException e) {
					log.debug("GC 予約で割り込みが発生しました。" + e);
				}
			}
		});

		// 非同期ユーティリティーのシャットダウン
		Asyncs.shutdown();
	}

	/**
	 * ワークベンチ復元前の処理です。
	 * <p>
	 * @param memento org.eclipse.ui.IMemento インスタンス
	 * @throws Exception 例外が発生した場合
	 */
	public static void setWindowPosition(Object memento) throws Exception {

		if (Pleiades.getPleiadesOption().isPreload()) {

			// プリロード起動時の表示位置に座標外の値をセット
			Method putInteger = memento.getClass().getMethod("putInteger", String.class, int.class);
			putInteger.invoke(memento, "width", 200);
			putInteger.invoke(memento, "x", -199);
		}
	}

	/**
	 * 起動トランスフォーマーをシャットダウンします。
	 */
	public static void shutdown() {

		new Thread("pleiades-shutdown") {
			@Override
			public void run() {

				if (log.isDebugEnabled()) {
					Analyses.flashLog("Eclipse 終了。");
				}

				try {
					// プロセス排他ロック
					Locks.lock();

					// 変換済みクラス・キャッシュを保存
					TransformedClassCache.getInstance().shutdown();

					// 変換除外クラス名キャッシュを保存
					ExcludesClassNameCache.getInstance().shutdown();

					// 翻訳プロパティーをキャッシュとして保存
					DynamicTranslationDictionary.getInstance().shutdown();

				} catch (Exception e) {
					log.warn(e, "キャッシュの更新に失敗しました。");

				} finally {

					// プロセス排他ロック解除
					Locks.release();

					// 非同期ユーティリティーのシャットダウン (通常はシャットダウン済み)
					Asyncs.shutdown();

					// プリロード用の自動終了
					if (Pleiades.getPleiadesOption().isPreload()) {
						System.exit(0);
					}
				}
			}
		}.start();
	}

	/**
	 * スプラッシュ画像パスを取得します。
	 * @return スプラッシュ画像パス。RCP の場合は null。
	 */
	private static String getSplashLocation() {

		long start = System.nanoTime();
		try {
			File home = Pleiades.getEclipseHome();
			File eclipse = new File(home, "eclipse");
			File eclipseExe = new File(home, "eclipse.exe");

			// RCP の場合、Pleiades スプラッシュは使用しない
			if (!eclipse.exists() && !eclipseExe.exists()) {
				return null;
			}

			// ユーザー・カスタム・スプラッシュ
			// (pleiades を格納した plugins ディレクトリーがある場所)
			File customFile = Files.getFile("../../../splash.bmp");
			if (customFile.exists()) {
				return customFile.getAbsolutePath().replace('\\', '/');
			}

			// Eclipse のバージョン取得
			String eclipseVersion = "";
			File pluginsFolder = new File(home, "plugins");
			for (File folder : pluginsFolder.listFiles()) {
				String folderName = folder.getName();
				if (folderName.startsWith("org.eclipse.osgi_")) {
					eclipseVersion = folderName.replaceFirst(".*?_(\\d+\\.\\d+).*", "$1");
					break;
				}
			}

			// Eclipse バージョンにあった bmp ファイル・パスを取得
			String fileName = "splash" + eclipseVersion + ".bmp";
			File file = Files.getFile(fileName);
			if (!file.exists()) {
				file = Files.getFile("splash.bmp");
			}
			String splashLocation = file.getAbsolutePath().replace('\\', '/');
			return splashLocation;

		} finally {
			Analyses.end(LauncherTransformer.class, "getSplashLocation", start);
		}
	}
}
