/*
    Bear Engine - Model editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/xml/code/model_mark_placement_node.cpp
 * \brief Implementation of the bf::xml::model_mark_placement_node class.
 * \author Julien Jorge
 */
#include "bf/xml/model_mark_placement_node.hpp"

#include "bf/mark.hpp"
#include "bf/mark_placement.hpp"
#include "bf/snapshot.hpp"
#include "bf/xml/reader_tool.hpp"

#include <claw/assert.hpp>
#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Read an xml node "mark_placement".
 * \param s (out) The snapshot in which we set the mark_placement.
 * \param node The item node.
 */
void bf::xml::model_mark_placement_node::read
( snapshot& s, const wxXmlNode* node ) const
{
  CLAW_PRECOND( node!=NULL );
  CLAW_PRECOND( node->GetName() == wxT("mark_placement") );

  const std::string label( reader_tool::read_string(node, wxT("label")) );

  if ( !s.has_mark(label) )
    claw::logger << claw::log_error << "There is no mark labeled '" << label
                 << "'" << std::endl;
  else
    {
      mark_placement& m = s.get_placement(label);

      m.set_position_x
        ( reader_tool::read_real_opt(node, wxT("x"), m.get_position_x()) );
      m.set_position_y
        ( reader_tool::read_real_opt(node, wxT("y"), m.get_position_y()) );
      m.set_depth_position
        ( reader_tool::read_int_opt(node, wxT("z"), m.get_depth_position()) );
      m.set_angle
        ( reader_tool::read_real_opt(node, wxT("angle"), m.get_angle()) );
      m.set_visibility
        ( reader_tool::read_bool_opt(node, wxT("visible"), m.is_visible()) );
    }
} // model_mark_placement_node::read()

/*----------------------------------------------------------------------------*/
/**
 * \brief Write an xml node "mark_placement".
 * \param m The mark_placement to write.
 * \param os The stream in which we write.
 */
void bf::xml::model_mark_placement_node::write
( const mark_placement& m, std::ostream& os ) const
{
  os << "<mark_placement label=\"" << m.get_mark()->get_label() << "\" x=\""
     << m.get_position_x() << "\" y=\"" << m.get_position_y() << "\" z=\""
     << m.get_depth_position() << "\" angle=\"" << m.get_angle()
     << "\" visible=\"" << m.is_visible() << "\"/>\n";
} // model_mark_placement_node::write()
