/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file derived_item_handle.hpp
 * \brief Safe way to point to an item that could die between two uses.
 * \author Julien Jorge
 */
#ifndef __UNIVERSE_DERIVED_ITEM_HANDLE_HPP__
#define __UNIVERSE_DERIVED_ITEM_HANDLE_HPP__

#include "universe/item_handle.hpp"

namespace bear
{
  namespace universe
  {
    /**
     * \brief Safe way to point an item that could die between two uses.
     * \author Julien Jorge
     *
     * \b Template \b parameters:
     *  - \a DerivedType: the final type of the pointed item.
     *  - \a ItemType: a class that inherits from universe::physical_item.
     *
     * \b Sample \b usage:
     *
     * Let's consider the following item class
     * \code
     * class my_item:
     *   public physical_item,
     *   public an_other_class
     * {
     *   // ...
     * }; // my_item
     * \endcode
     *
     * Let \c special_item be an item that deals only with instances of \c
     * an_other_class. \c special_item can keep a handle on an item of type \c
     * my_item and still consider the \c an_other_class part by using a \c
     * derived_item_handle.
     * \code
     * class special_item
     * {
     * public:
     *   void set_item( physical_item* item ) { m_item = item; }
     *
     * private:
     *   // The item on which we work.
     *   derived_item_handle<an_other_class, physical_item> m_item;
     *
     * }; // special_item
     * \endcode
     *
     * \sa derived_item_handle_maker, const_derived_item_handle_maker
     */
    template<typename DerivedType, typename ItemType>
    class derived_item_handle
    {
    public:
      typedef derived_item_handle<DerivedType, ItemType> self_type;

    public:
      derived_item_handle();
      derived_item_handle( ItemType* item );
      derived_item_handle( ItemType& item );
      derived_item_handle( const self_type& that );

      DerivedType* get() const;
      ItemType* get_item() const;

      DerivedType& operator*() const;
      DerivedType* operator->() const;

      self_type& operator=( ItemType* item );
      self_type& operator=( const self_type& that );

      bool operator==( const ItemType* item ) const;
      bool operator==( const DerivedType* item ) const;
      bool operator==( const self_type& that ) const;
      bool operator!=( const ItemType* item ) const;
      bool operator!=( const DerivedType* item ) const;
      bool operator!=( const self_type& that ) const;
      bool operator<( const self_type& that ) const;

    private:
      void cast_item();

    private:
      /** \brief The critical item. */
      base_item_handle<ItemType> m_item;

      DerivedType* m_derived;

    }; // class derived_item_handle

    class physical_item;

    /**
     * \brief A wrapper to easily create a derived_item_handle for a non
     *        constant class type inheriting from physical_item.
     */
    template<typename DerivedType>
    struct derived_item_handle_maker
    {
      typedef derived_item_handle<DerivedType, physical_item> handle_type;

    }; // struct derived_item_handle_maker

    /**
     * \brief A wrapper to easily create a derived_item_handle for a constant
     *        class type inheriting from physical_item.
     */
    template<typename DerivedType>
    struct const_derived_item_handle_maker
    {
      typedef
      derived_item_handle<const DerivedType, const physical_item> handle_type;

    }; // struct const derived_item_handle_maker

  } // namespace universe
} // namespace bear

#include "universe/impl/derived_item_handle.tpp"

#endif // __UNIVERSE_DERIVED_ITEM_HANDLE_HPP__
