/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dose;

import com.pixelmed.dicom.*;

public class CTDoseAcquisition {
	
	protected String seriesNumber;
	protected CTScanType scanType;
	protected ScanRange scanRange;
	protected String CTDIvol;
	protected String DLP;
	protected CTPhantomType phantomType;
	
	protected CTAcquisitionParameters acquisitionParameters;
	
	protected ContentItem contentItemFragment;
	
	public CTDoseAcquisition(String seriesNumber,CTScanType scanType,ScanRange scanRange,String CTDIvol,String DLP,CTPhantomType phantomType) {
		this.seriesNumber = seriesNumber;
		this.scanType = scanType;
		this.scanRange = scanRange;
		this.CTDIvol = CTDIvol;
		this.DLP = DLP;
		this.phantomType = phantomType;
		this.acquisitionParameters = null;
		this.contentItemFragment = null;
	}
	
	public String getSeriesNumber() { return seriesNumber; }
	public CTScanType getScanType() { return scanType; }
	public ScanRange getScanRange() { return scanRange; }
	public String getCTDIvol() { return CTDIvol; }
	public String getDLP() { return DLP; }
	public CTPhantomType getPhantomType () { return phantomType; }
	
	public void setAcquisitionParameters(CTAcquisitionParameters acquisitionParameters) { this.acquisitionParameters = acquisitionParameters; }
	public CTAcquisitionParameters getAcquisitionParameters() { return acquisitionParameters; }

	public boolean equals(Object o) {
		//System.err.println("Location.equals(): comparing "+this+" to "+o);
		boolean isEqual = false;
		if (o instanceof CTDoseAcquisition) {
			CTDoseAcquisition oda = (CTDoseAcquisition)o;
			isEqual = oda.getSeriesNumber().equals(this.getSeriesNumber())
			&& oda.getSeriesNumber().equals(this.getSeriesNumber())
			&& oda.getScanType().equals(this.getScanType())
			&& oda.getScanRange().equals(this.getScanRange())
			&& oda.getCTDIvol().equals(this.getCTDIvol())
			&& oda.getDLP().equals(this.getDLP())
			&& oda.getPhantomType().equals(this.getPhantomType());
			// do NOT check acquisitionParameters !
		}
		else {
			isEqual = false;
		}
		return isEqual;
	}
	
	public int hashCode() {
		return getSeriesNumber().hashCode()
		+ getScanType().hashCode()
		+ getScanRange().hashCode();	// sufficient to implement equals() contract
	}

	public String getDLPFromRangeAndCTDIvol() throws NumberFormatException {	// NB. Will NOT match specified DLP, due to overscan (helical) and slice thickness/gap (axial)
		double dlpFromRangeAndCTDIvol = Double.parseDouble(scanRange.getAbsoluteRange()) * Double.parseDouble(CTDIvol) / 10;		// in cm not mm
		java.text.DecimalFormat formatter = (java.text.DecimalFormat)(java.text.NumberFormat.getInstance());
		formatter.setMaximumFractionDigits(2);
		formatter.setMinimumFractionDigits(2);
		formatter.setDecimalSeparatorAlwaysShown(true);		// i.e., a period even if fraction is zero
		formatter.setGroupingUsed(false);					// i.e., no comma at thousands
		String formatted = formatter.format(dlpFromRangeAndCTDIvol);
//System.err.println("CTDoseAcquisition.getDLPFromRangeAndCTDIvol(): returns formatted string "+formatted+" for "+Double.toString(dlpFromRangeAndCTDIvol));
		return formatted;
	}
	
	public boolean specifiedDLPMatchesDLPFromRangeAndCTDIvol() {
		String computedDLP = getDLPFromRangeAndCTDIvol();
//System.err.println("CTDoseAcquisition.specifiedDLPMatchesDLPFromRangeAndCTDIvol(): comparing specified DLP "+DLP+" with computed "+computedDLP);
		return DLP != null && DLP.equals(computedDLP);
	}
	
	public String toString() {
		return toString(false);
	}
	
	public String toString(boolean pretty) {
		StringBuffer buffer = new StringBuffer();

		buffer.append("\tSeries=");
		buffer.append(seriesNumber);
		
		buffer.append("\t");
		buffer.append(scanType);
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Range=");
		}
		buffer.append(scanRange);
		buffer.append(" mm");
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("CTDIvol=");
		}
		buffer.append(CTDIvol);
		buffer.append(" mGy");
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("DLP=");
		}
		buffer.append(DLP);
		buffer.append(" mGycm");
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Phantom=");
		}
		buffer.append(phantomType);

		if (acquisitionParameters != null) {
			buffer.append("\n\t");
			buffer.append(acquisitionParameters.toString(pretty));
		}
		else {
			buffer.append("\n");
		}
		return buffer.toString();
	}

	public ContentItem getStructuredReportFragment(ContentItem root) throws DicomException {
		if (contentItemFragment == null) {
			ContentItemFactory cif = new ContentItemFactory();
			contentItemFragment = cif.new ContainerContentItem(root,"CONTAINS",new CodedSequenceItem("113819","DCM","CT Acquisition"),true/*continuityOfContentIsSeparate*/,"DCMR","10013");
//System.err.println("CTDoseAcquisition.getStructuredReportFragment(): acquisitionParameters=\n"+acquisitionParameters);
			{
				CodedSequenceItem targetRegion = acquisitionParameters == null ? null : acquisitionParameters.getAnatomy();
				cif.new CodeContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("123014","DCM","Target Region"),(targetRegion == null ? new CodedSequenceItem("T-D0010","SRT","Entire body") : targetRegion));
			}
			{
				String irradiationEventUID = acquisitionParameters == null ? null : acquisitionParameters.getIrradiationEventUID();
				cif.new UIDContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113769","DCM","Irradiation Event UID"),(irradiationEventUID == null ? new UIDGenerator().getNewUID() : irradiationEventUID));
			}
			cif.new CodeContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113820","DCM","CT Acquisition Type"),scanType.getCodedSequenceItem());
			if (acquisitionParameters != null) {
				acquisitionParameters.getStructuredReportFragment(contentItemFragment);
			}
			if (!scanType.equals(CTScanType.LOCALIZER)) {
				ContentItem dose = cif.new ContainerContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113829","DCM","CT Dose"),true/*continuityOfContentIsSeparate*/);
				cif.new NumericContentItem(dose,"CONTAINS",new CodedSequenceItem("113830","DCM","Mean CTDIvol"),CTDIvol,new CodedSequenceItem("mGy","UCUM","1.8","mGy"));
				if (phantomType != null) {
					cif.new CodeContentItem   (dose,"CONTAINS",new CodedSequenceItem("113835","DCM","CTDIw Phantom Type"),phantomType.getCodedSequenceItem());
				}
				cif.new NumericContentItem(dose,"CONTAINS",new CodedSequenceItem("113838","DCM","DLP"),DLP,new CodedSequenceItem("mGycm","UCUM","1.8","mGycm"));
			}
		}
		return contentItemFragment;
	}
}
