/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dose;

import com.pixelmed.dicom.*;
import com.pixelmed.utils.FloatFormatter;

public class CTAcquisitionParameters {
	
	protected String irradiationEventUID;
	protected CodedSequenceItem anatomy;
	protected String exposureTimeInSeconds;
	protected String scanningLengthInMM;
	protected String nominalSingleCollimationWidthInMM;
	protected String nominalTotalCollimationWidthInMM;
	protected String pitchFactor;
	protected String kvp;
	protected String tubeCurrent;
	protected String exposureTimePerRotation;
	
	protected ContentItem contentItemFragment;

	public CTAcquisitionParameters(String irradiationEventUID,CodedSequenceItem anatomy,String exposureTimeInSeconds,String scanningLengthInMM,
				String nominalSingleCollimationWidthInMM,String nominalTotalCollimationWidthInMM,String pitchFactor,
				String kvp,String tubeCurrent,String exposureTimePerRotation) {
		this.irradiationEventUID = irradiationEventUID;
		this.anatomy = anatomy;
		this.exposureTimeInSeconds = exposureTimeInSeconds;
		this.scanningLengthInMM = scanningLengthInMM;
		this.nominalSingleCollimationWidthInMM = nominalSingleCollimationWidthInMM;
		this.nominalTotalCollimationWidthInMM = nominalTotalCollimationWidthInMM;
		this.pitchFactor = pitchFactor;
		this.kvp = kvp;
		this.tubeCurrent = tubeCurrent;
		this.exposureTimePerRotation = exposureTimePerRotation;
	}
	
	public String getIrradiationEventUID() { return irradiationEventUID; }
	
	public CodedSequenceItem getAnatomy() { return anatomy; }
	
	public String getExposureTimeInSeconds() { return exposureTimeInSeconds; }
	
	public String getScanningLengthInMM() { return scanningLengthInMM; }
	
	public String getNominalSingleCollimationWidthInMM() { return nominalSingleCollimationWidthInMM; }
	
	public String getNominalTotalCollimationWidthInMM() { return nominalTotalCollimationWidthInMM; }
	
	public String getPitchFactor() { return pitchFactor; }
	
	public String getKVP() { return kvp; }
	
	public String getTubeCurrent() { return tubeCurrent; }
	
	public String getExposureTimePerRotation() { return exposureTimePerRotation; }
	
	public void deriveScanningLengthFromDLPAndCTDIVol(String dlp,String ctdiVol) {
		if (dlp != null && dlp.length() > 0 && ctdiVol != null && ctdiVol.length() > 0) {
			try {
				double dDLP = new Double(dlp).doubleValue();
				double dCTDIVol = new Double(ctdiVol).doubleValue();
				if (dDLP > 0 && dCTDIVol > 0) {	// don't want division by zero to produce NaN, and no point in producing length if no dose information ...
					scanningLengthInMM = FloatFormatter.toString(dDLP/dCTDIVol*10);	// DLP is in mGy.cm not mm
				}
			}
			catch (NumberFormatException e) {
				e.printStackTrace(System.err);
			}
		}
	}

	public String toString() {
		return toString(false);
	}

	public String toString(boolean pretty) {
		StringBuffer buffer = new StringBuffer();
		if (!pretty) {
			buffer.append("\tIrradiationEventUID=");
			buffer.append(irradiationEventUID);
		}

		buffer.append("\t");
		if (!pretty) {
			buffer.append("Anatomy=");
			buffer.append(anatomy);
		}
		else if (anatomy != null) {
			buffer.append(anatomy.getCodeMeaning());
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("ScanningLength=");
		}
		if (!pretty || (scanningLengthInMM != null && scanningLengthInMM.length() > 0)) {
			buffer.append(scanningLengthInMM);
			buffer.append(" mm");
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Collimation single/total=");
		}
		if (!pretty || (nominalSingleCollimationWidthInMM != null && nominalSingleCollimationWidthInMM.length() > 0) || (nominalTotalCollimationWidthInMM != null && nominalTotalCollimationWidthInMM.length() > 0)) {
			buffer.append(nominalSingleCollimationWidthInMM == null ? "" : nominalSingleCollimationWidthInMM);
			buffer.append("/");
			buffer.append(nominalTotalCollimationWidthInMM == null ? "" : nominalTotalCollimationWidthInMM);
			buffer.append(" mm");
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("PitchFactor=");
		}
		if (!pretty || (pitchFactor != null && pitchFactor.length() > 0)) {
			buffer.append(pitchFactor);
			buffer.append(":1");
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("KVP=");
		}
		if (!pretty || (kvp != null && kvp.length() > 0)) {
			buffer.append(kvp);
			buffer.append(" kVP");
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("TubeCurrent=");
		}
		if (!pretty || (tubeCurrent != null && tubeCurrent.length() > 0)) {
			buffer.append(tubeCurrent);
			buffer.append(" mA");
		}
		
		buffer.append("\t");
		if (!pretty) {
			buffer.append("Exposure time/per rotation=");
		}
		if (!pretty || (exposureTimeInSeconds != null && exposureTimeInSeconds.length() > 0) || (exposureTimePerRotation != null && exposureTimePerRotation.length() > 0)) {
			buffer.append(exposureTimeInSeconds == null ? "" : exposureTimeInSeconds);
			buffer.append("/");
			buffer.append(exposureTimePerRotation == null ? "" : exposureTimePerRotation);
			buffer.append(" s");
		}

		buffer.append("\n");

		return buffer.toString();
	}
	
	public ContentItem getStructuredReportFragment(ContentItem root) throws DicomException {
		if (contentItemFragment == null) {
			ContentItemFactory cif = new ContentItemFactory();
			contentItemFragment = cif.new ContainerContentItem(root,"CONTAINS",new CodedSequenceItem("113822","DCM","CT Acquisition Parameters"),true/*continuityOfContentIsSeparate*/);
			cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113824","DCM","Exposure Time"),exposureTimeInSeconds,new CodedSequenceItem("s","UCUM","1.8","s"));
			cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113825","DCM","Scanning Length"),scanningLengthInMM,new CodedSequenceItem("mm","UCUM","1.8","mm"));
			cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113826","DCM","Nominal Single Collimation Width"),nominalSingleCollimationWidthInMM,new CodedSequenceItem("mm","UCUM","1.8","mm"));
			cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113827","DCM","Nominal Total Collimation Width"),nominalTotalCollimationWidthInMM,new CodedSequenceItem("mm","UCUM","1.8","mm"));
			if (pitchFactor != null && pitchFactor.length() > 0) {
				cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113828","DCM","Pitch Factor"),pitchFactor,new CodedSequenceItem("{ratio}","UCUM","1.8","ratio"));
			}

			cif.new NumericContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113823","DCM","Number of X-Ray Sources"),"1",new CodedSequenceItem("{X-Ray sources}","UCUM","1.8","X-Ray sources"));
			ContentItem source = cif.new ContainerContentItem(contentItemFragment,"CONTAINS",new CodedSequenceItem("113831","DCM","CT X-Ray Source Parameters"),true/*continuityOfContentIsSeparate*/);
			cif.new TextContentItem(source,"CONTAINS",new CodedSequenceItem("113832","DCM","Identification Number of the X-Ray Source"),"1");
			cif.new NumericContentItem(source,"CONTAINS",new CodedSequenceItem("113733","DCM","KVP"),kvp,new CodedSequenceItem("kV","UCUM","1.8","kV"));
			cif.new NumericContentItem(source,"CONTAINS",new CodedSequenceItem("113833","DCM","Maximum X-Ray Tube Current"),tubeCurrent,new CodedSequenceItem("mA","UCUM","1.8","mA"));
			cif.new NumericContentItem(source,"CONTAINS",new CodedSequenceItem("113734","DCM","X-Ray Tube Current"),tubeCurrent,new CodedSequenceItem("mA","UCUM","1.8","mA"));
			cif.new NumericContentItem(source,"CONTAINS",new CodedSequenceItem("113834","DCM","Exposure Time per Rotation"),exposureTimePerRotation,new CodedSequenceItem("s","UCUM","1.8","s"));
			//>>>	CONTAINS	NUM	EV (113821, DCM, "X-ray Filter Aluminum Equivalent")	1	U		Units = EV (mm, UCUM, "mm")
		
		}
		return contentItemFragment;
	}
}

