#include "main.h"
#include "ee_thumb.h"
#include "viewer.h"
#include "prefs.h"


#define CELL_PADDING 10

/* DnD stuff */
static GtkTargetEntry dnd_types [] =
{ { "text/uri-list", 0, TARGET_URI_LIST } };

static gint n_dnd_types = sizeof (dnd_types) / sizeof(dnd_types[0]);

void drag_data_get (GtkWidget *widget, GdkDragContext *context,
		GtkSelectionData *selection_data, guint info, guint time)
{
		gchar *file;
		gchar *uri_list;

		file = gtk_object_get_data(GTK_OBJECT(widget), "file");

			uri_list = g_strconcat ("file:", file, NULL);
			gtk_selection_data_set (selection_data,
					  selection_data->target, 8,
					  (void *)uri_list, strlen((char *)uri_list));
			g_free (uri_list);
}

/* main function sets up gnome app and creates a window */
int main( int argc, char *argv[] )
{
	picwindow *pwindow;

	gnome_init("Picview", VERSION, argc, argv);
	gdk_imlib_init();
	ee_thumb_init_dirs();

	pwindow = create_window((gchar*)argv[1]);
	gtk_main ();
	write_prefs(pwindow);

	return 0;
}

/* callback to bring up preferences dialog */
void viewer_preferences_callback(GtkWidget *widget, gpointer data)
{
	show_prefs((picwindow*) data);
}

/* toggles viewing images in fullscreen mode or windowed mode
 * by changing picwindow.fullscreen */
void fullscreen_toggle(GtkWidget *widget, gpointer data)
{
	picwindow *pwindow;

	pwindow = (picwindow*) data;
	pwindow->fullscreen = !pwindow->fullscreen;
}

/* toggles using large or small previews
 * by changing picwindow.fullscreen */
void preview_size_toggle(GtkWidget *widget, gpointer data)
{
	picwindow *pwindow;

	pwindow = (picwindow*) data;
	pwindow->largesize = !pwindow->largesize;
	create_table(pwindow);
}

/* starts a slideshow. Passes the pwindow->file_list list to
 * view_images */
void slideshow_callback(GtkWidget *widget, gpointer data)
{
	picwindow *pwindow;

	pwindow = (picwindow*) data;
	view_images(pwindow->file_list);
}

/* displays a standard about box */
void about_callback(GtkWidget *widget, gpointer data)
{
	GtkWidget *about = NULL;

	gchar *authors[] = { "Scott Sams <sbsams@eos.ncsu.edu>", NULL }; 

	about = gnome_about_new(_("Picview"), VERSION,
		"Copyright (C) 1999 Scott Sams, The Free Software Foundation",
		(const gchar **) authors,
		_("Gdk-Imlib image preview viewer.\n\nhttp://www.digitallabyrinth.com/linux/picview/index.html\n"),
		"picview.png");
	gtk_widget_show(about);
}

/* creates the gnome menus and attaches callbacks. Adds the menu to picwindow.menubar */
void create_menus(picwindow *pwindow)
{
	GnomeUIInfo file_menu[] = {
		GNOMEUIINFO_MENU_EXIT_ITEM(close_application, NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo image_menu[] = {
		{GNOME_APP_UI_TOGGLEITEM, N_("_Full Screen"), NULL, fullscreen_toggle, pwindow, NULL, GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
		{GNOME_APP_UI_TOGGLEITEM, N_("_Large Previews"), NULL, preview_size_toggle, pwindow, NULL, GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
		GNOMEUIINFO_ITEM_DATA(N_("_Image Viewer..."), NULL, viewer_preferences_callback, pwindow, NULL),
		GNOMEUIINFO_ITEM_DATA(N_("Begin _Slideshow"), NULL, slideshow_callback, pwindow, NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo help_menu[] = {
		GNOMEUIINFO_MENU_ABOUT_ITEM(about_callback,NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo menubar[] = {
		GNOMEUIINFO_MENU_FILE_TREE(file_menu),
		GNOMEUIINFO_SUBTREE("_Image", image_menu),
		GNOMEUIINFO_MENU_HELP_TREE(help_menu),
		GNOMEUIINFO_END
	};

	/*create the menus for the application*/
	gnome_app_create_menus (GNOME_APP (pwindow->window), menubar);

	GTK_CHECK_MENU_ITEM(image_menu[0].widget)->active = pwindow->fullscreen;
	GTK_CHECK_MENU_ITEM(image_menu[1].widget)->active = pwindow->largesize;
}

void close_application( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	gtk_main_quit();
}

/* when a directory button is clicked, the window changes */
void dir_button_clicked( GtkWidget *widget, gpointer data )
{
	picwindow *pwindow = (picwindow*) data;
	gchar *filename = (gchar*) gtk_object_get_data(GTK_OBJECT(widget), "file");
	pwindow->path = filename;
	create_table(pwindow);
}

/* when a picture button is clicked, the image is displayed */
void button_clicked( GtkWidget *widget, gpointer data )
{
	picwindow *pwindow = (picwindow*) data;
	gchar *filename = (gchar*) gtk_object_get_data(GTK_OBJECT(widget), "file");
	gchar *command;
	GList *iterator;

	if (pwindow->fullscreen)
	{
		//view_image_fullscreen(filename, NULL);
		iterator = pwindow->file_list;
		while (iterator && strcmp(filename, iterator->data))
		{
			iterator = iterator->next;
		}
		
		view_images(iterator);
	}
	else
	{
		if (!pwindow->use_external_viewer)
			view_image_window(filename);
		else
		{
			signal(SIGCHLD, SIG_IGN);
			if (fork() == 0)
			{
				execlp(pwindow->viewer, pwindow->viewer, filename, NULL);
				fprintf(stderr, "could not execute viewer %s", pwindow->viewer);
				_exit( 127);
			}
		}
	}
}

/* when the window is resized, this callback determines if the icons should be rearranged */
void resize_cb(GtkWidget *widget, GdkEventConfigure *event)
{
	picwindow *pwindow = NULL;
	gint newcols;

	if (widget)
	{
		pwindow  = (picwindow*) (gtk_object_get_data(GTK_OBJECT(widget), "picwindow"));
		if (pwindow)
		{
			/* don't resize window if we are in the middle of drawing the icons */
			if (pwindow->redrawing)
				return;

			/* only resize the window at certain boundary points */
			newcols = (event->width - 30) / (pwindow->butt_width);
			if ((newcols != pwindow->cols) && (newcols != 0))
			{
				pwindow->cols = newcols;
				resize_table(pwindow);
			}
		}
	}
}

/* drag n drops from gmc. Any directory icon or url can be dropped into the main
 * window, and it will refresh with a view of that directory. pretty nifty */
void drop_callback(GtkWidget *widget, GdkDragContext *context,
              gint x, gint y, GtkSelectionData *selection_data,
              guint info, guint time, gpointer data)
{
	GList *name;
	picwindow *pwindow = (picwindow*) data;
	gint err;
	struct stat buff;

	switch (info)
	{
	case TARGET_URI_LIST:
		name = gnome_uri_list_extract_filenames (selection_data->data);
		if (name)
		{
			/* only accept directories */
			err = stat(name->data, &buff);
			if ((!err) && (buff.st_mode & S_IFDIR))
			{
					g_free(pwindow->path);
					pwindow->path = g_strconcat((gchar*) name->data, NULL);
					create_table(pwindow);
			}
		}
		gnome_uri_list_free_strings (name);
		break;
	default:
		break;
	}
}

/* creates the main gnome application window containing menubar, scrolled table, etc.
 * Takes an initial path as an argument */
picwindow *create_window(gchar *path)
{
	gchar *title_str;

	GtkWidget *window, *scrolled_window, *progress_bar, *vbox;
	picwindow *pwindow;

	pwindow = g_new0(picwindow, 1);
	read_prefs(pwindow);
	title_str = g_strconcat("Picview: ", path, NULL);

	/* create a new GNOME app */
	window = gnome_app_new ("Picview", title_str);
	g_free(title_str);

	gtk_signal_connect( GTK_OBJECT (window), "delete_event",
		GTK_SIGNAL_FUNC (close_application), NULL );
	gtk_container_set_border_width( GTK_CONTAINER (window), 0 );
	gtk_window_set_default_size(GTK_WINDOW(window), PWINDOW_DEFAULT_WIDTH, PWINDOW_DEFAULT_HEIGHT);
	gtk_widget_show( window );

	/* vbox holds everything in the gnome app */
	vbox = gtk_vbox_new(FALSE, 0);
	gnome_app_set_contents (GNOME_APP (window), vbox);
	gtk_widget_show (vbox);

	/* create a new scrolled window. */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 2);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, 1, 1, 2);
	gtk_widget_show (scrolled_window);

	/* progress bar */
	progress_bar = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(vbox), progress_bar, 0, 0, 2);
	gtk_widget_show (progress_bar);

	pwindow->window = window;
	pwindow->scrolled_window = scrolled_window;
	pwindow->progress_bar = progress_bar;
	if (path)
		pwindow->path = g_strconcat(path, NULL);
	pwindow->redrawing = FALSE;

	/* populate the table with pictures */
	pwindow->cols = 5;
	pwindow->rows = 4;
	pwindow->butt_width = 150;
	pwindow->butt_height = 100;

	create_menus(pwindow);
	create_table(pwindow);

	gtk_object_set_data(GTK_OBJECT(window), "picwindow", pwindow);
	gtk_signal_connect(GTK_OBJECT(window), "configure_event",
		GTK_SIGNAL_FUNC(resize_cb), NULL);
	gtk_drag_dest_set (GTK_WIDGET (window), GTK_DEST_DEFAULT_MOTION
		| GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
		dnd_types, n_dnd_types, GDK_ACTION_COPY);
	gtk_signal_connect (GTK_OBJECT (window), "drag_data_received",
		GTK_SIGNAL_FUNC(drop_callback), pwindow);

	return pwindow;
}

/* resizes the icon table to a new width of picwindow.cols by moving all
 * the buttons to a new table and destroying the old table */
void resize_table(picwindow *pwindow)
{
	GtkWidget *table, *button;
	GList *iterator;
	int i, j, p, width;

	if (pwindow->table == NULL)
		return;

	width = pwindow->cols;

	pwindow->redrawing = TRUE;
	
	table = gtk_table_new((g_list_length(pwindow->button_list) - 1)/width + 1, width, TRUE);
	gtk_table_set_row_spacings (GTK_TABLE (table), CELL_PADDING);
	gtk_table_set_col_spacings (GTK_TABLE (table), CELL_PADDING);

	iterator = pwindow->button_list;
	p = 0;
	while (iterator)
	{
		i = p % width;
		j = p / width;

		button = iterator->data;
		if (button)
		{
			/* unattach the button from the old table (without deleting the button)
			 * and re-attach it to the new table */
			gtk_widget_ref(button);
			gtk_container_remove(GTK_CONTAINER(pwindow->table), button);
			gtk_table_attach (GTK_TABLE (table), button, i, i+1, j, j+1, GTK_FILL, GTK_FILL, 0, 0);
			gtk_widget_unref(button);
		}

		iterator = iterator->next;
		p++;
	}

	/* remove the scrolled window's viewport widget, which deletes the old table */
	/* then add the new table to the scrolled window */
	//gtk_container_remove(GTK_CONTAINER(pwindow->scrolled_window), GTK_BIN(pwindow->scrolled_window)->child);	
	gtk_widget_destroy(GTK_BIN(pwindow->scrolled_window)->child);	
	pwindow->table = table;
	gtk_widget_show(pwindow->table);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (pwindow->scrolled_window), pwindow->table);

	while (gtk_events_pending())
		gtk_main_iteration();

	pwindow->redrawing = FALSE;
}

/* creates a new table full of icons in the main window. This is usually called
 * once at startup and whenever the directory changes. This is where all the
 * preview icons are loaded and made into buttons */
void create_table(picwindow *pwindow)
{
	GList *dirlist, *filelist, *iterator = NULL;
	GtkWidget *button;
	gchar *path, *filename, *fqpn;
	int i, j, p, width, numfiles, newcols;

	path = pwindow->path;

	if (pwindow->table != NULL)
	{
		gtk_widget_destroy(pwindow->table);	
	}

	if (pwindow->button_list != NULL)
	{
		g_list_free(pwindow->button_list);
		pwindow->button_list = NULL;
	}

	if (pwindow->file_list != NULL)
	{
		g_list_free(pwindow->file_list);
		pwindow->file_list = NULL;
	}

	dirlist = load_dir(path, TRUE);
	if (dirlist == NULL)
		return;

	filelist = load_dir(path, FALSE);
	if (filelist == NULL)
		return;

	pwindow->redrawing = TRUE;

	numfiles = g_list_length(filelist) + g_list_length(dirlist) - 1;
	width = pwindow->cols;

	/* create the table and put it in the scrolled window */
	pwindow->table = gtk_table_new((numfiles - 1)/width + 1, width, TRUE);
	gtk_table_set_row_spacings (GTK_TABLE (pwindow->table), CELL_PADDING);
	gtk_table_set_col_spacings (GTK_TABLE (pwindow->table), CELL_PADDING);

	gtk_widget_show(pwindow->table);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (pwindow->scrolled_window), pwindow->table);

	/* populate the table with the dirlist first */
	iterator = dirlist->next;
	p = 0; i = 0; j = 0;
	while (iterator)
	{
		filename = iterator->data;
		if (filename)
		{
			fqpn = g_strconcat(path, "/", filename, NULL);

			/* only update every 50th time */
			if (p % 50 == 0)
				while (gtk_events_pending())
					gtk_main_iteration();

			button = create_dir_button(path, filename, pwindow->largesize);
			gtk_progress_bar_update(GTK_PROGRESS_BAR(pwindow->progress_bar), p/((float)numfiles));

			if (button)
			{
				/* add the button to the button list so we can keep track of it later, when resizing */
				pwindow->button_list = g_list_append(pwindow->button_list, (gpointer) button);
				gtk_table_attach (GTK_TABLE (pwindow->table), button, i, i+1, j, j+1, GTK_FILL, GTK_FILL, 0, 0);
				gtk_object_set_data(GTK_OBJECT(button), "file", fqpn);
				gtk_signal_connect( GTK_OBJECT(button), "clicked",
					GTK_SIGNAL_FUNC(dir_button_clicked), pwindow);
			
				/* setup drag for this button */
				gtk_drag_source_set(button, GDK_BUTTON1_MASK, dnd_types, n_dnd_types, GDK_ACTION_COPY);
				gtk_signal_connect(GTK_OBJECT(button), "drag_data_get", GTK_SIGNAL_FUNC(drag_data_get), NULL);

				i++;
				if (i == width)
				{
					i = 0;
					j++;
				}
			}
		}
		iterator = iterator->next;
		p++;
	}

	/* populate the table with the filelist */
	iterator = filelist->next;
	while (iterator)
	{
		filename = iterator->data;
		if (filename)
		{
			fqpn = g_strconcat(path, "/", filename, NULL);

			/* only update every 50th time */
			if (p % 50 == 0)
				while (gtk_events_pending())
					gtk_main_iteration();

			button = create_preview_button(path, filename, pwindow->largesize);
			gtk_progress_bar_update(GTK_PROGRESS_BAR(pwindow->progress_bar), p/((float)numfiles));

			if (button)
			{
				/* add the button to the button list so we can keep track of it later, when resizing */
				pwindow->button_list = g_list_append(pwindow->button_list, (gpointer) button);
				/* add the file name to the file_list so we can use it later in a slideshow */
				pwindow->file_list = g_list_append(pwindow->file_list, fqpn);
				gtk_table_attach (GTK_TABLE (pwindow->table), button, i, i+1, j, j+1, GTK_FILL, GTK_FILL, 0, 0);
				gtk_object_set_data(GTK_OBJECT(button), "file", fqpn);
				gtk_signal_connect( GTK_OBJECT(button), "clicked",
					GTK_SIGNAL_FUNC(button_clicked), pwindow);
			
				/* setup drag for this button */
				gtk_drag_source_set(button, GDK_BUTTON1_MASK, dnd_types, n_dnd_types, GDK_ACTION_COPY);
				gtk_signal_connect(GTK_OBJECT(button), "drag_data_get", GTK_SIGNAL_FUNC(drag_data_get), NULL);

				i++;
				if (i == width)
				{
					i = 0;
					j++;
				}
			}
		}
		iterator = iterator->next;
		p++;
	}

	pwindow->button_list = g_list_first(pwindow->button_list);
	pwindow->file_list = g_list_first(pwindow->file_list);

	g_list_free(dirlist);
	g_list_free(filelist);

	gtk_progress_bar_update(GTK_PROGRESS_BAR(pwindow->progress_bar), 1.0);
	pwindow->redrawing = FALSE;

	while (gtk_events_pending())
		gtk_main_iteration();

	if (pwindow->button_list)
	{
		pwindow->butt_width = GTK_WIDGET(pwindow->button_list->data)->allocation.width + CELL_PADDING;
		pwindow->butt_height = GTK_WIDGET(pwindow->button_list->data)->allocation.height + CELL_PADDING;
	}

	/* only resize the window at certain boundary points */
	newcols = (pwindow->window->allocation.width - 30) / (pwindow->butt_width);
	if ((newcols != pwindow->cols) && (newcols != 0))
	{
		pwindow->cols = newcols;
		resize_table(pwindow);
	}
}

/* returns a button widget containing the directory and dirname. */
GtkWidget* create_dir_button(gchar *path, gchar *filename, gboolean large)
{
	GdkImlibImage* preview_img;
	GdkPixmap *pmap, *mask;
	gint width, height;
	gchar *fqpn;
	GtkWidget *button, *vbox, *label, *preview;

	fqpn = g_strconcat(path, "/", filename, NULL);

	preview_img = gdk_imlib_load_image("/usr/share/pixmaps/gnome-folder.png");
	if (!preview_img)
		return NULL;

	button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(filename);
	gtk_box_pack_end(GTK_BOX(vbox), label, 0, 0, 3);

	width = preview_img->rgb_width;
	height = preview_img->rgb_height;
	gdk_imlib_render(preview_img, width, height);
	pmap = gdk_imlib_move_image(preview_img);
	mask = gdk_imlib_move_mask(preview_img);

	preview = gtk_pixmap_new( pmap, mask);
	gtk_box_pack_end(GTK_BOX(vbox), preview, 0, 0, 3);
	gtk_widget_show( preview );

	gdk_imlib_free_pixmap(pmap);
	gdk_imlib_destroy_image(preview_img);

	gtk_container_add(GTK_CONTAINER(button), vbox);
	gtk_widget_show( label );
	gtk_widget_show( vbox );
	gtk_widget_show (button);

	return button;
}

/* returns a button widget containing the preview and filename. This
 * is where the thumbnail is loaded using imlib */
GtkWidget* create_preview_button(gchar *path, gchar *filename, gboolean large)
{
	GdkImlibImage* preview_img;
	GdkPixmap *pmap, *mask;
	gint width, height;
	GtkWidget *button, *vbox, *label, *preview;

	if (large)
		preview_img = ee_thumb_find(path, filename, "previews", 180, 160);
	else
		preview_img = ee_thumb_find(path, filename, "icons", 80, 60);

	if (!preview_img)
		return NULL;

	button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(filename);
	gtk_box_pack_end(GTK_BOX(vbox), label, 0, 0, 3);

	width = preview_img->rgb_width;
	height = preview_img->rgb_height;
	gdk_imlib_render(preview_img, width, height);
	pmap = gdk_imlib_move_image(preview_img);
	mask = gdk_imlib_move_mask(preview_img);

	preview = gtk_pixmap_new( pmap, mask);
	gtk_box_pack_end(GTK_BOX(vbox), preview, 0, 0, 3);
	gtk_widget_show( preview );

	gdk_imlib_free_pixmap(pmap);
	gdk_imlib_destroy_image(preview_img);

	gtk_container_add(GTK_CONTAINER(button), vbox);
	gtk_widget_show( label );
	gtk_widget_show( vbox );
	gtk_widget_show (button);

	return button;
}

/* returns a list of all the files, or directories in a directory */
/* if get_dirs is true, only directories will be returned */
GList *load_dir(gchar *path, gboolean get_dirs)
{
	DIR *dir;
	struct dirent *file;
	struct stat buff;
	gint err;
	gchar *fqpn;
	GList *filelist;

	if (path == NULL)
		return NULL;
	dir = opendir(path);
	if (!dir)
		return NULL;

	filelist = g_list_alloc();
	filelist->data = "";

	file = readdir(dir);
	while (file)
	{
		if ((strcmp(file->d_name, "..") == 0) || (file->d_name[0] != '.'))
		{
			fqpn = g_strconcat(path, "/", file->d_name, NULL);
			err = stat(fqpn, &buff);
			if (get_dirs)
			{
				if (buff.st_mode & S_IFDIR)
				{
					filelist = g_list_insert_sorted(filelist, (gpointer) g_strdup(file->d_name), (GCompareFunc) valcomp);
				}
			}
			else
			{
				if (!(buff.st_mode & S_IFDIR))
				{
					filelist = g_list_insert_sorted(filelist, (gpointer) g_strdup(file->d_name), (GCompareFunc) valcomp);
				}
			}
			g_free(fqpn);
		}
		file = readdir(dir);
	}
	closedir(dir);
	filelist->data = NULL;

	return filelist;
}

/* helper function used for sorting the glist */
gint valcomp(gconstpointer a, gconstpointer b)
{
	return strcmp((const char *) a, (const char *) b);
}
