/*
 *  Program: pgn-extract: a Portable Game Notation (PGN) extractor.
 *  Copyright (C) 1994-2005 David Barnes
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  David Barnes may be contacted as D.J.Barnes@kent.ac.uk
 *  http://www.cs.kent.ac.uk/people/staff/djb/
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "bool.h"
#include "mymalloc.h"
#include "defs.h"
#include "typedef.h"
#include "tokens.h"
#include "taglist.h"
#include "lex.h"
#include "moves.h"
#include "map.h"
#include "lists.h"
#include "output.h"
#include "end.h"
#include "grammar.h"
#include "hashing.h"
#include "argsfile.h"

/* The maximum length of an output line.  This is conservatively
 * slightly smaller than the PGN export standard of 80.
 */
#define MAX_LINE_LENGTH 75

/* Define a file name relative to the current directory representing
 * a file of ECO classificiations.
 */
#ifndef DEFAULT_ECO_FILE
#define DEFAULT_ECO_FILE "eco.pgn"
#endif

    /* This structure holds details of the program state
     * available to all parts of the program.
     * This goes against the grain of good structured programming
     * principles, but most of these fields are set from the program's
     * arguments and are read-only thereafter. If I had done this in
     * C++ there would have been a cleaner interface!
     */
StateInfo GlobalState = {
    FALSE,                /* check_only (-r) */
    TRUE,                 /* verbose (-s) */
    TRUE,                 /* keep_NAGs (-N) */
    TRUE,                 /* keep_comments (-C) */
    TRUE,                 /* keep_variations (-V) */
    FALSE,                /* seven_tag_roster (-7) */
    TRUE,                 /* match_permutations (-v) */
    FALSE,                /* positional_variations (-x) */
    FALSE,                /* use_soundex (-S) */
    FALSE,                /* suppress_duplicates (-D) */
    FALSE,                /* suppress_originals (-U) */
    FALSE,                /* check_tags */
    FALSE,                /* add_ECO (-e) */
    FALSE,                /* parsing_ECO_file (-e) */
    DONT_DIVIDE,          /* ECO_level (-E) */
    SAN,                  /* output_format (-W) */
    MAX_LINE_LENGTH,      /* max_line_length (-w) */
    FALSE,                /* use_virtual_hash_table (-Z) */
    FALSE,                /* check_move_bounds (-b) */
    0,                    /* depth_of_positional_search */
    0,                    /* num_games_processed */
    0,                    /* num_games_matched */
    0,                    /* games_per_file (-#) */
    1,                    /* next_file_number */
    0,                    /* lower_move_bound */
    10000,                /* upper_move_bound */
    FALSE,		  /* output_FEN_string */
    (char *)NULL,         /* current_input_file */
    NORMALFILE,           /* current_file_type */
    DEFAULT_ECO_FILE,     /* eco_file (-e) */
    (FILE *)NULL,         /* outputfile (-o, -a). Default is stdout */
    (char *)NULL,         /* output_filename (-o, -a) */
    (FILE *)NULL,         /* logfile (-l). Default is stderr */
    (FILE *)NULL,         /* duplicate_file (-d) */
    (FILE *)NULL,         /* non_matching_file (-n) */
};

         /* Prepare the output file handles in GlobalState. */
static void
init_default_global_state(void)
{
    GlobalState.outputfile = stdout;
    GlobalState.logfile = stderr;
    set_output_line_length(GlobalState.max_line_length);
}

int
main(int argc, char *argv[])
{   int argnum;

    /* Prepare global state. */
    init_default_global_state();
    /* Prepare the GameHeader. */
    init_game_header();
    /* Prepare the tag lists for -t/-T matching. */
    init_tag_lists();
    /* Prepare the hash tables for transposition detection. */
    init_hashtab();
    /* Initialise the lexical analyser's tables. */
    init_lex_tables();
    /* Allow for some arguments. */
    for(argnum = 1; argnum < argc; argnum++){
        const char *argument = argv[argnum];
        if(argument[0] == '-'){
            switch(argument[1]){
                case SEVEN_TAG_ROSTER_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case TAG_ROSTER_ARGUMENT:
		    process_argument(argument[1],&argument[2]);
                    break;
                case FILE_OF_ARGUMENTS_ARGUMENT:
                   {  /* We expect a file name containing arguments. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case APPEND_TO_OUTPUT_FILE_ARGUMENT:
                    { /* We expect an output file to which to append. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
		    }
                    break;
                case BOUNDS_ARGUMENT:
                    { const char *bound = &(argument[2]);
		      process_argument(argument[1],bound);
                  }
                  break;
                case CHECK_FILE_ARGUMENT:
                    { /* We expect a list of files as the game sources.
                       * These files are CHECKFILEs, to act as checks
                       * for duplicates in any NORMALFILEs.
                       */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case DONT_KEEP_COMMENTS_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case DUPLICATES_FILE_ARGUMENT:
                    { /* We expect an output file for duplicate games. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case DONT_KEEP_DUPLICATES_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case USE_ECO_FILE_ARGUMENT:
                    { /* A filename of ECO lines is optional. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case ECO_OUTPUT_LEVEL_ARGUMENT:
                    /* Set an ECO output level. */
		    process_argument(argument[1],&argument[2]);
                    break;
                case FILE_OF_FILES_ARGUMENT:
                    { /* We expect a list of files as the game sources. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
		case OUTPUT_FEN_STRING_ARGUMENT:
		    process_argument(argument[1],"");
		    break;
                case HELP_ARGUMENT:
		    process_argument(argument[1],&argument[2]);
                    break;
                case ALTERNATIVE_HELP_ARGUMENT:
		    process_argument(HELP_ARGUMENT,"");
                    break;
                case WRITE_TO_LOG_FILE_ARGUMENT:
		    /* Create new log file. */
                case APPEND_TO_LOG_FILE_ARGUMENT:
		    /* Append to log file. */
		    process_argument(argument[1],&argument[2]);
                    break;
                case NON_MATCHING_GAMES_ARGUMENT:
                    { /* We expect an output file for unmatched games. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case DONT_KEEP_NAGS_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case WRITE_TO_OUTPUT_FILE_ARGUMENT:
                    { /* We expect an output file to write to. */
                      const char *filename = &(argument[2]);
		      process_argument(argument[1],filename);
                    }
                    break;
                case DONT_MATCH_PERMUTATIONS_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case CHECK_ONLY_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case KEEP_SILENT_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case USE_SOUNDEX_ARGUMENT:
                    /* Use soundex matches for player tags. */
		    process_argument(argument[1],"");
                    break;
                case TAG_EXTRACTION_ARGUMENT:
                    /* A single tag extraction criterion. */
		    process_argument(argument[1],&argument[2]);
                    break;
                case SUPPRESS_ORIGINALS_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case DONT_KEEP_VARIATIONS_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case LINE_WIDTH_ARGUMENT:
		    process_argument(argument[1],&argument[2]);
                    break;
                case OUTPUT_FORMAT_ARGUMENT:
		    process_argument(argument[1],&argument[2]);
                    break;
                case TAGS_ARGUMENT:
                case MOVES_ARGUMENT:
                case POSITIONS_ARGUMENT:
                case ENDINGS_ARGUMENT:
                    {   /* From the command line, we require an
		         * associated file argument.
			 * Check this here, as it is not the case
			 * when reading arguments from an argument file.
			 */
                        const char *filename =  &(argument[2]);
			const char argument_letter = argument[1];
                        if(*filename == '\0'){
			      /* Try to pick it up from the next argument. */
			      if((argnum+1) < argc){
				  ++argnum;
				  filename = argv[argnum];
			      }
			    /* Make sure the filename does not look
			     * like the next argument.
			     */
			    if((*filename == '\0') ||
			       (*filename == '-')){
				fprintf(GlobalState.logfile,
					"Usage: -%cfilename or -%c filename\n",
					argument_letter,argument_letter);
				exit(1);
			    }
			}
			else{
			    process_argument(argument_letter,filename);
                        }
                    }
                    break;
                case USE_VIRTUAL_HASH_TABLE_ARGUMENT:
		    process_argument(argument[1],"");
                    break;
                case GAMES_PER_FILE_ARGUMENT:
		    process_argument(argument[1],&argument[2]);
                    break;
                default:
                    fprintf(GlobalState.logfile,
			    "Unknown flag %s. Use -%c for usage details.\n",
			    argument,HELP_ARGUMENT);
                    break;
            }
        }
        else{
            /* Should be a file name containing games. */
            add_filename_to_source_list(argument,NORMALFILE);
        }
    }
    /* Prepare the hash tables for duplicate detection. */
    init_duplicate_hash_table();

    if(GlobalState.add_ECO){
        /* Read in a list of ECO lines in order to classify the games. */
        if(open_eco_file(GlobalState.eco_file)){
            /* Indicate that the ECO file is currently being parsed. */
            GlobalState.parsing_ECO_file = TRUE;
            yyparse(ECOFILE);
            GlobalState.parsing_ECO_file = FALSE;
        }
        else{
            fprintf(GlobalState.logfile,"Unable to open the ECO file %s.\n",
                        GlobalState.eco_file);
            exit(1);
        }
    }

    /* Open up the first file to act as Lex's source of input. */
    if(!open_first_file()){
        exit(1);
    }
    yyparse(GlobalState.current_file_type);
    /* Remove any temporary files. */
    clear_duplicate_hash_table();
    if(GlobalState.verbose){
        fprintf(GlobalState.logfile,"%lu game%s matched out of %lu.\n",
                        GlobalState.num_games_matched,
                        GlobalState.num_games_matched == 1?"":"s",
                        GlobalState.num_games_processed);
    }
    if((GlobalState.logfile != stderr) && (GlobalState.logfile != NULL)){
        (void) fclose(GlobalState.logfile);
    }
    return 0;
}
