/*
 * detailed_ia64_old_smpl.c - detailed sampling module for the all IA-64 PMU models
 *                            using perfmon v2.0 interface
 *
 * Copyright (c) 2005-2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"
#include "pfmon_smpl_ia64_old.h"
#include <perfmon/perfmon_default_smpl.h> /* IA-64 compatibility ONLY! */

static int has_btb; /* set to 1 if PMU has Branch Trace Buffer or Execution Trace Buffer */

extern int print_ita_reg(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, unsigned long val);
extern int print_ita2_reg(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, unsigned long val);
extern int print_mont_reg(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, unsigned long val);

static int (*print_reg)(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, unsigned long val);

static int
print_ia64_reg(pfmon_event_set_t *set, pfmon_smpl_desc_t *csmpl, int rnum, unsigned long val)
{
	return fprintf(csmpl->smpl_fp, "\tPMD%-3d:0x%016lx\n", rnum, val);
}

static int
detailed_ia64_process_samples(pfmon_sdesc_t *sdesc)
{
	pfm_default_smpl_hdr_t *hdr;
	pfm_default_smpl_entry_t *ent;
	pfmon_smpl_desc_t *csmpl;
	pfmon_event_set_t *active_set;
	FILE *fp;
	unsigned long *msk;
	size_t count, i;
	uint64_t entry;
	void *hash_desc;
	unsigned long *reg;
	unsigned int j, n;
	int ret;
	unsigned short ovfl_pmd; /* undefined PMD */

	csmpl      = &sdesc->csmpl;
	hdr        = csmpl->smpl_hdr;
	fp         = csmpl->smpl_fp;
	hash_desc  = csmpl->sym_hash;
	ent        = (pfm_default_smpl_entry_t *)(hdr+1);
	entry      = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;
	count      = hdr->hdr_count;
	active_set = sdesc->sets; /* only one set when sampling */

	DPRINT(("hdr_count=%lu hdr=%p active_set=%u\n", count, hdr, active_set->id));

	for(i=0; i < count; i++) {

		ovfl_pmd = ent->ovfl_pmd;
		ret = fprintf(fp, "entry %"PRIu64" PID:%d TID:%d CPU:%d STAMP:0x%"PRIx64" OVFL:%d LAST_VAL:%"PRIu64" SET:%u IIP:",
			entry,
			ent->tgid,
			ent->pid,
			ent->cpu,
			ent->tstamp,
			ovfl_pmd, 
			-ent->last_reset_val,
			ent->set);

		pfmon_print_address(fp, hash_desc, &options.primary_syms, PFMON_TEXT_SYMBOL, ent->ip);

		reg = (unsigned long *)(ent+1);

		msk = active_set->rev_smpl_pmds[ovfl_pmd].smpl_pmds;
		n   = active_set->rev_smpl_pmds[ovfl_pmd].num_smpl_pmds;

		for(j=0; n; j++) {	
			if (pfmon_bv_isset(msk, j)) {
				ret = (*print_reg)(active_set, csmpl, j, *reg);
				reg++;
				n--;
			}
		}
		/* fprintf() error detection */
		if (ret == -1) goto error;

		/*
		 * entries are contiguously stored
		 */
		ent  = (pfm_default_smpl_entry_t *)reg;	
		entry++;
	}
	
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += count;
	} else {
		csmpl->entry_count += count;
	}
	csmpl->last_count = count;
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
error:
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

/*
 * Allocate space for the optional BTB trace buffer
 */
static int
detailed_ia64_initialize_session( pfmon_smpl_desc_t *csmpl)
{
	unsigned int num_pmds;

	if (has_btb == 0) return 0;
	/*
	 * let's be generous and consider all PMDS to be potentially BTB
	 */
	pfm_get_num_pmds(&num_pmds);

	csmpl->data = calloc(1, sizeof(unsigned long)*num_pmds);

	return csmpl->data == NULL ? -1 : 0;
}

static int
detailed_ia64_terminate_session(pfmon_sdesc_t *sdesc)
{
	pfmon_smpl_desc_t *csmpl = &sdesc->csmpl;
	if (csmpl->data) free(csmpl->data);
	return 0;
}

static int
detailed_ia64_initialize_module(void)
{
	switch(options.pmu_type) {
		case PFMLIB_ITANIUM_PMU:
			has_btb = 1;
			print_reg = print_ita_reg;
			break;
		case PFMLIB_ITANIUM2_PMU:
			has_btb = 1;
			print_reg = print_ita2_reg;
			break;
		case PFMLIB_MONTECITO_PMU:
			has_btb = 1;
			print_reg = print_mont_reg;
			break;
		default:
			print_reg = print_ia64_reg;
	}
	return 0;
}

pfmon_smpl_module_t detailed_ia64_old_smpl_module = {
	.name		    = "detailed-ia64",
	.pmu_mask	    = PFMON_PMU_MASK(PFMLIB_GEN_IA64_PMU),
	.description	    = "Itanium processors detailed sampling",
	.process_samples    = detailed_ia64_process_samples,
	.initialize_session = detailed_ia64_initialize_session,
	.terminate_session  = detailed_ia64_terminate_session,
	.initialize_module  = detailed_ia64_initialize_module,
	.init_ctx_arg	    = default_smpl_init_ctx_arg,
	.check_version	    = default_smpl_check_version,
	.check_new_samples  = default_smpl_check_new_samples,
	.flags		    = PFMON_SMPL_MOD_FL_LEGACY,
	.uuid		    = PFM_DEFAULT_SMPL_UUID
};
