/*
 * pfmon.c 
 *
 * Copyright (C) 2001-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <ctype.h>
#include <regex.h>
#include <limits.h>

extern pfmon_support_t pfmon_itanium2;
extern pfmon_support_t pfmon_itanium;
extern pfmon_support_t pfmon_generic_ia64;

static pfmon_support_t *pfmon_cpus[]={
#ifdef CONFIG_PFMON_ITANIUM2
	&pfmon_itanium2,
#endif
#ifdef CONFIG_PFMON_ITANIUM
	&pfmon_itanium,
#endif
#ifdef CONFIG_PFMON_GENERIC_IA64
	&pfmon_generic_ia64,		/* must always be last of IA-64 choices */
#endif
	NULL
};

#define PFMON_FORCED_GEN	"pfmon_gen"

pfmon_support_t		*pfmon_current;	/* current pfmon support */
program_options_t	options;	/* keep track of global program options */


static	pfarg_context_t *master_ctx;			/* master perfmon context */
static	pfarg_reg_t	 master_pd[PFMON_MAX_PMDS];	/* master PMDs arguments  */
static	pfarg_reg_t	 master_pc[PFMON_MAX_PMCS];	/* master PMCs arguments  */

/*
 * does the final preparation on the pmc arguments
 * and also initializes the pmds arguments.
 *
 * the function does not make the perfmonctl() call to
 * install pmcs or pmds.
 *
 * As such this function is supposed to be called only once.
 */
static int
prepare_pmd_registers(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	unsigned int i;

	/*
	 * install initial value, long and short overflow rates
	 */
	for(i=0; i < evt->inp.pfp_event_count; i++) {

		pd[i].reg_num         = pc[i].reg_num;
		pd[i].reg_value       = options.long_rates[i].value;
		pd[i].reg_short_reset = options.short_rates[i].value;
		pd[i].reg_long_reset  = options.long_rates[i].value;
		pd[i].reg_random_mask = options.long_rates[i].mask;	/* mask is per monitor */
		pd[i].reg_random_seed = options.long_rates[i].seed;	/* seed is per monitor */

		vbprintf("[pmd%u ival=0x%"PRIx64
			 "  long_rate=0x%"PRIx64
			 " short_rate=0x%"PRIx64
			 " mask=0x%"PRIx64
			 " seed=%"PRIu64
			 " rand=%c]\n",
			pd[i].reg_num,
			pd[i].reg_value,
			pd[i].reg_long_reset,
			pd[i].reg_short_reset,
			pd[i].reg_random_mask,
			pd[i].reg_random_seed,
			pc[i].reg_flags & PFM_REGFL_RANDOM ? 'y' : 'n');
	}
	return 0;
}

static int
prepare_pmc_registers(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	unsigned long tmp_smpl_pmds = 0UL;
	unsigned long m;
	unsigned int i;

	/*
	 * nothing special to do if not sampling
	 */
	if (options.opt_use_smpl == 0) return 0;

	for(i=0; i < evt->inp.pfp_event_count; i++) {
		/*
		 * The counters for which a sampling period has been
		 * set must have their notify flag set unless requested
		 * otherwise by user in which case the
		 * buffer will saturate: you stop when the buffer becomes
		 * full, i.e., collect the first samples only.
		 *
		 * Counters for which no sampling period is set are
		 * considered part of the set of PMC/PMD to store
		 * in each sample.
		 */
		if (options.long_rates[i].flags & PFMON_RATE_VAL_SET) {

			if (options.opt_no_ovfl_notify == 0) 
				pc[i].reg_flags |= PFM_REGFL_OVFL_NOTIFY;

			/*
			 * set randomization flag
			 */
			if (options.events[i].flags & PFMON_MONITOR_FL_RANDOMIZE)
				pc[i].reg_flags |= PFM_REGFL_RANDOM;
		} else {
			/*
			 * accumulate list of all PMC/PMD pairs that we have
			 * to record in each sample.
			 */
			tmp_smpl_pmds |= M_PMD(pc[i].reg_num) | options.smpl_pmds[i];
		}
	}
	/*
	 * some common PMD may have already been requested by module specific
	 * code (as part of post_options).
	 */
	options.common_smpl_pmds |= tmp_smpl_pmds;

	/*
	 * update smpl_pmds for all sampling periods
	 * we need to wait until we know know all the pmcs involved
	 */
	for(i=0; i < evt->inp.pfp_event_count; i++) {

		if ((options.long_rates[i].flags & PFMON_RATE_VAL_SET) == 0) continue;

		m  = options.smpl_pmds[i];
		m |=  options.common_smpl_pmds;

		pc[i].reg_smpl_pmds[0]  = m;

		options.rev_smpl_pmds[pc[i].reg_num] = m;

		options.smpl_pmds[i] = m;

		m = options.opt_reset_non_smpl ? options.common_smpl_pmds : 0UL;
		master_pc[i].reg_reset_pmds[0] |= options.common_reset_pmds | m;
	}
	
	if (options.opt_verbose) {
		DPRINT(("common_smpl_pmds=0x%lx common_reset_pmds=0x%lx\n", 
			options.common_smpl_pmds, options.common_reset_pmds));
		for(i=0; i < evt->inp.pfp_event_count; i++) {
			vbprintf("[pmc%u smpl_pmds=0x%lx reset_pmds=0x%lx]\n",
				master_pc[i].reg_num,
				master_pc[i].reg_smpl_pmds[0],
				master_pc[i].reg_reset_pmds[0]);
		}
	}
	return 0;
}

/*
 * simply install the values for both PMCs and PMDs into the context
 */
int
install_registers(pfmon_sdesc_t *sdesc, pfmon_lib_param_t *evt)
{
	/*
	 * now program the PMC registers
	 */
	if (perfmonctl(sdesc->ctxid, PFM_WRITE_PMCS, sdesc->pc, sdesc->pc_count) == -1) {
		warning("perfmonctl error WRITE_PMCS %s\n", strerror(errno));
		return -1;
	}
	/*
	 * and the PMD registers
	 */
	if (perfmonctl(sdesc->ctxid, PFM_WRITE_PMDS, sdesc->pd, sdesc->pd_count) == -1) {
		warning( "perfmonctl error WRITE_PMDS %s\n", strerror(errno));
		return -1;
	}
	
	/*
	 * Give the PMU specific code a chance to install specific registers
	 */
	if (pfmon_current->pfmon_install_registers) {
		if (pfmon_current->pfmon_install_registers(sdesc->ctxid, evt, sdesc->pc, sdesc->pd) == -1) {
			warning("model specific install_registers failed\n");
			return -1;
		}
	}
	return 0;
}

/*
 * Prepare measurement, setup PMC/PMD values, setup context and sampling module
 */
static int
do_measurements(pfmon_lib_param_t *evt, char **argv)
{
	pfmon_ctx_arg_t *ctx_arg;		/* area where master context + fmt arg are stored */
	unsigned int 	i;
	unsigned int	n, max_pmds_sample = 0; /* max number of PMD for a sample */
	int 		ret;

	if (options.opt_syst_wide) evt->inp.pfp_flags = PFMLIB_PFP_SYSTEMWIDE;

	/*
	 * assign events to counters, configure additional PMCs
	 * count may be greater than pfp_count when non trivial features are used
	 * We are guaranteed that the first n entries contains the n counters
	 * specified on the command line. The optional PMCs always come after.
	 */
	ret = pfm_dispatch_events(&evt->inp, evt->mod_inp, &evt->outp, evt->mod_outp);
	if (ret != PFMLIB_SUCCESS) {
		fatal_error("cannot configure events: %s\n", pfm_strerror(ret));
	}
	/*
	 * in case we just want to check for a valid event combination, we
	 * exit here
	 */
	if (options.opt_check_evt_only) exit(0);

	/*
	 * propagate libpfm setup to PMC table
	 */
	for(i=0; i < evt->outp.pfp_pmc_count; i++) {
		master_pc[i].reg_num   = evt->outp.pfp_pmcs[i].reg_num;
		master_pc[i].reg_value = evt->outp.pfp_pmcs[i].reg_value;
	}

	gen_reverse_table(evt, options.rev_pc);

	/*
	 * final model-specific chance to setup for PMCs and PMDs (no actual perfmonctl() call).
	 * we use the master copies.
	 */
	ret = 0;
	if (pfmon_current->pfmon_prepare_registers) 
		ret = pfmon_current->pfmon_prepare_registers(evt, master_pc, master_pd);
	if (ret) return ret;

	ret = prepare_pmc_registers(evt, master_pc, master_pd);
	if (ret) return ret;

	ret = prepare_pmd_registers(evt, master_pc, master_pd);
	if (ret) return ret;

	if (options.opt_use_smpl) {
		/*
		 * size is total size: generic context + module specific arguments
		 */
		if (options.smpl_mod->get_fmt_arg_size) {
			options.ctx_arg_size = (*options.smpl_mod->get_fmt_arg_size)();
		}

		/*
		 * find the monitor which has the largest number of PMDs to record
		 * for each overflow. This determines the maximum size of an entry.
		 */
		for(i=0; i < evt->inp.pfp_event_count; i++) {
			if ((options.long_rates[i].flags & PFMON_RATE_VAL_SET) == 0) continue;
			n = bit_weight(master_pc[i].reg_smpl_pmds[0]);
			if (n > max_pmds_sample) max_pmds_sample = n;
		}
		DPRINT(("ctx_arg_size=%lu max_pmds_samples=%u\n", options.ctx_arg_size, max_pmds_sample));
	}

	/*
	 * at a minimum, we need a generic context argument
	 */
	if (options.ctx_arg_size == 0) options.ctx_arg_size = sizeof(pfarg_context_t);

	ctx_arg = (pfmon_ctx_arg_t *)malloc(options.ctx_arg_size);
	if (ctx_arg == NULL) fatal_error("cannot allocate context argument\n");

	memset(ctx_arg, 0, options.ctx_arg_size);

	/*
	 * pointer to generic context argument
	 */
	master_ctx = (pfarg_context_t *) ctx_arg;

	/*
	 * initialize various context flags
	 */
	if (options.opt_syst_wide) {
		master_ctx->ctx_flags  = PFM_FL_SYSTEM_WIDE | PFM_FL_OVFL_NO_MSG; 
		// excl_idle will move to event set properties
		// ctx->ctx_flags |= options.opt_excl_idle ? PFM_FL_EXCL_IDLE: 0;
	} 
	master_ctx->ctx_flags |= options.opt_block    ? PFM_FL_NOTIFY_BLOCK : 0;

	/*
	 * extra work to do when sampling
	 */
	if (options.opt_use_smpl) {
		/*
	 	 * give the sampling module a chance to review PMC/PMD programming
	 	 */
		if (options.smpl_mod->validate_events) {
			ret = (*options.smpl_mod->validate_events)(evt, master_pc, master_pd);
			if (ret) return ret;
		}
		/*
	 	 * initialize module specific context arguments
	 	 * (must be done after generic). 
	 	 */
		if (options.smpl_mod->initialize_ctx_arg) {
			ret = (*options.smpl_mod->initialize_ctx_arg)(ctx_arg, max_pmds_sample);
			if (ret) return ret;
		}
	}

	DPRINT(("ctx_flags=0x%lx\n", master_ctx->ctx_flags)); 

	if (options.opt_syst_wide) {
		ret = measure_system_wide(evt, ctx_arg, master_pc, master_pd, argv);
	} else {
		ret = measure_task(evt, ctx_arg, master_pc, master_pd, argv);
	}
	return ret;
}

static void
show_detailed_event_name(unsigned int idx, const char *name)
{
	pfmlib_regmask_t used_counters, impl_pmds;
	unsigned int num_pmds;
	int i;
	int code;

	memset(&used_counters, 0, sizeof(used_counters));
	memset(&impl_pmds, 0, sizeof(impl_pmds));

	pfm_get_num_pmds(&num_pmds);

	pfm_get_event_code(idx, &code);
	pfm_get_event_counters(idx, &used_counters);

	printf("%s code=0x%02x counters=[ ", name, code);

	/*
	 * cannot used pfm_get_num_counters() because we are interested
	 * in the number of counters supported by this event
	 */
	for (i=0; i < num_pmds; i++) {
		if (PFMLIB_REGMASK_ISSET(&used_counters, i)) printf("%d ", i);
	}

	putchar(']');
	putchar(' ');

	if (pfmon_current->pfmon_detailed_event_name) {
		(*pfmon_current->pfmon_detailed_event_name)(i);
	}
	putchar('\n');
}

/*
 * mode=0 : just print event name
 * mode=1 : print name + other information (some of which maybe model specific)
 */
static void
pfmon_list_all_events(char *pattern, int mode)
{
	regex_t preg;
	unsigned int i, count;
	char name[PFMON_MAX_EVTNAME_LEN];

	pfm_get_num_events(&count);

	if (pattern) {
		int done = 0;

		if (regcomp(&preg, pattern, REG_ICASE|REG_NOSUB)) {
			fatal_error("error in regular expression for event \"%s\"\n", pattern);
		}

		for(i=0; i < count; i++) {
			pfm_get_event_name(i, name, PFMON_MAX_EVTNAME_LEN);

			if (regexec(&preg, name, 0, NULL, 0) == 0) {
				if (mode == 0) 
					printf("%s\n", name);
				else
					show_detailed_event_name(i, name);
				done = 1;
			}
		}
		if (done == 0) fatal_error("event not supported\n");
	} else {
		for(i=0; i < count; i++) {
			pfm_get_event_name(i, name, PFMON_MAX_EVTNAME_LEN);
			if (mode == 0) 
				printf("%s\n", name);
			else
				show_detailed_event_name(i, name);

		}
	}
}

/*
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option pfmon_common_options[]={
	{ "event-info", 1, 0, 1},
	{ "show-events", 2, 0, 2 },
	{ "kernel-level", 0, 0, 3 },
	{ "user-level", 0, 0, 4 },
	{ "events", 1, 0, 5 },
	{ "help", 0, 0, 6 },
	{ "version", 0, 0, 7 },
	{ "outfile", 1, 0, 8 },
	{ "long-show-events", 2, 0, 9 },
	{ "info", 0, 0, 10},
	/* 11 slot if free (used to be smpl-entries) */
	{ "smpl-outfile", 1, 0, 12},
	{ "long-smpl-periods", 1, 0, 13},
	{ "short-smpl-periods", 1, 0, 14},
	{ "cpu-mask", 1, 0, 15}, /* obsolete */
	{ "session-timeout", 1, 0, 16},
	{ "trigger-address", 1, 0, 17}, /* obsolete */
	{ "priv-levels", 1, 0, 18},
	{ "symbol-file", 1, 0, 19},
	{ "smpl-module", 1, 0, 20},
	{ "smpl-module-info", 1, 0, 21},
	{ "sysmap-file", 1, 0, 22},
	{ "smpl-periods-random", 1, 0, 23},
	{ "trigger-code-start-addresses", 1, 0, 24},
	{ "trigger-start-delay", 1, 0, 25},
	{ "attach-task", 1, 0, 26},
	{ "follow-exec", 2, 0, 27 },
	{ "follow-exec-exclude", 2, 0, 28 },
	{ "trigger-code-stop-addresses", 1, 0, 29},
	{ "cpu-list", 1, 0, 30},
	{ "trigger-data-start-addresses", 1, 0, 31},
	{ "trigger-data-stop-addresses", 1, 0, 32},

	{ "verbose", 0, &options.opt_verbose, 1 },
	{ "append", 0, &options.opt_append, 1},
	{ "overflow-block",0, &options.opt_block, 1},
	{ "system-wide", 0, &options.opt_syst_wide, 1},
	{ "debug", 0, &options.opt_debug, 1 },
	{ "aggregate-results", 0, &options.opt_aggr, 1 },

	{ "with-header", 0, &options.opt_with_header, 1},
	{ "us-counter-format",0, &options.opt_print_cnt_mode, 1},
	{ "eu-counter-format",0, &options.opt_print_cnt_mode, 2},
	{ "hex-counter-format",0, &options.opt_print_cnt_mode, 3},
	{ "show-time",0, &options.opt_show_rusage, 1},
	{ "check-events-only",0, &options.opt_check_evt_only, 1},
	{ "smpl-print-counts",0, &options.opt_smpl_print_counts, 1},
	{ "reset-non-smpl-periods", 0, &options.opt_reset_non_smpl, 1 },
	{ "follow-fork", 0, &options.opt_follow_fork, 1 },
	{ "follow-vfork", 0, &options.opt_follow_vfork, 1 },
	{ "follow-pthread", 0, &options.opt_follow_pthread, 1 },
	{ "follow-all", 0, &options.opt_follow_all, 1 },
	{ "no-cmd-output", 0, &options.opt_cmd_no_verbose, 1 },
	{ "trigger-code-repeat", 0, &options.opt_code_trigger_repeat, 1},
	{ "trigger-code-follow", 0, &options.opt_code_trigger_follow, 1},
	{ "trigger-data-repeat", 0, &options.opt_data_trigger_repeat, 1},
	{ "trigger-data-follow", 0, &options.opt_data_trigger_follow, 1},
	{ "trigger-data-ro", 0, &options.opt_data_trigger_ro, 1},
	{ "trigger-data-wo", 0, &options.opt_data_trigger_wo, 1},
	{ "restart-wait", 0, &options.opt_block_restart, 1}, /* for debug only */
	{ "exec-split-results", 0, &options.opt_split_exec, 1},
	{ "resolve-addresses", 0, &options.opt_addr2sym, 1},
	{ "saturate-smpl-buffer", 0, &options.opt_no_ovfl_notify, 1},
	{ 0, 0, 0, 0}
};

static struct option *pfmon_cmd_options = pfmon_common_options;
static size_t pfmon_option_base_size	= sizeof(pfmon_common_options);

static void
usage(char **argv)
{
	printf("usage: %s [OPTIONS]... COMMAND\n", argv[0]);

	printf(	"-h, --help\t\t\t\tdisplay this help and exit\n"
		"-V, --version\t\t\t\toutput version information and exit\n"
		"-l[regex], --show-events[=regex]\tdisplay all or a matching subset of the events\n"
		"--long-show-events[=regex]\t\tdisplay all or a matching subset of the events with info\n"
		"-i event, --event-info=event\t\tdisplay information about an event (numeric code or regex)\n"
		"-u, -3 --user-level\t\t\tmonitor at the user level for all events (default: on)\n"
		"-k, -0 --kernel-level\t\t\tmonitor at the kernel level for all events (default: off)\n"
		"-1\t\t\t\t\tmonitor execution at privilege level 1 (default: off)\n"
		"-2\t\t\t\t\tmonitor execution at privilege level 2 (default: off)\n"
		"-e, --events=ev1,ev2,...\t\tselect events to monitor (no space)\n"
		"-I,--info\t\t\t\tlist supported PMU models and compiled in sampling output formats\n"
		"-t secs, --session-timeout=secs\t\tduration of the system wide session in seconds\n"
		"-S format, --smpl-module-info=format\tdisplay information about a sampling output format\n"
		"--debug\t\t\t\t\tenable debug prints\n"
		"--verbose\t\t\t\tprint more information during execution\n"
		"--outfile=filename\t\t\tprint results in a file\n"
		"--append\t\t\t\tappend results to outfile\n"
		"--overflow-block\t\t\tblock the task when sampling buffer is full (default: off)\n"
		"--system-wide\t\t\t\tcreate a system wide monitoring session (default: per-task)\n"
		"--smpl-outfile=filename\t\t\tfile to save the sampling results\n"
		"--long-smpl-periods=val1,val2,...\tset sampling period after user notification\n"
		"--short-smpl-periods=val1,val2,...\tset sampling period\n"
		"--with-header\t\t\t\tgenerate a header for results\n"
		"--cpu-mask=0xn\t\t\t\tobsolete use --cpu-list instead\n"
		"--cpu-list=num,num1-num2,...\t\tspecify list, via numbers, of CPUs for system-wide session (default: all)\n"
		"--aggregate-results\t\t\taggregate counts and sampling buffer outputs for multi CPU monitoring\n"
		"--trigger-code-start-address=addr\tstart monitoring only when code address is executed\n"
		"--trigger-code-stop-address=addr\tstop monitoring when code address is executed\n"
		"--trigger-data-start-address=addr\tstart monitoring only when data address is accessed\n"
		"--trigger-data-stop-address=addr\tstop monitoring when data address code is accessed\n"
		"--trigger-code-repeat\t\t\tstart/stop monitoring each time trigger start/stop are executed\n"
		"--trigger-code-follow\t\t\tstart/stop code trigger applied to all monitored task (default first only)\n"
		"--trigger-data-repeat\t\t\tstart/stop monitoring each time trigger start/stop are accessed\n"
		"--trigger-data-follow\t\t\tstart/stop data trigger applied to all monitored task (default first only)\n"
		"--trigger-data-ro\t\t\tdata trigger activated on read access (default read-write)\n"
		"--trigger-data-wo\t\t\tdata trigger activated on write access (default read-write)\n"
		"--trigger-start-delay=secs\t\tnumber of seconds before activating monitoring\n"
		"--priv-levels=lvl1,lvl2,...\t\tset privilege level per event (any combination of [0123uk]))\n"
		"--us-counter-format\t\t\tprint counters using commas (e.g.: 1,024)\n"
		"--eu-counter-format\t\t\tprint counters using points (e.g.: 1.024)\n"
		"--hex-counter-format\t\t\tprint counters in hexadecimal (e.g.: 0x400)\n"
		"--smpl-module=name\t\t\tselect sampling module, use -I to list modules\n"
		"--show-time\t\t\t\tshow real,user, and system time for the command executed\n"
		"--symbol-file=filename\t\t\tELF image containing a symbol table\n"
		"--sysmap-file=filename\t\t\tSystem.map-format file containing a symbol table\n"
		"--check-events-only\t\t\tverify combination of events and exit (no measurement)\n"
		"--smpl-periods-random=mask1:seed1,...\tapply randomization to long and short periods\n"
		"--smpl-print-counts\t\t\tprint counters values when sampling session ends (default: no)\n"
		"--attach-task pid\t\t\tmonitor process identified by pid\n"
		"--reset-non-smpl-periods\t\ton overflow reset counters not used as sampling periods\n"
		"--follow-fork\t\t\t\tmonitoring continues across fork\n"
		"--follow-vfork\t\t\t\tmonitoring continues across vfork\n"
		"--follow-pthread\t\t\tmonitoring continues across pthread_create\n"
		"--follow-exec[=pattern]\t\t\tfollow exec with optional command pattern\n"
		"--follow-exec-exclude=pattern\t\tfollow exec but exclude commands matching the pattern\n"
		"--follow-all\t\t\t\tfollow fork, vfork, exec, pthreads\n"
		"--no-cmd-output\t\t\t\tredirect all output of executed commands to /dev/null\n"
		"--exec-split-results\t\t\tgenerate separate results output on execve()\n"
		"--resolve-addresses\t\t\ttry to resolve code/data addresses to symbols\n"
		"--saturate-smpl-buffer\t\t\tonly collect samples until buffer becomes full\n"
	);
	if (pfmon_current->pfmon_usage) pfmon_current->pfmon_usage();

	pfmon_smpl_mod_usage();
}

int
pfmon_register_pmu_options(struct option *cmd, size_t new_sz)
{
	size_t prev_sz;
	char *o;

	if (cmd == NULL || new_sz == 0) return 0;

	/* don't account for last (null) element */
	prev_sz = pfmon_option_base_size - sizeof(struct option);

	o = (char *)malloc(prev_sz + new_sz);
	if (o == NULL) return -1;

	/* do not copy null element at the end */
	memcpy(o, pfmon_cmd_options, prev_sz);

	/* copy starting at the previous last null element slot */
	memcpy(o+prev_sz, cmd, new_sz);

	/* 
	 * every new set of commands MUST be NULL terminated
	 */

	pfmon_cmd_options  = (struct option *)o;
	/*
	 * size of generic+pmu-specific options
	 * this is the base size for smpl module options
	 */
	pfmon_option_base_size = prev_sz + new_sz;

	//printf("added %ld pmu-specific options\n", (new_sz/sizeof(struct option))-1);

	return 0;
}

int
pfmon_register_smpl_mod_options(struct option *cmd, size_t new_sz)
{
	size_t prev_sz;
	char *o;

	if (cmd == NULL || new_sz == 0) return 0;

	/* don't account for last (null) element */
	prev_sz = pfmon_option_base_size - sizeof(struct option);

	o = (char *)malloc(prev_sz + new_sz);
	if (o == NULL) return -1;

	/* do not copy null element at the end */
	memcpy(o, pfmon_cmd_options, prev_sz);

	/* copy starting at the previous last null element slot */
	memcpy(o+prev_sz, cmd, new_sz);

	/* 
	 * every new set of commands MUST be NULL terminated
	 */

	pfmon_cmd_options      = (struct option *)o;

	//printf("added %ld sampling module options\n", (new_sz/sizeof(struct option))-1);
	//{struct option *p = o; while(p->name) { printf("option: %s\n", p->name); p++; } }

	return 0;
}

static void
pfmon_detect(void)
{
	pfmon_support_t **p = pfmon_cpus;
	int type;

	pfm_get_pmu_type(&type);

	while (*p) {
		if ((*p)->pmu_type == type) break;
		p++;
	}

	if (*p == NULL) fatal_error("no detected PMU support\n");

	pfmon_current = *p;

	vbprintf("pfmon will use %s PMU support\n", (*p)->name);
}

/*
 * We use the command name as the hint for forced generic
 * mode. We cannot use an option because, the command line 
 * options depends on the detected support.
 */
static int
check_forced_generic(char *cmd)
{
	char *p;

	p = strrchr(cmd, '/');
	if (p) cmd = p + 1;

	return strcmp(cmd, PFMON_FORCED_GEN) ? 0 : 1;
}

static void
pfmon_get_version(void)
{
	pfarg_features_t ft;

	if (perfmonctl(0, PFM_GET_FEATURES, &ft, 1) == -1) {
		if (errno == ENOSYS) {
			fatal_error("host kernel does not have perfmon support\n");
		}
		fatal_error("you need at least kernel 2.6.0-test5 to run this version of pfmon\n");
	}

	options.pfm_version      = ft.ft_version;
}

/*
 * sym_type: PFMON_TEXT_SYMBOL, PFMON_DATA_SYMBOL
 */
static void 
parse_trigger_list(char *list, int sym_type, int is_start, pfmon_trigger_t *trg, unsigned int max, unsigned int *used)
{
	uintptr_t addr;
	char *endptr, *p;
	unsigned int count = 0;
	int ret;

	while (list && count < max) {
		p = strchr(list, ',');
		if (p) *p = '\0';

		if (isdigit(*list)) {
			endptr = NULL;
			addr   = (uintptr_t)strtoul(list, &endptr, 0);
			if (*endptr != '\0') goto error_address;
		} else {
			ret = find_sym_addr(list, sym_type, &addr, NULL);
			if (ret) goto error_symbol;
		}
		if (p) *p++ = ',';

		trg->brk_address    = addr;
		trg->trg_attr_start = is_start;

		trg++;

		count++;

		list = p;
	}

	if (list && count == max) goto error_many;

	*used = count;

	return;

error_symbol:
	fatal_error("cannot find address of symbol %s\n", list);
	/* no return */
error_address:
	fatal_error("invalid address %s\n", list);
	/* no return */
error_many:
	fatal_error("too many triggers defined, cannot use %s\n", list);
	/* no return */
}

static void
setup_trigger_addresses(void)
{
	pfmon_trigger_t *trg1, *trg2;
	uintptr_t addr;
	unsigned int max;
	unsigned int count;
	unsigned int i, j;
	unsigned int is_func;
	int rw;

	max   = PFMON_MAX_TRIGGER_CODE;
	trg1  = options.code_triggers;
	count = 0;

	if (options.code_trigger_start) {
		parse_trigger_list(options.code_trigger_start, PFMON_TEXT_SYMBOL, 1, trg1, max, &count);
		/*
		 * we have some triggered start, therefore we do not start right away
		 */
		options.opt_dont_start = 1;
	}

	trg1 += count;
	max  -= count;
	count = 0;

	if (options.code_trigger_stop) {
		parse_trigger_list(options.code_trigger_stop, PFMON_TEXT_SYMBOL, 0, trg1, max, &count);
	}
	trg1 += count;
	max  -= count;

	options.num_code_triggers = PFMON_MAX_TRIGGER_CODE - max;

	if (options.num_code_triggers > (options.nibrs>>1))
		fatal_error("not enough code debug registers to fit all code triggers, max=%u\n", options.nibrs>>1);

	max   = PFMON_MAX_TRIGGER_DATA;
	trg1  = options.data_triggers;
	count = 0;

	if (options.data_trigger_start) {
		parse_trigger_list(options.data_trigger_start, PFMON_DATA_SYMBOL, 1, trg1, max, &count);
		/*
		 * we have some triggered start, therefore we do not start right away
		 */
		options.opt_dont_start = 1;
	}

	trg1 += count;
	max  -= count;
	count = 0;

	if (options.data_trigger_stop) {
		parse_trigger_list(options.data_trigger_stop, PFMON_DATA_SYMBOL, 0, trg1, max, &count);
	}
	trg1 += count;
	max  -= count;

	options.num_data_triggers = PFMON_MAX_TRIGGER_DATA - max;

	if (options.num_data_triggers > (options.ndbrs>>1))
		fatal_error("not enough data debug registers to fit all code triggers, max=%u\n", options.ndbrs>>1);

	/*
	 * sanity checks on code triggers
	 */
	for (i=0; i < options.num_code_triggers; i++) {

		trg1     = options.code_triggers+i;
		addr     = trg1->brk_address;
		is_func  = trg1->trg_attr_start && is_exact_sym(addr, PFMON_TEXT_SYMBOL);

		if (pfmon_validate_code_trigger_address(addr)) fatal_error("");

		for (j=0; j < options.num_code_triggers; j++) {
			trg2 = options.code_triggers+j;
			if (j != i && trg2->brk_address == addr) {
				if (is_func && trg2->trg_attr_start == 0) {
					trg1->trg_attr_func = 1;
					trg1->stop_trg_idx  = j;
					trg2->trg_attr_func = 1;
				} else if (trg2->trg_attr_func == 0) {
					fatal_error("cannot  have twice the same code trigger %p\n", addr);
				}
			}
		}
	}

	for (i=0; i < options.num_code_triggers; i++) {
		options.code_triggers[i].trg_attr_repeat  = options.opt_code_trigger_repeat;
		options.code_triggers[i].trg_attr_inherit = options.opt_code_trigger_follow;

		vbprintf("%-5s code trigger @%p\n", 
			options.code_triggers[i].trg_attr_start ? "start" : "stop",
			(void *)options.code_triggers[i].brk_address);
	}

	/* default RW */
	rw = 0x3;
	if (options.opt_data_trigger_ro) rw = 0x2;
	if (options.opt_data_trigger_wo) rw = 0x1;

	/*
	 * sanity checks on data triggers
	 */
	for (i=0; i < options.num_data_triggers; i++) {
		addr = options.data_triggers[i].brk_address;

		if (pfmon_validate_data_trigger_address(addr)) fatal_error("");

		for (j=0; j < options.num_data_triggers; j++) {
			if (j != i && options.data_triggers[j].brk_address == addr)
				fatal_error("cannot  have twice the same data trigger %p\n", addr);
		}
	}

	for (i=0; i < options.num_data_triggers; i++) {
		options.data_triggers[i].trg_attr_repeat  = options.opt_data_trigger_repeat;
		options.data_triggers[i].trg_attr_inherit = options.opt_data_trigger_follow;
		options.data_triggers[i].trg_attr_rw      = rw;

		vbprintf("%-5s data %s trigger @%p\n", 
			options.data_triggers[i].trg_attr_start ? "start" : "stop",
			rw == 0x3 ? "read-write" : (rw == 0x1 ? "write-only" : "read-write"),
			(void *)options.data_triggers[i].brk_address);
	}
}

static int
pfmon_print_event_info(char *event)
{
	regex_t preg;
	int done = PFMLIB_ERR_NOTFOUND;
	unsigned int i, count;
	char name[PFMON_MAX_EVTNAME_LEN];

	if (isdigit(*event)) {
		done = pfm_print_event_info(event, printf);
		goto skip_regex;
	}

	if (regcomp(&preg, event, REG_ICASE|REG_NOSUB)) {
		fatal_error("error in regular expression for event \"%s\"\n", event);
	}

	pfm_get_num_events(&count);
	for(i=0; i < count; i++) {
		pfm_get_event_name(i, name, PFMON_MAX_EVTNAME_LEN);
		if (regexec(&preg, name, 0, NULL, 0) == 0) {
			pfm_print_event_info_byindex(i, printf);
			done = PFMLIB_SUCCESS;
		}
	}
skip_regex:
	if (done != PFMLIB_SUCCESS)
		fatal_error("event \"%s\" not found\n", event);

	return 0;
}

static void
pfmon_show_info(void)
{
	unsigned int version;

	pfmon_print_simple_cpuinfo(stdout, "detected host CPUs: ");
	pfm_list_supported_pmus(printf);
	pfmon_list_smpl_modules();
	pfm_get_version(&version);
	printf("pfmlib version: %u.%u\n", PFMLIB_MAJ_VERSION(version), PFMLIB_MIN_VERSION(version));
	printf("kernel perfmon version: %u.%u\n", PFM_VERSION_MAJOR(options.pfm_version), PFM_VERSION_MINOR(options.pfm_version));
}

static void
pfmon_propagate2libary(pfmon_lib_param_t *evt, unsigned int count, pfmon_monitor_t *events)
{
	unsigned int i;

	evt->inp.pfp_event_count = count;
	for(i=0; i < count; i++) {
		evt->inp.pfp_events[i].event = events[i].event;
		evt->inp.pfp_events[i].plm   = events[i].plm;
	}
}

static void
segv_handler(int n, struct siginfo *info, struct sigcontext *sc)
{
	pfmon_segv_handler_info(info, sc);
	pfmon_backtrace();
	fatal_error("pfmon got a fatal SIGSEGV signal\n");
}

static void
setup_common_signals(void)
{
	struct sigaction act;

	memset(&act,0,sizeof(act));
	act.sa_handler = (sig_t)segv_handler;
	sigaction (SIGSEGV, &act, 0);
}

static void
pfmon_initialize(char **argv, pfmon_lib_param_t *evt)
{
	long long_val;

	pfmon_arch_initialize();

	setup_common_signals();

	if (pfm_initialize() != PFMLIB_SUCCESS) fatal_error("cannot initialize library\n");

	if (check_forced_generic(argv[0])) {

		if (options.opt_support_gen == 0) 
			fatal_error("pfmon cannot be forced to generic mode\n");

		if (pfm_force_pmu(options.libpfm_generic) != PFMLIB_SUCCESS)
			fatal_error("failed to force  generic mode (support may not be available).\n");
	}

	pfmon_detect();

	long_val = sysconf(_SC_NPROCESSORS_ONLN);
	if (long_val == -1) 
		fatal_error("cannot figure out the number of online processors\n");

	options.online_cpus = long_val;

	long_val = sysconf(_SC_CLK_TCK);
	if (long_val == -1) 
		fatal_error("cannot figure out the clock tick\n");

	options.clock_tick  = long_val;

	options.page_size   = getpagesize();

	if (pfmon_current->pfmon_initialize) 
		pfmon_current->pfmon_initialize(evt);

	load_config_file();

	pfmon_smpl_initialize();
}

static void
setup_plm(pfmon_lib_param_t *evt)
{
	char *arg;
	unsigned int cnt=0;
	int val;

	/*
	 * if not specified, force defaut priv level to PLM3 (user)
	 */
	if (options.opt_plm == 0) {
		options.opt_plm = pfmon_current->default_plm;
		vbprintf("measuring at %s privilege level ONLY\n", priv_level_str(options.opt_plm));
	}

	/* set default privilege level: used when not explicitly specified for an event */
	evt->inp.pfp_dfl_plm = options.opt_plm;

	/*
	 * set default priv level for all events
	 */
	for(cnt=0; cnt < evt->inp.pfp_event_count; cnt++) {
		evt->inp.pfp_events[cnt].plm = options.opt_plm;
		options.events[cnt].plm  = options.opt_plm;
	}
	if (options.priv_lvl_str == NULL) return;
	/*
	 * adjust using per-event settings
	 */
	for (cnt=0, arg = options.priv_lvl_str ; *arg; ) {
		if (cnt == evt->inp.pfp_event_count) goto too_many;
		val = 0;
		while (*arg && *arg != ',') {
			switch (*arg) {
				case 'k':
				case '0': val |= 1; break;
				case '1': val |= 2; break;
				case '2': val |= 4; break;
				case '3': 
				case 'u': val |= 8; break;
				default: goto error;
			}
			arg++;
		}
		if (*arg) arg++;

		if (val) {
			evt->inp.pfp_events[cnt].plm = val;
			options.events[cnt].plm  = val;
		}
		cnt++;
	}
	return;
error:
	fatal_error("unknown per-event privilege level %c ([ku0123])\n", *arg);
	/* no return */
too_many:
	fatal_error("too many per-event privilege levels specified, max=%d\n", evt->inp.pfp_event_count);
	/* no return */
}

static int
pfmon_check_extra_options(int c, char *optarg, pfmon_lib_param_t *evt)
{
	if (pfmon_current->pfmon_parse_options && pfmon_current->pfmon_parse_options(c, optarg, evt) == 0) {
		return 0;
	}

	if (options.smpl_mod && options.smpl_mod->parse_options) {
		return options.smpl_mod->parse_options(c, optarg, evt);
	}
	return -1;
}

static void
populate_cpumask(char *cpu_list)
{
	char *p;
	unsigned long start_cpu, end_cpu = 0;
	unsigned long i, count = 0;

	if (cpu_list == NULL)  {
		/*
		 * The limit is mostly driven by the affinity support in NPTL and glibc __CPU_SETSIZE.
		 * the kernel interface does not expose any limitation.
	         */
		if (options.online_cpus >= PFMON_MAX_CPUS)
			fatal_error("pfmon can only handle to %u CPUs\n", PFMON_MAX_CPUS);

		for(i=0; i < options.online_cpus; i++) {
			PFMON_CPUMASK_SET(options.cpu_mask, i);
		}
		options.selected_cpus = options.online_cpus;

		return;
	} 

	while(isdigit(*cpu_list)) { 
		p = NULL;
		start_cpu = strtoul(cpu_list, &p, 0); /* auto-detect base */

		if (start_cpu == ULONG_MAX || (*p != '\0' && *p != ',' && *p != '-')) goto invalid;

		if (*p == '-') {
			cpu_list = ++p;
			p = NULL;

			end_cpu = strtoul(cpu_list, &p, 0); /* auto-detect base */
			
			if (end_cpu == ULONG_MAX || (*p != '\0' && *p != ',')) goto invalid;
			if (end_cpu < start_cpu) goto invalid_range; 
		} else {
			end_cpu = start_cpu;
		}

		if (start_cpu >= PFMON_MAX_CPUS || end_cpu >= PFMON_MAX_CPUS) goto too_big;

		for (; start_cpu <= end_cpu; start_cpu++) {

			if (start_cpu >= options.online_cpus) goto not_online; /* XXX: assume contiguous range of CPUs */

			if (PFMON_CPUMASK_ISSET(options.cpu_mask, start_cpu)) continue;

			PFMON_CPUMASK_SET(options.cpu_mask, start_cpu);

			count++;
		}

		if (*p) ++p;

		cpu_list = p;
	}

	options.selected_cpus = count;

	if (options.opt_verbose) {
		vbprintf("selected CPUs (%lu:%lu): ", count, options.online_cpus);
		for(i=0; count;i++) {
			if (PFMON_CPUMASK_ISSET(options.cpu_mask, i) == 0) continue;
			vbprintf("CPU%lu ", i);
			count--;
		}
		vbprintf("\n");
	}
	return;
invalid:
	fatal_error("invalid cpu list argument: %s\n", cpu_list);
	/* no return */
not_online:
	fatal_error("cpu %lu is not online\n", start_cpu);
	/* no return */
invalid_range:
	fatal_error("cpu range %lu - %lu is invalid\n", start_cpu, end_cpu);
	/* no return */
too_big:
	fatal_error("pfmon is limited to %u CPUs\n", PFMON_MAX_CPUS);
	/* no return */
}

static void
pfmon_verify_cmdline_options(int argc, char **argv)
{
	int has_code_data_triggers = 0;

	if (optind == argc && options.opt_syst_wide == 0 && options.opt_check_evt_only == 0 && options.opt_attach == 0)
		fatal_error("you need to specify a command to measure\n");

	if (options.opt_attach && optind != argc) 
		fatal_error("you cannot attach to a task AND launch a program to monitor at the same time\n");

	/*
 	 * propagate in case all needs to be activated
 	 */
	if (options.opt_follow_all) {
		options.opt_follow_exec  = 1;
		options.opt_follow_vfork =
		options.opt_follow_fork  =
		options.opt_follow_pthread = 1;
		options.opt_follows = 1;
	} else if (options.opt_follow_fork
		  || options.opt_follow_vfork
		  || options.opt_follow_pthread
		  || options.opt_follow_exec) {
		options.opt_follows = 1;
	}

	if (options.code_trigger_start
	  || options.data_trigger_start
	  || options.code_trigger_stop
	  || options.data_trigger_stop)
		has_code_data_triggers = 1;

	if (options.opt_syst_wide) {

		populate_cpumask(options.cpu_list);

		if (optind != argc && options.session_timeout > 0)
			fatal_error("you cannot use both a timeout and command in system-wide mode\n");

		if (options.opt_block == 1) 
			fatal_error("cannot use blocking mode in system wide monitoring\n");

		if (options.code_trigger_start)
			fatal_error("cannot use a code trigger start address in system wide mode\n");

		if (options.data_trigger_start)
			fatal_error("cannot use a data trigger start address in system wide mode\n");

		if (options.code_trigger_stop)
			fatal_error("cannot use a code trigger stop address in system wide mode\n");

		if (options.data_trigger_stop)
			fatal_error("cannot use a data trigger stop address in system wide mode\n");

		if (options.opt_follow_exec || options.opt_follow_fork || options.opt_follow_vfork || options.opt_follow_pthread)
			warning("no --follow-* option has any effect in system-wide mode\n");

		if (options.opt_split_exec)
			warning("--exec-split has not effect in system-wide mode\n");

	} else {
		/* wait4 use RUSAGE_BOTH */
		if (options.opt_show_rusage && (options.opt_follow_fork || options.opt_follow_vfork || options.opt_follow_pthread))
			fatal_error("show-time cannot be used when following execution across fork/vfork/clone\n");

		if (options.opt_data_trigger_ro && options.opt_data_trigger_wo)
			fatal_error("cannot use --data-trigger-ro and --data-trigger-wo at the same time\n");

		//if (options.opt_split_exec && options.opt_follow_all == 0 && options.opt_follow_exec == 0)
		//	fatal_error("the --exec-split option can only be used in conjunction with --follow-all or --follow-exec\n");

		if (options.trigger_delay)
			fatal_error("cannot use --trigger-start-delay in per-task mode\n");

		if (options.opt_follow_exec && options.opt_split_exec && options.opt_addr2sym) 
			warning("only resolving symbols from first program (binary)\n");
		
		if (options.opt_split_exec && options.opt_follow_exec == 0) {
			warning("--exec-split is ignored, you need to use --follow-exec to activate\n");
			options.opt_split_exec = 0;
		}

		if (options.opt_split_exec && options.opt_follow_exec && options.opt_aggr) 
			fatal_error("--exec-split cannot be used with --aggregate-results\n");

		if (options.opt_follow_exec && has_code_data_triggers && (options.opt_code_trigger_follow ||options.opt_data_trigger_follow) )
			fatal_error("cannot use code/data trigger follow with --follow-exec option\n");

	}

	if (options.opt_attach) {
		if (options.opt_syst_wide)
			fatal_error("cannot attach to a process in system-wide mode\n");
		if (optind != argc) 
			warning("command is ignored when attaching to a task\n");
		if (options.opt_show_rusage)
			fatal_error("--show-time does not work when attaching to a task\n");
	}

	/*
	 * try to use the command to get the symbols
	 * XXX: require absolute path
	 */
	if (options.symbol_file == NULL) options.symbol_file = argv[optind];

	/*
	 * make sure we measure at least one event
	 */
	if (options.monitor_count == 0) {
		char *dfl_event;

		dfl_event = pfmon_current->default_event;
		if (dfl_event == NULL) 
			fatal_error("you need to specify at least one event name via -e option\n");

		if (pfm_find_event_byname(dfl_event, &options.events[0].event) != PFMLIB_SUCCESS)
			fatal_error("default event %s does not exist\n", dfl_event); 

		options.monitor_count      = 1;
		options.max_event_name_len = strlen(dfl_event);

		vbprintf("defaulting to event: %s\n", dfl_event);
	}
}

int
main(int argc, char **argv)
{
	pfmon_lib_param_t evt;	/* hold most configuration data */
	pfmlib_options_t pfmlib_options;
	char *endptr = NULL;
	char *long_smpl_args = NULL, *short_smpl_args = NULL, *smpl_random_args = NULL;
	pfmon_smpl_module_t *smpl_mod = NULL;
	unsigned long long_val;
	int c, r, ret;

	memset(&evt, 0, sizeof(evt));
	memset(&pfmlib_options, 0, sizeof(pfmlib_options));

	pfmon_initialize(argv, &evt);

	pfmon_get_version();

	while ((c=getopt_long(argc, argv,"+0123kuvhe:Il::i:Vt:S:p:", pfmon_cmd_options, 0)) != -1) {
		switch(c) {
			case   0: continue; /* fast path for options */

			case 'v': options.opt_verbose = 1;
				  break;
			case   1:
			case 'i':
				exit(pfmon_print_event_info(optarg));
			case   2:
			case 'l':
				pfmon_list_all_events(optarg, 0);
				exit(0);
			case '1':
				options.opt_plm |= PFM_PLM1;
				break;
			case '2':
				options.opt_plm |= PFM_PLM2;
				break;
			case '3':
 			case   4:
 			case 'u':
 				options.opt_plm |= PFM_PLM3;
				break;
			case '0':
			case   3:
			case 'k':
				options.opt_plm |= PFM_PLM0;
				break;
			case   5:
			case 'e':
				if (evt.inp.pfp_event_count) fatal_error("events already defined\n");
				options.monitor_count = gen_event_list(optarg, options.events);
				break;
			case   6:
			case 'h':
				usage(argv);
				exit(0);
			case 'V':
			case   7:
				printf("pfmon version " PFMON_VERSION " Date: " __DATE__ "\n"
					"Copyright (C) 2001-2003 Hewlett-Packard Company\n");
				exit(0);
			case   8:
				options.outfile = optarg;
				break;
			case   9:
				pfmon_list_all_events(optarg, 1);
				exit(0);
			case  10:
			case 'I':
				pfmon_show_info();
				exit(0);
			case  12:
				if (options.smpl_outfile) fatal_error("sampling output file specificed twice\n");
				options.smpl_outfile = optarg;
				break;
			case  13:
				/* need to wait until we have the events */
				if (long_smpl_args) fatal_error("long sampling rates specificed twice\n");
				long_smpl_args = optarg;
				break;
			case  14:
				/* need to wait until we have the events */
				if (short_smpl_args) fatal_error("short sampling rates specificed twice\n");
				short_smpl_args = optarg;
				break;
			case 15:
				fatal_error("--cpu-mask obsolete option, use --cpu-list instead\n");
				break;
			case 't':
			case 16 :
				if (options.session_timeout) fatal_error("too many timeouts\n");
				if (*optarg == '\0') fatal_error("--session-timeout needs an argument\n");
			  	long_val = strtoul(optarg,&endptr, 10);
				if (*endptr != '\0') 
					fatal_error("invalid number of seconds for timeout: %s\n", optarg);

				if (long_val >= UINT_MAX) 
					fatal_error("timeout is too big, must be < %u\n", UINT_MAX);

				options.session_timeout = (unsigned int)long_val;

				break;
			case 17 :
				fatal_error("--trigger-address is obsolete, use --trigger-code-start-address instead\n");
				break;
			case 18 :
				if (options.priv_lvl_str) fatal_error("per event privilege levels already defined");
				options.priv_lvl_str = optarg;
				break;
			case 19 :
				if (options.symbol_file) {
					if (options.opt_sysmap_syms)
						fatal_error("Cannot use --sysmap-file and --symbol-file at the same time\n");
					fatal_error("symbol file already defined\n");
				}
				if (*optarg == '\0') fatal_error("you must provide a filename for --symbol-file\n");

				options.symbol_file = optarg;
				break;
			case 20:
				if (*optarg == '\0') fatal_error("--smpl-module needs an argument\n");
				/*
				 * check if the user already specified a format, but we can override default
				 */
				if (options.smpl_mod && smpl_mod)
					fatal_error("sampling output format already defined\n");

				r = pfmon_find_smpl_module(optarg, &smpl_mod, 0);
				if (r == -1)
					fatal_error("invalid sampling output format %s\n", optarg);

				/* 
				 * initialize module right away to register options, among other things
				 */
				if (smpl_mod->initialize_module && (*smpl_mod->initialize_module)() != 0) {
					fatal_error("failed to intialize sampling module%s\n", smpl_mod->name);
				}
				options.smpl_mod = smpl_mod;
				break;
			case 'S':
			case 21 : 
				  if (*optarg == '\0') fatal_error("--smpl-module-info needs an argument\n");
				  r = pfmon_find_smpl_module(optarg, &smpl_mod, 1);
				  if (r != PFMLIB_SUCCESS)
					fatal_error("invalid sampling output format %s: %s\n", optarg, pfm_strerror(r));
				  pfmon_smpl_module_info(smpl_mod);
				  exit(0);
			case 22 : 
				/* 
				 * Despite /proc/kallsyms, System.map is still useful because it includes data symbols
				 */
				if (options.symbol_file) {
					if (options.opt_sysmap_syms == 0)
						fatal_error("Cannot use --sysmap-file and --symbol-file at the same time\n");
					fatal_error("sysmap file already defined\n");
				}
				if (*optarg == '\0') fatal_error("you must provide a filename for --sysmap-file\n");
				options.opt_sysmap_syms = 1;
				options.symbol_file     = optarg;
				break;
			case 23 :
				/* need to wait until we have the events */
				if (smpl_random_args) fatal_error("randomization parameters specified twice\n");
				smpl_random_args = optarg;
				break;
			case 24 : 
				if (options.trigger_delay) fatal_error("cannot use a code trigger start address with a trigger delay\n");
				if (options.code_trigger_start) fatal_error("code trigger start specificed twice\n");
				if (*optarg == '\0') fatal_error("--trigger-code-start needs an argument\n");
				options.code_trigger_start = optarg;
				break;
			case 25 :
				if (options.code_trigger_start || options.code_trigger_stop) 
					fatal_error("cannot use a trigger delay with a trigger code\n");

				if (options.trigger_delay) fatal_error("trigger start address specificed twice\n");
				if (*optarg == '\0') fatal_error("--trigger-start-delay needs an argument\n");
				long_val = strtoul(optarg,&endptr, 10);
				if (*endptr != '\0') 
					fatal_error("invalid trigger delay : %s\n", optarg);
				if (long_val >= UINT_MAX) 
					fatal_error("trigger delay is too big, must be < %u\n", UINT_MAX);

				options.trigger_delay = (unsigned int)long_val;

				break;
			case 'p':
			case 26 :
				if (options.opt_attach) {
					fatal_error("attach-task or -p specified more than once\n");
				}
				if (*optarg == '\0') fatal_error("you must provide a process id with --attach-task or -p\n");
				options.opt_attach = 1;
				options.attach_pid = atoi(optarg);
				break;
			case 27 :
				if (options.fexec_pattern) fatal_error("cannot have two patterns for --follow-exec\n");
				options.fexec_pattern   = optarg;
				options.opt_follow_exec = 1;
				break;
			case 28 :
				if (options.fexec_pattern) fatal_error("cannot have an exclude pattern and a pattern for --follow-exec\n");
				if (*optarg == '\0') fatal_error("--follow-exec-exlcude needs an argument\n");
				options.fexec_pattern 	     = optarg;
				options.opt_follow_exec      = 1;
				options.opt_follow_exec_excl = 1;
				break;
			case 29 : 
				if (options.trigger_delay) fatal_error("cannot use a code trigger stop address with a trigger delay\n");
				if (options.code_trigger_stop) fatal_error("code trigger stop specificed twice\n");
				if (*optarg == '\0') fatal_error("--trigger-code-stop needs an argument\n");
				options.code_trigger_stop = optarg;
				break;
			case 30 :
				if (options.cpu_list) fatal_error("cannot specify --cpu-list more than once\n");
				if (*optarg == '\0') fatal_error("--cpu-list needs an argument\n");
				options.cpu_list = optarg;
				break;
			case 31: 
				if (options.trigger_delay) fatal_error("cannot use a code trigger start address with a trigger delay\n");
				if (options.data_trigger_start) fatal_error("data trigger start specificed twice\n");
				if (*optarg == '\0') fatal_error("--trigger-data-start needs an argument\n");
				options.data_trigger_start = optarg;
				break;

			case 32 : 
				if (options.trigger_delay) fatal_error("cannot use a code trigger stop address with a trigger delay\n");
				if (options.data_trigger_stop) fatal_error("data trigger stop specificed twice\n");
				if (*optarg == '\0') fatal_error("--trigger-data-stop needs an argument\n");
				options.data_trigger_stop = optarg;
				break;
			default:
				if (pfmon_check_extra_options(c, optarg, &evt)) fatal_error("");
		}
	}

	if (PFM_VERSION_MAJOR(options.pfm_version) !=  PFM_VERSION_MAJOR(PFM_VERSION)) {
		fatal_error("perfmon version mismatch, kernel is %u.%u, pfmon needs %u.x\n", 
			    PFM_VERSION_MAJOR(options.pfm_version),
			    PFM_VERSION_MINOR(options.pfm_version),
			    PFM_VERSION_MAJOR(PFM_VERSION));
	}

	pfmon_verify_cmdline_options(argc, argv);

	/*
	 * propagate monitor parameters to library
	 */
	pfmon_propagate2libary(&evt, options.monitor_count, options.events);

	DPRINT(("%s main process id is %d\n", argv[0], getpid()));

	load_kernel_syms();

	/*
	 * try to load symbol table of the command in per-process mode
	 */
	if (options.opt_syst_wide == 0) {
		load_elf_syms(options.symbol_file);
	}

	setup_plm(&evt);

	pfmon_setup_sampling_rates(long_smpl_args, short_smpl_args, smpl_random_args);

	/*
	 * propagate debug/verbose options to library
	 */
	if (options.opt_debug)   pfmlib_options.pfm_debug = 1;
	if (options.opt_verbose) pfmlib_options.pfm_verbose = 1;
	pfm_set_options(&pfmlib_options);

	/* setup trigger addresses */
	setup_trigger_addresses();

	/* used in sampling output header */
	options.argv    = argv;
	options.command = argv+optind;

	if (pfmon_current->pfmon_post_options && pfmon_current->pfmon_post_options(&evt) == -1) 
		fatal_error("");

	/*
	 * if sampling, then check that the sampling module support the
	 * kind of measurement that is requested.
	 *
	 * This is meant to check the command line options of pfmon
	 * as visible via the options data structure.
	 *
	 * At this point we know that if opt_use_smpl is set then we have
	 * a valid sampling module pointed to be smpl_mod.
	 */
	if (options.opt_use_smpl && options.smpl_mod->validate_options) {
		ret = (*options.smpl_mod->validate_options)();
		if (ret) return ret;
	}

	return do_measurements(&evt, argv+optind);
}
