/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/ 

#include <config.h>

#include <string.h>
#include <stdlib.h>

#include <gtk/gtkeventbox.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkprogressbar.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>

#include <pan/gui.h>
#include <pan/status-item-view.h>
#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static void status_item_view_class_init (StatusItemViewClass *klass);
static void status_item_view_init (StatusItemView *object);
static void status_item_view_destroy (GtkObject *object);
static void status_item_view_shutdown (GtkObject *object);

static gchar* string_shorten_nolock (GdkFont     * font,
                                     gint          width,
                                     const gchar * original,
                                     gint          original_len);

static void status_item_view_unregister (StatusItemView *view);

static void progress_cb      (gpointer, gpointer, gpointer);
static void status_cb        (gpointer, gpointer, gpointer);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

extern GtkTooltips * ttips;                                                                                      

/***********
************  Public
***********/

/***********
************  Private
***********/

static GtkObjectClass *parent_class = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE ROUTINES
************/

static gint
set_percentage_mainthread (gpointer user_data)
{
	ArgSet * argset = (ArgSet*) user_data;
	StatusItemView * siv = (StatusItemView *)(argset_get (argset, 0));
	const int of_100 = GPOINTER_TO_INT (argset_get (argset, 1));

	/* update the percentage */
	pan_lock ();
	gtk_progress_set_percentage (GTK_PROGRESS(siv->progress), (gfloat)(of_100/100.0));
	pan_unlock ();

	/* cleanup */
	argset_free (argset);
	return 0;
}

static void
progress_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	const gint of_100 = GPOINTER_TO_INT(call_arg);
	if (0<=of_100 && of_100<=100)
		gui_queue_add (set_percentage_mainthread, argset_new2 (user_data, GINT_TO_POINTER(of_100)));
}

/************
*************  PUBLIC ROUTINES
************/

GtkType
status_item_view_get_type (void)
{
	static GtkType status_item_view_type = 0;

	if (!status_item_view_type)
	{
		static const GtkTypeInfo status_item_view_info =
		{
			"StatusItemView",
			sizeof(StatusItemView),
			sizeof(StatusItemViewClass),
			(GtkClassInitFunc) status_item_view_class_init,
			(GtkObjectInitFunc) status_item_view_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		status_item_view_type = gtk_type_unique (
			GTK_TYPE_VBOX,
			&status_item_view_info);
	}

	return status_item_view_type;
}

/*****
******
*****/

static void
status_item_view_class_init (StatusItemViewClass *klass)
{
	GtkObjectClass * object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class->destroy = status_item_view_destroy;
	object_class->shutdown = status_item_view_shutdown;
}

static void
status_item_view_init (StatusItemView *view)
{
	view->progress = NULL;
	view->item = NULL;
	view->last_status = g_string_new (NULL);
}


/*****
******
*****/

static void
status_item_view_shutdown (GtkObject* object)
{
	StatusItemView *view = STATUS_ITEM_VIEW(object);
	status_item_view_unregister (view);

	(*GTK_OBJECT_CLASS (parent_class)->shutdown) (object);
}

static void
status_item_view_destroy (GtkObject* object)
{
	StatusItemView *item = STATUS_ITEM_VIEW(object);
	debug1 (DEBUG_QUEUE, "%p status_item_view_destroy", item);

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_STATUS_ITEM_VIEW (object));
	g_return_if_fail (GTK_OBJECT_CONSTRUCTED (object));

	g_string_truncate (item->last_status, 0);
	g_string_free (item->last_status, TRUE);

	(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkWidget*
status_item_view_new (void)
{
	StatusItemView * view;

        /* gtk instantiation */
	view = STATUS_ITEM_VIEW(gtk_type_new(TYPE_STATUS_ITEM_VIEW));
	debug1 (DEBUG_QUEUE, "%p status_item_view_new", view);

	/* parent setup */
	GTK_CONTAINER(view)->border_width = 0;
	GTK_BOX(view)->spacing = 0;
	GTK_BOX(view)->homogeneous = FALSE;

	/* status_item_view setup: progress bar */
	view->progress = gtk_progress_bar_new ();
	gtk_progress_set_format_string (GTK_PROGRESS(view->progress), "");
	gtk_progress_set_show_text (GTK_PROGRESS(view->progress), TRUE);

	/* give the progressbar an eventbox backer for tooltips */
	view->eventbox = gtk_event_box_new();
	gtk_container_add (GTK_CONTAINER(view->eventbox), view->progress);
	gtk_box_pack_start (GTK_BOX(view), view->eventbox, TRUE, TRUE, 0);

	/* return what we've got */
	gtk_widget_show_all (GTK_WIDGET(view));
	return GTK_WIDGET(view);
}

/*****
******
*****/

static void
status_item_view_unregister (StatusItemView *view)
{
	StatusItem *item = NULL;

	item = STATUS_ITEM(view->item);
	if (!item)
		return;

	/* stop listening to the item... */
	pan_callback_remove (item->progress, progress_cb, view);
	pan_callback_remove (item->status, status_cb, view);
	pan_callback_remove (item->error, status_cb, view);

	/* unref the item... */
	pan_object_unref(PAN_OBJECT(item));
	view->item = NULL;
}

void
status_item_view_set_item_nolock (StatusItemView* view, StatusItem* item)
{
	debug_enter ("status_item_view_set_item_nolock");

	if (item != view->item)
	{
		GtkProgress* prog = GTK_PROGRESS(view->progress);

		status_item_view_unregister (view);

		/* update last_status; update progressbar w/o a resize */
		gui_queue_add (set_percentage_mainthread, argset_new2 (view, NULL));
		g_string_assign (view->last_status, "");
		replace_gstr (&prog->format, g_strdup(""));
		GTK_PROGRESS_CLASS (GTK_OBJECT(prog)->klass)->update (prog);
		gtk_tooltips_set_tip (ttips, GTK_WIDGET(view->eventbox), NULL, NULL);

		if (item != NULL)
		{
			gchar * description = status_item_describe (item);
			pan_object_ref (PAN_OBJECT(item));
			pan_callback_add (item->progress, progress_cb, view);
			pan_callback_add (item->status, status_cb, view);
			pan_callback_add (item->error, status_cb, view);
			gtk_tooltips_set_tip (ttips, GTK_WIDGET(view->eventbox), description, NULL);
			g_free (description);
		}

		view->item = item;

		/* show the last status text */
		if (item !=NULL)
		{
			gchar * pch = item->status_text;
			if (is_nonempty_string(pch))
				status_cb (NULL, pch, view);
		}
	}

	debug_exit ("status_item_view_set_item_nolock");
}

/*****
******
*****/


static gint
status_cb_mainthread (gpointer data)
{
	StatusItemView * view;

	pan_lock ();
	view = STATUS_ITEM_VIEW(data);
	if (view!=NULL && view->item!=NULL)
	{
		GtkProgress * prog = GTK_PROGRESS(view->progress);
		gchar * s = string_shorten_nolock (gtk_widget_get_style (view->progress)->font,
		                                   view->progress->allocation.width,
		                                   view->last_status->str,
		                                   view->last_status->len);
		replace_gstr (&prog->format, s);
		/* update progressbar w/o a resize */
		GTK_PROGRESS_CLASS (GTK_OBJECT(prog)->klass)->update (prog);
	}
	pan_unlock ();

	return 0;
}

static void
status_cb (gpointer call_object, gpointer call_arg, gpointer user_data)
{
	StatusItemView * siv = STATUS_ITEM_VIEW (user_data);
	g_string_assign (siv->last_status, (const gchar*) call_arg);
	gui_queue_add (status_cb_mainthread, siv);
}

/**
***
**/

static void
string_remove_char (gchar * str, gint index)
{
	gchar *p = str+index;
	for (;;) {
		++p;
		p[-1] = *p;
		if (!*p)
			break;
	}
}

static gchar*
string_shorten_nolock (GdkFont           * font,
                       gint                max_width,
                       const gchar       * original,
                       gint                original_len)
{
	gchar *clone = NULL;
	size_t len;
	gint i, j;
	gboolean i_turn = TRUE;
	const int DOTLEN = 3;
	gint target_width = max_width - 4 - 30; /* 30 is just a magic safety number */

	/* if original is NULL, so is shortened */
	if (!original)
		return NULL;

	/* if original fits, use it */
	if (gdk_text_width (font, original, original_len) < target_width)
		return g_strdup (original);

	/* add in the middle "..." */
	len = original_len;
	clone = g_malloc (original_len + DOTLEN + 1);
	i = original_len/2;
	memcpy (clone, original, i);
	memcpy (clone+i, "...", DOTLEN);
	j = i+3;
	memcpy (clone+j, original+i, original_len+1-i);
	original = NULL;

	/* start trimming out text until it fits */	
	len = original_len + DOTLEN;
	while (gdk_text_width(font, clone, len) >= target_width)
	{
		int *pi = i_turn ? &i : &j;
		*pi -= 1;
		if (*pi<0 || *pi>=len)
			break;
		string_remove_char (clone, *pi);
		i_turn = !i_turn;
		--len;
	}

	return clone;
}
