#include <cassert>

#include <qmessagebox.h>
#include <qaction.h>
#include <QMainWindow>
#include <QMutex>

#include <unistd.h>		// for getuid and geteuid
#include <sys/types.h>	//

#include <wibble/operators.h>
#include <ept/debtags/debtags.h>
#include <ept/debtags/tag.h>

#include "debtagsplugincontainer.h"

// NUtil
#include "helpers.h"

// NApplication
#include "applicationfactory.h"
#include "runcommand.h"

// NPlugin
#include <iprovider.h>
#include <plugincontainer.h>
#include <iprogressobserver.h>
#include "debtagsplugin.h"
#include "relatedplugin.h"
#include "debtagspluginfactory.h"

// NTagModel
#include "vocabularymodel.h"


// Debug: Modeltest
#include <modeltest.h>


#include "debtagssettingswidget.h"

extern "C" 
{ 
	NPlugin::PluginContainer* new_debtagsplugin() 
	{
		return new NPlugin::DebtagsPluginContainer;
	} 

	NPlugin::PluginInformation get_pluginInformation()
	{
		return NPlugin::PluginInformation("debtagsplugin", "2.4", "Benjamin Mesing");
	} 
}

/** Initialize the plugin. */
__attribute__ ((constructor)) void init() 
{
}
      
      
// __attribute__ ((destructor)) void fini() 
// {
//   /* code here is executed just before dlclose() unloads the module */
// } 


namespace NPlugin
{

DebtagsPluginContainer::DebtagsPluginContainer()
{
	// this assumes, that only one instance of the DebtagsPluginContainer is created
	// (no two versions of libapt-front may be opened at any time)
	// This is ok for our purpose - though usually there should be a singleton ensuring
	// this constraint...
	DebtagsPluginFactory::getInstance()->setContainer(this);
	_pCommand = 0;
	_pRelatedPlugin = 0;
	_pDebtagsPlugin = 0;
	_pSettingsWidget = 0;
	addPlugin("DebtagsPlugin");
	addPlugin("RelatedPlugin");
	
	_debtagsEnabled=false;
	
}
 
DebtagsPluginContainer::~DebtagsPluginContainer()
{
	delete _pCommand;
//	qDebug("Closing global aptFront Cache");
//	cache::Global::get().close();
}

/////////////////////////////////////////////////////
// PluginContainer Interface
/////////////////////////////////////////////////////


bool DebtagsPluginContainer::init(IProvider* pProvider)
{
	BasePluginContainer::init(pProvider, DebtagsPluginFactory::getInstance());
	updateDebtags();
	_pVocabularyModel = new NTagModel::VocabularyModel(this);

	new ModelTest(_pVocabularyModel, this);
	
	
	if (debtagsEnabled())
	{
		// use dynamic cast here because of the virtual base class 
		// (static_cast is not allowed there)
		_pRelatedPlugin = dynamic_cast<RelatedPlugin*>(requestPlugin("RelatedPlugin"));
		_pDebtagsPlugin = dynamic_cast<DebtagsPlugin*>(requestPlugin("DebtagsPlugin"));
	}
	
	return debtagsEnabled();
}

vector< pair<QString, QAction*> > DebtagsPluginContainer::actions()
{
	vector< pair<QString, QAction*> > result;
	return result;
}

QWidget* DebtagsPluginContainer::getSettingsWidget(QWidget* pParent)
{
	_pSettingsWidget = new DebtagsSettingsWidget(_pVocabularyModel, pParent, "DebtagsSettingsWidget");
 	return _pSettingsWidget;
}

void DebtagsPluginContainer::applySettings()
{
	// nothing to be done here, settings are applied automatically
}



/////////////////////////////////////////////////////
// BasePluginContainer Interface
/////////////////////////////////////////////////////


QDomElement DebtagsPluginContainer::loadContainerSettings(const QDomElement source)
{
	if (source.tagName() != "ContainerSettings")
		return source;
	float settingsVersion;
	NXml::getAttribute(source, settingsVersion, "settingsVersion", 0.0f);
	
	QDomNodeList hiddenFacets = source.elementsByTagName("HiddenFacet");
	for (int i=0; i < hiddenFacets.count(); ++i)
	{
		string hiddenFacet = toString( hiddenFacets.item(i).toElement().text() );
		_pVocabularyModel->setFacetHidden(true, hiddenFacet);
	}
	if (debtagsEnabled())
		updateVocabulary();
	return NXml::getNextElement(source);
}

void DebtagsPluginContainer::saveContainerSettings(NXml::XmlData& outData, QDomElement parent) const
{
	qDebug("saveContainerSettings called");
	typedef ept::debtags::Facet Facet;
	QDomElement containerElement = outData.addElement(parent, "ContainerSettings");
	outData.addAttribute(containerElement, 0.1f, "settingsVersion");
	set<Facet> hiddenFacets = _pVocabularyModel->hiddenFacets();
	for (set<Facet>::const_iterator it = hiddenFacets.begin(); it != hiddenFacets.end(); ++it)
	{
		QDomElement hiddenFacetElement = outData.addElement(containerElement, "HiddenFacet");
		outData.addText(hiddenFacetElement, it->name());
	}
}





/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPluginContainer::updateVocabulary(bool informPlugins)
{
	if (informPlugins)
	{
		if (_pDebtagsPlugin)
			_pDebtagsPlugin->debtagsDataChanged();
		if (_pRelatedPlugin)
			_pRelatedPlugin->debtagsDataChanged();
	}
}

void DebtagsPluginContainer::updateDebtags()
{
	NUtil::IProgressObserver* pProgressObserver = provider()->progressObserver();
	if (pProgressObserver)
		pProgressObserver->setText("Loading Debtags Plugin");
/*	try 
	{
		// this is set to 0 if no listener is available
		cache::Global::get().close();
		qDebug("Opening aptfront cache for debtags");
		cache::Global::get().open(cache::Cache::OpenDefault
				| cache::Cache::OpenReadOnly
				| cache::Cache::OpenTags
				| cache::Cache::OpenDebtags);
	}
	catch (Tagcoll::SystemException e)*/
	
	if (!collection()->hasData())
	{
		setDebtagsEnabled(false);	// disable the debtags system
		provider()->reportError(
			tr("Tag Database Not Available" ),
			tr(
				"<p>The tag database is not available and the debtags plugin was disabled!</p>"
				"<p>"
				"You must execute <tt><b>debtags update</b></tt> as root on the commandline to download "
				"the database. If debtags is not on your system you can install it via "
				"<tt><b>apt-get install debtags</b></tt><br>"
				"Afterwards you can enable the debtags plugin via the plugin menu -> Control Plugins."
				"</p>")
/*				"<p>"
				"The original error message reported by debtags was:<br>"
				"<tt>"
			) +
			toQString(e.desc()) +
			"</tt>"
			"</p>"*/
		);
		return;
	}
/*	catch (exception::Error e)
	{
		setDebtagsEnabled(false);	// disable the debtags system
		provider()->reportError(
			tr("Unknown Error when reading the Debtags database" ),
			tr("An unknown error occured when reading the debtags database.<br>"
				"The error reported by debtags is:.<br><tt>") +
			toQString(e.what()) +
			"</tt>"
		);
		return;
	}*/

	setDebtagsEnabled(true);
	if (debtagsEnabled())
	{
		// load the new vocabulary
		updateVocabulary(false);
	}
	// inform the plugins about the new collections
	if (_pDebtagsPlugin)
		_pDebtagsPlugin->debtagsDataChanged();
	if (_pRelatedPlugin)
		_pRelatedPlugin->debtagsDataChanged();
	if (pProgressObserver)
		pProgressObserver->setProgress(100);
}

void DebtagsPluginContainer::setDebtagsEnabled(bool enabled)
{
	_debtagsEnabled = enabled;
}

const ept::debtags::Debtags* DebtagsPluginContainer::collection() const	
{
	return &(provider()->debtags());
}

const std::set<DebtagsPluginContainer::Facet> DebtagsPluginContainer::facets() const
{
	const Vocabulary& tags = provider()->debtags().vocabulary();
	return tags.facets(); 
}




}	// namespace NPlugin

#undef emit
/*
#include <ept/cache/debtags/tagmap.tcc>
#include <ept/cache/debtags/vocabulary.tcc>
#include <ept/cache/apt/index.tcc>
*/
